/*  Screem:  screem-tag_file.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include "config.h"

#include "screem-tagfile.h"

#include <string.h>
#include <libgnome/gnome-macros.h>
#include <libgnome/gnome-i18n.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreemodelsort.h>
#include <glib/gmarkup.h>
#include <glib/gqueue.h>
#include <gdk/gdk.h>

#include "fileops.h"
#include "screem-file-browser.h"

static void screem_tag_file_class_init( ScreemTagFileClass *klass );
static void screem_tag_file_instance_init( ScreemTagFile *tag_file );
static void screem_tag_file_finalize( GObject *object );
static void screem_tag_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_tag_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data );
static void screem_tag_file_icon_change( ScreemFileBrowser *browser, 
					const gchar *uri,
					const gchar *mime_type,
					GtkTreeIter *it,
					gpointer data );
static void screem_tag_file_free_node_info( NodeInfo *info );
static void screem_tag_file_clear( ScreemTagFile *tagfile );
static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data );

static void screem_tag_file_start_element( GMarkupParseContext *context,
					const gchar *element,
					const gchar **attributes,
					const gchar **values,
					gpointer data,
					GError **error );
static void screem_tag_file_end_element( GMarkupParseContext *context,
					const gchar *element,
					gpointer data,
					GError **error );
static void screem_tag_file_text( GMarkupParseContext *context,
				const gchar *text,
				gsize len,
				gpointer data,
				GError **error );
static void screem_tag_file_error( GMarkupParseContext *context,
				GError *error,
				gpointer data );
static void screem_tag_file_parse_node( ScreemTagFilePrivate *priv,
					const gchar *element,
					const gchar **attributes,
					const gchar **values );
static void screem_tag_file_parse_insert( ScreemTagFilePrivate *priv,
					  const gchar **attributes,
					  const gchar **values );

static void screem_tag_file_parse_param( ScreemTagFilePrivate *priv,
					 const gchar **attributes,
					 const gchar **values );

enum {
	PROP_0,
	PROP_ACTIVE
};

struct ScreemTagFilePrivate {
	ScreemTagFile *tfile;
	
	gchar *filename;

	gchar *name;
	gchar *description;

	gchar *type;
	
	ScreemFileBrowser *browser;
	/* the unsorted browser model */
	GtkTreeModel *model;
	
	gboolean active;

	gboolean loaded;

	/* current auto complete prefix */
	const gchar *prefix;

	ScreemTagFileFormat format;

	GQueue *stack;
	/* current parent path */
	GtkTreePath *path;
};

static GMarkupParser screem_parser_cbs = {
	screem_tag_file_start_element,
	screem_tag_file_end_element,
	screem_tag_file_text,
	NULL,
	screem_tag_file_error
};

typedef enum {
	PARSER_STATE_NONE = 0,
	PARSER_STATE_REF,
	PARSER_STATE_GROUP,
	PARSER_STATE_ENTRY,
	PARSER_STATE_FUNCTION,
	PARSER_STATE_INSERT,
	PARSER_STATE_DESCRIPTION,
	PARSER_STATE_PARAM,
	PARSER_STATE_PROPERTY
} ScreemTagTreeParserState;

static const gchar *tag_file_tags[] = {
	"",
	"ref",
	"group",
	"entry",
	"function",
	"insert",
	"description",
	"param",
	"property",
};

GNOME_CLASS_BOILERPLATE( ScreemTagFile, screem_tag_file,
			  GObject,
			  G_TYPE_OBJECT )

static void screem_tag_file_class_init( ScreemTagFileClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *spec;

	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_tag_file_finalize;
	obj_class->get_property = screem_tag_file_get_prop;
	obj_class->set_property = screem_tag_file_set_prop;

	spec = g_param_spec_boolean( "active", "active",
				_( "Is the Tag File active or no" ),
				TRUE, 
				G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( obj_class, PROP_ACTIVE,
					spec );
}

static void screem_tag_file_instance_init( ScreemTagFile *tag_file )
{
	ScreemTagFilePrivate *priv;
	
	priv = tag_file->priv = g_new0( ScreemTagFilePrivate, 1 );
	priv->tfile = tag_file;

	priv->browser = screem_file_browser_new();
	priv->model = screem_file_browser_get_model( priv->browser );
	priv->model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( priv->model ) );

	g_object_set_data( G_OBJECT( priv->model ), "tagfile", tag_file );
	
	screem_file_browser_set_mode( priv->browser,
					FILE_BROWSE_RECURSE );
	screem_file_browser_set_sort_func( priv->browser,
					   screem_tag_file_compare_func,
					   NULL );
	
	g_signal_connect( G_OBJECT( priv->browser ),
			  "icon_change", 
			  G_CALLBACK( screem_tag_file_icon_change ),
			  NULL );

}

static void screem_tag_file_finalize( GObject *object )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_TAGFILE( object ) );

	tag_file = SCREEM_TAGFILE( object );

	priv = tag_file->priv;

	screem_tag_file_clear( tag_file );

	g_free( priv->filename );
	
	g_object_unref( priv->browser );
	
	g_free( priv );
	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (object) );
}

static void screem_tag_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			priv->active = g_value_get_boolean( value );
			if( priv->active && ! priv->loaded ) {
				screem_tag_file_load( tag_file,
						SCREEM_TAG_TREE_FORMAT);
			}
			break;
		default:
			break;
	}
}

static void screem_tag_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			g_value_set_boolean( value, priv->active );
			break;
		default:
			break;
	}
}

/* static stuff */
static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data )
{
        NodeInfo *ainfo;
        NodeInfo *binfo;
	int ret = 0;

	gtk_tree_model_get( model, a, 
			FILE_BROWSER_DATA_COL, &ainfo, -1 );
	gtk_tree_model_get( model, b, 
			FILE_BROWSER_DATA_COL, &binfo, -1 );

	if( ainfo && ! binfo ) {
		ret = 1;
	} else if( binfo && !ainfo ) {
		ret = -1;
	} else if( ! ainfo && ! binfo ) {
		ret = 0;
	} else {
		ret = strcmp( ainfo->name, binfo->name );
	}
	
	return ret;
}

static void screem_tag_file_icon_change( ScreemFileBrowser *browser, 
					const gchar *uri,
					const gchar *mime_type,
					GtkTreeIter *it,
					gpointer data )
{
	GtkTreeModel *model;
	NodeInfo *info;
	GdkPixbuf *pixbuf;
	
	model = screem_file_browser_get_model( browser );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );

	gtk_tree_model_get( model, it, 
			FILE_BROWSER_DATA_COL, &info,
			-1 );
	if( info ) {
		pixbuf = NULL;
		if( info->file ) {
			pixbuf = screem_file_browser_get_icon( browser,
								info->file,
								16, 16, FALSE, NULL );
		} else if( gtk_tree_model_iter_has_child( model, it ) ) {
			pixbuf = screem_file_browser_get_icon( browser,
								"/",
								16, 16, FALSE, NULL );
		}
		gtk_tree_store_set( GTK_TREE_STORE( model ), 
				it, FILE_BROWSER_ICON_COL,
				pixbuf, -1 );
		if( pixbuf ) {
			g_object_unref( pixbuf );
		}
	}
}

static void screem_tag_file_free_node_info( NodeInfo *info )
{
	GSList *list;
	NodeAttribute *attr;
	
	g_free( info->name );
	g_free( info->open );
	g_free( info->close );
	g_free( info->file );
	g_free( info->text );
	g_free( info->description );

	for( list = info->attributes; list; list = list->next ) {
		attr = (NodeAttribute*)list->data;

		g_free( attr->name );
		g_free( attr->type );
		g_free( attr->defvalue );
		g_free( attr->description );
		g_free( attr );
	}
	if( info->attributes ) {
		g_slist_free( info->attributes );
	}
	
	g_free( info );
}

static void screem_tag_file_clear( ScreemTagFile *tagfile )
{
	ScreemTagFilePrivate *priv;
	
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	priv = tagfile->priv;
	
	gtk_tree_model_foreach( priv->model,
			screem_tag_file_clear_foreach,
			NULL );

	gtk_tree_store_clear( GTK_TREE_STORE( priv->model ) );
}

static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	NodeInfo *info;
	
	gtk_tree_model_get( model, iter, 
			FILE_BROWSER_DATA_COL, &info, 
			-1 );
	if( info ) {
		screem_tag_file_free_node_info( info );
	}

	return FALSE;
}

static void screem_tag_file_start_element( GMarkupParseContext *context,
					const gchar *element,
					const gchar **attributes,
					const gchar **values,
					gpointer data,
					GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	switch( state ) {

		/* no parser state, valid elements == <ref> */
		case PARSER_STATE_NONE:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_REF ],
					element ) ) {
				/* got ref state */


				state = PARSER_STATE_REF;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_node( priv,
						element,
						attributes,
						values );
			} else {
				/* error */
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_UNKNOWN_ELEMENT,
					element );
			}
			
			break;

		/* <ref> allows <group> */
		case PARSER_STATE_REF:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_GROUP ],
					element ) ) {
				/* got group state */

				state = PARSER_STATE_GROUP;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_node( priv,
						element,
						attributes,
						values );
			} 
			/* not in our DTD, but simplifies the
			 * current doctype node */
			else if( ! strcmp( tag_file_tags[ PARSER_STATE_ENTRY ],
						element ) ) {
				state = PARSER_STATE_ENTRY;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_node( priv, 
						element,
						attributes,
						values );
			} else {
				/* error */
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					element );
			}

			break;
	
		/* <group> allows <group> <entry> or <function> */
		case PARSER_STATE_GROUP:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_GROUP ],
					element ) ) {
				/* got group state */

				state = PARSER_STATE_GROUP;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_ENTRY ],
						element ) ) {
				/* got entry state */

				state = PARSER_STATE_ENTRY;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_FUNCTION ],
						element ) ) {
				/* got function state */

				state = PARSER_STATE_FUNCTION;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else {
				/* error */
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					element );
				break;
			}
			screem_tag_file_parse_node( priv, element,
						attributes,
						values );
			break;
	
		/* <entry> allows <insert> <description> <property> */
		case PARSER_STATE_ENTRY:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_INSERT ],
					element ) ) {
				/* got insert state */
				screem_tag_file_parse_insert( priv,
						attributes,
						values );

				state = PARSER_STATE_INSERT;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_DESCRIPTION ],
						element ) ) {
				/* got description state */

				state = PARSER_STATE_DESCRIPTION;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_PROPERTY ],
						element ) ) {
				/* got property state */

				state = PARSER_STATE_PROPERTY;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_param( priv,
						attributes,
						values );
			} else {
				/* error */
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					element );
			}
			break;

		/* <function> allows <insert> <description> <param> */
		case PARSER_STATE_FUNCTION:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_INSERT ],
					element ) ) {
				/* got insert state */
				screem_tag_file_parse_insert( priv,
						attributes,
						values );

				state = PARSER_STATE_INSERT;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_DESCRIPTION ],
						element ) ) {
				/* got description state */

				state = PARSER_STATE_DESCRIPTION;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_PARAM ],
						element ) ) {
				/* got param state */

				state = PARSER_STATE_PARAM;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_param( priv,
						attributes,
						values );
			} else {
				/* error, does comply with out
				 * DTD, however that is because
				 * we don't support <return> <dialog>
				 * <info> in the Bluefish reference
				 * files, ignore the error
				 * 
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					element );
				*/

			}
			break;

		/* <insert> allows only #PCDATA 
		 * <description> allows only #PCDATA
		 * <param> allows only #PCDATA
		 * <property> allows only #PCDATA */
		case PARSER_STATE_INSERT:
		case PARSER_STATE_DESCRIPTION:
		case PARSER_STATE_PARAM:
		case PARSER_STATE_PROPERTY:
			/* error */
			g_set_error( error, 
				G_MARKUP_ERROR, 
				G_MARKUP_ERROR_PARSE,
				element );
			break;
	}
}

static void screem_tag_file_end_element( GMarkupParseContext *context,
					const gchar *element,
					gpointer data,
					GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	if( ! strcmp( tag_file_tags[ state ], element ) ) {

		switch( state ) {
			case PARSER_STATE_REF:
			case PARSER_STATE_GROUP:
			case PARSER_STATE_ENTRY:
			case PARSER_STATE_FUNCTION:
				gtk_tree_path_up( priv->path );
				break;
		}
		g_queue_pop_head( stack );
	} else {
		/* error, but as we accept Bluefish function
		 * files it is possible to receive an end
		 * tag for one of the tags we don't support,
		 * ignore the error
		g_set_error( error, 
			G_MARKUP_ERROR, 
			G_MARKUP_ERROR_PARSE,
			element );
			*/
	}
}

static void screem_tag_file_text( GMarkupParseContext *context,
				const gchar *text,
				gsize len,
				gpointer data,
				GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;

	GtkTreeIter it;

	NodeInfo *info;
	NodeAttribute *attr;

	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	info = NULL;
	if( priv->path ) {
		/* get parent iter */
		if( ! gtk_tree_model_get_iter( priv->model, 
					&it, priv->path ) ) {
			/* error */
			return;
		}
		gtk_tree_model_get( priv->model, &it,
				FILE_BROWSER_DATA_COL, &info, -1 );
	} else {
		/* error */
		return;
	}

	switch( state ) {
		case PARSER_STATE_INSERT:
			info->text = g_strdup( text );
			info->text = g_strstrip( info->text );
			break;
		case PARSER_STATE_DESCRIPTION:
			info->description = g_strdup( text );
			info->description = g_strstrip( info->description );
			break;
		case PARSER_STATE_PARAM:
		case PARSER_STATE_PROPERTY:
			if( info->attributes ) {
				attr = (NodeAttribute*)info->attributes->data;
				if( attr ) {
					attr->description = g_strdup( text );
					attr->description = g_strstrip( attr->description );

				}
			}
			break;
		default:
			/* ignore */
			break;
	}
}

static void screem_tag_file_error( GMarkupParseContext *context,
				GError *error,
				gpointer data )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );
	
}

static void screem_tag_file_parse_node( ScreemTagFilePrivate *priv,
					const gchar *element,
					const gchar **attributes,
					const gchar **values )
{
	ScreemTagTreeParserState state;
	GQueue *stack;

	GtkTreeIter it;
	GtkTreeIter *parent;

	GdkPixbuf *pixbuf;
	const gchar *name;
	const gchar *description;
	const gchar *type;
	gint i;
	NodeInfo *info;
	
	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	parent = NULL;
	if( priv->path ) {
		if( ! gtk_tree_model_get_iter( priv->model, 
					&it, priv->path ) ) {
			/* error */
			return;
		}
		parent = &it;
	} else if( state != PARSER_STATE_REF ) {
		/* error, can only not have a path for <ref> */
		return;
	}

	gtk_tree_store_append( GTK_TREE_STORE( priv->model ),
					&it, parent );
	pixbuf = NULL;
	name = description = type = NULL;
	for( i = 0; attributes[ i ]; ++ i ) {
		if( ! strcmp( "name", attributes[ i ] ) ) {
			name = values[ i ];
		} else if( ! strcmp( "description", attributes[ i ] ) ) {
			description = values[ i ];
		} else if( ! strcmp( "type", attributes[ i ] ) ) {
			type = values[ i ];
		}
	}
	if( ! name ) {
		name = _( "Unknown Resource" );
	}
	
	switch( state ) {
		case PARSER_STATE_REF:
			pixbuf = screem_file_browser_get_icon( priv->browser,
					"/", 16, 16, FALSE, NULL );
			g_free( priv->name );
			g_free( priv->description );
			g_free( priv->type );
			priv->name = g_strdup( name );
			priv->description = NULL;
			if( description ) {
				priv->description = g_strdup( description );
			}
			if( type ) {
				priv->type = g_strdup( type );
			}
			break;
		case PARSER_STATE_GROUP:
			pixbuf = screem_file_browser_get_icon( priv->browser,
					"/", 16, 16, FALSE, NULL );
			break;
		case PARSER_STATE_ENTRY:
		case PARSER_STATE_FUNCTION:
			description = NULL;
			break;
		default:
			/* error */
			return;
			break;
	}
	priv->path = gtk_tree_model_get_path( priv->model, &it );

	info = g_new0( NodeInfo, 1 );
	info->tfile = priv->tfile;
	info->name = g_strdup( name );
	if( description ) {
		info->description = g_strdup( description );
	}
	
	gtk_tree_store_set( GTK_TREE_STORE( priv->model ),
			&it,
			FILE_BROWSER_NAME_COL, name,
			FILE_BROWSER_ICON_COL, pixbuf,
			FILE_BROWSER_DATA_COL, info,
			-1 );
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
}

static void screem_tag_file_parse_insert( ScreemTagFilePrivate *priv,
					const gchar **attributes,
					const gchar **values )
{
	ScreemTagTreeParserState state;
	GQueue *stack;

	GtkTreeIter it;

	gint i;
	NodeInfo *info;

	const gchar *attr;
	const gchar *val;
	GdkPixbuf *pixbuf;
	gchar *pathname;
	
	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	info = NULL;
	if( priv->path ) {
		/* get parent iter */
		if( ! gtk_tree_model_get_iter( priv->model, 
					&it, priv->path ) ) {
			/* error */
			return;
		}
		gtk_tree_model_get( priv->model, &it,
				FILE_BROWSER_DATA_COL, &info, -1 );
	} else {
		/* error */
		return;
	}

	info->isfunc = ( state == PARSER_STATE_FUNCTION );
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];
		
		if( ! strcmp( "href", attr ) ) {
			/* external reference */
			pathname = relative_to_full( val,
						RESOURCE_PATH );
			info->file = pathname;
			pixbuf = screem_file_browser_get_icon( priv->browser,
					pathname, 16, 16, FALSE, NULL );
			gtk_tree_store_set( GTK_TREE_STORE( priv->model ),
					&it,
					FILE_BROWSER_ICON_COL,
					pixbuf, -1 );
		} else if( ! strcmp( "open", attr ) ) {
			/* open text */
			info->open = g_strdup( val );	
		} else if( ! strcmp( "close", attr ) ) {
			/* close text */
			info->close = g_strdup( val );
		}
	}

}


static void screem_tag_file_parse_param( ScreemTagFilePrivate *priv,
					 const gchar **attributes,
					 const gchar **values )
{
	ScreemTagTreeParserState state;
	GQueue *stack;

	GtkTreeIter it;

	gint i;
	NodeInfo *info;

	NodeAttribute *attribute;
	const gchar *attr;
	const gchar *val;
	
	stack = priv->stack;
	
	state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );

	info = NULL;
	if( priv->path ) {
		/* get parent iter */
		if( ! gtk_tree_model_get_iter( priv->model, 
					&it, priv->path ) ) {
			/* error */
			return;
		}
		gtk_tree_model_get( priv->model, &it,
				FILE_BROWSER_DATA_COL, &info, -1 );
	} else {
		/* error */
		return;
	}

	attribute = g_new0( NodeAttribute, 1 );
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		if( ! strcmp( "name", attr ) ) {
			attribute->name = g_strdup( val );
			attribute->name = g_strstrip( attribute->name );
		} else if( ! strcmp( "required", attr ) ) {
			attribute->required = ( *val == '1' );
		} else if( ! strcmp( "vallist", attr ) ) {
			attribute->vallist = ( *val == '1' );
		} else if( ! strcmp( "type", attr ) ) {
			attribute->type = g_strdup( val );
		} else if( ! strcmp( "default", attr ) ) {
			attribute->defvalue = g_strdup( val );
			attribute->defvalue = g_strstrip( attribute->defvalue );
		}
	}
	info->attributes = g_slist_prepend( info->attributes, 
					attribute );

}

static gboolean screem_tag_file_autocomplete_fill( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	GSList **ret;
	ScreemTagFile *tfile;
	NodeInfo *info;
	const gchar *str;
	const gchar *prefix;

	tfile = SCREEM_TAGFILE( g_object_get_data( G_OBJECT( model ),
						"tagfile" ) );
	prefix = tfile->priv->prefix;
	ret = (GSList**)data;

	gtk_tree_model_get( model, iter,
			FILE_BROWSER_DATA_COL, &info,
			-1 );

	str = info->open;
	if( ! str ) {
		str = info->close;
	}
	if( ! str ) {
		str = info->text;
	}
	if( str && 
	   ! strncmp( str, prefix, strlen( prefix ) ) ) {
		*ret = g_slist_prepend( *ret, (gchar*)str );
	}

	return FALSE;
}

/* public stuff */

ScreemTagFile *screem_tag_file_new( void )
{
	ScreemTagFile *tag_file;

	tag_file = g_object_new( SCREEM_TYPE_TAGFILE, NULL );

	return tag_file;
}

gboolean screem_tag_file_load( ScreemTagFile *tagfile,
		ScreemTagFileFormat format )
{
	GString *file;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), FALSE );

	ret = FALSE;
	file = load_file( tagfile->priv->filename );
	if( file ) {
		tagfile->priv->format = format;
		ret = screem_tag_file_set_from_string( tagfile, 
						file->str,
						file->len );
		g_string_free( file, TRUE );
	}
	
	return ret;	
}

gboolean screem_tag_file_load_with_uri( ScreemTagFile *tagfile,
				const gchar *uri,
				ScreemTagFileFormat format )
{
	screem_tag_file_set_uri( tagfile, uri );

	return screem_tag_file_load( tagfile, format );
}

gboolean screem_tag_file_set_from_string( ScreemTagFile *tagfile,
					const gchar *str,
					guint length )
{
	ScreemTagFilePrivate *priv;
	GMarkupParseContext *ctx;
	gboolean ret;
	GMarkupParser *parser_cbs;

	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), FALSE );
	g_return_val_if_fail( str != NULL, FALSE );
	
	priv = tagfile->priv;
	
	if( length == 0 ) {
		length = strlen( str );
	}
	
	ret = FALSE;

	screem_tag_file_clear( tagfile );
	
	switch( priv->format ) {
		default:
			parser_cbs = &screem_parser_cbs;
			priv->stack = g_queue_new();
			priv->path = NULL;
			g_queue_push_head( priv->stack,
					GINT_TO_POINTER( PARSER_STATE_NONE ) );
			break;
	}
	
	ctx = NULL;
	if( parser_cbs ) {
		ctx = g_markup_parse_context_new( parser_cbs, 0, 
						  tagfile, NULL );
	}
	if( ctx ) {
		ret = g_markup_parse_context_parse( ctx, str, 
						length, NULL );
		if( ret ) {
			ret = g_markup_parse_context_end_parse( ctx,
					NULL );
		}
		g_markup_parse_context_free( ctx );

		priv->loaded = ret;
	}
	switch( priv->format ) {
		default:
			g_queue_free( priv->stack );
			gtk_tree_path_free( priv->path );
			break;
	}

	
	return ret;	
}

GtkTreeModel *screem_tag_file_get_model( ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->model;
}

const gchar *screem_tag_file_get_name( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->name;
}

const gchar *screem_tag_file_get_desc( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->description;
}

const gchar *screem_tag_file_get_uri( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->filename;
}

const gchar *screem_tag_file_for_type( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->type;
}

void screem_tag_file_set_name( ScreemTagFile *tagfile, 
				const gchar *name )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( name != NULL );

	g_free( tagfile->priv->name );
	tagfile->priv->name = g_strdup( name );
}

void screem_tag_file_set_desc( ScreemTagFile *tagfile,
				const gchar *desc )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( desc != NULL  );

	g_free( tagfile->priv->description );
	tagfile->priv->description = g_strdup( desc );
}

void screem_tag_file_set_uri( ScreemTagFile *tagfile,
				const gchar *uri )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( uri != NULL );

	g_free( tagfile->priv->filename );
	tagfile->priv->filename = g_strdup( uri );
}

GSList *screem_tag_file_autocomplete( ScreemTagFile *tagfile,
					const gchar *prefix )
{
	ScreemTagFilePrivate *priv;
	GSList *ret;
	
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	priv = tagfile->priv;
	
	ret = NULL;
	priv->prefix = prefix;
	gtk_tree_model_foreach( priv->model,
			(GtkTreeModelForeachFunc)screem_tag_file_autocomplete_fill,
			&ret );

	return ret;			
}

