/*  Screem:  screem-page-model.c
 *
 *  The ScreemPageModel object
 *  
 *  This object is responsible for keeping a tree view of a ScreemPage
 *  up to date with changes made to the GtkTextBuffer of the page
 *  or at least that is the intention.
 *
 *  It will also be responsible for keeping track of the DTD in use
 *  for the page rather than the horrible method of the ScreemWindow
 *  handling this in combination with ScreemPage
 *
 *  The problem here though is coming up with an algorithm for
 *  dynamically updating the tree on text changes
 * 
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include <libgnome/gnome-i18n.h>

#include <libcroco/libcroco.h>

#include "screem-application.h"

#include "screem-page-model.h"

#include "screem-dtd.h"
#include "screem-markup.h"
#include "screem-page.h"
#include "screem-search.h"

static void screem_page_model_class_init( ScreemPageModelClass *klass );
static void screem_page_model_init( ScreemPageModel *page );
static void screem_page_model_finalize( GObject *object );
static void screem_page_model_set_prop( GObject *object, guint prop_id,
				  	const GValue *value, GParamSpec *spec );
static void screem_page_model_get_prop( GObject *object, guint prop_id,
				  	GValue *value, GParamSpec *spec );

static void screem_page_model_page_set( ScreemPageModel *model );
static void insert_text( GtkTextBuffer *buffer, GtkTextIter *it,
			 const gchar *text, gint length, ScreemPageModel *model );
static void delete_range( GtkTextBuffer *buffer, GtkTextIter *it,
			  GtkTextIter *eit, ScreemPageModel *model );

static void screem_page_doctype_check( ScreemPageModel *model, gint pos, 
				       gint length );
static void screem_page_charset_check( ScreemPageModel *model, gint pos, 
				       gint length );

static void screem_page_model_dtd_set( ScreemPageModel *model );

typedef enum {
	PROP_0,
	PROP_APPLICATION,
	PROP_PAGE,
	PROP_DTD,
	PROP_CHARSET,
	PROP_DOCTYPE
} ScreemPageModelProps;

typedef struct {
	GtkTreeModel *model;
	GtkTreeIter document;
	GtkTreeIter external;
	GtkTreeIter editable;

	GSList *stack;
	
	GtkTreeIter *parent;
	gchar *pname;
	ScreemDTDTagExistance pclose;
	const ScreemDTDElement *pelem;
	
	guint pos;
	guint ppos;
	gchar *text;
	const gchar *offset_text;
	const gchar *poffset_text;
	gchar *name;
	gchar *tag;
	guint start;
	guint end;
	guint last;

	ScreemDTD *dtd;
	gboolean gotroot;

	guint sourceid;

	GCompareFunc compare;
} ScreemPageModelParser;

typedef struct {
	GtkTreeModel *model;
	GtkTreeIter *current;
	CRInput *input;
	ScreemPage *page;
} ScreemPageModelCSSParser;

struct ScreemPageModelPrivate {
	ScreemApplication *application;
	
	ScreemPage *page;

	ScreemDTD *dtd;

	gint doctype_pos;
	gchar *doctype;

	gint charset_pos;
	gint charset_len;
	gchar *charset;

	ScreemPageModelParser parser;

	GtkTreeIter *context;
};

ScreemPageModel *screem_page_model_new( ScreemPage *page, GObject *app )
{
	ScreemPageModel *model;
	GType type;

	type = screem_page_model_get_type();

	model = SCREEM_PAGE_MODEL( g_object_new( type, 
						 "page", page,
						 "application", app,
						 NULL ) );

	return model;
}

void screem_page_model_force_check( ScreemPageModel *model )
{
	if( screem_page_is_markup( model->private->page ) ) {
		screem_page_doctype_check( model, 0, 0x7fffffff );
		
		screem_page_charset_check( model, 0, 0x7fffffff );
	}
}

/* static stuff */
static void screem_page_model_page_set( ScreemPageModel *model )
{
	GtkTreeIter it;

	gtk_tree_store_clear( GTK_TREE_STORE( model ) );
	
	gtk_tree_store_append( GTK_TREE_STORE( model ), &it, NULL );
	gtk_tree_store_set( GTK_TREE_STORE( model ), &it,
			    SCREEM_PAGE_MODEL_NAME, "[Document]",
			    SCREEM_PAGE_MODEL_VALID, TRUE,
			    -1 );

	g_signal_connect_after( G_OBJECT( model->private->page ), "insert_text",
			  G_CALLBACK( insert_text ), model );
	g_signal_connect_after( G_OBJECT( model->private->page ), "delete_range",
				G_CALLBACK( delete_range ), model );

}

static void insert_text( GtkTextBuffer *buffer, GtkTextIter *it,
			 const gchar *text, gint length, ScreemPageModel *model )
{
	guint pos;

	/* - length as we are connected after, so the iter will be advanced
	 * by length chars */
	pos = gtk_text_iter_get_offset( it ) - length + 1;

	/* hmm, can end up with - pos here, possibly due to the initial
	 * newline chars in the buffer? */
	if( pos < 0 ) {
		pos = 0;
	}
	
	if( screem_page_is_markup( model->private->page ) ) {
		screem_page_doctype_check( model, pos, length );
		screem_page_charset_check( model, pos, length );
	}
}

static void delete_range( GtkTextBuffer *buffer, GtkTextIter *it,
			  GtkTextIter *eit, ScreemPageModel *model )
{
	guint pos;
	guint length;
	
	pos = gtk_text_iter_get_offset( it );
	length = gtk_text_iter_get_offset( eit ) - pos;
	
	if( screem_page_is_markup( model->private->page ) ) {
		screem_page_doctype_check( model, pos, length );
		screem_page_charset_check( model, pos, length );
	}

}

static void screem_page_doctype_check( ScreemPageModel *model, gint pos, 
				       gint length )
{
	ScreemPageModelPrivate *private;
	gboolean set;

	private = model->private;

	set = FALSE;

	if( ! private->doctype ) {
		set = TRUE;
	} else { 
		gint len;
		
		len = strlen( private->doctype );
			
		if( ( pos <= ( private->doctype_pos + len ) ) &&
		    ( pos + length ) >= private->doctype_pos ) {
			set = TRUE;
		}
	}
	if( set ) {
		/* set the doctype */
		gchar *text;
		gchar *doctype;
		
		text = screem_page_get_data( private->page );
		
		doctype = find_text( text,
				     "<![Dd][Oo][Cc][Ty][Yy][Pp][Ee] ",
				     NULL, NULL );
		if( doctype ) {
			GString *tag;
			
			tag = g_string_new( NULL );
			
			private->doctype_pos = doctype - text;
			
			doctype --;
			do {
				doctype ++;
				g_string_append_c( tag, *doctype );
			} while( *doctype != '\0' && *doctype != '>' );
			g_object_set( G_OBJECT( model ), "doctype", tag->str, NULL );
			g_string_free( tag, TRUE );
		}
			
		g_free( text );
	}
}

static void screem_page_charset_check( ScreemPageModel *model, gint pos, 
				       gint length )
{
	ScreemPageModelPrivate *private;
	gboolean set = FALSE;
	GConfClient *client;

	private = model->private;

	if( ! private->charset ) {
		set = TRUE;
	} else { 
		gint len;

		len = private->charset_len;

		if( ( pos <= ( private->charset_pos + len ) ) &&
		    ( pos + length ) >= private->charset_pos ) {
			set = TRUE;
		}
	}
	if( set ) {
		/* set the charset */
		gchar *text;
		gchar *tag;
		gchar *name;
		gint pos;
		gint start;
		gint end;
		gint len;

		pos = 0;
		text = screem_page_get_data( private->page );
		len = strlen( text );
		
		g_free( private->charset );
		private->charset = NULL;
		private->charset_pos = 0;

		/* first we look for <?xml encoding="" ?>
		   then we check <meta http-equiv="content-type" />
		   then we check <meta name="charset" /> 
		   if we have an <?xml ?> without an encoding="" we
		   enforce UTF-8 no matter what <meta /> exist */

		while( pos < len &&
			( tag = screem_markup_next_tag( text, pos,
						       &start, &end, 
						       &name ) ) ) {
			g_assert( name );
			pos = end;
			if( ! strcmp( "?xml", name ) ) {
				/* xml PI element */
				GSList *list;
				GSList *attr;

				list=screem_markup_build_attributes_list(tag,
									 NULL);
				for( attr = list; attr; attr = attr->next ) {
					if(! strcmp( "encoding", 
						     attr->next->data ) &&
					   attr->data ) {
						g_object_set( G_OBJECT( model ), "charset", attr->data, NULL );
						break;
					} else {
						attr = NULL;
						break;
					}
					attr = attr->next;
				}
				
				if( ! attr ) {
					private->charset_pos = start;
					private->charset_len = end - start;
					g_object_set( G_OBJECT( model ), "charset", "UTF-8", NULL );
				}

				g_slist_foreach( list, (GFunc)g_free, NULL );
				g_slist_free( list );
				pos = len;
			} else if( ! g_strcasecmp( "meta", name ) ) {
				/* meta element */
				GSList *list;
				GSList *attr;
				gboolean content;
				
				list=screem_markup_build_attributes_list(tag,
									 NULL);
				content = FALSE;

				for( attr = list; attr; attr = attr->next ) {
					if( ! strcmp( "http-equiv", 
						      attr->next->data ) && 
					    attr->data &&
					    ! strcmp( "content-type",
						      attr->data ) ) {
						content = TRUE;
						break;
					} else if(! strcmp( "name", 
							    attr->next->data) &&
						  attr->data && 
						  ! strcmp( "charset",
							    attr->data)){
						break;
					}
					attr = attr->next;
				}
				if( attr ) {
					for( attr = list; attr; 
					     attr = attr->next ) {
						if( ! strcmp( "content",
							      attr->next->data ) ) {
							break;
						}
						attr = attr->next;
					}
				}
				if( attr && attr->data && content ) {
					gchar *val;

					val = strstr( attr->data, "charset=" );
					if( val ) {
						GString *tmp;

						val += strlen( "charset=" );
						tmp = g_string_new( NULL );
						while( *val != ' ' &&
						       *val != '\0' &&
						       *val != '"' ) {
							g_string_append_c( tmp,
									   *val);
							val ++;
						}
						private->charset_pos = start;
						private->charset_len = end - 
							start;
						pos =  len;
						g_object_set( G_OBJECT( model ), "charset", tmp->str, NULL );
						g_string_free( tmp, TRUE );
					}
				} else if( attr ) {
					private->charset_pos = start;
					private->charset_len = end - start;
					pos = len;
					g_object_set( G_OBJECT( model ), "charset", attr->data, NULL );
				}

				g_slist_foreach( list, (GFunc)g_free, NULL );
				g_slist_free( list );
			} else if( g_strcasecmp( "html", name ) &&
				   g_strcasecmp( "head", name ) &&
				   g_strcasecmp( "title", name ) &&
				   g_strcasecmp( "link", name ) &&
				   g_strcasecmp( "script", name ) &&
				   g_strcasecmp( "style", name ) &&
				   g_strcasecmp( "!--", name ) &&
				   g_strcasecmp( "base", name ) &&
				   g_strcasecmp( "isindex", name ) &&
				   g_strcasecmp( "!DOCTYPE", name ) &&
				   *name != '/' && *name != '?' ) {
				/* limiting case, bit of a hack for html,
				   valid xml docs are supposed to have the
				   <?xml ?> element and have indeterminate 
				   element names so we can't stop if the doc is
				   invalid */
				pos = len;
			}
			g_free( tag );
			g_free( name );
		}
	
		g_free( text );
	
		if( ! private->charset ) {
			client = gconf_client_get_default();
			model->private->charset = 
				gconf_client_get_string( client,
							"/apps/screem/editor/default_charset",
							NULL );
			g_object_unref( client );
		}
	}
}

static void screem_page_model_dtd_set( ScreemPageModel *model )
{
	ScreemPageModelPrivate *priv;
	ScreemDTDDB *db;
	ScreemDTD *dtd;
	
	g_return_if_fail( SCREEM_IS_PAGE_MODEL( model ) );
	priv = model->private;
	
	db = screem_application_get_dtd_db( priv->application );
	dtd = NULL;
	if( priv->doctype ) {
		dtd = screem_dtd_db_get_dtd_from_doctype( db, 
							priv->doctype );
	}
	if( ! dtd ) {
		dtd = screem_dtd_db_get_default_dtd( db );
	}
	if( dtd != priv->dtd ) {
		g_object_set( G_OBJECT( model ), "dtd", dtd, NULL );
	}
}

/*********************************************/
static gboolean build_tree( ScreemPageModel *model );
static gboolean screem_page_model_clear_tree( GtkTreeModel *model, 
					     GtkTreePath *path,
					     GtkTreeIter *iter, 
					     gpointer data );
static void screem_page_model_build_text_node( ScreemPageModelParser *parser );
static void screem_page_model_build_close( ScreemPageModelParser *parser );
static void screem_page_model_build_open( ScreemPageModelParser *parser );
static void screem_page_model_parser_cleanup( ScreemPageModelParser *parser );
static gboolean screem_page_model_build_step( ScreemPageModelParser *parser );
static void screem_page_model_set_invalid( ScreemPage *page,
				guint start, guint end,
				const gchar *name,
				const gchar *pname,
				const gchar *attr,
				const gchar *value );
static gboolean select_context( ScreemPageModel *model,
				GtkTreeIter **it,
				guint pos,
				guint *start, guint *end,
				gboolean select_text,
				gboolean build );

static void screem_page_model_css_build( ScreemPage *page );


void screem_page_model_build_model( ScreemPageModel *model )
{
	build_tree( model );
}

void screem_page_model_ensure_built( ScreemPageModel *model )
{
	ScreemPageModelPrivate *priv;
	
	priv = model->private;
	
	
	/* make sure full tree is fully built, we remove
	   the idle handler, but don't set the handle to 0
	   as this will be done by while building in the
	   while loop */
	if( priv->parser.sourceid ) {
		g_source_remove( priv->parser.sourceid );
	}

	/* we need to leave the thread lock as
	 * screem_page_model_build_step() is normally
	 * called via an idle handler */
	gdk_threads_leave();
	while( priv->parser.sourceid ) {
		screem_page_model_build_step( &priv->parser );
	}
gdk_threads_enter();
}

gboolean screem_page_model_select_context( ScreemPageModel *model,
					  guint pos,
					  guint *start, guint *end,
					  gboolean select_text )
{
	guint tstart;
	guint tend;
	gboolean ret;

	if( ! start ) {
		start = &tstart;
	}
	if( ! end ) {
		end = &tend;
	}

	ret = select_context( model,
			NULL, pos, start, end, select_text, TRUE );

	return ret;
}

gchar *screem_page_model_query_context( ScreemPageModel *model,
		guint pos, gboolean query_text,
		gboolean build, 
		guint *depth, guint *start, guint *end )
{
	ScreemPageModelPrivate *priv;
	guint tstart;
	guint tend;
	guint tdepth;
	gchar *ret;
	GtkTreePath *path;

	if( ! start ) {
		start = &tstart;
	}
	if( ! end ) {
		end = &tend;
	}
	if( ! depth ) {
		depth = &tdepth;
	}
	*depth = 0;

	priv = model->private;
	ret = NULL;

	/* FIXME: priv->context should always be non NULL
	   if select_context() returns TRUE, 
	   however we don't handle <!DOCTYPE ... [ .. ]>
	   very well, and can cause it to be NULL, at least I think
	   that is what is causing a crash */
	if( select_context( model, NULL, pos, start, end, query_text,
				build ) && priv->context ) {

		gtk_tree_model_get( GTK_TREE_MODEL( model ), 
				priv->context,
				SCREEM_PAGE_MODEL_NAME, &ret, -1 );

		path = gtk_tree_model_get_path( GTK_TREE_MODEL( model ),
					priv->context );

		/* -1 as we are always under the [document] node */
		*depth = gtk_tree_path_get_depth( path );
		if( *depth > 0 ) {
			(*depth) --;
		}
		gtk_tree_path_free( path );

		gtk_tree_iter_free( priv->context );
		priv->context = NULL;
	}

	return ret;
}

static gboolean build_tree( ScreemPageModel *model )
{
	ScreemPageModelPrivate *priv;
	ScreemPage *page;
	gboolean feature_markup;
	ScreemPageModelParser *parser;
	GtkTextTagTable *table;
	GtkTextTag *tag;
	GtkTextIter it;
	GtkTextIter eit;
	const gchar *mime_type;
	
	priv = model->private;
	page = priv->page;
	
gdk_threads_enter();
	
	screem_page_model_emit_building( page );

	table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
	tag = gtk_text_tag_table_lookup( table,
					SCREEM_INVALID_MARKUP_TAG );
	if( tag ) {
		gtk_text_buffer_get_start_iter( GTK_TEXT_BUFFER( page ),
						&it );
		gtk_text_buffer_get_end_iter( GTK_TEXT_BUFFER( page ),
						&eit );
		gtk_text_buffer_remove_tag( GTK_TEXT_BUFFER( page ), 
				tag, &it, &eit );
	}
	/* this will stop any current build + cleanup after it */
	screem_page_model_parser_cleanup( &priv->parser );

	gtk_tree_model_foreach( GTK_TREE_MODEL( model ), 
				screem_page_model_clear_tree, 
				NULL );
	gtk_tree_store_clear( GTK_TREE_STORE( model ) );
	
	mime_type = screem_page_get_mime_type( page );
	feature_markup = screem_page_is_markup( page );

	if( feature_markup ) {
		/* init parser */
		parser = &priv->parser;
		parser->dtd = screem_page_get_dtd( page );
		g_object_get( G_OBJECT( parser->dtd ),
				"compare", &parser->compare, NULL );
		parser->model = GTK_TREE_MODEL( model );
		gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
					&parser->document, NULL );
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ), 
				    &parser->document,
				    SCREEM_PAGE_MODEL_NAME,
				    _( "[Document]" ),
				    SCREEM_PAGE_MODEL_VALID, TRUE,
				    -1 );
		gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
					&parser->editable,
					NULL ); //&parser->document );
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
					&parser->editable,
					SCREEM_PAGE_MODEL_NAME,
					_( "[Regions]" ),
					SCREEM_PAGE_MODEL_VALID, TRUE,
					-1 );
			
		gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
					&parser->external,
					NULL );//&parser->document );
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
					&parser->external,
					SCREEM_PAGE_MODEL_NAME,
					_( "[Links]" ),
					SCREEM_PAGE_MODEL_VALID, TRUE,
					-1 );
			
		parser->parent = &parser->document;
		parser->pclose = SCREEM_DTD_MUST;
		parser->text = screem_page_get_data( page );
		parser->offset_text = parser->text;
		parser->last = gtk_text_buffer_get_char_count( GTK_TEXT_BUFFER( page ) ); 

		/* add parser idle handler */
		parser->sourceid = g_idle_add_full( G_PRIORITY_LOW,
				(GSourceFunc)screem_page_model_build_step,
				parser, NULL );
	} else if( ! strcmp( "text/css", mime_type ) ) {
		screem_page_model_css_build( page );
	} else {
		screem_page_model_emit_built( page );
	}
		
	gdk_threads_leave();
	
	return FALSE;
}

static void screem_page_model_css_selector( CRDocHandler *handler,
					CRSelector *selector )
{
	guchar *sel;
	ScreemPageModelCSSParser *data;
	CRInputPos pos;
	GtkTextIter it;
	guint start;
	gunichar c;
	
	data = handler->app_data;

	if( data->current ) {
		gtk_tree_iter_free( data->current );
		data->current = NULL;
	}
	data->current = g_new0( GtkTreeIter, 1 );

	gtk_tree_store_append( GTK_TREE_STORE( data->model ),
				data->current, NULL );
	sel = cr_selector_to_string( selector );
	start = 0;
	if( cr_input_get_cur_pos( data->input, &pos ) == CR_OK ) {
		gtk_text_buffer_get_iter_at_line_offset( GTK_TEXT_BUFFER( data->page ), &it, pos.line, pos.col -1 );

		c = ' ';
		while( gtk_text_iter_backward_char( &it ) &&
			g_unichar_isspace( c ) ) {	
			c = gtk_text_iter_get_char( &it );
		}
		start = gtk_text_iter_get_offset( &it ) + 1;
		start ++;
		start -= strlen( sel );
	}

	gtk_tree_store_set( GTK_TREE_STORE( data->model ),
			data->current,
			SCREEM_PAGE_MODEL_NAME, sel,
			SCREEM_PAGE_MODEL_START, start,
			SCREEM_PAGE_MODEL_END, start + strlen( sel ),
			-1 );

	g_free( sel );
}

static void screem_page_model_css_end_selector( CRDocHandler *handler,
		CRSelector *selector )
{
	ScreemPageModelCSSParser *data;
	CRInputPos pos;
	GtkTextIter it;
	guint end;
	
	data = handler->app_data;
	
	if( data->current &&
	    cr_input_get_cur_pos( data->input, &pos ) == CR_OK ) {
		gtk_text_buffer_get_iter_at_line( GTK_TEXT_BUFFER( data->page ), &it, pos.line );
		end = gtk_text_iter_get_offset( &it );
		end += pos.col;
		
		gtk_tree_store_set( GTK_TREE_STORE( data->model ),
			data->current,
			SCREEM_PAGE_MODEL_END, end,
			-1 );
	}
}

static void screem_page_model_css_property( CRDocHandler *handler,
					GString *name,
					CRTerm *value )
{
	GtkTreeIter it;
	GtkTreeIter vit;
	guchar *val;
	ScreemPageModelCSSParser *data;
	CRInputPos pos;
	GtkTextIter tit;
	guint start;
	guint valstart;
	guint end;
	gunichar c;

	data = handler->app_data;
	
	if( data->current ) {
		val = cr_term_to_string( value );

		start = valstart = end = 0;
		if( cr_input_get_cur_pos( data->input, 
					  &pos ) == CR_OK ) {
			gtk_text_buffer_get_iter_at_line_offset( GTK_TEXT_BUFFER( data->page ), &tit, pos.line, pos.col );
			start = gtk_text_iter_get_offset( &tit );
			end = start;
			start -= strlen( val );
			
			gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( data->page ), &tit, start );
			c = ' ';
			while( gtk_text_iter_backward_char( &tit ) &&
				g_unichar_isspace( c ) ) {	
				c = gtk_text_iter_get_char( &tit );
			}
			start = gtk_text_iter_get_offset( &tit ) + 1;
			valstart = start + 2;
			start -= name->len;
		}
		
		gtk_tree_store_append( GTK_TREE_STORE( data->model ),
				&it, data->current );
		gtk_tree_store_set( GTK_TREE_STORE( data->model ), &it,
				SCREEM_PAGE_MODEL_NAME, name->str,
				SCREEM_PAGE_MODEL_START, start,
				SCREEM_PAGE_MODEL_END, end,
				-1 );
		gtk_tree_store_append( GTK_TREE_STORE( data->model ),
				&vit, &it );
		gtk_tree_store_set( GTK_TREE_STORE( data->model ), &vit,
				SCREEM_PAGE_MODEL_NAME, val,
				SCREEM_PAGE_MODEL_START, valstart,
				SCREEM_PAGE_MODEL_END, end, 
				-1 );
		g_free( val );
	}
}

static void screem_page_model_css_import( CRDocHandler *handler,
		GList *media_list, GString *uri, GString *ns )
{
	ScreemPageModelCSSParser *data;
	GtkTreeIter it;
	GtkTextIter tit;
	CRInputPos pos;
	guint start;
	guint end;
	gunichar c;
	gboolean fixed;
	guint lastspace;
	gunichar p;

	data = handler->app_data;

	start = end = lastspace = 0;
	if( cr_input_get_cur_pos( data->input, &pos ) == CR_OK ) {
		gtk_text_buffer_get_iter_at_line_offset( GTK_TEXT_BUFFER( data->page ), &tit, pos.line, pos.col );
		start = gtk_text_iter_get_offset( &tit );
		end = start;

		c = p = '\0';
		fixed = FALSE;
		while( gtk_text_iter_backward_char( &tit ) &&
			c != '@' ) {	
			c = gtk_text_iter_get_char( &tit );
			if( c == ';' && ! fixed ) {
				end = gtk_text_iter_get_offset( &tit );
				fixed = TRUE;
			} else if( g_unichar_isspace( c ) &&
				! g_unichar_isspace( p ) ) {
				lastspace = gtk_text_iter_get_offset( &tit );
			}
			p = c;
		}
		start = gtk_text_iter_get_offset( &tit );
	}

	gtk_tree_store_append( GTK_TREE_STORE( data->model ),
			&it, NULL );
	gtk_tree_store_set( GTK_TREE_STORE( data->model ),
			&it,
			SCREEM_PAGE_MODEL_NAME, "@import",
			SCREEM_PAGE_MODEL_START, start,
			SCREEM_PAGE_MODEL_END, end,
			-1 );
	gtk_tree_store_append( GTK_TREE_STORE( data->model ),
			&it, &it );
	gtk_tree_store_set( GTK_TREE_STORE( data->model ),
			&it,
			SCREEM_PAGE_MODEL_NAME, uri->str,
			SCREEM_PAGE_MODEL_START, lastspace + 1,
			SCREEM_PAGE_MODEL_END, end,
			-1 );
}

static void screem_page_model_css_build( ScreemPage *page )
{
	CRParser *parser;
	CRInput *input;
	CRDocHandler *sac;
	gchar *buf;
	ScreemPageModelCSSParser data;

	buf = screem_page_get_data( page );

	data.model = screem_page_get_model( page );
	data.current = NULL;	
		
	input = cr_input_new_from_buf( buf,
			gtk_text_buffer_get_char_count( GTK_TEXT_BUFFER( page ) ),
				CR_UTF_8, TRUE );
	data.input = input;
	data.page = page;
	parser = cr_parser_new_from_input( input );
	if( parser ) {
		sac = cr_doc_handler_new();
		if( sac ) {
			sac->start_document = NULL;
			sac->end_document = NULL;
			sac->start_selector = screem_page_model_css_selector;
			sac->property = screem_page_model_css_property;
			sac->end_selector = screem_page_model_css_end_selector;
			sac->import_style = screem_page_model_css_import;
			sac->app_data = &data;

			cr_parser_set_sac_handler( parser, sac );

			cr_parser_parse( parser );
		}
		cr_parser_destroy( parser );
	}
	/* input and sac are destroyed by cr_parser_destroy,
	 * do not destroy them here */

	screem_page_model_emit_built( page );

}

static gboolean screem_page_model_clear_tree( GtkTreeModel *model, 
					     GtkTreePath *path,
					     GtkTreeIter *iter, 
					     gpointer data )
{

	return FALSE;
}

static void screem_page_model_build_text_node( ScreemPageModelParser *parser )
{
	/* add text node */
	GtkTreeIter tit;
	gchar *txt;
	gboolean valid;
	
	txt = g_strndup( parser->poffset_text,
			g_utf8_offset_to_pointer( parser->poffset_text,
				parser->pos - parser->ppos ) -
			parser->poffset_text );
		
	/* chomp / chop any white space, if all 
	   white space then we won't even bother
	   putting it in the tree */
	txt = g_strstrip( txt );

	if( *txt != '\0' ) {
		valid = screem_dtd_valid_child_element( parser->dtd, 
				parser->pelem, "PCDATA" );
		if( ! valid ) {
			valid = screem_dtd_valid_child_element( parser->dtd,
						     parser->pelem,
						     "CDATA");
		}

		gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
				       &tit, parser->parent );
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ), 
				    &tit,
				    SCREEM_PAGE_MODEL_NAME,
				    _( "[text]" ),
				    SCREEM_PAGE_MODEL_TEXT,
				    txt,
				    SCREEM_PAGE_MODEL_START,
				    parser->ppos,
				    SCREEM_PAGE_MODEL_END,
				    parser->pos,
				    SCREEM_PAGE_MODEL_VALID,
				    valid,
				    -1 );
	}
	g_free( txt );
}

static void screem_page_model_build_close( ScreemPageModelParser *parser )
{
	ScreemPageModel *pmodel;
	gboolean match;
	gboolean valid;
	guint start;
	guint end;
	gchar *name;
	gchar *pname;

	GtkTreeIter it;
	
	pmodel = SCREEM_PAGE_MODEL( parser->model );
	
	/* pop stack, change parent until
	   we hit the correct close tag or
	   we run out of stack */
	match = FALSE;
	g_free( parser->pname );
	end = parser->pos;
	while( parser->stack && ! match ) {
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				    parser->parent,
				   SCREEM_PAGE_MODEL_END, 
				   end, -1 );
		gtk_tree_model_get( GTK_TREE_MODEL( parser->model ),
				parser->parent,
				SCREEM_PAGE_MODEL_NAME, &name,
				SCREEM_PAGE_MODEL_VALID, &valid, 
				SCREEM_PAGE_MODEL_START, &start,
				-1 );
		pname = NULL;
		if( gtk_tree_model_iter_parent( GTK_TREE_MODEL( parser->model ),
					&it, parser->parent ) ) {
			gtk_tree_model_get( GTK_TREE_MODEL( parser->model ),
					&it, SCREEM_PAGE_MODEL_NAME, &pname,
					-1 );
		}
		
		if( ! valid ) {
			screem_page_model_set_invalid( pmodel->private->page,
					start, end, name, pname,
					NULL, NULL );
		}
		g_free( pname );
		
		parser->stack = g_slist_remove( parser->stack, 
					parser->stack->data );
		match = ! parser->compare( parser->name + 1, name );
		g_free( name );
		gtk_tree_iter_free( parser->parent );
		if( parser->stack ) {
			parser->parent = parser->stack->data;
		}
	}

	parser->pname = NULL;
	parser->pelem = NULL;
	if( ! parser->stack ) {
		parser->parent = &parser->document;
		parser->pclose = SCREEM_DTD_MUST;
	} else if( parser->parent != &parser->document ) {
		gtk_tree_model_get( parser->model,
				  parser->parent,
				  SCREEM_PAGE_MODEL_NAME,
				  &parser->pname, -1 );
		parser->pelem = screem_dtd_valid_element( parser->dtd,
						parser->pname );
		parser->pclose = screem_dtd_element_close_element_state( parser->dtd,
						 parser->pelem );
	}
}

static void screem_page_model_build_open( ScreemPageModelParser *parser )
{
	ScreemPageModel *pmodel;
	gchar firstc;
	gboolean valid;
	gboolean iscomment;
	ScreemDTDTagExistance close_state;
	GtkTreeIter it;
	GSList *attrs;
	GSList *tmp;
	gchar *name;
	gchar *value;
	const ScreemDTDElement *elem;
	
	const gchar *xmlclosecheck;
	gboolean hasuri;
	
	pmodel = SCREEM_PAGE_MODEL( parser->model );
	
	firstc = *parser->name;
	
	if( ( iscomment = ! strcmp( "!--", parser->name ) ) ) {
		g_free( parser->name );
		parser->name = g_strdup( _( "[Comment]" ) );
	}

	elem = screem_dtd_valid_element( parser->dtd, parser->name );

	close_state = screem_dtd_element_close_element_state( parser->dtd,
						elem );

	valid = FALSE;
	if( firstc == '!' || firstc == '?' || firstc == '%' ) {
		valid = TRUE;
	} else if( parser->parent == &parser->document &&
		! parser->gotroot &&
		screem_dtd_is_root_element( parser->dtd, parser->name )) {

		valid = TRUE;
		parser->gotroot = TRUE;
	} else if( parser->pelem ) {
		valid = screem_dtd_valid_child_element( parser->dtd,
					parser->pelem,
					parser->name );
	}
	
	/* if closing the parent tag is optional, and 
	   valid is FALSE, pop the parent */
	while( parser->parent != &parser->document && 
		parser->pclose == SCREEM_DTD_SHOULD &&
	       ! valid ) {
		/* assume this closes the parent */
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				parser->parent, SCREEM_PAGE_MODEL_END, 
				parser->start, -1 );
		parser->stack = g_slist_remove( parser->stack, 
						parser->stack->data );
		gtk_tree_iter_free( parser->parent );
		if( parser->stack ) {
			parser->parent = parser->stack->data;
				
			g_free( parser->pname );
			gtk_tree_model_get( parser->model, 
					parser->parent,
					SCREEM_PAGE_MODEL_NAME,
					&parser->pname, -1 );
			parser->pelem = screem_dtd_valid_element( parser->dtd,
						parser->pname );
			parser->pclose = screem_dtd_element_close_element_state( parser->dtd, parser->pelem );
			valid = screem_dtd_valid_child_element( parser->dtd,
					parser->pelem,
					parser->name );
		} else {
			parser->parent = &parser->document;
			g_free( parser->pname );
			parser->pname = NULL;
			parser->pelem = NULL;
		}
	}
			
	/* we are dealing with an opening tag */
	gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
			       &it, parser->parent );

	/* set values for the iter */
	if( iscomment ) {
		GtkTreeIter rit;
		gint n;
		gchar *ed_name;
		guint len;
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				&it,
				SCREEM_PAGE_MODEL_NAME, parser->name,
				SCREEM_PAGE_MODEL_START,parser->start,
				SCREEM_PAGE_MODEL_END, parser->end,
				SCREEM_PAGE_MODEL_TEXT, parser->tag,
				-1 );

		/* is it an editable region comment ? */
		if( ! strncmp( "<!-- #BeginEditable ",
				parser->tag,
				strlen( "<!-- #BeginEditable " ) ) ) {
			/* it is */
			gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
					&rit, &parser->editable );
			ed_name = find_text( parser->tag, "\"[^\"]*\"",
						NULL, &len );
			ed_name = g_strndup( ed_name, len );

			gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
					&rit,
					SCREEM_PAGE_MODEL_NAME, ed_name,
					SCREEM_PAGE_MODEL_START, 
					parser->start,
					SCREEM_PAGE_MODEL_END,
					parser->end,
					SCREEM_PAGE_MODEL_VALID, TRUE,
					-1 );
			g_free( ed_name );
		} else if( ! strcmp( "<!-- #EndEditable --", 
					parser->tag ) ) {
			/* it ends the last one */
			n = gtk_tree_model_iter_n_children( parser->model,
					&parser->editable );
			if( ( n > 0 ) &&
			    gtk_tree_model_iter_nth_child( parser->model,
				    &rit, &parser->editable, n - 1 ) ) {
				gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
							&rit,
							SCREEM_PAGE_MODEL_END,
							parser->end,
							-1 );
			}
		}
	} else {
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				&it,
				SCREEM_PAGE_MODEL_NAME, parser->name,
				SCREEM_PAGE_MODEL_START,parser->start,
				SCREEM_PAGE_MODEL_END, parser->end,
				-1 );
	}
	
	/* check attributes + validate 
	 *
	 * TODO: emit the invalid signal only across the positions
	 * for invalid attributes, instead of invalidating the whole
	 * tag
	 *
	 * */
	if( elem ) {
		hasuri = FALSE;
		attrs = screem_markup_build_attributes_list( parser->tag, NULL );
		for( tmp = attrs; tmp; tmp = tmp->next ) {
			value = tmp->data;
			tmp = tmp->next;
			name = tmp->data;

			valid &= ( screem_dtd_valid_element_attr( parser->dtd,
							elem,
							name ) != NULL );
			hasuri |= screem_dtd_attr_is_uri( parser->dtd,
					parser->name, name );
			g_free( name );
			g_free( value );
		}
		g_slist_free( attrs );
		if( hasuri ) {
			GtkTreeIter lit;
			gtk_tree_store_append( GTK_TREE_STORE( parser->model ),
					&lit, &parser->external );
			gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
					&lit,
					SCREEM_PAGE_MODEL_VALID, valid,
					SCREEM_PAGE_MODEL_NAME, 
					parser->name,
					SCREEM_PAGE_MODEL_START,
					parser->start,
					SCREEM_PAGE_MODEL_END, parser->end,
					-1 );
		}
	}
	gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				&it,
				SCREEM_PAGE_MODEL_VALID, valid,
				-1 );
	if( ! valid ) {
		screem_page_model_set_invalid( pmodel->private->page,
					parser->start, 
					parser->end, 
					parser->name, 
					parser->pname,
					NULL, NULL );
	}

	/* put it on the stack as a parent if needed,
	   we don't need to for:
	   1) comments / doctypes
	   2) tags closing themselves, eg <meta /> 
	   3) HTML tags which forbid closing, same as
	      2 but we need to lookup in the doctype */

	/* 2 g_utf8_prev_char() is safe, as offset_text is end + 1,
	 * and end must be > start therefore -2 >= start */
	xmlclosecheck = parser->offset_text;
	xmlclosecheck = g_utf8_prev_char( xmlclosecheck );
	xmlclosecheck = g_utf8_prev_char( xmlclosecheck );
	if(  ( firstc != '!' ) && ( firstc != '?' ) &&
	    ( *xmlclosecheck != '/' ) &&
		( close_state != SCREEM_DTD_MUST_NOT ) ) {
		parser->parent = gtk_tree_iter_copy( &it );
		parser->stack = g_slist_prepend( parser->stack, 
						parser->parent );
		g_free( parser->pname );
		gtk_tree_model_get( parser->model,
				  parser->parent,
				  SCREEM_PAGE_MODEL_NAME,
				  &parser->pname, -1 );
		parser->pelem = screem_dtd_valid_element( parser->dtd,
						parser->pname );
		parser->pclose = screem_dtd_element_close_element_state( parser->dtd, parser->pelem );
	}
}

static void screem_page_model_parser_cleanup( ScreemPageModelParser *parser )
{
	GSList *cleanup;
	GtkTreeModel *model;

	model = parser->model;
	
	g_free( parser->pname );
	parser->pelem = NULL;

	/* cleanup the stack */
	cleanup = parser->stack;
	while( cleanup ) {
		gtk_tree_store_set( GTK_TREE_STORE( parser->model ),
				   parser->parent,
				   SCREEM_PAGE_MODEL_END, 
				   parser->pos, -1 );
		cleanup = g_slist_remove( cleanup, cleanup->data ); 
		if( parser->parent != &parser->document ) {
			gtk_tree_iter_free( parser->parent );
		}
		if( cleanup ) {
			parser->parent = cleanup->data;
		}
	}
	g_free( parser->text );
	parser->offset_text = NULL;

	if( parser->sourceid ) {
		g_source_remove( parser->sourceid );
		parser->sourceid = 0;
	}

	memset( parser, 0, sizeof( ScreemPageModelParser ) );

	parser->model = model;
}

static gboolean screem_page_model_build_step( ScreemPageModelParser *parser )
{
	gboolean ret;
	ScreemPageModel *pmodel;

	/*clock_t start;
	clock_t end;
	
	start = clock();
	while( ret ) {*/
	
	ret = TRUE;
	
	parser->ppos = parser->pos;

	parser->name = NULL;
	
	parser->start = 0;
	parser->end = 0;
	
	parser->tag = screem_markup_next_tag( parser->offset_text, 
				0, 
				&parser->start, &parser->end, 
				&parser->name );

	parser->poffset_text = parser->offset_text;
	parser->offset_text = 
		g_utf8_offset_to_pointer( parser->offset_text, 
					  parser->end + 1 );

	parser->start += parser->pos;
	parser->end += parser->pos + 1;
	
	if( ! parser->tag ) {
		parser->pos = parser->last;
	} else {
		parser->pos = parser->start;
	}

gdk_threads_enter();

	if( parser->pos > parser->ppos ) {
		/* TODO?
		 *
		 * If we are in a <script> we could possibly
		 * split the text node based on blocks, 
		 * e.g. { } 
		 *
		 * Likewise if we are in <style> we could
		 * split out the individual CSS classes etc
		 */
		screem_page_model_build_text_node( parser );
	}
		
	if( parser->tag ) {
		parser->pos = parser->end;
		if( *parser->name == '/' ) {
			screem_page_model_build_close( parser );
		} else {
			screem_page_model_build_open( parser );
		}
	}
	g_free( parser->name );
	g_free( parser->tag );

	if( parser->pos >= parser->last ) {
		ret = FALSE;
		parser->sourceid = 0;

		pmodel = SCREEM_PAGE_MODEL( parser->model );
		screem_page_model_parser_cleanup( parser );
		screem_page_model_emit_built( pmodel->private->page );
	}

	gdk_threads_leave();
/*	}
	end = clock();
	g_print( "TIME: %f\n", ( end - start ) * 1.0 / CLOCKS_PER_SEC );*/
	
	return ret;
}

static void screem_page_model_set_invalid( ScreemPage *page,
				guint start, guint end,
				const gchar *name,
				const gchar *pname,
				const gchar *attr,
				const gchar *value )
{
	GtkTextTagTable *table;
	GtkTextTag *tag;
	GtkTextIter it;
	GtkTextIter eit;
	GConfClient *client;
	gboolean highlight;
	
	client = gconf_client_get_default();
	highlight = gconf_client_get_bool( client,
			"/apps/screem/editor/error_highlight",
			NULL );
	g_object_unref( client );

	if( highlight ) {
		table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
		tag = gtk_text_tag_table_lookup( table, 
				SCREEM_INVALID_MARKUP_TAG );
		if( tag ) {
			gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( page ),
							&it, start );
			gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( page ),
							&eit, end );
			gtk_text_buffer_apply_tag( GTK_TEXT_BUFFER( page ), tag,
						&it, &eit );
		}
	}
}

static gboolean select_context( ScreemPageModel *model,
				GtkTreeIter **it,
				guint pos,
				guint *start, guint *end,
				gboolean select_text,
				gboolean build )
{
	ScreemPageModelPrivate *priv;
	GtkTreeModel *tmodel;
	gboolean loop;
	GtkTreeIter *tmp;
	GtkTreeIter root;
	
	priv = model->private;
	tmodel = GTK_TREE_MODEL( model );

	if( build ) {
		/* leave gdk lock, build_tree() is generally
		 * called via timeout so claims the lock itself */
		gdk_threads_leave();
		build_tree( model );
gdk_threads_enter();
	}
	screem_page_model_ensure_built( model );
	
	loop = TRUE;
	if( ! it ) {
		if( priv->context ) {
			gtk_tree_iter_free( priv->context );
			priv->context = NULL;
		}

		loop = gtk_tree_model_get_iter_first( tmodel, &root );
		if( loop && gtk_tree_model_iter_has_child( tmodel, &root ) ) {
			GtkTreeIter *parent;
			
			parent = gtk_tree_iter_copy( &root );

			loop = gtk_tree_model_iter_children( tmodel, &root, 
							     parent );
			if( loop ) {
				tmp = &root;
				it = &tmp;
			}
			gtk_tree_iter_free( parent );
		} else {
			loop = FALSE;
		}
	}
	
	while( loop ) {
		gint estart;
		gint eend;

		gtk_tree_model_get( tmodel, *it, 
				SCREEM_PAGE_MODEL_START, &estart,
				SCREEM_PAGE_MODEL_END, &eend, -1 );

		if( pos <= eend && pos >= estart ) {
			/* in here somewhere */
			gboolean istext;
			GValue value = { 0 };

			gtk_tree_model_get_value( tmodel, *it,
						  SCREEM_PAGE_MODEL_TEXT,
						  &value );
			istext = ( g_value_get_string( &value ) != NULL );
			g_value_unset( &value );

			if( ! istext || select_text ) {
				*start = estart;
				*end = eend + 1;
				if( priv->context ) {
					gtk_tree_iter_free( priv->context );
				}
				priv->context = 
					gtk_tree_iter_copy( *it );
			}
			
			if( gtk_tree_model_iter_has_child( tmodel, *it ) ) {
				GtkTreeIter child;
				
				if( gtk_tree_model_iter_children( tmodel,
								  &child,
								  *it ) ) {
					GtkTreeIter *cptr;
					
					cptr = &child;
					select_context( model, 
							&cptr, pos,
							start, end,
							select_text,
							FALSE );
				}
			}
			break;
		} else {
			loop = gtk_tree_model_iter_next( tmodel, *it );
		}
	}

	return loop;
}

/* G Object stuff */
#define PARENT_TYPE GTK_TYPE_TREE_STORE

static gpointer parent_class;

static void screem_page_model_class_init( ScreemPageModelClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_page_model_finalize;
	object_class->get_property = screem_page_model_get_prop;
	object_class->set_property = screem_page_model_set_prop;

	g_object_class_install_property(object_class,
					PROP_APPLICATION,
					g_param_spec_object("application",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE|
							    G_PARAM_CONSTRUCT)
					);
	g_object_class_install_property(object_class,
					PROP_PAGE,
					g_param_spec_object("page",
							    "Page",
							    "The Page",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_DTD,
					g_param_spec_object("dtd",
							    "DTD",
							    "The page DTD",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_CHARSET,
					g_param_spec_string("charset",
							    "Page character set",
							    "The character set of the page",
							    "",
							    G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_DOCTYPE,
					g_param_spec_string("doctype",
							    "Page doctype",
							    "The doctype of the page",
							    "",
							    G_PARAM_READWRITE)
					
					);

}

static void screem_page_model_init( ScreemPageModel *model )
{
	GConfClient *client;
	GType model_types[] = {
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT,
		G_TYPE_UINT,
		G_TYPE_BOOLEAN,
		GDK_TYPE_PIXBUF
	};
	GtkTreeIter it;
	
	model->private = g_new0( ScreemPageModelPrivate, 1 );

	client = gconf_client_get_default();
	model->private->charset = 
		gconf_client_get_string( client,
					"/apps/screem/editor/default_charset",
					NULL );
	g_object_unref( client );

	gtk_tree_store_set_column_types( GTK_TREE_STORE( model ),
					 SCREEM_PAGE_MODEL_MAX,
					 model_types );

	gtk_tree_store_append( GTK_TREE_STORE( model ), &it, NULL );
	gtk_tree_store_set( GTK_TREE_STORE( model ), &it,
			    SCREEM_PAGE_MODEL_NAME, "[Document]",
			    SCREEM_PAGE_MODEL_VALID, TRUE,
			    -1 );
}

static void screem_page_model_finalize( GObject *object )
{
	ScreemPageModel *model;

	model = SCREEM_PAGE_MODEL( object );

	screem_page_model_parser_cleanup( &model->private->parser );
	
	g_free( model->private->charset );
	g_free( model->private->doctype );

	g_free( model->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_page_model_set_prop( GObject *object, guint prop_id,
					const GValue *value, GParamSpec *spec )
{
	ScreemPageModel *model;
	const gchar *str;
	ScreemDTDDB *db;
	
	model = SCREEM_PAGE_MODEL( object );

	switch( prop_id ) {
		case PROP_APPLICATION:
			model->private->application = SCREEM_APPLICATION( g_value_get_object( value ) );
			db = screem_application_get_dtd_db( model->private->application );
	
			model->private->dtd = screem_dtd_db_get_default_dtd( db );
			break;
		case PROP_PAGE:
			model->private->page = SCREEM_PAGE( g_value_get_object( value ) );
			screem_page_model_page_set( model );
			break;
		case PROP_DTD:
			model->private->dtd = SCREEM_DTD( g_value_get_object( value ) );
			g_object_set( G_OBJECT( model->private->page ), "dtd",
					model->private->dtd, NULL );
			break;
		case PROP_CHARSET:
			g_free( model->private->charset );
			model->private->charset = NULL;
			str = g_value_get_string( value );
			if( str ) {
				model->private->charset = g_strdup( str );
			}
			break;
		case PROP_DOCTYPE:
			g_free( model->private->doctype );
			model->private->doctype = NULL;
			str = g_value_get_string( value );
			if( str ) {
				model->private->doctype = g_strdup( str );
			}
			screem_page_model_dtd_set( model );
			break;
		default:
			g_warning( "Unknown property\n" );
	}
}

static void screem_page_model_get_prop( GObject *object, guint prop_id,
				  	GValue *value, GParamSpec *spec )
{
	ScreemPageModel *model;

	model = SCREEM_PAGE_MODEL( object );

	switch( prop_id ) {
		case PROP_APPLICATION:
			g_value_set_object( value, model->private->application );
			break;
		case PROP_PAGE:
			g_value_set_object( value, model->private->page );
			break;
		case PROP_DTD:
			g_value_set_object( value, model->private->dtd );
			break;
		case PROP_CHARSET:
			if( model->private->charset ) {
				g_value_set_string( value, model->private->charset );
			}
			break;
		case PROP_DOCTYPE:
			if( model->private->doctype ) {
				g_value_set_string( value, model->private->doctype );
			}
			break;
		default:
			break;
	}
}

GType screem_page_model_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemPageModelClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_page_model_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemPageModel ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_page_model_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemPageModel",
					       &info, 0 );
	}

	return type;
}
