/*  Screem:  screem-helper.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>

#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <gtk/gtkdialog.h>

#include "screem-window.h"
#include "screem-helper.h"
#include "screem-page.h"

#include "libegg/menu/egg-menu.h"

#include "screemmarshal.h"

static void screem_helper_class_init( ScreemHelperClass *klass );
static void screem_helper_init( ScreemHelper *helper );
static void screem_helper_finalize( GObject *object );

static void screem_helper_set_property(GObject *object, guint prop_id,
				       const GValue *value, GParamSpec *pspec);
static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec );
static void screem_helper_verb_cb( EggAction *action, gpointer user_data );

static gboolean prepare( GSource *source, gint *timeout );
static gboolean check( GSource *source );
static gboolean dispatch( GSource *source, GSourceFunc cb,
			  gpointer user_data );
static void destroy( GSource *source );
static gboolean read_from_pipe( gpointer data );
static gboolean err_from_pipe( gpointer data );

enum {
	PROP_BOGUS,
	PROP_PATHNAME,
	PROP_ICONPATH,
	PROP_HELPER_NAME,
	PROP_INPUT_MODE,
	PROP_OUTPUT_MODE
};

enum {
	HELPER_FAILED,
	HELPER_ERROR,
	LAST_SIGNAL
};
static guint screem_helper_signals[LAST_SIGNAL] = { 0 };

struct ScreemHelperPrivate {
	/* a list of ScreemWindows */
	GSList *list;

	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	HelperInputMode imode;
	HelperOutputMode omode;

	/* the following two variables are set in screem_helper_execute(), 
	   only valid in that context + the source cb */
	ScreemPage *page;

      	pid_t pid;
	gint inpipe[ 2 ];
	gint outpipe[ 2 ];
	gint errpipe[ 2 ];
};

typedef struct {
	GSource source;
	GPollFD fd;
	ScreemHelper *helper;
	ScreemWindow *window;
	GString *buffer;
	GMainLoop *loop;
} ScreemHelperSource;

ScreemHelper* screem_helper_new( const gchar *name, 
				 const gchar *pathname,
				 const gchar *iconpath,
				 HelperInputMode imode,
				 HelperOutputMode omode )
{
	ScreemHelper *helper;
	GType type;
	
	type = screem_helper_get_type();

	helper = SCREEM_HELPER( g_object_new( type, 
					      "name", name,
					      "pathname", pathname,
					      "iconpath", iconpath,
					      "input", imode,
					      "output", omode,
					      NULL ) );

	return helper;
}

void screem_helper_execute( ScreemHelper *helper, ScreemWindow *window )
{
	gchar *pathname;
	ScreemHelperPrivate *private;
	ScreemPage *page;
	FILE *out;

	gint status;

	GMainLoop *loop;
	GMainContext *context;
	ScreemHelperSource *hsource;
	ScreemHelperSource *esource;
	GSource *source;
	GSource *source2;
	GSourceFuncs funcs = { prepare, check, dispatch, destroy };
	gchar *data;

	HelperInputMode imode;
	HelperOutputMode omode;

	GConfClient *client;
	gchar *gladepath;
	GladeXML *xml;
	GtkWidget *widget;

	gchar *exec;

	const gchar *pagepath;
	
	client = gconf_client_get_default();
	
	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	
	
	private = helper->private;
	page = private->page = screem_window_get_document( window );

	g_object_get( G_OBJECT( helper ), 
		      "pathname", &pathname, 
		      "input", &imode,
		      "output", &omode,
		      NULL );

	pipe( private->inpipe );
	pipe( private->outpipe );
	pipe( private->errpipe );
	switch( ( private->pid = fork() ) ) {
	case -1:
		/* error */
		break;
	case 0:
		/* child */
		close( 0 );
		dup( private->outpipe[ 0 ] );
		close( private->outpipe[ 1 ] );

		close( 1 );
		dup( private->inpipe[ 1 ] );
		close( private->inpipe[ 0 ] );

		close( 2 );
		dup( private->errpipe[ 1 ] );
		close( private->errpipe[ 0 ] );

		exec = strstr( pathname, " %f" );
		if( exec ) {
			gchar *temp;
	
			pagepath = screem_page_get_pathname( page );
			if( ! strncmp( "file://", pagepath,
					strlen( "file://" ) ) ) {

				pagepath += strlen( "file://" );
			}
			
			temp = g_strndup( pathname, exec - pathname );
			exec = g_strconcat( temp,
					    " ", pagepath,
					    exec + strlen( " %f" ),
					    NULL );
			g_free( temp );
			
		} else {
			exec = g_strdup( pathname );
		}
		
		
		execl( "/bin/sh", "/bin/sh", "-c", exec, NULL );
		_exit( 0 );
		break;
	default:
		/* parent */
		close( private->inpipe[ 1 ] );
		close( private->outpipe[ 0 ] );
		close( private->errpipe[ 1 ] );

		switch( imode ) {
		case SCREEM_HELPER_STDIN_SELECTION:
			out = fdopen( private->outpipe[ 1 ], "w" );
			data = screem_window_get_selection( window );
			fprintf( out, "%s", data );
			g_free( data );
			fclose( out );
			break;
		case SCREEM_HELPER_STDIN_USER:
			out = fdopen( private->outpipe[ 1 ], "w" );
			xml = glade_xml_new( gladepath, "helper_user_input",
						NULL );
			widget = glade_xml_get_widget( xml, "helper_name" );
			gtk_label_set_text( GTK_LABEL( widget ), private->name );
			widget = glade_xml_get_widget( xml, 
							"helper_user_input" );
			if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
				GtkWidget *widg;

				widg = glade_xml_get_widget( xml, 
							     "use_selection" );
				if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widg ) ) ) {
					data = screem_window_get_selection( window );
				} else {
					GtkTextBuffer *buffer;
					GtkTextIter it;
					GtkTextIter eit;
					
					widg = glade_xml_get_widget( xml, "user_data" );
					buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widg ) );
					gtk_text_buffer_get_bounds( buffer, &it, &eit );
					data = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
				}
				fprintf( out, "%s", data );
				g_free( data );
				fclose( out );
			} else {
				/* canceled, stop the helper */
				kill( private->pid, SIGKILL );
			}
			gtk_widget_destroy( widget );
		
			g_object_unref( xml );

			break;
		case SCREEM_HELPER_STDIN_NONE:
			close( private->outpipe[ 1 ] );
			break;
		}

		/* we start a new g_main_loop() and add
		   the file descriptors to be listened to in
		   the main loop */
		loop = g_main_loop_new( g_main_context_default(), FALSE );
		context = g_main_loop_get_context( loop );

		source = g_source_new( &funcs, sizeof( ScreemHelperSource ) );
		hsource = (ScreemHelperSource*)source;
		hsource->fd.fd = private->inpipe[ 0 ];
		hsource->fd.events = G_IO_IN | G_IO_HUP | G_IO_ERR;
		hsource->helper = helper;
		hsource->window = window;
		hsource->buffer = g_string_new( NULL );
		hsource->loop = loop;
		g_source_set_callback( source, read_from_pipe,
				       NULL, NULL );
		g_source_add_poll( source, &hsource->fd );
		g_source_attach( source, context );

		source2 = g_source_new( &funcs, sizeof( ScreemHelperSource ) );
		esource = (ScreemHelperSource*)source2;
		esource->fd.fd = private->errpipe[ 0 ];
		esource->fd.events = G_IO_IN | G_IO_HUP | G_IO_ERR;
		esource->helper = helper;
		esource->window = window;
		esource->buffer = NULL;
		esource->loop = loop;
		g_source_set_callback( source2, err_from_pipe,
				       NULL, NULL );
		g_source_add_poll( source2, &esource->fd );
		g_source_attach( source2, context );

		GDK_THREADS_LEAVE();
		g_main_loop_run( loop );
		GDK_THREADS_ENTER();

		waitpid( private->pid, &status, 0 );

		data = g_strdup( hsource->buffer->str );

		g_source_destroy( source );
		g_source_destroy( source2 );

		g_main_loop_unref( loop );

		close( private->inpipe[ 0 ] );
		close( private->errpipe[ 0 ] );

		if( WIFEXITED( status ) != 0 ) {
			if( status == 0 ) {
				/* success */
				switch( omode ) {
				case SCREEM_HELPER_STDOUT_SELECTION:
					screem_window_replace_selection( window, data );
					break;
				case SCREEM_HELPER_STDOUT_INSERT:
					break;
				case SCREEM_HELPER_STDOUT_NONE:
					/* will have been displayed in
					   the windows message display
					   by the callback */
					break;
				}
			} else {
				/* helper returned an error */
				g_signal_emit( G_OBJECT( helper ),
					       screem_helper_signals[ HELPER_ERROR ], 0, status );
			}
		} else {
			/* helper didn't exit normally */
			g_signal_emit( G_OBJECT( helper ),
				       screem_helper_signals[ HELPER_FAILED ],
				       0 );
		}
		g_free( data );

		break;
	}

	g_free( pathname );
	g_free( gladepath );
	g_object_unref( client );
}

void screem_helper_add_to_window( ScreemHelper *helper, ScreemWindow *window )
{
	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	gchar *uiname;
	gchar *ui;
	static EggActionGroupEntry entries[] = {
		{ "action", "label",
	  	GTK_STOCK_EXECUTE, NULL, "tip",
	  	G_CALLBACK( screem_helper_verb_cb ), NULL },
	};
	EggActionGroupEntry *entry;
	EggAction *action;
	guint merge_id;

	helper->private->list = 
		g_slist_append( helper->private->list, window );
	
	g_object_get( G_OBJECT( helper ),
		      "name", &name, "pathname", &pathname, 
		      "iconpath", &iconpath, NULL );

	uiname = g_strconcat( "Exec_Helper_", name, NULL );

	entry = entries;
	entry->name = uiname;
	entry->label = g_strdup( name );
	entry->tooltip = g_strdup( pathname );
	entry->user_data = helper;
	
	ui = g_strconcat( "<Root><menu><submenu name=\"Tools\" verb=\"Tools\"><submenu name=\"Helpers\" verb=\"Helpers\">",
			  "<menuitem name=\"", uiname, "\" verb=\"", uiname, "\" />",
			  "</submenu></submenu></menu></Root>",
			  NULL );
			  
	egg_action_group_add_actions( EGG_ACTION_GROUP( window->action_group ),
					entry, 1 );
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						uiname );
	merge_id = egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( window->merge ),
							ui, strlen( ui ), NULL );
	g_object_set_data( G_OBJECT( action ), "merge_id",
				GUINT_TO_POINTER( merge_id ) );

	g_object_set_data( G_OBJECT( action ), "window", window );

	g_free( uiname );
	g_free( ui );

	g_free( pathname );
	g_free( name );
	g_free( iconpath );
	
	egg_menu_merge_ensure_update( EGG_MENU_MERGE( window->merge ) );
}

void screem_helper_remove_from_window( ScreemHelper *helper,
				       ScreemWindow *window )
{
	gchar *name;
	gchar *uiname;
	EggAction *action;
	guint merge_id;

	g_return_if_fail( SCREEM_IS_HELPER( helper ) );
	
	helper->private->list = g_slist_remove( helper->private->list, 
						window );

	g_object_get( G_OBJECT( helper ), "name", &name, NULL );

	uiname = g_strconcat( "Exec_Helper_", name, NULL );

	g_free( name );
	
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						uiname );
	merge_id = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( action ),
					"merge_id" ) );
	egg_menu_merge_remove_ui( EGG_MENU_MERGE( window->merge ), merge_id );
	egg_action_group_remove_action( EGG_ACTION_GROUP( window->action_group ),
					action );
	g_free( uiname );

	egg_menu_merge_ensure_update( EGG_MENU_MERGE( window->merge ) );
}


gchar *screem_helper_get_gconf_path( ScreemHelper *helper )
{
	const gchar *base_path = "/apps/screem/helpers_data/";
	gchar *name;
	gchar *path;

	if( helper ) {
		g_object_get( G_OBJECT( helper ), "name", &name, NULL );
	} else {
		name = NULL;
	}

	path = g_strconcat( base_path, name, NULL );

	g_free( name );
	
	return path;
}

/* static stuff */
static void screem_helper_verb_cb( EggAction *action, gpointer user_data )
{
	ScreemHelper *helper;
	ScreemWindow *window;
	
	helper = SCREEM_HELPER( user_data );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( action ), "window" ) );
	screem_helper_execute( helper, window );
}


static gboolean prepare( GSource *source, gint *timeout )
{
	*timeout = 0;

	return TRUE;
}

static gboolean check( GSource *source )
{
	ScreemHelperSource *hsource;

	hsource = (ScreemHelperSource*)source;

	if( hsource->buffer &&
	    ( (hsource->fd.revents & G_IO_HUP) || 
	      (hsource->fd.revents & G_IO_ERR) ) ) {
		g_main_loop_quit( hsource->loop );
	}
	
	return ( ! hsource->buffer || hsource->fd.revents & G_IO_IN );
}

static gboolean dispatch( GSource *source, GSourceFunc cb,
			  gpointer user_data )
{
	g_assert( cb );

	return (*cb)( source );
}

static void destroy( GSource *source )
{
	ScreemHelperSource *hsource = (ScreemHelperSource*)source;

	if( hsource->buffer ) {
		g_string_free( hsource->buffer, TRUE );
	}
}

static gboolean read_from_pipe( gpointer data )
{
	ScreemHelperSource *source = (ScreemHelperSource*)data;
	ScreemHelper *helper;
	gint size;
	gchar buffer[ BUFSIZ + 1 ];

	helper = SCREEM_HELPER( source->helper );

	size = read( source->fd.fd, buffer, BUFSIZ );
	if( size > 0 ) {
		buffer[ size ] = '\0';
		if( helper->private->omode == SCREEM_HELPER_STDOUT_NONE ) {
			screem_window_show_message( source->window,
						    buffer );
		} else {
			g_string_append( source->buffer, buffer );
		}
	} else if( errno != EINTR ) {
		g_main_loop_quit( source->loop );
	}

	return TRUE;
}

static gboolean err_from_pipe( gpointer data )
{
	ScreemHelperSource *source = (ScreemHelperSource*)data;
	ScreemHelper *helper;
	ScreemWindow *window;
	gint size;
	gchar buffer[ BUFSIZ ];

	helper = SCREEM_HELPER( source->helper );
	window = SCREEM_WINDOW( source->window );

	size = read( source->fd.fd, buffer, BUFSIZ );
	if( size > 0 ) {
		buffer[ size ] = '\0';
		screem_window_show_error( window, buffer );
	} else if( errno != EINTR ) {
	}

	return TRUE;
}

/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_helper_class_init( ScreemHelperClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_helper_finalize;

	object_class->set_property = screem_helper_set_property;
	object_class->get_property = screem_helper_get_property;

	g_object_class_install_property( object_class,
					 PROP_PATHNAME,
					 g_param_spec_string("pathname",
							     "Helper Path",
							     "The Pathname to the Helper",
							     "",
							     G_PARAM_READWRITE)
					 );
	g_object_class_install_property( object_class,
					 PROP_ICONPATH,
					 g_param_spec_string("iconpath",
							     "Icon Path",
							     "The Pathname to the image file for the Helper icon",
							     "",
							     G_PARAM_READWRITE)
					 );
	g_object_class_install_property( object_class,
					 PROP_HELPER_NAME,
					 g_param_spec_string("name",
							     "Helper name",
							     "The name of the Helper",
							     "",
							     G_PARAM_READWRITE)
					 );
	g_object_class_install_property( object_class,
					 PROP_INPUT_MODE,
					 g_param_spec_int( "input",
							   "Input Mode",
							   "The input mode of the Helper",
							   SCREEM_HELPER_STDIN_NONE,
							   SCREEM_HELPER_STDIN_USER,
							   SCREEM_HELPER_STDIN_SELECTION,
							   G_PARAM_READWRITE )
					 );
	g_object_class_install_property( object_class,
					 PROP_OUTPUT_MODE,
					 g_param_spec_int( "output",
							   "Output Mode",
							   "The output mode of the Helper",
							   SCREEM_HELPER_STDOUT_NONE,
							   SCREEM_HELPER_STDOUT_INSERT,
							   SCREEM_HELPER_STDOUT_SELECTION,
							   G_PARAM_READWRITE )
					 );


	screem_helper_signals[ HELPER_FAILED ] = 
		g_signal_new( "helper_failed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_failed ),
			      NULL, NULL,
			      screem_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );

	screem_helper_signals[ HELPER_ERROR ] = 
		g_signal_new( "helper_error",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_error ),
			      NULL, NULL,
			      screem_marshal_VOID__INT,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT );
}

static void screem_helper_init( ScreemHelper *helper )
{
	helper->private = g_new0( ScreemHelperPrivate, 1 );

}

static void screem_helper_finalize( GObject *object )
{
	ScreemHelper *helper;

	helper = SCREEM_HELPER( object );

	/* we need to remove outselves from all windows */
	while( helper->private->list ) {
		ScreemWindow *window = 
			SCREEM_WINDOW( helper->private->list->data );

		screem_helper_remove_from_window( helper, window );
	}

	if( helper->private->name )
		g_free( helper->private->name );
	if( helper->private->pathname )
		g_free( helper->private->pathname );
	if( helper->private->iconpath )
		g_free( helper->private->iconpath );

	g_free( helper->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}


static void screem_helper_set_property( GObject *object, guint prop_id,
					const GValue *value, GParamSpec *pspec)
{
	ScreemHelper *helper;
	ScreemHelperPrivate *private;
	const gchar *val;
	gint mode;
	
	helper = SCREEM_HELPER( object );
	private = helper->private;

	switch( prop_id ) {
	case PROP_PATHNAME:
		val = g_value_get_string( value );
		if( private->pathname )
			g_free( private->pathname );
		if( val ) {
			private->pathname = g_strdup( val );
		} else {
			private->pathname = NULL;
		}
		break;
	case PROP_ICONPATH:
		val = g_value_get_string( value );
		if( private->iconpath )
			g_free( private->iconpath );
		if( val ) {
			private->iconpath = g_strdup( val );
		} else {
			private->iconpath = NULL;
		}
		break;
	case PROP_HELPER_NAME:
		{
			/* we can't allow private->name to contain
			   spaces */
			GString *string;
			
			string = g_string_new( NULL );
			val = g_value_get_string( value );

			while( *val ) {
				gboolean space;

				space = isspace( *val );
				
				if( ! space && ( *val != '_' ) ) {
					g_string_append_c( string, *val );
				} else if( space || ( *val == '_' &&
						      *( val + 1 ) != '_' ) ) {
					/* append __ as _ needs escaping */
					g_string_append( string, "__" );
				} else {
					g_string_append( string, "__" );
					val ++;
				}
				val ++;
			}
			
			if( private->name ) {
				g_free( private->name );
			}
			private->name = string->str;
			g_string_free( string, FALSE );
			break;
		}
	case PROP_INPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDIN_USER ) {
			mode = SCREEM_HELPER_STDIN_NONE;
		}
		private->imode = mode;
		break;
	case PROP_OUTPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDOUT_INSERT ) {
			mode = SCREEM_HELPER_STDOUT_NONE;
		}
		private->omode = mode;
		break;
	default:
		break;
	}
}

static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *private;

	helper = SCREEM_HELPER( object );
	private = helper->private;

	switch( prop_id ) {
	case PROP_PATHNAME:
		if( private->pathname ) {
			g_value_set_string( value, private->pathname );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_ICONPATH:
		if( private->iconpath ) {
			g_value_set_string( value, private->iconpath );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_HELPER_NAME:
		if( private->name ) {
			g_value_set_string( value, private->name );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_INPUT_MODE:
		g_value_set_int( value, private->imode );
		break;
	case PROP_OUTPUT_MODE:
		g_value_set_int( value, private->omode );
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID( object, prop_id, pspec );
		break;
	}
}



GType screem_helper_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemHelperClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_helper_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemHelper ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_helper_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemHelper",
					       &info, 0 );
	}

	return type;
}
