/*  Screem:  screem-editor.c
 *
 *  The editor widget
 *
 *  Copyright (C) 1999-2003  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <ctype.h>

#include <gmodule.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <gtksourceview/gtksourceview.h>
#include <gtksourceview/gtksourcebuffer.h>
#include <gtksourceview/gtksourceprintjob.h>
#include <gtk/gtkselection.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkdnd.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmain.h>
#include <gtk/gtktable.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkeventbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtksizegroup.h>
#include <gtk/gtktooltips.h>

#include <gdk/gdkkeysyms.h>

#include <gconf/gconf-client.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-url.h>

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-editor.h"
#include "screem-site.h"
#include "screem-markup.h"
#include "screem-page.h"
#include "screem-plugin.h"
#include "screem-search.h"

#include "screem-tagtree.h"

#include "fileops.h"
#include "support.h"

#include "pageUI.h"

#include "libegg/menu/egg-menu.h"

#include "screemmarshal.h"

enum {
	DUMMY_SIGNAL,
	LAST_SIGNAL
};

static guint screem_editor_signals[ LAST_SIGNAL ] = { 0 };

typedef enum _Drop_actions {
        RELATIVE_PATH,
        FULL_PATH,
        TAG,
        ATTRIBUTE,
        INLINE     /* inserts the file contents (text files only) */
} Drop_actions;

typedef struct {
	ScreemEditor *editor;
	guint pos;
	gboolean adj;
	gdouble vadj;
} ScreemEditorDelaySetPos;

typedef void (*AutocompleteAfterFunc)( ScreemEditor *editor );

static void screem_editor_init( ScreemEditor *editor );
static void screem_editor_class_init( ScreemEditorClass *editor );
static void screem_editor_finalize( GObject *object );
static void screem_editor_window_set( ScreemView *view );
static void screem_editor_size_request( GtkWidget *widget, 
					GtkRequisition *req );
static void screem_editor_realize( GtkWidget *widget );
static void screem_editor_show( GtkWidget *widget );
static void screem_editor_hide( GtkWidget *widget );


static void screem_editor_display( ScreemView *view );
static void screem_editor_display_page( ScreemEditor *editor, 
					ScreemPage *page );

static gboolean screem_editor_mark_set( GtkTextBuffer *buffer,
					const GtkTextIter *it,
					GtkTextMark *mark,
					gpointer data );
static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      ScreemEditor *editor );
static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time, ScreemEditor *editor );

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event,
					ScreemEditor *editor );
gboolean html_key_press( ScreemEditor *editor, GdkEventKey *event, 
			 ScreemEditorMacro *macro );

static gboolean screem_editor_tip( ScreemEditor *editor );
static gboolean screem_editor_tooltip( ScreemEditor *editor );

static void screem_editor_colour_notify( GConfClient *client,
					 guint cnxn_id,
					 GConfEntry *entry,
					 gpointer data );
static void screem_editor_wrap_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_tabwidth_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_autocomplete_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );

static void screem_editor_clip_received( GtkClipboard *clipboard,
					 GtkSelectionData *selection_data,
					 gpointer data );

static void screem_editor_toggle_overwrite( GtkTextView *view,
					    ScreemEditor *editor );

static gboolean build_attribute_list( ScreemEditor *editor );
static void build_attribute_entries( ScreemEditor *editor,
				     ScreemDTD *dtd, const gchar *name,
				     GtkWidget *box,
				     GSList *attr );
static void replace_attr( GtkWidget *entry, gchar *attr_name );


static gboolean screem_editor_switch_mime( GtkWidget *widget, 
					   GdkEventButton *event,
					   ScreemEditor *editor );
static void screem_editor_mime_switch( GtkWidget *widget, 
				       ScreemEditor *editor );

static gboolean html_editor_tip( ScreemEditor *editor, ScreemWindow *window,
				 gint pos, gchar *text );

static gboolean screem_editor_drop_uris( ScreemEditor *editor,
					 Drop_actions action,
					 const gchar *selectionData,
					 guint info );

static gboolean set_pos( ScreemEditorDelaySetPos *delay );
static void screem_editor_set_pos_delayed( ScreemEditor *editor, 
					guint pos, 
					gboolean adj, gdouble val );

static void screem_editor_buffer_change( ScreemPage *page, GParamSpec *spec,
					 ScreemEditor *editor );

static gboolean screem_editor_autocomplete_popup_event( GtkWidget *widget, 
							GdkEventKey *event,
							gpointer data );
static void screem_editor_autocomplete_popup( ScreemEditor *editor, 
					      const GSList *list, 
					      guint32 term,
					      const gchar *str,
					      AutocompleteAfterFunc after );

static void screem_editor_attr_after( ScreemEditor *editor );

static GtkWidget *screem_editor_attribute_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element );
static GtkWidget *screem_editor_allowed_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element );
static void screem_editor_insert_attr_cb( GtkWidget *item, const gchar *attribute );
static void screem_editor_insert_attribute( ScreemEditor *editor, 
				     const gchar *attribute );

static void screem_editor_learn_more( GtkWidget *widget,
				ScreemEditor *editor );

struct ScreemEditorPrivate {
	GConfClient *client;

	/* previous cursor position mark name, based
	 * off the name of the window we are in */
	gchar *markname;
	
	/* the text widget */
	GtkWidget *view;
	GtkWidget *sw;

	/* current page */
	ScreemPage *page;

	gboolean popup_open;
	gboolean inserted;

	/* screem_editor_cursor_moved() timeout handle */
	gint handle;

	/* previous position for build_attributes_list */
	gint ppos;

	/* gconf notify handles, back, text, font,
	 * themecolours, themefont, wordwrap, tab width,
	 * autocomplete */
	gint notifies[ 8 ];

	gint tooltip_timer;
	
	GtkWidget *attr_menu;

	GtkWidget *overwriteMode;
	GtkWidget *cursorPosition;
	GtkWidget *mime;

	gboolean overwrite;


	GtkWidget *attrtab;
	GtkWidget *attributes;

	GtkWidget *types;

	gboolean autocomplete;
};

typedef enum _Drop_types {
	TARGET_URI_LIST,
	TARGET_URL,
	TARGET_MOZ,
	TARGET_COLOUR,
	TARGET_TEXT,
	TARGET_UNICODE,
} Drop_types;

static const GtkTargetEntry drop_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST },
	{ "text/x-moz-url", 0, TARGET_MOZ },
        { "_NETSCAPE_URL", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_URL },
        { "x-url/ftp", 0, TARGET_URL },
        { "application/x-color", 0, TARGET_COLOUR },
	{ "text/plain", 0, TARGET_TEXT },
	{ "UTF8_STRING", 0, TARGET_TEXT },
	{ "text/unicode", 0, TARGET_UNICODE }
};
static const gint num_drop_types = sizeof(drop_types) / sizeof(drop_types [0]);

/* DnD menu */
static GnomeUIInfo editor_dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("Insert relative filename"),
          N_("Insert relative filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert complete filename"),
          N_("Insert complete filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert tag"), N_("Insert tag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert tag attibute"),
          N_("If the drop is into a tag then the data is converted into\
 an attribute for that tag"), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert inline"),
          N_("Insert the text from the file into the page"), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

typedef gboolean (*Tipcb)( ScreemEditor *editor, gint pos, gchar *text );

static gboolean screem_editor_popup_key( GtkWidget *widget, 
						ScreemEditor *editor );
static gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event,
					ScreemEditor *editor);

enum {
	ON = 0,
	OFF,
	BY_SET
};

ScreemEditor *screem_editor_new( ScreemWindow *window )
{
	ScreemEditor *editor;
	GType type;

	type = screem_editor_get_type();

	editor = SCREEM_EDITOR( g_object_new( type, "window", window,
					      NULL ) );

	return editor;
}

static gboolean screem_editor_popup_key( GtkWidget *widget, ScreemEditor *editor )
{
	screem_editor_popup( widget, NULL, editor );

	return TRUE;
}

gboolean screem_editor_button_release( GtkWidget *widget, 
				       GdkEventButton *event,
				       ScreemEditor *editor )
{
	ScreemEditorPrivate *private;

	private = editor->private;

	if( event && event->button == 1 &&
	    event->window ==
	    gtk_text_view_get_window( GTK_TEXT_VIEW( private->view ),
				      GTK_TEXT_WINDOW_LEFT ) ) {
		/* left click on the line numbers,
		   we want to be able to drag and select
		   entire lines, so lets cheat on pass the
		   event onto the main window, this callback is needed
		   so the drag is released properly */
		GdkEventButton child_event;

		child_event = *event;
		child_event.window =
			gtk_text_view_get_window( GTK_TEXT_VIEW(private->view),
						  GTK_TEXT_WINDOW_TEXT );
		gtk_widget_event( private->view, (GdkEvent*)&child_event );
	}

	return FALSE;
}

static gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event,
					ScreemEditor *editor )
{
	gint cpos;
	gint pos2;
    	gint pos;
	gint tag_pos;
	gint tag_len;
	gchar *text;
	gchar *tag;
	gchar *tag2;
	GtkWidget *menu;
	GtkWidget *submenu;
	GtkWidget *menu_item;
	GtkTextBuffer *buffer;

	gboolean tag_known;

	ScreemPage *page;
	ScreemDTD *dtd;

	const gchar *pathname;
	const gchar *mime_type;

	gint start;
	gint len;
	gboolean sel;

	ScreemEditorPrivate *private;
	ScreemApplication *app;
	ScreemWindow *window;
	
	gboolean feature_markup;

	GSList *addeditems = NULL;
	GSList *tmp;

	GtkTextTagTable *table;
	GtkTextTag *itag;
	GtkTextIter it;
	
	private = editor->private;
	page = private->page;

	if( ! page ) {
		return FALSE;
	}

	table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
	itag = gtk_text_tag_table_lookup( table, 
			SCREEM_INVALID_MARKUP_TAG );
	
	pathname = screem_page_get_pathname( page );

	if( event && event->button == 1 &&
	    event->window ==
	    gtk_text_view_get_window( GTK_TEXT_VIEW( private->view ),
				      GTK_TEXT_WINDOW_LEFT ) ) {
		/* left click on the line numbers,
		   we want to be able to drag and select
		   entire lines, so lets cheat on pass the
		   event onto the main window */
		GdkEventButton child_event;

		child_event = *event;
		child_event.window =
			gtk_text_view_get_window( GTK_TEXT_VIEW(private->view),
						  GTK_TEXT_WINDOW_TEXT );
		/* we do this so we select from the start of the line */
		child_event.x = 0;
		gtk_widget_event( private->view, (GdkEvent*)&child_event );
		return TRUE;
	}

	g_object_get( G_OBJECT( editor ), "window", &window, NULL );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );

	if( event && event->button == 1 && event->type == GDK_2BUTTON_PRESS ) {
		/* double click event */
		gint x;
		gint y;
		gint bx;
		gint by;
		gchar *dirname;
		gchar *word;
		gchar *file;
		gchar *pattern;

		x = (gint)event->x + 0.5;
		y = (gint)event->y + 0.5;
		gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(private->view ),
						      GTK_TEXT_WINDOW_TEXT,
						      x, y, &bx, &by );
		gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(private->view),
						   &it, bx, by);

		pos2 = gtk_text_iter_get_offset( &it );

		word = screem_editor_get_word( editor, pos2 );
		
		if( pathname ) {
			gchar *temp;
			temp = gnome_vfs_get_local_path_from_uri( pathname );
			if( temp ) {
				dirname = g_dirname( temp );
				g_free( temp );
			} else {
				dirname = g_strdup( "" );
			}
		} else {
			dirname = g_strdup( "" );
		}

		file = screem_support_ctags( dirname, word, &pattern );

		if( file ) {
			ScreemSite *site;

			site = screem_window_get_current( window );
			screem_page_open_with_filename( site, window,
							file );
			page = screem_site_locate_page( site, file );
			if( page && pattern ) {
				/* perform search for pattern */
				gint len;
				gchar *found;

				text = screem_page_get_data( page );
				found = find_text( text, pattern, NULL, &len );
				if( found ) {
					screem_editor_set_pos_delayed( editor, found - text, FALSE, 0.0 );
				}

				g_free( text );
			}
			g_free( file );
			if( pattern ) {
				g_free( pattern );
			}
		} else {

		}

		g_free( dirname );
		g_free( word );

		g_object_unref( app );
		return FALSE;
	}

	if( event && event->button != 3 ) {
		g_object_unref( app );
		return FALSE;
	}

	private->popup_open = TRUE;

	if( event ) {
		g_signal_stop_emission_by_name( G_OBJECT( widget ),
						"button_press_event" );
	} else {
		g_signal_stop_emission_by_name( G_OBJECT( widget ),
						"popup_menu" );
	}
	
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( private->view ) );
	mime_type = screem_page_get_mime_type( page );
	cpos = screem_editor_get_pos( editor );

	/* get position of click */
	if( event ) {
		gint x;
		gint y;
		gint bx;
		gint by;

		x = (gint)event->x + 0.5;
		y = (gint)event->y + 0.5;
		gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(private->view ),
						      GTK_TEXT_WINDOW_TEXT,
						      x, y, &bx, &by );
		gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(private->view),
						   &it, bx, by);
		pos2 = pos = gtk_text_iter_get_offset( &it );
	} else {
		pos2 = pos = cpos;
	}

	sel = screem_editor_has_selection( editor, &start, &len );
	screem_editor_set_pos( editor, pos );

	if( sel ) {
		screem_editor_select_region( editor, start, len - start );
	}

	/* create the menu */
       	menu = egg_menu_merge_get_widget( EGG_MENU_MERGE( window->merge ),
					"/popups/editormenu" );
	
	/* id the tag we are inside so we can offer attribute editing */
	
	text = screem_page_get_data( page );

	feature_markup = screem_page_is_markup( page );

	if( feature_markup &&
	    screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		tag_pos = pos;
		tag = screem_markup_next_tag( text, pos, NULL, &pos, &tag2 );
		tag_len = pos - tag_pos + 1;

		dtd = screem_page_get_dtd( page );
		menu_item = screem_editor_attribute_menu( editor, dtd, tag2 );
		if( menu_item ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ), 
						menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}
		tag_known = (gboolean)menu_item;

		/* add the wizard option */
		if( (menu_item = screem_plugin_tag_wizard( app, 
							   window,
							   tag2, tag,
							   tag_pos, 
							   tag_len ) ) ) {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}

		g_free( tag );

		/* add the learn option */
		if( itag && gtk_text_iter_has_tag( &it, itag ) ) {
			menu_item = gtk_menu_item_new_with_label( _( "Learn More..." ) );
			gtk_widget_show( menu_item );
			gtk_menu_append( GTK_MENU( menu ), menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
			g_signal_connect( G_OBJECT( menu_item ), 
					"activate", 
					G_CALLBACK( screem_editor_learn_more ), 
					editor );
		}
		g_free( tag2 );
	} else if( feature_markup ) {
		/* add the insert option */
		tag2 = screem_page_query_context( page, pos, FALSE,
			TRUE, NULL, NULL, NULL );

		dtd = screem_page_get_dtd( page );

		if( tag2 && 
		    ( menu_item = screem_editor_allowed_menu( editor, dtd, tag2 ) ) ) {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ), 
					       menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}
	} else if( ! strcmp( "text/x-perl", mime_type ) ) {
		/* create perl menu */
		/*menu_item = perl_menu();
		  gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ), menu_item );
		
		addeditems = g_slist_prepend( addeditems, menu_item );
		*/
		
	} else if( ! strcmp( "text/x-java", mime_type ) ) {
		/* create java menu */
		/*menu_item = java_menu();
		  gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ), menu_item );
		
		addeditems = g_slist_prepend( addeditems, menu_item );
		*/
	}
	g_free( text );

	menu_item = gtk_menu_item_new_with_mnemonic( _( "Input _Methods" ) );
	addeditems = g_slist_prepend( addeditems, menu_item );
	gtk_widget_show( menu_item );
	submenu = gtk_menu_new ();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), submenu );
	gtk_menu_shell_append( GTK_MENU_SHELL( menu ), menu_item );
	gtk_im_multicontext_append_menuitems( GTK_IM_MULTICONTEXT( GTK_TEXT_VIEW(private->view)->im_context ),
						GTK_MENU_SHELL( submenu ) );

	/* display menu */
	private->inserted = FALSE;
	gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0, private->view );

	for( tmp = addeditems; tmp; tmp = tmp->next ) {
		gtk_container_remove( GTK_CONTAINER( menu ), 
					GTK_WIDGET( tmp->data ) );
	}
	g_slist_free( addeditems );
	
	if( ! private->inserted )
		screem_editor_set_pos( editor, cpos );
	
	if( sel ) {
		screem_editor_select_region( editor, start, len - start );
	}

	private->popup_open = FALSE;

	g_object_unref( app );
		
	return TRUE;
}

void screem_editor_insert( ScreemEditor *editor, gint pos, const gchar *text )
{
	gint len;
	gint start;
	const gchar *pathname;
	const gchar *mime_type;

	GtkTextBuffer *buffer;
	GtkTextIter it;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( ! text || ! editor->private->page )
		return;

	pathname = screem_page_get_pathname( editor->private->page );
	mime_type = screem_page_get_mime_type( editor->private->page );

	len = gtk_text_buffer_get_char_count( buffer );
	if( pos == -1 ) {
		pos = screem_editor_get_pos( editor );
	}

	if( pos > len ) {
		pos = len;
	}

	start = pos;

	len = strlen( text );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	gtk_text_buffer_insert( buffer, &it, text, len );

	gtk_widget_grab_focus( GTK_WIDGET( editor ) );
}

static void focus( GtkWidget *widget )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( widget );
	gtk_widget_grab_focus( editor->private->view );
}

static gboolean screem_editor_focus( GtkWidget *widget )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( widget );

gdk_threads_enter();
	
	gtk_widget_grab_focus( editor->private->view );

	gdk_threads_leave();

	return FALSE;
}

gchar *screem_editor_get_text( ScreemEditor *editor, guint start, guint len )
{
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	guint end = len;

	buffer =
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( end == 0 ) {
		end = gtk_text_buffer_get_char_count( buffer );
	} else {
		end += start;
	}

	gtk_text_buffer_get_iter_at_offset( buffer, &it, start );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, end );

	return gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
}

gboolean screem_editor_has_selection( ScreemEditor *editor, guint *start, 
				      guint *end )
{
	guint temp;
	gboolean ret;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( ! start )
		start = &temp;
	if( ! end )
		end = &temp;

	ret = gtk_text_buffer_get_selection_bounds( buffer, &it, &eit );

	*start = gtk_text_iter_get_offset( &it );
	*end = gtk_text_iter_get_offset( &eit );

	return ret;
}

void screem_editor_select_region( ScreemEditor *editor, guint start, guint len )
{
        GtkTextBuffer* buffer = NULL;
        GtkTextIter it;
	GtkTextIter eit;

        buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( len == 0 ) {
		len = gtk_text_buffer_get_char_count( buffer );
	}

	gtk_text_buffer_get_iter_at_offset( buffer, &it, start );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, start + len );

        gtk_text_buffer_place_cursor( buffer, &eit );

        gtk_text_buffer_move_mark( buffer,
				   gtk_text_buffer_get_mark(buffer, 
							    "selection_bound"),
				   &it );
	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );
}

void screem_editor_cut( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	gtk_text_buffer_cut_clipboard( buffer,
				       gtk_clipboard_get( GDK_NONE ),
				       TRUE );

	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );
}

void screem_editor_copy( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	gtk_text_buffer_copy_clipboard( buffer,
					gtk_clipboard_get( GDK_NONE ) );
	
	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );
}

void screem_editor_paste( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	gtk_text_buffer_paste_clipboard( buffer,
					 gtk_clipboard_get( GDK_NONE ),
					 NULL,
					 TRUE );

	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );

}

void screem_editor_paste_encoded( ScreemEditor *editor )
{
	gtk_clipboard_request_contents( gtk_clipboard_get( GDK_NONE ),
					gdk_atom_intern( "UTF8_STRING", 
							 FALSE ),
					screem_editor_clip_received, editor );
}

void screem_editor_clear_selection( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	gtk_text_buffer_delete_selection( buffer, FALSE, TRUE );

	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );

}

void screem_editor_set_pos( ScreemEditor *editor, guint pos )
{
	GtkTextView *view;
	GtkTextBuffer *buffer;
	GtkTextIter it;

	view = GTK_TEXT_VIEW( editor->private->view );

	buffer = gtk_text_view_get_buffer( view );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );

	gtk_text_buffer_place_cursor( buffer, &it );

	gtk_text_view_scroll_to_iter( view, &it, 0.0, FALSE, 0.0, 0.0 );
}

guint screem_editor_get_pos( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;
	GtkTextMark *mark;
	GtkTextIter it;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));
	mark = gtk_text_buffer_get_insert( buffer );

	gtk_text_buffer_get_iter_at_mark( buffer, &it, mark );

	return gtk_text_iter_get_offset( &it );
}

void screem_editor_delete_forward( ScreemEditor *editor, guint pos, guint len )
{
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));
	if( len == 0 ) {
		len = gtk_text_buffer_get_char_count( buffer );
	}
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, pos + len );

	gtk_text_buffer_delete( buffer, &it, &eit );
}

void screem_editor_insert_markup( ScreemEditor *editor, 
				  const gchar *open_element, 
				  const gchar *close_element )
{
	gboolean has_selection;
	guint start;
	guint end;
	gint pos;
	gint ipos;
	ScreemPage *page;

	page = editor->private->page;
	
	if( ! page )
		return;

	pos = screem_editor_get_pos( editor );
	ipos = pos;

	has_selection = screem_editor_has_selection( editor, &start, &end );

	if( has_selection ) {
		if( open_element ) {
			screem_editor_insert( editor, start, open_element );
                        end += strlen( open_element );
		}
		pos = end;
		if( close_element ) {
                        screem_editor_insert( editor, end, close_element );
                        pos += strlen( close_element );
                }
	} else {
		if( open_element ) {
			screem_editor_insert( editor, pos, open_element );
			pos += strlen( open_element );
		}
		if( close_element ) {
			screem_editor_insert( editor, pos, close_element );
			pos += strlen( close_element );
		}

		if( open_element )
			pos = ipos + strlen( open_element );
		else if( close_element )
			pos = ipos + strlen( close_element );
	}
	screem_editor_set_pos( editor, pos );

	if( editor->private->popup_open )
		editor->private->inserted = TRUE;
}

void screem_editor_insert_file( ScreemEditor *editor, const gchar *filename )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemPage *page;
	gchar *data;
	gint pos;

	g_object_get( G_OBJECT( editor ), "window", &window, NULL );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	page = screem_page_new( G_OBJECT( app ) );

	screem_page_set_pathname( page, filename );

	screem_page_load( page );

	pos = screem_editor_get_pos( editor );
	data = screem_page_get_data( page );
	screem_editor_insert( editor, pos, data );
	
	g_free( data );
	
	g_object_unref( page );
	g_object_unref( app );
}

void screem_editor_undo( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemPage *page;

	editor = SCREEM_EDITOR( view );
	page = editor->private->page;

	if( page && gtk_source_buffer_can_undo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_undo( GTK_SOURCE_BUFFER( page ) );
	}
}

void screem_editor_redo( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemPage *page;

	editor = SCREEM_EDITOR( view );
	page = editor->private->page;

	if( page && gtk_source_buffer_can_redo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_redo( GTK_SOURCE_BUFFER( page ) );
	}
}

gint screem_editor_auto_indent( ScreemEditor *editor, gint pos )
{
	guint depth;
	guint start;
	guint end;
	GString *indent;
	GtkTextIter it;
	GtkTextIter *copy;
	GtkTextBuffer *buffer;
	gunichar c;
	guint offset;
	gboolean corrected;

	guint sel_start;
	guint sel_end;
	gint sel_offset;
	gboolean built;

	gchar *temp;
	
	if( ! screem_editor_has_selection( editor, &sel_start, &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer =gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( sel_offset = 0, built = FALSE; sel_start <= sel_end;
		++ sel_start ) {
		pos = sel_start;
		gtk_text_buffer_get_iter_at_line( buffer, &it, pos );
		pos = gtk_text_iter_get_offset( &it );

		temp = screem_page_query_context( SCREEM_PAGE( buffer ),
				pos - sel_offset, FALSE, ! built, 
				&depth, &start, &end );
		g_free( temp );
		built = TRUE;
	
		/* if the first char on a line is < then we will have
		   depth 1 > than we should */
		corrected = FALSE;
		if( depth != 0 && gtk_text_iter_get_char( &it ) == '<' ) {
			depth --;
			corrected = TRUE;
		}
		
		copy = gtk_text_iter_copy( &it );
		offset = 0;
		
		do {
			offset ++;
			c = gtk_text_iter_get_char( copy );
			if( ! g_unichar_isspace( gtk_text_iter_get_char( copy ) ) ||
			    c == '\n' ) {
				break;
			}
			gtk_text_iter_forward_char( copy );
		} while( ! gtk_text_iter_is_end( copy ) );
		
		gtk_text_buffer_delete( buffer, &it, copy );
		gtk_text_iter_free( copy );
		gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
		
		/* handle close of context correctly */
		copy = gtk_text_iter_copy( &it );
		c = gtk_text_iter_get_char( copy );
		if( c == '<' ) {
			gtk_text_iter_forward_char( copy );
			if( ! gtk_text_iter_is_end( copy ) ) {
				c = gtk_text_iter_get_char( copy );
			}
			if( c == '/' ) {
				/* closing tag, does it close the context? */
				do {
					gtk_text_iter_forward_char( copy );
					if( gtk_text_iter_is_end( copy ) ) {
						break;
					}
					c = gtk_text_iter_get_char( copy );
				} while( c != '>' );
			}
			pos = gtk_text_iter_get_offset( copy ) + 1;
			pos += offset;
			/* is pos outside the current context? */
			if( pos >= end && depth != 0 && ! corrected ) {
				depth --;
			}
		}
		gtk_text_iter_free( copy );
		
		indent = g_string_new_len( "", depth );
		memset( indent->str, '\t', depth );

		gtk_text_buffer_insert( buffer, &it, indent->str, indent->len );
		
		sel_offset -= offset;
		sel_offset += 1 + indent->len;
		
		g_string_free( indent, TRUE );
	}
	
	pos = gtk_text_iter_get_offset( &it );
	pos += sel_offset;

	return pos;
}

void screem_editor_indent( ScreemEditor *editor, guint pos )
{
	guint sel_start;
	guint sel_end;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	
	if( ! screem_editor_has_selection( editor, &sel_start, 
					   &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( ; sel_start <= sel_end; ++ sel_start ) {
		pos = sel_start;
		gtk_text_buffer_get_iter_at_line( buffer, &it, pos );
		pos = gtk_text_iter_get_offset( &it );
		
		gtk_text_buffer_insert( buffer, &it, "\t", 1 );
	}
}

void screem_editor_unindent( ScreemEditor *editor, guint pos )
{
	guint sel_start;
	guint sel_end;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	
	if( ! screem_editor_has_selection( editor, &sel_start, 
					   &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( ; sel_start <= sel_end; ++ sel_start ) {
		gtk_text_buffer_get_iter_at_line( buffer, &it, sel_start );
		eit = it;
		gtk_text_iter_forward_char( &eit );

		if( gtk_text_iter_get_char( &it ) == '\t' ) {
			gtk_text_buffer_delete( buffer, &it, &eit );
		}
	}
}


void screem_editor_goto_line( ScreemEditor *editor, gint line )
{
	ScreemPage *page;
	GtkTextIter it;
	GtkTextView *view;

	page = editor->private->page;

	if( ! page ) {
		return;
	}

	view = GTK_TEXT_VIEW(editor->private->view);
	gtk_text_buffer_get_iter_at_line( GTK_TEXT_BUFFER( page ), &it, line - 1 );
	gtk_text_view_scroll_to_iter( view, &it, 0.0, FALSE, 0.0, 0.0 );
}

void screem_editor_encode_text( ScreemEditor *editor, gboolean urienc )
{
	gchar *text;
	gchar *ret;
	guint start = 0;
	guint len = 0;
	ScreemPage *page;

	page = editor->private->page;

	if( ! page )
		return;

	if( screem_editor_has_selection( editor, &start, &len ) )
		len = len - start;

	text = screem_editor_get_text( editor, start, len );

	if( ! urienc ) {
		ret = screem_markup_encode_text( text );
	} else {
		ret = gnome_vfs_escape_string( text );		
	}

	screem_editor_delete_forward( editor, start, len );
	screem_editor_insert( editor, start, ret );

	g_free( ret );
	g_free( text );
}

gchar* screem_editor_get_word( ScreemEditor *editor, gint pos )
{
	ScreemEditorPrivate *priv;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	gunichar c;
	gchar *word;

	priv = editor->private;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	c = gtk_text_iter_get_char( &it );
	if( c == 0 || c == '\n' || c == '\r' ) {
		pos --;
	}
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );

	c = gtk_text_iter_get_char( &it );
	if( g_unichar_isspace( c ) ) {
		return NULL;
	}
	
	for( ; ! gtk_text_iter_is_start( &it ); gtk_text_iter_backward_char( &it ) ) {
		c = gtk_text_iter_get_char( &it );
		if( c != '_' && ! g_unichar_isalnum( c ) ) {
			gtk_text_iter_forward_char( &it );
			break;
		}
	}
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, pos );
	for( ; ! gtk_text_iter_is_end( &eit ); gtk_text_iter_forward_char( &eit ) ) {
		c = gtk_text_iter_get_char( &eit );
		if( c != '_' && ! g_unichar_isalnum( c ) ) {
			break;
		}
	}

	word = NULL;
	if( gtk_text_iter_compare( &it, &eit ) ) {
		word = gtk_text_iter_get_visible_text( &it, &eit );
	}

	return word;
}

static void screem_editor_set_colours( ScreemEditor *editor )
{
	GdkColor back;
	GdkColor text;
	GConfClient *client;
	ScreemEditorPrivate *private;
 	PangoFontDescription *font = NULL;
	gchar *value;

	private = editor->private;
	client = private->client;

	if( ! gconf_client_get_bool( client, "/apps/screem/editor/themecolours", NULL ) ) {
		value = gconf_client_get_string( client, "/apps/screem/editor/back",
						 NULL );
		if( ! value ) {
			value = g_strdup( "#ffffffffffff" );
		}
		gdk_color_parse( value, &back );
		g_free( value );
	
		gtk_widget_modify_base( GTK_WIDGET( private->view ),
					GTK_STATE_NORMAL, &back );

		value = gconf_client_get_string( client, "/apps/screem/editor/text",
						 NULL );
		if( ! value ) {
			value = g_strdup( "#000000000000" );
		}
		gdk_color_parse( value, &text );

		g_free( value );

		gtk_widget_modify_text( GTK_WIDGET( private->view ),
					GTK_STATE_NORMAL, &text );
	} else {
		GtkRcStyle *rc_style;

		rc_style = gtk_widget_get_modifier_style( GTK_WIDGET( private->view ) );
		rc_style->color_flags [GTK_STATE_NORMAL] = 0;
		rc_style->color_flags [GTK_STATE_SELECTED] = 0;
   		rc_style->color_flags [GTK_STATE_ACTIVE] = 0;
		gtk_widget_modify_style( GTK_WIDGET( private->view), rc_style );
	}

	if( ! gconf_client_get_bool( client, "/apps/screem/editor/themefont", NULL ) ) {
		value = gconf_client_get_string( client, "/apps/screem/editor/font",
						 NULL );
		if( value ) {
			font = pango_font_description_from_string( value );
			gtk_widget_modify_font( GTK_WIDGET( private->view ), font );
			pango_font_description_free( font );
			g_free( value );
		}
	} else {
		GtkRcStyle *rc_style;
	
		rc_style = gtk_widget_get_modifier_style( GTK_WIDGET( private->view ) );
		if( rc_style->font_desc ) {
			pango_font_description_free( rc_style->font_desc );
		}
		rc_style->font_desc = NULL;
		gtk_widget_modify_style( GTK_WIDGET( private->view ), rc_style );
	}
}

static void screem_editor_print_page_cb( GtkSourcePrintJob *job, 
					 ScreemEditor *editor )
{
	ScreemWindow *window;
	gchar *msg;
	
	g_object_get( G_OBJECT( editor ), "window", &window, NULL );

	msg = g_strdup_printf( "Printing %.2f%%",
				100.0 * gtk_source_print_job_get_page( job ) /
				gtk_source_print_job_get_page_count( job ) );

	screem_window_show_message( window, msg );

	g_free( msg );
}
static void screem_editor_print_finish_cb( GtkSourcePrintJob *job,
					ScreemEditor *editor )
{
	GnomePrintJob *gjob;
	GtkWidget *preview;

	gjob = gtk_source_print_job_get_print_job( job );

	if( g_object_get_data( G_OBJECT( job ), "preview" ) ) {
		preview = gnome_print_job_preview_new( gjob, _( "Print Preview - Screem" ) );
		gtk_widget_show( preview );
	} else {
                gnome_print_job_print( gjob );
	}
	g_object_unref( gjob );
	g_object_unref( job );

}

static GtkSourcePrintJob* screem_editor_print_setup_job( ScreemEditor *editor ) 
{
	GConfClient *client;
	GtkSourcePrintJob *job;
	GtkSourceView *view;
	ScreemPage *page;
	const gchar *pathname;

	client = gconf_client_get_default();
	
	view = GTK_SOURCE_VIEW( editor->private->view );
	page = editor->private->page;

	job = gtk_source_print_job_new( NULL );
	gtk_source_print_job_setup_from_view( job, view );

	if( gconf_client_get_bool( client, 
				"/apps/screem/editor/print/wrap_lines",
				NULL ) ) {
		gtk_source_print_job_set_wrap_mode( job, GTK_WRAP_CHAR );
		if( gconf_client_get_bool( client,
					"/apps/screem/editor/print/split",
					NULL ) ) {
			gtk_source_print_job_set_wrap_mode( job, GTK_WRAP_WORD );
		}
	}

	gtk_source_print_job_set_highlight( job, 
			gconf_client_get_bool( client,
				"/apps/screem/editor/print/highlight",
				NULL ) );
	
	if( gconf_client_get_bool( client, "/apps/screem/editor/print/lines",
				   NULL ) ) {
		gfloat lines;

		lines = gconf_client_get_float( client,
					"/apps/screem/editor/print/nlines",
					NULL );
		gtk_source_print_job_set_print_numbers( job, (gint)lines );
	}
	
	gtk_source_print_job_set_header_format( job,
						"Printed on %A",
						NULL,
						"%F",
						TRUE );
	
	pathname = screem_page_get_pathname( page );
	if( ! pathname ) {
		pathname = _( "Untitled" );
	}
	
	gtk_source_print_job_set_footer_format( job,
						"%T",
						pathname,
						"Page %N/%Q",
						TRUE );
	if( gconf_client_get_bool( client, 
				"/apps/screem/editor/print/headers",
				NULL ) ) {
		gtk_source_print_job_set_print_header( job, TRUE );
		gtk_source_print_job_set_print_footer( job, TRUE );
	}

	
	return job;
}

static void screem_editor_print( ScreemView *view, gboolean preview )
{
	ScreemEditor *editor;
	ScreemPage *page;
	GtkTextIter it;
	GtkTextIter eit;
	GtkSourcePrintJob *job;
	
	editor = SCREEM_EDITOR( view );
	
	job = screem_editor_print_setup_job( editor );
	page = editor->private->page;
		
	if( preview ) {
		gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER( page ), &it, &eit );
	} else {
		GtkWidget *dialog;
		GnomePrintConfig *config;
		gint lines;
		gint res;
	
		gint first;
		gint last;
		GnomePrintRangeType type;
		
		config = gtk_source_print_job_get_config( job );
		
		dialog = g_object_new( GNOME_TYPE_PRINT_DIALOG, 
					"print_config", config, NULL );
	
		gnome_print_dialog_construct( GNOME_PRINT_DIALOG( dialog ),
					      _( "Screem - Print Document " ),
					      GNOME_PRINT_DIALOG_RANGE |
					      GNOME_PRINT_DIALOG_COPIES );

		lines = gtk_text_buffer_get_line_count( GTK_TEXT_BUFFER( page ) );
		gnome_print_dialog_construct_range_page( GNOME_PRINT_DIALOG( dialog ),
							 GNOME_PRINT_RANGE_ALL,/* ||
							 GNOME_PRINT_RANGE_RANGE, */
						 	1, lines, "A", "Lines" );

		res = gtk_dialog_run( GTK_DIALOG( dialog ) );
		switch( res ) {
			case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
				preview = FALSE;
				break;
			case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
				preview = TRUE;
				break;
			default:
				gtk_widget_destroy( dialog );
				g_object_unref( job );
				return ;
				break;
		}

		type = gnome_print_dialog_get_range( GNOME_PRINT_DIALOG( dialog ) );
		gnome_print_dialog_get_range_page( GNOME_PRINT_DIALOG( dialog ),
						   &first, &last );

		gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER( page ), &it, &eit );

		gtk_widget_destroy( dialog );
	}
	g_object_set_data( G_OBJECT( job ), "preview", 
			   GINT_TO_POINTER( preview ) );

	if( gtk_source_print_job_print_range_async( job, &it, &eit ) ) {
		g_signal_connect( job, "begin_page", 
				  G_CALLBACK( screem_editor_print_page_cb ),
				  editor );
		g_signal_connect( job, "finished",
				  G_CALLBACK( screem_editor_print_finish_cb ),
				  editor );
	}
}

/* static stuff */
static void screem_editor_display( ScreemView *view )
{
	ScreemPage *page;

	g_object_get( G_OBJECT( view ), "page", &page, NULL );

	screem_editor_display_page( SCREEM_EDITOR( view ), page );

	if( page ) {
		g_object_unref( page );
	}
}

static void screem_editor_display_page( ScreemEditor *editor, 
					ScreemPage *page )
{
	gint ppos;
	GtkTextMark *mark;
	GtkTextIter it;
	GtkTextBuffer *buffer;
	
	editor->private->ppos = -1;

	gtk_widget_set_sensitive( editor->private->mime,
				page != NULL );
	gtk_widget_set_sensitive( editor->private->cursorPosition,
				page != NULL );
	gtk_widget_set_sensitive( editor->private->overwriteMode,
				page != NULL );
	
	if( page ) {
		const gchar *pathname;
		GSList *types;
		GSList *tmp;
		GtkWidget *item;
		gchar *type;
		const gchar *showtype;

		pathname = screem_page_get_pathname( page );
		type = NULL;
		if( pathname ) {
			type = screem_get_mime_type( pathname );
		}
		if( ! type ) {
			type = g_strdup( "text/plain" );
		}
		types = screem_page_get_mime_types( page );
		types = g_slist_prepend( types, type );

		if( editor->private->types ) {
			gtk_widget_destroy( editor->private->types );
		}

		editor->private->types = gtk_menu_new();

		for( tmp = types; tmp; tmp = tmp->next ) {
			showtype = gnome_vfs_mime_get_description( tmp->data );
			if( ! showtype ) {
				showtype = tmp->data;
			}
			item = gtk_menu_item_new_with_label( showtype );
			gtk_widget_show( item );
			g_object_set_data_full( G_OBJECT( item ), 
						"label", tmp->data,
						(GDestroyNotify)g_free);
			g_signal_connect( G_OBJECT( item ), "activate",
					  G_CALLBACK( screem_editor_mime_switch ),
					  editor );
			gtk_menu_shell_append( GTK_MENU_SHELL( editor->private->types ), item );
		}
		g_slist_free( types );
	} 
	
	/* disconnect mark_set signal from the buffer */
	ppos = screem_editor_get_pos( editor );
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );
	g_signal_handlers_disconnect_matched( G_OBJECT( buffer ),
					      G_SIGNAL_MATCH_DATA,
					      0, 0, NULL, NULL, editor );
	
	gtk_text_view_set_buffer( GTK_TEXT_VIEW( editor->private->view ),
				  NULL );
	gtk_text_view_set_editable( GTK_TEXT_VIEW( editor->private->view ),
				    FALSE );
	
	
	if( editor->private->page ) {
		/* set mark for cursor position */
		gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( editor->private->page ), &it, ppos );
		gtk_text_buffer_create_mark( GTK_TEXT_BUFFER( editor->private->page ), 
				editor->private->markname, &it, TRUE );
		
		/* disconnect from page signals */
		g_signal_handlers_disconnect_matched( G_OBJECT( editor->private->page ),
						      G_SIGNAL_MATCH_DATA,
						      0, 0, NULL, NULL, editor );
	}
	editor->private->page = NULL;

	gtk_button_set_label( GTK_BUTTON( editor->private->mime ), 
					  _( "No Page" ) );

	if( page ) {
		const gchar *type;
		const gchar *showtype;

		editor->private->page = page;

		/* we need to know if the page changes buffer, so
		 * the editor knows it needs to get the new one, this
		 * happens on pathname or mime type change */
		g_signal_connect( G_OBJECT( page ), "notify::pathname",
				  G_CALLBACK( screem_editor_buffer_change ),
				  editor );
		g_signal_connect( G_OBJECT( page ), "notify::mime-type",
				  G_CALLBACK( screem_editor_buffer_change ),
				  editor );
	
		gtk_text_view_set_buffer( GTK_TEXT_VIEW( editor->private->view ),
					  GTK_TEXT_BUFFER( page ) );
		
		gtk_text_view_set_editable( GTK_TEXT_VIEW( editor->private->view ),
					    TRUE );
		type = screem_page_get_mime_type( page );
		if( ! type ) {
			type = "text/plain";
		}
		showtype = gnome_vfs_mime_get_description( type );
		if( ! showtype ) {
			showtype = type;
		}
		gtk_button_set_label( GTK_BUTTON( editor->private->mime ),
				      showtype );
		buffer = GTK_TEXT_BUFFER( page );
	} else {
		gtk_button_set_label( GTK_BUTTON( editor->private->mime ),
				      "text/plain" );


		/* yes we did set this to NULL, but that means we
		   get a newed one back so we can actually clear the display
		   it will be unrefed when we set it to a proper one later */
		buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );

		gtk_text_view_set_buffer( GTK_TEXT_VIEW( editor->private->view ), buffer );
	}

	/* connect mark_set signal */
	g_signal_connect( G_OBJECT( buffer ),
			  "mark_set",
			  G_CALLBACK( screem_editor_mark_set ),
			  editor );

	mark = gtk_text_buffer_get_mark( buffer, 
			editor->private->markname );
	ppos = 0;
	if( mark ) {
		gtk_text_buffer_get_iter_at_mark( buffer, &it, mark );
		ppos = gtk_text_iter_get_offset( &it );
	}
	screem_editor_set_pos_delayed( editor, ppos, FALSE, 0.0 );
}

static gboolean screem_editor_mark_set( GtkTextBuffer *buffer,
					const GtkTextIter *it,
					GtkTextMark *mark,
					gpointer data )
{
	ScreemEditor *editor;
	ScreemPage *page;

	g_return_val_if_fail( SCREEM_IS_EDITOR( data ), FALSE );

	editor = SCREEM_EDITOR( data );


	page = editor->private->page;

	if( page ) {
		gint row;
		gint col;
		gchar *mesg;

		if( editor->private->handle ) {
			g_source_remove( editor->private->handle );
		}
	
		editor->private->handle = 
			g_timeout_add( 2000, (GSourceFunc)build_attribute_list,
				       editor );
		
		gtk_statusbar_pop( GTK_STATUSBAR( editor->private->cursorPosition ),
				   0 );
		
		row = gtk_text_iter_get_line( it );
		col = gtk_text_iter_get_line_offset( it );
		mesg = g_strdup_printf( _("  Ln %d, Col. %d"), row + 1, col + 1 );

		gtk_statusbar_push( GTK_STATUSBAR( editor->private->cursorPosition ),
				    0, mesg );
		g_free( mesg );
	}

	return TRUE;
}

static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      ScreemEditor *editor )
{
	GdkDragAction action;
       
	GdkModifierType modifiers;
        gdk_window_get_pointer (NULL, NULL, NULL, &modifiers);

	
        if( context->suggested_action != GDK_ACTION_ASK )
                action = GDK_ACTION_COPY;
        else
                action = GDK_ACTION_ASK;

	if ((modifiers & GDK_MOD1_MASK) != 0)
                action = GDK_ACTION_ASK;
        
        gdk_drag_status( context, action, time );

        return TRUE;
}

static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time, ScreemEditor *editor )
{
	ScreemPage *page;
	GtkWidget *popup;
	gint item;
	gint pos;
	guint16 *colours;
	gchar *text = NULL;
	gchar *temp;
	Drop_actions action;

	const gchar *pathname;

	gint bx;
	gint by;
	GtkTextIter it;

	gboolean handled;
	ScreemWindow *window;
	
	handled = FALSE;

	page = editor->private->page;

	if( ! page ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
		
	pathname = screem_page_get_pathname( page );

	/* if we are doing a right drag then we need to ask the user
	   what to do */
	if( context->action == GDK_ACTION_ASK ) {
                popup = gnome_popup_menu_new( editor_dnd_menu );
                item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0, 
				editor->private->view );
                switch( item ) {
                case 0: /* insert relative filename */
                        action = RELATIVE_PATH;
                        break;
                case 1: /* insert complete filename */
                        action = FULL_PATH;
                        break;
                case 2: /* insert tag */
                        action = TAG;
                        break;
                case 3: /* insert tag attribute */
                        action = ATTRIBUTE;
                        break;
                case 4: /* insert text inline */
                        action = INLINE;
                        break;
                default:
			gtk_drag_finish( context, FALSE, FALSE, time ); 
			return;
                        break;
                }
		gtk_widget_destroy( popup );
        } else {
                action = RELATIVE_PATH;
	}

	/* position the cursor at the drop location */
	gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(editor->private->view ),
					      GTK_TEXT_WINDOW_WIDGET,
					      x, y, &bx, &by );
	gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(editor->private->view),
					   &it, bx, by);
	pos = gtk_text_iter_get_offset( &it );

	screem_editor_set_pos( editor, pos );


	if( info == TARGET_MOZ || info == TARGET_UNICODE ) {
		glong read;
		glong written;

		text = g_utf16_to_utf8( (const gunichar2*)selectionData->data,
					selectionData->length,
					&read, &written,
					NULL );
	} else {
		text = g_strdup( selectionData->data );
	}

	/* handle the different drop types */
	switch( info ) {
	case TARGET_MOZ:
	case TARGET_URL:
	case TARGET_URI_LIST:
		handled = screem_editor_drop_uris( editor, 
						   action,
						   text,
						   info );

		break;
	case TARGET_TEXT:
	case TARGET_UNICODE:
		screem_editor_insert( editor, pos, text );
		g_free( text );
		handled = TRUE;
		break;
	case TARGET_COLOUR:
		colours = (guint16*)selectionData->data;
                temp = g_strdup_printf( "#%.2x%.2x%.2x",colours[ 0 ] >> 8, 
                                        colours[ 1 ] >> 8, colours[ 2 ] >> 8 );
		switch( action ) {
		case TAG:
                        text = g_strdup_printf( "<font color=\"%s\"> </font>",
                                                temp );
                        g_free( temp );
                        break;
		case ATTRIBUTE:
			text = screem_page_get_data( page );
                        if( screem_markup_is_tag( text, pos, NULL, NULL ) ) {
                                /* we are inside a tag */
                                g_free( text );
                                text = g_strdup_printf( " color=\"%s\"", temp );
                                screem_editor_insert_attribute( editor, text );
                                g_free( text );
				gtk_drag_finish( context, TRUE, FALSE, time );
                                return;
                        } else {
                                /* not in a tag, but we will insert it anyway*/
                                g_free( text );
                                text = g_strdup_printf( " color=\"%s\"", temp );
                                g_free( temp );
                        }
                        break;
		default:
			text = temp;
			break;
		}
		if( text ) {
			screem_editor_insert( editor, pos, text );
			g_free( text );
			handled = TRUE;
		}
		break;
	default:
		/* we don't know about this type, and as such
		   this case should never execute */
		g_object_get( G_OBJECT( editor ), "window", &window, NULL );
		screem_window_show_error( SCREEM_WINDOW( window ),
					  _("Error: invalid drop type for editor occured\n") );
		g_free( text );
		break;
	}

	gtk_drag_finish( context, handled, FALSE, time );
}

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event, 
					ScreemEditor *editor )
{
	ScreemPage *page;

	const gchar *pathname;

	GList *list;
	GModule *self;

	gchar *string;
	ScreemEditorMacro *macro;
	gint pos;

	gboolean press_handled;

	ScreemEditorPrivate *private;
	ScreemApplication *application;
	GConfClient *client;

	GtkTextBuffer *buffer;
	gboolean feature_markup;

	ScreemWindow *window;

	const gchar *mime_type;
	
	private = editor->private;
	client = private->client;

	page = private->page;

	if( ! page )
		return FALSE;

	press_handled = FALSE;

	g_object_get( G_OBJECT( editor ), "window", &window, NULL );
	g_object_get( G_OBJECT( window ), "app", &application, NULL );

	pathname = screem_page_get_pathname( page );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	/* does string match any of the editor keys set by the user? */
	string = NULL;
	if( application->macros ) {
		string = convert_keysym_state_to_string( event->keyval, 
							 event->state );
	}
	macro = NULL;

	if( string ) {
		for( list = application->macros; list; 
				list = list->next ) {
			macro = (ScreemEditorMacro*)list->data;
			if( ! strcmp( string, macro->key_name ) ) {
				if( macro->action == TEXT ) {
					pos = screem_editor_get_pos( editor );
					screem_editor_insert( editor, pos, 
							      macro->text );
					pos += strlen( macro->text );
					screem_editor_set_pos( editor, pos );
					g_free( string );
					g_object_unref( application );
					return TRUE;
				}
				break;
			} else {
				macro = NULL;
			}
		}
		g_free( string );
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( private->view ) );
	feature_markup = screem_page_is_markup( page );

	if( feature_markup ) {
		press_handled = html_key_press( editor, event, NULL );
	}

	/* offer tag tree auto completion */
	mime_type = screem_page_get_mime_type( page );
	if( private->autocomplete &&
		event->string && *event->string && ! press_handled ) {
		GSList *comp;
		gchar *tmp;
		gchar *word;
		gint pos;
		guint len;
		
		pos = screem_editor_get_pos( editor );
		word = screem_editor_get_word( editor, pos );

		comp = NULL;
		tmp = NULL;
		len = 0;
		if( word ) {
			len = strlen( word );
		}
		if( len > 2 ) {
			/* get completions */
			tmp = g_strdup_printf( "%s%s", 
					word, event->string );
			comp = screem_tag_tree_autocomplete( mime_type, tmp );
		}
		g_free( word );
		
		screem_editor_autocomplete_popup( editor, comp, 0, tmp, NULL );
		g_free( tmp );
		if( comp ) {
			g_slist_free( comp );
		}

	}
	
	g_object_unref( application );
	
	return press_handled;
}

gboolean html_key_press( ScreemEditor *editor, GdkEventKey *event, 
			 ScreemEditorMacro *macro )
{
	gchar *text;
	gchar *tag = NULL;
	gchar *tag2;
	const gchar *tmp;
	const gchar *tmp2;
	gchar *format;
	gint pos;
	gint start;
	gint len;
	gint len2;
	gboolean empty = FALSE;
	ScreemPage *page;
	gchar *charset;
	gboolean isent;
	
	ScreemEditorPrivate *private;
	GConfClient *client;
	ScreemDTD *dtd;
	ScreemWindow *window;

	gint entinsert;
	const ScreemDTDElement *elem;
	const GSList *list;
	GSList *tlist;
	
	/* we don't want to do any checking if they only key being
	pressed is a modifier, or cursor keys.
	this should speed things up a little, as we avoid checking
	the shift on its own when a < is being typed etc. */
	if( ( event->keyval >= GDK_Shift_L &&
		event->keyval <= GDK_Hyper_R ) || 
		( event->keyval >= GDK_Home &&
		  event->keyval <= GDK_End ) )	{
			return FALSE;
	}

	private = editor->private;
	client = private->client;

	pos = screem_editor_get_pos( editor );
	page = private->page;

	text = screem_page_get_data( page );

	/* Are we in an inline for scripting languages, 
	   ie  <? ?>  <% %> <?php ?> <!-- --> if so we don't want
	   any features, apart from possibly auto indent */
	
	if( ( screem_markup_is_tag( text, pos, &start, NULL ) ) &&
	    event->keyval != GDK_Tab ) {
		tmp = g_utf8_offset_to_pointer( text, start );
		tmp2 = g_utf8_next_char( tmp );
	    	if( strncmp( "<?xml ", tmp, strlen( "<?xml " ) ) &&
		    ( *tmp2 == '?' || *tmp2 == '%' || *tmp2 == '!' ) ) {
			g_free( text );
			return FALSE;
		}
	}

	switch( event->keyval ) {
	case GDK_Return:
		g_free( text );
		if( gconf_client_get_bool( client, "/apps/screem/editor/auto_indent", NULL ) ) {
			screem_editor_insert( editor, pos, "\n" );
			pos ++;
			screem_editor_auto_indent( editor, pos );
			g_signal_stop_emission_by_name(G_OBJECT(private->view),
						       "key_press_event" );
			return TRUE;
		}
		break;
	case GDK_space:
		if( ( ! gconf_client_get_bool( client, "/apps/screem/editor/inline_tagging", NULL ) ) ||
		    ( ! screem_markup_is_tag( text, pos, &start, NULL ) ) ||
		      ( screem_markup_is_attribute( text, pos, NULL, NULL ) ) ) {
			g_free( text );
			return FALSE;
		}

		if( start > 0 ) {
			start --;
		}
		tag = screem_markup_next_tag(text, start, &start, NULL, &tag2);
		if( ! tag ) {
			return FALSE;
		}
		len = strlen( tag );

		len2 = strlen( tag2 );
		g_free( text );

		if( pos <= start + len2 ) {
			/* not in the tag, or space pressed
			   in the tag name */
			g_free( tag2 );
			g_free( tag );
			return FALSE;
		}
		/* if intelliclose is enabled, and the tag can't be
		   closed, and we are an XML document we should put
		   in the closing / part of the tag if it isn't already
		   present */
		dtd = screem_page_get_dtd( page );
		if( gconf_client_get_bool( client,
					   "/apps/screem/editor/intelliclose",
					   NULL  ) ) {
			ScreemDTDTagExistance close;
			const gchar *id;

			id = NULL;
			close = SCREEM_DTD_MUST;
			if( dtd ) {
				id = g_object_get_data( G_OBJECT( dtd ),
							"type" );
				close = screem_dtd_element_close_state( dtd,
									tag2 );
			}
			if( close == SCREEM_DTD_MUST_NOT && id && 
			    ( ( ! strncmp( "<?xml ",
					   text, strlen( "<?xml " ) ) ) ||
			      ( strstr( id, " XHTML " ) || 
				strstr( id, " XML " ) ) ) &&
			    tag[ len - 2 ] != '/' ) {
				screem_editor_insert( editor, start + len - 1,
						      " /" );
				screem_editor_set_pos( editor, pos );
			}
		}
		elem = screem_dtd_valid_element( dtd, tag2 );
		list = NULL;
		tlist = NULL;
		if( elem ) {
			list = screem_dtd_element_get_attrs( elem );
			tlist = NULL;
			if( list ) {
				tlist = g_slist_copy( (GSList*)list );
				list = tlist;
			}
			for( ; tlist; tlist = tlist->next ) {
				tlist->data = (gpointer)screem_dtd_attribute_get_name( tlist->data );
			}
			tlist = (GSList*)list;
		}
		screem_editor_autocomplete_popup( editor, list, GDK_equal, NULL,
						  screem_editor_attr_after );
		if( tlist ) {
			g_slist_free( tlist );
		}

		g_free( tag2 );
		g_free( tag );
		break;
	case GDK_slash:
		if( ! gconf_client_get_bool(client,
					    "/apps/screem/editor/intelliclose",
					    NULL ) ||  
		    ( ! screem_markup_is_tag( text, pos, &start, NULL ) )  ) {
			g_free( text );
			return FALSE;
		}
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( ( ( start < ( pos - 1 ) ) && *tmp2 != '<' )
		    || *tmp != '>' ) {
			g_free( text );
			return FALSE;
		}
	
		screem_editor_insert( editor, pos, "/" );
		tag2 = screem_page_query_context( page, pos, FALSE,
			TRUE, NULL, NULL, NULL );

		if( tag2 && 
		    ! screem_markup_next_tag_close( text, tag2, start + 1 ) ) {
			/* we found a tag that needed closing */
			screem_editor_insert( editor, pos + 1, tag2 );
			g_signal_stop_emission_by_name(G_OBJECT(private->view),
						       "key_press_event");
			screem_editor_set_pos( editor, pos + strlen( tag2 ) + 2 );
			g_free( tag2 );
		} else {
			g_object_get( G_OBJECT( editor ), "window", &window, NULL );
			if( tag2 ) {
				tag =g_strdup_printf(_("%s is already closed"),
						     tag2 );
			} else {
				tag=g_strdup_printf(_("no tag needs closing"));
			}
			screem_window_show_message( SCREEM_WINDOW( window ),
						    tag );
			g_free( tag );
			g_free( tag2 );
		}

		g_free( text );
	   		
		return TRUE;
		break;
	case GDK_greater:
	case GDK_less:
	case GDK_ampersand:
	case GDK_quotedbl:
		tag = NULL;
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
		if( isent ) {
			if( empty ) {
				tag = (gchar*)screem_markup_char_to_ent( event->keyval );
				tag = g_strdup( tag );
			}
		} else if( event->keyval == GDK_ampersand ) {
			tag = g_strdup( "&;" );
		} else if( event->keyval == GDK_quotedbl ) {
			if( *tmp2 == '=' &&
			    ( *tmp == ' ' || *tmp == '>' ||
				*tmp == '/' ) )  {
				tag = g_strdup( "\"\"" );
			}
		} else if( screem_markup_is_attribute( text, pos - 1, NULL, NULL ) ) {
			/* in a tag, we need an entity for <>" */
			tag = (gchar*)screem_markup_char_to_ent( event->keyval );
			tag = g_strdup_printf( "&%s;", tag );
		} else if( event->keyval == GDK_less && *tmp != '>' ) {
			tag = g_strdup( "<>" );
		}
		if( tag ) {
			screem_editor_insert( editor, pos, tag );
			if( isent && empty ) {
				pos += strlen( tag );
			}
			pos ++;
			screem_editor_set_pos( editor, pos );
			g_free( tag );
		}
		g_free( text );

		if( tag && event->keyval == GDK_less ) {
			/* get list of valid elements,
			 * FIXME: should only offer elements which
			 * are valid in the current context */
			dtd = screem_page_get_dtd( page );
			list = screem_dtd_get_elements( dtd );
			tlist = g_slist_copy( (GSList*)list );
			for( list = tlist; tlist; tlist = tlist->next ) {
				tlist->data = (gpointer)screem_dtd_element_get_name( tlist->data );
			}

			screem_editor_autocomplete_popup( editor, list, GDK_slash, NULL, NULL );

			g_slist_free( tlist );
		}

		
		return ( tag != NULL );
		break;
	case GDK_Tab:
		tag = NULL;
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
		if( isent ) {
			if( empty ) {
				tag = (gchar*)screem_markup_char_to_ent( event->keyval );
				tag = g_strdup( tag );
				screem_editor_insert( editor, pos, tag );
				if( isent && empty ) {
					pos += strlen( tag );
				}
				pos ++;
				screem_editor_set_pos( editor, pos );
				g_free( tag );
			}
		} else if( gconf_client_get_bool( client, "/apps/screem/editor/auto_indent", NULL ) ) {
			screem_editor_auto_indent( editor, pos );
			tag = (gchar*)0x1;
		}
		g_free( text );
		return ( tag != NULL );
		break;
	default:
		/* get entity name for the key press */
		tag = (gchar*)screem_markup_char_to_ent( event->keyval );
		
		if( tag ) {
			tag = g_strdup( tag );
		} else {
			guint32 uni;

			uni = gdk_keyval_to_unicode( event->keyval );
			
			/* we should try an insert the numerical entity
			   if needed here */
			if( uni > 0x7F && g_unichar_isprint( uni ) ) {
				tag = g_strdup_printf( "#%i", uni );
			}
		}
		
		/* are we performing entity insertion? */
                entinsert = gconf_client_get_int( client,
                                                  "/apps/screem/editor/entityinsertion", NULL );
		charset = screem_page_get_charset( page );
		if( entinsert == BY_SET && 
		    ( ! charset || ! g_strcasecmp( "ISO-8859-1", charset ) ) ) {
			entinsert = OFF;
		}
		g_free( charset );
		/* are we in an entity? */
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
	
		/* if we are in an entity:
		   if it is empty enable entity insertion,
		   if not disable entity insertion,
		   also set up the correct format string for insertion */
		if( isent ) {
			if( empty ) {
				entinsert = ON;
			} else {
				entinsert = OFF;
			}
			format = "%s";
		} else if( *tmp == ';' ) {
			format = "&%s";
		} else if( pos > 0 && *tmp2 == '&' ) {
			format = "%s;";
		} else {
			format = "&%s;";
		}
		
		if( entinsert == ON && tag ) {
			gchar *temp;
			
			temp = tag;
			tag = g_strdup_printf( format, tag );
			g_free( temp );
			
			screem_editor_insert( editor, pos, tag );
			pos += strlen( tag );
			if( ! empty ) {
				pos ++;
			}
			screem_editor_set_pos( editor, pos );
		}
		if( tag ) {
			g_free( tag );
		}
		g_free( text );
		return ( entinsert == ON && ( tag != NULL ) );
		break;
	}
	return FALSE;
}

static gboolean screem_editor_tip( ScreemEditor *editor )
{
	ScreemEditorPrivate *private;

	private = editor->private;

	if( private->tooltip_timer ) {
		g_source_remove( private->tooltip_timer );
	}
	private->tooltip_timer = g_timeout_add( 2000, 
						(GSourceFunc)screem_editor_tooltip,
						editor );

	return FALSE;
}

static gboolean screem_editor_tooltip( ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	gint pos;
	gchar *text;
	ScreemPage *page;
	const gchar *mime_type;
	GModule *self;
	gboolean handled;

	GdkWindow *win;
	gint x;
	gint y;
	GdkModifierType mask;
	gint bx;
	gint by;
	GtkTextIter it;

	GtkTextBuffer *buffer;
	gboolean feature_markup;

	priv = editor->private;
	page = priv->page;

	if( ! page ) {
		return FALSE;
	}

gdk_threads_enter();
	
	mime_type = screem_page_get_mime_type( page );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	win = gtk_text_view_get_window( GTK_TEXT_VIEW( priv->view ),
				        GTK_TEXT_WINDOW_TEXT);
	gdk_window_get_pointer( win, &x, &y, &mask );

	gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(priv->view),
					      GTK_TEXT_WINDOW_TEXT,
					      x, y, &bx, &by );
	gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(priv->view),
					   &it, bx, by);
	pos = gtk_text_iter_get_offset( &it );

	text = screem_page_get_data( page );

	handled = FALSE;

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	feature_markup = screem_page_is_markup( page );

	if( feature_markup ) {
		ScreemWindow *window;

		g_object_get( G_OBJECT( editor ), "window", &window, NULL );
		handled = html_editor_tip( editor, window, pos, text );
	}

	g_free( text );

	priv->tooltip_timer = 0;
	
	gdk_threads_leave();
	
	return FALSE;
}

static gboolean html_editor_tip( ScreemEditor *editor, ScreemWindow *window,
				 gint pos, gchar *text )
{
	ScreemPage *page;
	ScreemDTD *dtd;
	gchar *tag;
	gchar *name;
	const ScreemDTDElement *element;
	gchar *message;

	page = editor->private->page;
	
	if( screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		tag = screem_markup_next_tag( text, pos, NULL, &pos, &name );
		/* so we can tooltip PHP/ASP etc functions */
		if( tag[ 1 ] == '?' || tag[ 1 ] == '!' ) {
			g_free( name );
			return FALSE;
		}
		/* we have the tag, look it up in the DTD */
		dtd = screem_page_get_dtd( page );
		element = screem_dtd_valid_element( dtd, name );
		if( ! element && *name != '/' ) {
			message = g_strdup_printf( _("%s is not a valid element of this documents DTD" ), name );
			screem_window_show_message( window, message );

			g_free( message );
		} else if( *name != '/' ) {
			/* its valid, but is it allowed here? */
			message = g_strdup( "" );

			g_free( message );
		}

		g_free( name );
		g_free( tag );
		return TRUE;
	}

	return FALSE;
}

static void screem_editor_colour_notify( GConfClient *client,
					 guint cnxn_id,
					 GConfEntry *entry,
					 gpointer data )
{
	screem_editor_set_colours( SCREEM_EDITOR( data ) );
}

static void screem_editor_wrap_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	GtkWrapMode wmode;
	
	editor = SCREEM_EDITOR( data );

	wmode = GTK_WRAP_NONE;
	if( entry->value && entry->value->type == GCONF_VALUE_BOOL &&
		gconf_value_get_bool( entry->value ) ) {

		wmode = GTK_WRAP_WORD;
	}
	gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( editor->private->view ), 
				     wmode);
}

static void screem_editor_tabwidth_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	gfloat width;	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_FLOAT ) {

		width = gconf_value_get_float( entry->value );
		
		gtk_source_view_set_tabs_width( GTK_SOURCE_VIEW( editor->private->view ), (gint)width );

	}
}

static void screem_editor_autocomplete_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		editor->private->autocomplete = gconf_value_get_bool( entry->value );
	}

}


static void screem_editor_clip_received( GtkClipboard *clipboard,
					 GtkSelectionData *selection_data,
					 gpointer data )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( data );

	if( selection_data->length > 0 ) {
		gchar *text = 
			screem_markup_encode_text( selection_data->data );
		screem_editor_insert( editor, -1, text );
		g_free( text );
	}
}

static void screem_editor_toggle_overwrite( GtkTextView *view,
					    ScreemEditor *editor )
{
	gchar *mesg;

	gtk_statusbar_pop( GTK_STATUSBAR( editor->private->overwriteMode ),
			   0 );

	editor->private->overwrite = ! editor->private->overwrite;

	if( editor->private->overwrite ) {
		mesg = g_strdup( _( "  OVR" ) );
	} else {
		mesg = g_strdup( _( "  INS" ) );
	}

	gtk_statusbar_push( GTK_STATUSBAR( editor->private->overwriteMode ),
			    0, mesg );

	g_free( mesg );
}

static gboolean build_attribute_list( ScreemEditor *editor )
{
    	GtkWidget *box;
	GtkBoxChild *bchild;
    
	GSList *set_attr;

	gchar *name;

	gint pos;
	gchar *text;
	gchar *tag;

	GList *list;

	ScreemDTD *dtd;
	ScreemPage *page;

	gboolean istag;

gdk_threads_enter();
	
	editor->private->handle = 0;

	page = editor->private->page;

	box = editor->private->attributes;

	text = NULL;
	istag = FALSE;
	if( page && screem_page_is_markup( page ) ) {
		/* id which tag we are in */
		pos = screem_editor_get_pos( editor );
		text = screem_page_get_data( page );
		
		istag = screem_markup_is_tag( text, pos, &pos, NULL );
	}
	if( istag ) {	
		if( editor->private->ppos == pos ) {
			g_free( text );
			gdk_threads_leave();
			return FALSE;
		}
		editor->private->ppos = pos;
	
		gtk_widget_hide( box );
		while( ( list = GTK_BOX( box )->children ) ) {
			bchild = list->data;
			gtk_container_remove( GTK_CONTAINER( box ),
					      bchild->widget );
		}
		
		tag = screem_markup_next_tag( text, pos, NULL, &pos, &name );
		if( tag ) {
			set_attr = screem_markup_build_attributes_list( tag, NULL );
			set_attr = g_slist_reverse( set_attr );
		} else {
			name = NULL;
			set_attr = NULL;
		}
		
		g_free( text );
		g_free( tag );
		
		if( name ) {
			GtkWidget *label;

			tag = g_strdup_printf( "<span size=\"xx-large\">&lt;%s&gt;</span>",
					name );
			label = gtk_label_new( tag );
			gtk_label_set_use_markup( GTK_LABEL( label ), TRUE );
			gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.0 );
			gtk_box_pack_start( GTK_BOX( box ), label, FALSE, FALSE, 0 );
			g_free( tag );
	
			dtd = screem_page_get_dtd( page );
			build_attribute_entries( editor, dtd, name, box,
						set_attr );
			/* destroy all of set_attr */
			g_slist_foreach( set_attr, (GFunc)g_free, NULL );
			g_slist_free( set_attr );

			g_free( name );
		}
		
		gtk_widget_set_sensitive( box, TRUE );
	} else {
		editor->private->ppos = -1;
		g_free( text );
		gtk_widget_hide( box );
		while( ( list = GTK_BOX( box )->children ) ) {
			bchild = list->data;
			gtk_container_remove( GTK_CONTAINER( box ),
					      bchild->widget );
		}
		gtk_widget_set_sensitive( box, FALSE );
	}
	gtk_widget_show_all( box );

	gdk_threads_leave();
	
	return FALSE;
}

static void build_attribute_entries( ScreemEditor *editor,
				     ScreemDTD *dtd, const gchar *name,
				     GtkWidget *box,
				     GSList *attr )
{
    	GtkWidget *line;
	GtkWidget *label;
	GtkWidget *combo;
	GtkWidget *entry;

	GList *combo_list;
	GSList *list;
	GtkSizeGroup *group;

	const ScreemDTDElement *element;
	const GSList *attrs;

	if( ! dtd ) {
		return;
	}

	element = screem_dtd_valid_element( dtd, name );
	if( ! element ) {
		return;
	}

	attrs = screem_dtd_element_get_attrs( element );

	group = gtk_size_group_new( GTK_SIZE_GROUP_BOTH );
	
	for( ;attrs; attrs = attrs->next ) {
		ScreemDTDAttribute *att;
		const gchar *attr_name;
		const GSList *values;
		gchar *tmp;
		
		att = (ScreemDTDAttribute*)attrs->data;
		attr_name = screem_dtd_attribute_get_name( att );
		
		values = screem_dtd_attribute_get_values( att );

		/* add entrys */
		line = gtk_table_new( 2, 1, FALSE );
		if( screem_dtd_attribute_get_required( att ) ) {
			tmp = g_strconcat( "<b>", attr_name, "</b>", NULL );
		} else {
			tmp = g_strdup( attr_name );
		}
		label = gtk_label_new( tmp );
		gtk_label_set_use_markup( GTK_LABEL( label ), TRUE );
		gtk_label_set_justify( GTK_LABEL( label ), GTK_JUSTIFY_LEFT );
		gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.5 );
		gtk_table_attach( GTK_TABLE( line ), label, 0, 1, 0, 1,
				  GTK_FILL, 
				  GTK_FILL, 4, 4 );
		gtk_size_group_add_widget( group, label );

		combo = gtk_combo_new();
		gtk_widget_set_size_request( combo, 128, -1 );
		entry = GTK_COMBO( combo )->entry;
		gtk_table_attach( GTK_TABLE( line ), combo, 1, 2, 0, 1,
				  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 4 );
		gtk_box_pack_start( GTK_BOX( box ), line, FALSE, FALSE, 0 );
		
		/* add combo entries */
		combo_list = NULL;
		for( ; values; values = values->next ) {
			combo_list = g_list_prepend( combo_list, values->data );
		}
		if( combo_list ) {
			combo_list = g_list_reverse( combo_list );
			gtk_combo_set_popdown_strings( GTK_COMBO( combo ), 
						       combo_list );
			g_list_free( combo_list );
		}
		
		/* 
		 * set the entry value to whatever value the
		 * attribute currently has in the tag, or make it
		 * empty if the attribute isn't used
		 */

		/* find name in the attr list */
		for( list = attr; list; list = list->next ) {
			if( ! strcmp( attr_name, (gchar*)list->data ) )
				break;
			list = list->next;
		}
		
		if( ! list ) {
			gtk_entry_set_text( GTK_ENTRY( entry ), "" );
		} else if( ! list->next ) {
			gtk_entry_set_text( GTK_ENTRY( entry ), "1" );
		} else {
			gtk_entry_set_text( GTK_ENTRY( entry ),
					    list->next->data );
		}

		g_object_set_data( G_OBJECT( entry ), "editor", editor );

		gtk_combo_disable_activate( GTK_COMBO( combo ) );
		g_signal_connect( G_OBJECT( entry ), "activate",
				  G_CALLBACK( replace_attr ), 
				  (gpointer)attr_name );

		gtk_widget_show_all( line );
	}
}

static void replace_attr( GtkWidget *entry, gchar *attr_name )
{
	ScreemEditor *editor;
	const gchar *text;
	gchar *attr;
	gint pos;

	editor = SCREEM_EDITOR( g_object_get_data( G_OBJECT( entry ),
						   "editor" ) );

	text = gtk_entry_get_text( GTK_ENTRY( entry ) );

	attr = g_strconcat( attr_name, "=\"", text, "\"", NULL );

	pos = screem_editor_get_pos( editor );
	screem_editor_insert_attribute( editor, attr );
	screem_editor_set_pos( editor, pos );

	g_free( attr );
}


static gboolean screem_editor_switch_mime( GtkWidget *widget, 
					   GdkEventButton *event,
					   ScreemEditor *editor )
{
	gnome_popup_menu_do_popup_modal( editor->private->types,
					 0, 0, event, 0, widget );
	gtk_widget_hide( editor->private->types );

	return TRUE;
}

static gboolean screem_editor_set_mime( ScreemEditor *editor )
{
	ScreemPage *page;
	const gchar *mime_type;

gdk_threads_enter();

	mime_type = g_object_get_data( G_OBJECT( editor ), "mime_type" );
	page = editor->private->page;
	screem_page_set_mime_type( page, mime_type );

	gdk_threads_leave();
	
	return FALSE;
}

static void screem_editor_mime_switch( GtkWidget *widget, 
				       ScreemEditor *editor )
{
	const gchar *mime_type;
	ScreemPage *page;

	page = editor->private->page;

	mime_type = g_object_get_data( G_OBJECT( widget ), "label" );

	g_object_set_data( G_OBJECT( editor ), "mime_type", 
		(gchar*)mime_type );

	g_idle_add_full( G_PRIORITY_HIGH_IDLE, 
			(GSourceFunc)screem_editor_set_mime,
			editor, NULL );
}

static gboolean screem_editor_drop_uris( ScreemEditor *editor,
					 Drop_actions action,
					 const gchar *selectionData,
					 guint info )
{
	gboolean online;
	GList *uris;
	GList *tmp;
	GnomeVFSURI *uri;
	
	ScreemPage *page;
	const gchar *pathname;
	gchar *mime_type;
	gboolean ret;

	guint pos;
	guint i;

	gchar **astext;
	
	g_object_get( G_OBJECT( editor ), "online", &online, NULL );
	page = editor->private->page;

	ret = FALSE;

	pathname = screem_page_get_pathname( page );

	uris = gnome_vfs_uri_list_parse( selectionData );
	astext = g_strsplit( selectionData, "\r\n", -1 );
	
	for( i = 0, tmp = uris; tmp; tmp = tmp->next, ++ i ) {
		gchar *url;
		uri = (GnomeVFSURI*)tmp->data;
		
		url = gnome_vfs_uri_to_string( uri, 0 );
		if( online || gnome_vfs_uri_is_local( uri ) ) {
			mime_type = screem_get_mime_type( url );
		} else {
			mime_type = NULL;
		}
		/* INLINE action selected, can only do this,
		   if we have a mime type that is for a text file */
		if( action == INLINE && 
		    screem_page_is_mime_type_page( mime_type ) ) {
			screem_editor_insert_file( editor,
						   url );
			g_free( url );
			url = NULL;
			ret = TRUE;
		} else if( action != FULL_PATH && pathname ) {
			gchar *dir;
			gchar *temp;
			
			dir = g_dirname( pathname );
			temp = url;
			url = relative_path( url, dir );
			g_free( dir );
				g_free( temp );
		}
		if( action == TAG ) {
			gchar *temp;

			temp = url;
			if( mime_type && ! strncmp(mime_type, "image/", 
						   strlen("image/") ) ) {
				url = g_strdup_printf( "<img src=\"%s\" alt=\"\">", temp );
			} else if( info == TARGET_MOZ && tmp->next ) {
				const gchar *title;

				/* astext should be 1/2 the size of
				   the url list */
				title = strchr( astext[ ( i / 2 ) ],
						'\n' );
				if( title ) {
					title ++;
					if( ! strcmp( title, temp ) ) {
						title = "";
					}
				} else {
					title = "";
				}
				url = g_strdup_printf( "<a href=\"%s\">%s</a>", temp, title );
			} else {
				url = g_strdup_printf( "<a href=\"%s\"> </a>",
						       temp );
			}
			g_free( temp );
		}
		if( url ) {
			pos = screem_editor_get_pos( editor );
			screem_editor_insert( editor, pos, url );
			g_free( url );
			ret = TRUE;
		}
		g_free( mime_type );	
		gnome_vfs_uri_unref( uri );

		/* skip title for text/x-moz-url */
		if( info == TARGET_MOZ && ! ( i % 2 ) ) {
			if( tmp->next ) {
				tmp = tmp->next;
				gnome_vfs_uri_unref( tmp->data );
			}
		}
	}
	g_list_free( uris );
	g_strfreev( astext );

	return ret;
}

static gboolean set_pos( ScreemEditorDelaySetPos *delay )
{
	ScreemEditor *editor;
	GtkTextView *view;

	g_return_val_if_fail( delay != NULL, TRUE );
	g_return_val_if_fail( SCREEM_IS_EDITOR( delay->editor ), TRUE );
	
	editor = delay->editor;
	view = GTK_TEXT_VIEW( editor->private->view );

gdk_threads_enter();
	
	screem_editor_set_pos( delay->editor, delay->pos );
	if( delay->adj ) {
		gtk_adjustment_set_value( view->vadjustment, delay->vadj );
	}
	g_free( delay );

	gdk_threads_leave();
	
	return FALSE;
}

static void screem_editor_set_pos_delayed( ScreemEditor *editor, 
					guint pos, 
					gboolean adj, gdouble val )
{
	ScreemEditorDelaySetPos *delay;

	g_return_if_fail( SCREEM_IS_EDITOR( editor ) );
	
	delay = g_new0(ScreemEditorDelaySetPos, 1 );
	delay->editor = editor;
	delay->pos = pos;
	delay->adj = adj;
	if( adj ) {
		delay->vadj = val;
	}
	g_idle_add_full( G_PRIORITY_DEFAULT_IDLE, 
			 (GSourceFunc)set_pos, delay, NULL );
	g_idle_add_full( G_PRIORITY_DEFAULT_IDLE,
			 (GSourceFunc)screem_editor_focus, 
			 editor, NULL );
}

static void screem_editor_buffer_change( ScreemPage *page, GParamSpec *spec,
					 ScreemEditor *editor )
{
	screem_editor_display_page( editor, page );
}

static void screem_editor_autocomplete_popup_delete( GtkWidget *widget,
						     gpointer data )
{
	gchar *str;
	ScreemEditor *editor;
	AutocompleteAfterFunc after;
	
	str = g_object_get_data( G_OBJECT( widget ), "string" );
	after = g_object_get_data( G_OBJECT( widget ), "after" );

	editor = SCREEM_EDITOR( data );
	screem_editor_insert( editor, -1, str );

	if( after && *str != '\0' ) {
		after( editor );
	}
	
	g_free( str );
}

static gboolean screem_editor_autocomplete_activate( GtkWidget *widget,
						     GtkTreePath *path,
						     GtkTreeViewColumn *coli,
						     gpointer data )
{
	return screem_editor_autocomplete_popup_event( widget, NULL, data );
}

static gboolean screem_editor_autocomplete_popup_event( GtkWidget *widget, 
							GdkEventKey *event,
							gpointer data )
{
	ScreemEditor *editor;
	GdkEventKey child;
	GtkWidget *popup;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gchar *str;
	gchar *temp;

	guint32 uni;
	
	gboolean dosel;
	gboolean ret;
	
	guint32 term;
	gsize len;
	guint pos;
	
	editor = SCREEM_EDITOR( data );

	pos = screem_editor_get_pos( editor );
	
	popup = gtk_widget_get_toplevel( widget );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	str = g_object_get_data( G_OBJECT( widget ), "string" );

	ret = dosel = FALSE;

	term = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget ), 
				"term" ) );
	if( ! event ) {
		event = &child;
		event->keyval = GDK_Return;
	} else if( event->keyval == term && *str == '\0' ) {
		gtk_widget_destroy( popup );
		ret = TRUE;
		/* pass term onto the editor */
		child = *event;
		child.window =	gtk_text_view_get_window( GTK_TEXT_VIEW(editor->private->view),
						  	  GTK_TEXT_WINDOW_TEXT );
		gtk_widget_event( editor->private->view, (GdkEvent*)&child );
		return ret;	
	}

	switch( event->keyval ) {
		case GDK_space:
			child = *event;
			child.window =	gtk_text_view_get_window( GTK_TEXT_VIEW(editor->private->view),
							  	  GTK_TEXT_WINDOW_TEXT );
			gtk_widget_event( editor->private->view, (GdkEvent*)&child );
			/* deliberate fall through */
		case GDK_Escape:
			g_free( str );
			g_object_set_data( G_OBJECT( widget ),
					   "string", g_strdup( "" ) );
			gtk_widget_destroy( popup );
			ret = TRUE;
			break;
		case GDK_Up:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				GtkTreePath *path;

				path = gtk_tree_model_get_path( model, &it );

				if( ! gtk_tree_path_prev( path ) ) {
					gtk_tree_model_get_iter_first( model, &it );
				} else {
					gtk_tree_model_get_iter( model, &it, path );
				}

				gtk_tree_path_free( path );
			} else {
				gtk_tree_model_get_iter_first( model, &it );
			}
			gtk_tree_selection_select_iter( sel, &it );
			path = gtk_tree_model_get_path( model, &it );
			column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
							   0 );
			gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW(  widget ),
						      path, column, FALSE, 
						      0.0, 0.0 );
			gtk_tree_path_free( path );
			ret = TRUE;
			break;
		case GDK_Down:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				gtk_tree_model_iter_next( model, &it );

			} else {
				gtk_tree_model_get_iter_first( model, &it );
			}
			gtk_tree_selection_select_iter( sel, &it );
			path = gtk_tree_model_get_path( model, &it );
			if( path ) {
				column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
								0 );
				gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ),
							      path, column, FALSE, 
							      0.0, 0.0 );
				gtk_tree_path_free( path );
			}
			ret = TRUE;
			break;
		case GDK_Return:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				GValue value = { 0 };
				temp = str;
				
				/* delete previously inserted text */
				len = g_utf8_strlen( str, -1 );
				if( len > 0 ) {
					screem_editor_delete_forward( editor, pos - len, len );
				}
				gtk_tree_model_get_value( model, &it, 0, &value );
				str = g_strdup( g_value_get_string( &value ) );
				g_value_unset( &value );
				g_object_set_data( G_OBJECT( widget ), "string", str );
				g_free( temp );
			}
			gtk_widget_destroy( popup );
			ret = TRUE;
			break;
		case GDK_BackSpace:
			len = g_utf8_strlen( str, -1 );
			if( len > 0 ) {
				temp = g_malloc0( strlen( str ) );
				g_utf8_strncpy( temp, str, len - 1 );
				g_free( str );
				str = temp;
				g_object_set_data( G_OBJECT( widget ), 
						   "string", str );
				ret = TRUE;
				dosel = TRUE;
				screem_editor_delete_forward( editor, pos - 1, 1 );
			}
			break;
		default:
			uni = gdk_keyval_to_unicode( event->keyval );
			if( g_unichar_isprint( uni ) ) {
				gchar out[ 6 ];

				len = g_utf8_strlen( str, -1 );
				memset( out, '\0', 6 );
				g_unichar_to_utf8( uni, out );
				temp = str;
				str = g_strconcat( str, out, NULL );
				g_free( temp );
				g_object_set_data( G_OBJECT( widget ), 
						   "string", str );
				ret = TRUE;
				dosel = TRUE;
				screem_editor_insert( editor, pos, out );
			}
			break;
	}
	
	/* possibly need to select a new row in the list view */
	if( dosel ) {
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
		if( gtk_tree_model_get_iter_first( model, &it ) ) {
			GValue value = { 0 };
			gboolean selected;
			selected = FALSE;
			do {
				gtk_tree_model_get_value( model, &it, 0, &value );
				if( ! strncmp( str, g_value_get_string( &value ), 
						strlen( str ) ) ) {
					gtk_tree_selection_select_iter( sel, &it );
					path = gtk_tree_model_get_path( model, &it );
					column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
									0 );
					gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ),
								      path, column, FALSE, 
								      0.0, 0.0 );
					gtk_tree_path_free( path );
					g_value_unset( &value );
					selected = TRUE;
					break;
				}
				g_value_unset( &value );
			} while( gtk_tree_model_iter_next( model, &it ) );
			if( ! selected ) {
				g_free( str );
				g_object_set_data( G_OBJECT( widget ),
						   "string", g_strdup( "" ) );
				gtk_widget_destroy( popup );
				ret = TRUE;
			}
		}
	}
	
	return ret;
}

static void screem_editor_autocomplete_popup( ScreemEditor *editor, const GSList *list, guint32 term, const gchar *str,
						AutocompleteAfterFunc after )
{
	ScreemWindow *window;
	GtkWidget *popup;
	GtkWidget *sw;
	GtkWidget *widget;
	GtkListStore *store;
	GtkTreeIter it;
	GtkTextBuffer *buffer;
	GtkTextIter tit;	
	gint pos;
	GdkRectangle rect;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
     
	g_return_if_fail( SCREEM_IS_EDITOR( editor ) );
	
	if( ! list ) {
		return;
	}
	
	g_object_get( G_OBJECT( editor ), "window", &window, NULL );

	/* get rect for cursor position */
	widget = editor->private->view;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
	pos = screem_editor_get_pos( editor );
	gtk_text_buffer_get_iter_at_offset( buffer, &tit, pos );
	gtk_text_view_get_iter_location( GTK_TEXT_VIEW( widget ), &tit, &rect );

	
	popup = gtk_window_new( GTK_WINDOW_POPUP );

	store = gtk_list_store_new( 1, G_TYPE_STRING );

	while( list ) {
		gtk_list_store_append( store, &it );
		gtk_list_store_set( GTK_LIST_STORE( store ), &it, 0, list->data, -1 );
		list = list->next;
	}

	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( popup ), sw );
	
	widget = gtk_tree_view_new_with_model( GTK_TREE_MODEL( store ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "completion" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_columns_autosize( GTK_TREE_VIEW( widget ) );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 0, NULL );
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ), FALSE );
	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( widget ), TRUE );
	g_object_unref( store );

	gtk_container_add( GTK_CONTAINER( sw ), widget );

	gtk_window_set_transient_for( GTK_WINDOW( popup ), GTK_WINDOW( window ) );
	gtk_window_set_skip_taskbar_hint( GTK_WINDOW( popup ), TRUE );
	gtk_window_set_skip_pager_hint( GTK_WINDOW( popup ), TRUE );
	gtk_window_set_decorated( GTK_WINDOW( popup ), FALSE );
	gtk_window_set_modal( GTK_WINDOW( popup ), TRUE );

	gtk_window_set_position( GTK_WINDOW( popup ), GTK_WIN_POS_CENTER_ON_PARENT );

	gtk_window_set_focus( GTK_WINDOW( popup ), widget );

	g_signal_connect( G_OBJECT( widget ), "key_press_event",
			  G_CALLBACK( screem_editor_autocomplete_popup_event ),
			  editor );
	g_signal_connect( G_OBJECT( widget ), "destroy",
			  G_CALLBACK( screem_editor_autocomplete_popup_delete ),
			  editor );
	g_signal_connect_after( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK( screem_editor_autocomplete_activate ),
			  editor );
			
	if( ! str ) {
		str = "";
	}
	g_object_set_data( G_OBJECT( widget ), "string", g_strdup( str ) );
	g_object_set_data( G_OBJECT( widget ), "term", GINT_TO_POINTER( term ) );
	g_object_set_data( G_OBJECT( widget ), "after", after );
	gtk_widget_show_all( popup );

	gtk_widget_set_size_request( popup, -1, 128 );	     
}

static void screem_editor_attr_after( ScreemEditor *editor )
{
	ScreemPage *page;
	GtkTextBuffer *buffer;
	gchar *attr;
	gchar *elem;
	GtkTextIter it;
	GtkTextIter *eit;
	guint pos;

	page = editor->private->page;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );
	pos = screem_editor_get_pos( editor );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	eit = gtk_text_iter_copy( &it );
	if( gtk_text_iter_backward_word_start( &it ) ) {
		attr = gtk_text_buffer_get_text( buffer, &it, eit, TRUE );
		while( gtk_text_iter_backward_char( &it ) &&
			gtk_text_iter_get_char( &it ) != '<' ) {}
	
		elem = NULL;
		if( gtk_text_iter_forward_char( &it ) ) {
			gunichar c;
			gtk_text_iter_free( eit );
			eit = gtk_text_iter_copy( &it );
			do {
				c = gtk_text_iter_get_char( &it );
			} while( c != ' ' && c != '>' && 
				gtk_text_iter_forward_char( &it ) );
			elem = gtk_text_buffer_get_text( buffer, eit, &it, TRUE );
		}	
	} else {
		attr = elem = NULL;
	}
	gtk_text_iter_free( eit );
	
	screem_editor_insert( editor, -1, "=\"\"" );

	pos = screem_editor_get_pos( editor );
	pos --;
	screem_editor_set_pos( editor, pos );

	if( attr ) {
		if( elem ) {
			ScreemDTD *dtd;
			const ScreemDTDAttribute *attribute;
			const GSList *list;
			
			dtd = screem_page_get_dtd( page );

			attribute = screem_dtd_valid_attr( dtd, elem, attr );
			list = NULL;
			if( attribute ) {
				list = screem_dtd_attribute_get_values( attribute );
			}
			screem_editor_autocomplete_popup( editor, list, 0, NULL, NULL );

			g_free( elem );
		}
		
		g_free( attr );
	}

}

static GtkWidget *screem_editor_attribute_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element )
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *subitem;
	GtkWidget *submenu;

	const ScreemDTDElement *el;
	const GSList *list;
	const GSList *values;

	if( ! dtd ) {
		return NULL;
	}

	el = screem_dtd_valid_element( dtd, element );

	if( ! el ) {
		return NULL;
	}

	item = gtk_menu_item_new_with_label( element );
	g_object_set_data( G_OBJECT( item ), "editor", editor );
	gtk_widget_show( item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), menu );

	list = screem_dtd_element_get_attrs( el );
	for( ; list; list = list->next ) {
		const ScreemDTDAttribute *attr;
		const gchar *name;
		const gchar *default_value;
		
		attr = (const ScreemDTDAttribute*)list->data;
		name = screem_dtd_attribute_get_name( attr );
		default_value = screem_dtd_attribute_get_default( attr );

		subitem = gtk_menu_item_new_with_label( name );
		g_object_set_data( G_OBJECT( subitem ), "editor", editor );
		gtk_widget_show( subitem );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), subitem );

		submenu = NULL;
		values = screem_dtd_attribute_get_values( attr );
		if( values || default_value ) {
			submenu = gtk_menu_new();
			gtk_menu_item_set_submenu( GTK_MENU_ITEM( subitem ),
						   submenu );
		}
		for( ; values || default_value; values = values->next ) {
			gchar *val;
			gchar *data;

			if( values ) {
				data = g_strdup( values->data );
			} else {
				data = g_strdup( default_value );
				default_value = NULL;
			}
			val = NULL;
			if( *data == '%' ) {
				val = screem_dtd_lookup_entity( dtd,
								data, 
								TRUE );
			}
			if( ! val ) {
				val = g_strdup( data );
			}
			g_free( data );
			
			if( ! values ) {
				data = g_strdup_printf( "%s (%s)", val,
							_( "Default" ) );
				subitem = gtk_separator_menu_item_new();
				gtk_widget_show( subitem );
				gtk_menu_shell_prepend( GTK_MENU_SHELL( submenu ), subitem );
				subitem = gtk_menu_item_new_with_label( data );
				g_free( data );
			} else {
				subitem = gtk_menu_item_new_with_label( val );
			}
			g_object_set_data( G_OBJECT( subitem ), 
					   "editor", editor );
			gtk_widget_show( subitem );
			
			data = g_strconcat( name, "=\"", val, "\"",
					    NULL );
			g_free( val );
			g_signal_connect_data( G_OBJECT( subitem ),
					       "activate",
					       G_CALLBACK(screem_editor_insert_attr_cb),
					       data,
					       (GClosureNotify)g_free,
					       0 );
			if( values ) {
				gtk_menu_shell_append( GTK_MENU_SHELL(submenu),
						       subitem );
			} else {
				gtk_menu_shell_prepend( GTK_MENU_SHELL( submenu ), subitem );
				break;
			}
		}
	}

	return item;
}

static GtkWidget *screem_editor_allowed_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element )
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *subitem;
	const ScreemDTDElement *el;
	
	item = NULL;

	if( dtd ) {
		el = screem_dtd_valid_element( dtd, element );
		
		if( el ) {
			const GSList *list;
			
			item = gtk_menu_item_new_with_label( _( "Valid Elements") );
			gtk_widget_show( item );

			menu = NULL;

			list = screem_dtd_element_get_allowed( el );

			if( list ) {
				menu = gtk_menu_new();
				gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),
							   menu );
			}
			for( ; list; list = list->next ) {
				subitem = gtk_menu_item_new_with_label( (const gchar*)list->data );
				gtk_widget_show( subitem );
				gtk_menu_append( GTK_MENU( menu ), subitem );
			}
		}
	}

	return item;
}

static void screem_editor_insert_attr_cb( GtkWidget *item, const gchar *attribute )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( g_object_get_data( G_OBJECT( item ),
						   "editor" ) );
	
	screem_editor_insert_attribute( editor, attribute );
}

static void screem_editor_insert_attribute( ScreemEditor *editor, 
				     const gchar *attribute )
{
	gint pos;
	gint cpos;
        gint start;
        gchar *insert;
        gchar *text;
        gchar *tag;
        gchar *attr;
	ScreemPage *page;

	gchar *temp;

	page = editor->private->page;
	
	if( ! page || ! attribute )
		return;

        cpos = pos = screem_editor_get_pos( editor );
        text = screem_page_get_data( page );

        if( ! screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		g_free( text );
		return;
        }
        start = pos;
        /* pos == the start of the tag */
        tag = screem_markup_next_tag( text, pos, NULL, &pos, NULL );

  	/* does the tag already have this attribute?
           = match:  [A-Za-z]*=\"[^"]*\"
           other: [A-Za-z]*
        */
      	temp = g_strconcat( " ", attribute, NULL );
        if( ( attr = find_text( temp, "=", NULL, NULL ) ) ) {
                attr = g_strndup( temp, attr - temp );
                insert = g_strdup_printf( "%s=\\\"[^\\\"]*\\\"", attr );
                g_free( attr );
        } else {
                insert = g_strdup( temp );
        }

        /* now we try and replace it */
        if( ! ( attr = find_text( tag, insert, temp, NULL ) ) ) {
                /* we didn't find it so we just insert it */
                g_free( insert );
		/* pos is at the end of the tag, however, we may
		   be dealing with xml, hence we need to insert
		   the attribute before a possible closing / */
		if( text[ pos - 1 ] == '/' ) {
			pos --;
		}
		insert = g_strdup_printf( " %s", attribute );
                screem_editor_insert( editor, pos, insert );
		pos += strlen( attribute ) + 1;
	} else {
                /* erase all the tag's attributes */
		g_free( tag );
                tag = attr;
		screem_editor_delete_forward( editor, start, pos - start + 1 );
                insert = g_strdup( tag );
                screem_editor_insert( editor, start, insert );
		pos = start + strlen( insert );
        }

        g_free( text );
	g_free( tag );
	g_free( insert );
	g_free( temp );

	editor->private->inserted = TRUE;

	screem_editor_set_pos( editor, pos );
}

static gboolean predicate( gunichar ch, gpointer data )
{
	return ( ch == (gunichar)data );
}

static void screem_editor_learn_more( GtkWidget *widget,
				ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	guint pos;
	GtkTextIter it;
	GtkTextIter eit;
	GtkTextBuffer *buffer;
	GtkTextTagTable *table;
	GtkTextTag *tag;
	gchar *txt;
	gchar *tmp;
	gchar *tmp2;
	gchar *attr;
	ScreemDTD *dtd;
	const gchar *publicid;
	gchar *url;
	
	priv = editor->private;
	pos = screem_editor_get_pos( editor );
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	table = gtk_text_buffer_get_tag_table( buffer );
	tag = gtk_text_tag_table_lookup( table, 
			SCREEM_INVALID_MARKUP_TAG );
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	eit = it;

	gtk_text_iter_backward_to_tag_toggle( &it, tag );
	gtk_text_iter_forward_to_tag_toggle( &eit, tag );

	/* it and eit are now the invalid part, this may be
	 * a full tag, or just an attribute in a tag */
	tmp = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );

	if( *tmp != '<' ) {
		/* txt is an attribute, we want the tag name as well */
		gtk_text_iter_backward_find_char( &it, predicate, 
						GUINT_TO_POINTER( '<' ),
						NULL );
		gtk_text_iter_forward_char( &it );
		eit = it;
		gtk_text_iter_forward_word_end( &eit );
		attr = tmp;
		txt = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
	} else {
		txt = g_utf8_strchr( tmp, -1, ' ' );
		tmp2 = g_utf8_strchr( tmp, -1, '>' );
		if( ( ! txt ) || ( tmp2 < txt ) ) {
			txt = tmp2;
			if( ! txt ) {
				txt = tmp + strlen( tmp ) - 1;
			}
		}
		txt = g_strndup( tmp + 1, txt - tmp - 1 );
		g_free( tmp );
		attr = NULL;
	}

	dtd = screem_page_get_dtd( SCREEM_PAGE( buffer ) );
	publicid = screem_dtd_get_public_id( dtd );

	/* txt == tag name, attr = attribute name or NULL,
	 * publicid is the DOCTYPE public identifier in use */
	url = g_strconcat( "http://www.screem.org/invalid.php?",
			"doctype=", publicid,
			"&tag=", txt,
			"&attr=", attr,
			NULL );
	
	gnome_url_show( url, NULL );
	
	g_free( url );
	g_free( txt );
	g_free( attr );
}

/* G Object stuff */

#define PARENT_TYPE SCREEM_TYPE_VIEW

static gpointer parent_class;

static void screem_editor_class_init( ScreemEditorClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_editor_finalize;

	widget_class->show = screem_editor_show;
	widget_class->hide = screem_editor_hide;
	widget_class->realize = screem_editor_realize;
	widget_class->size_request = screem_editor_size_request;

	widget_class->grab_focus = focus;
}

static void screem_editor_init( ScreemEditor *editor )
{
	GtkWidget *sw;
	GtkWidget *text;
	GdkPixbuf *pixbuf;
	GtkWrapMode wmode;
	gfloat tabwidth;
	ScreemEditorPrivate *private;

	SCREEM_VIEW( editor )->window_set = screem_editor_window_set;
	SCREEM_VIEW( editor )->display = screem_editor_display;
	SCREEM_VIEW( editor )->print = screem_editor_print;
	SCREEM_VIEW( editor )->undo = screem_editor_undo;
	SCREEM_VIEW( editor )->redo = screem_editor_redo;

	private = editor->private = g_new0( ScreemEditorPrivate, 1 );

	private->client = gconf_client_get_default();
	gconf_client_add_dir( private->client, "/apps/screem/editor",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL );

	private->handle = 0;
	private->popup_open = FALSE;
	private->inserted = FALSE;
	private->tooltip_timer = 0;

	
	private->view = text = gtk_source_view_new();
	gtk_source_view_set_show_line_numbers( GTK_SOURCE_VIEW(private->view),
						TRUE );
	gtk_source_view_set_show_line_markers( GTK_SOURCE_VIEW(private->view),
						TRUE );
	tabwidth = gconf_client_get_float( private->client,
			"/apps/screem/editor/tabwidth",
			NULL );
	gtk_source_view_set_tabs_width( GTK_SOURCE_VIEW( private->view ), (gint)tabwidth );
	gtk_source_view_set_auto_indent( GTK_SOURCE_VIEW( private->view ),
					TRUE );
	pixbuf = gtk_widget_render_icon( GTK_WIDGET( private->view ),
					 "Screem_Bookmark",
					 GTK_ICON_SIZE_MENU, "" );
	gtk_source_view_set_marker_pixbuf( GTK_SOURCE_VIEW( private->view ),
					   BOOKMARK_MARKER,
					   pixbuf );

	wmode = GTK_WRAP_NONE;
	if( gconf_client_get_bool( private->client,
				"/apps/screem/editor/wordwrap",
				NULL ) ) {
		wmode = GTK_WRAP_WORD;
	}
	gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( text ), 
				     wmode );
	gtk_text_view_set_editable( GTK_TEXT_VIEW( text ),
				    FALSE );
		
	gtk_widget_show( text );

	sw = private->sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_widget_show( sw );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), text );

	g_signal_connect( G_OBJECT( text ), "button_press_event",
			  G_CALLBACK( screem_editor_popup ), editor );
	
	g_signal_connect( G_OBJECT( text ), "button_release_event",
			  G_CALLBACK( screem_editor_button_release ),
			  editor );

	g_signal_connect( G_OBJECT( text ), "popup_menu",
			  G_CALLBACK( screem_editor_popup_key ), editor );
	
	g_signal_connect( G_OBJECT( text ), "key_press_event",
			  G_CALLBACK( screem_editor_keypress ), editor );
	
	gtk_drag_dest_set( text,
			   GTK_DEST_DEFAULT_MOTION | 
			   GTK_DEST_DEFAULT_HIGHLIGHT,
			   drop_types, num_drop_types,
			   GDK_ACTION_COPY | GDK_ACTION_ASK );
	
	g_signal_connect( G_OBJECT( text ), "drag_data_received",
			  G_CALLBACK( screem_editor_drop ), editor );
	g_signal_connect( G_OBJECT( text ), "drag_motion",
			  G_CALLBACK( screem_editor_motion ), editor );

	g_signal_connect( G_OBJECT( editor->private->view ),
			  "toggle_overwrite",
			  G_CALLBACK( screem_editor_toggle_overwrite ),
			  editor );

	private->autocomplete = gconf_client_get_bool( private->client,
			"/apps/screem/editor/autocomplete",
			NULL );
	
	private->notifies[ 0 ] = 
		gconf_client_notify_add( private->client,
					 "/apps/screem/editor/back",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies[ 1 ] = 
		gconf_client_notify_add( private->client,
					 "/apps/screem/editor/text",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies[ 2 ] = 
		gconf_client_notify_add( private->client,
					 "/apps/screem/editor/font",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies[ 3 ] =
		gconf_client_notify_add( private->client,
					 "/apps/screem/editor/themecolours",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies[ 4 ] =
		gconf_client_notify_add( private->client,
					 "/apps/screem/editor/themefont",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies[ 5 ] =
		gconf_client_notify_add( private->client,
				"/apps/screem/editor/wordwrap",
				screem_editor_wrap_notify,
				editor, NULL, NULL );
	private->notifies[ 6 ] =
		gconf_client_notify_add( private->client,
				"/apps/screem/editor/tabwidth",
				screem_editor_tabwidth_notify,
				editor, NULL, NULL );
	private->notifies[ 7 ] =
		gconf_client_notify_add( private->client,
				"/apps/screem/editor/autocomplete",
				screem_editor_autocomplete_notify,
				editor, NULL, NULL );
	
	g_signal_connect_data( G_OBJECT( text ),
			       "motion_notify_event",
			       G_CALLBACK( screem_editor_tip ), 
			       editor,
			       NULL, G_CONNECT_SWAPPED );

	gtk_container_add( GTK_CONTAINER( editor ), sw );

}

static void screem_editor_finalize( GObject *object )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	int i;
	
	editor = SCREEM_EDITOR( object );
	private = editor->private;

	g_free( private->markname );
	
	for( i = 0; i < 6; i ++ ) {
		gconf_client_notify_remove( private->client,
					    private->notifies[ i ] );
	}
	g_object_unref( private->client );
	
	if( private->handle ) {
		g_source_remove( private->handle );
	}
	if( private->tooltip_timer ) {
		g_source_remove( private->tooltip_timer );
	}

	if( private->types ) {
		gtk_widget_destroy( private->types );
	}

	g_free( private );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_editor_window_set( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	ScreemWindow *window;
	GtkWidget *sw;
	GtkTooltips *tip;

	editor = SCREEM_EDITOR( view );
	private = editor->private;

	g_object_get( G_OBJECT( view ), "window", &window, NULL );

	private->markname = g_strdup_printf( "ScreemEditor_%s\n",
					"fixme" );
	
	private->overwriteMode = gtk_statusbar_new();
	gtk_statusbar_set_has_resize_grip( GTK_STATUSBAR( private->overwriteMode ),
						TRUE );
	gtk_widget_show( private->overwriteMode );
	gtk_box_pack_end( GTK_BOX( window->status ), private->overwriteMode,
				FALSE, FALSE, 0 );
	gtk_widget_set_size_request( private->overwriteMode, 80, 10 );
	
	private->cursorPosition = gtk_statusbar_new();
	gtk_statusbar_set_has_resize_grip( GTK_STATUSBAR( private->cursorPosition ),
						FALSE );
	gtk_widget_show( private->cursorPosition );
	gtk_widget_set_size_request( private->cursorPosition, 150, 10 );	
	gtk_box_pack_end( GTK_BOX( window->status ), private->cursorPosition, 
				FALSE, FALSE, 0 );

	private->mime = gtk_button_new();
	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, private->mime,
			_( "Change type of document" ), "" );
	gtk_button_set_relief( GTK_BUTTON( private->mime  ), GTK_RELIEF_NONE );
	gtk_widget_show( private->mime  );
	gtk_box_pack_end( GTK_BOX( window->status ), private->mime,
				FALSE, FALSE, 0 );
	g_signal_connect( G_OBJECT( private->mime ), "button_press_event",
			  G_CALLBACK( screem_editor_switch_mime ),
			  editor );
	gtk_widget_set_size_request( private->mime , 120, 10 );

	private->overwrite = TRUE;
	screem_editor_toggle_overwrite( GTK_TEXT_VIEW( private->view ),
					editor );
	
	/* add attributes tab */
	private->attrtab = sw = 
		gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				       GTK_POLICY_AUTOMATIC, 
				       GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	private->attributes = gtk_vbox_new( 0, FALSE );
	gtk_container_set_border_width( GTK_CONTAINER( private->attributes ), 6 );
	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW(sw),
					       private->attributes );
	gtk_widget_show_all( sw );

	screem_window_add_dock_item( window, _( "Attributes" ), 
					GTK_STOCK_INDEX,
					"Attributes",
					sw, GTK_POS_RIGHT );
	
}	

static void screem_editor_size_request( GtkWidget *widget, GtkRequisition *req )
{
	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void screem_editor_realize( GtkWidget *widget )
{
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void screem_editor_show( GtkWidget *widget )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	
	editor = SCREEM_EDITOR( widget );
	private = editor->private;
	
	screem_editor_set_colours( editor );

	gtk_widget_set_sensitive( private->attrtab, TRUE );

	GTK_WIDGET_CLASS( parent_class )->show( widget );

	gtk_widget_show( private->view );
	gtk_widget_show( private->sw );
	
	gtk_widget_set_sensitive( private->overwriteMode, TRUE );
	gtk_widget_set_sensitive( private->cursorPosition, TRUE );
	gtk_widget_set_sensitive( private->mime, TRUE );
}

static void screem_editor_hide( GtkWidget *widget )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	
	editor = SCREEM_EDITOR( widget );
	private = editor->private;
	
	gtk_widget_set_sensitive( private->overwriteMode, FALSE );
	gtk_widget_set_sensitive( private->cursorPosition, FALSE);
	gtk_widget_set_sensitive( private->mime, FALSE );
	
	gtk_widget_set_sensitive( private->attrtab, FALSE );
	
	GTK_WIDGET_CLASS( parent_class )->hide( widget );
}

GType screem_editor_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemEditorClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_editor_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemEditor ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_editor_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemEditor",
					       &info, 0 );
	}

	return type;
}
