/*  Server-Side Includes Wizard for screem.
 *  Copyright (C) 1999 Lee Mallabone
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <libgnome/gnome-i18n.h>

#include <gtk/gtknotebook.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkentry.h>

#include <glade/glade.h>

#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "libegg/menu/egg-menu.h"

typedef enum {
	CONFIGURATION = 0,
	VARIABLES,
	EXECUTE,
	ECHO,
	INCLUDE,
	FILE_INFORMATION,
	CONDITIONALS
} SsiDirectives;

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

	GladeXML *xml;
	GtkWidget *dialog;
} SsiWizard;

void ssi_wizard_type_switch( GtkWidget *widget );
void ssi_wizard_clicked( GtkWidget *dialog, gint button, gpointer data );

/* keep track of how many Ssi wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Ssi Wizard"
#define AUTHOR  "Lee Mallabone, David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );
void ssi_wizard_display( EggAction *action, gpointer user_data );
static void ssi_wizard_create( SsiWizard *wizard );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	SsiWizard *Ssi_wizard;

	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"SsiWizard\" verb=\"SsiWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"SsiWizard\" verb=\"SsiWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "SsiWizard", "label",
	  	GTK_STOCK_ADD, NULL, "tip",
	  	G_CALLBACK( ssi_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;
	gchar *label;
	gchar *tip;

	Ssi_wizard = g_new0( SsiWizard, 1 );
	Ssi_wizard->window = SCREEM_WINDOW( window );
	Ssi_wizard->editor = SCREEM_EDITOR( editor );
	
	label = g_strdup( _( "SSI Wizard" ) );
	tip = g_strdup( _( "Insert a Server Side Include Directive" ) );
	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = Ssi_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( Ssi_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( Ssi_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	
	g_free( label );
	g_free( tip );

	ssi_wizard_create( Ssi_wizard );

	wizards = g_list_append( wizards, Ssi_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	SsiWizard *Ssi_wizard;

	for( list = wizards; list; list = list->next ) {
		Ssi_wizard = (SsiWizard*)list->data;
		if( Ssi_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* Ssi_wizard is the one to erase */
	wizards = g_list_remove( wizards, Ssi_wizard );
	g_free( Ssi_wizard );
}

/* End of required section */

void ssi_wizard_display( EggAction *action, gpointer user_data )
{
	SsiWizard *wizard;
	ScreemPage *page;

	wizard = (SsiWizard*)user_data;

	page = screem_window_get_document( wizard->window );

	if( page ) {
		gtk_widget_show_all( wizard->dialog );
		gdk_window_raise( wizard->dialog->window );
	}
}

static void ssi_wizard_create( SsiWizard *wizard )
{
	GtkWidget *widget;
	GtkWidget *menu;
	gint i;

	static const gchar *types[] = {
		"Configuration",
		"Variables",
		"Execute",
		"Echo",
		"Include",
		"File Information",
		"Conditionals",
		NULL
	};

	static const gchar *sizes[] = {
		"Bytes", "bytes",
		"Kb or Mb", "abbrev",
		NULL
	};

	static const gchar *execute[] = {
		"CGI", "cgi",
		"Shell", "cmd",
		NULL
	};

	static const gchar *echo[] = {
		"Date (GMT)", "DATE_GMT",
		"Date (Local)", "DATE_LOCAL",
		"Document Name", "DOCUMENT_NAME",
		"Document URI", "DOCUMENT_URI",
		"Last Modified", "LAST_MODIFIED",
		NULL
	};

	wizard->xml = glade_xml_new( GLADE_PATH"/ssiwizard.glade",
				     "ssiwizard", NULL );
	wizard->dialog = glade_xml_get_widget( wizard->xml,
					       "ssiwizard" );
	g_object_set_data( G_OBJECT( wizard->dialog ), "ssi_wizard", wizard );

	widget = glade_xml_get_widget( wizard->xml, "directives" );
	gtk_notebook_set_show_tabs( GTK_NOTEBOOK( widget ), FALSE );

	widget = glade_xml_get_widget( wizard->xml, "directive_type" );
	menu = gtk_menu_new();
	for( i = 0; types[ i ]; ++ i ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( types[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "type", GINT_TO_POINTER( i ) );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( wizard->xml, "config_size" );
	menu = gtk_menu_new();
	for( i = 0; sizes[ i ]; i += 2 ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( sizes[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "size",(gpointer) sizes[ i + 1 ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( wizard->xml, "execute_menu" );
	menu = gtk_menu_new();
	for( i = 0; execute[ i ]; i += 2 ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( execute[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "exec",(gpointer) execute[ i + 1 ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( wizard->xml, "echo_menu" );
	menu = gtk_menu_new();
	for( i = 0; echo[ i ]; i += 2 ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( echo[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "echo",(gpointer) echo[ i + 1 ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );


	glade_xml_signal_autoconnect( wizard->xml );
}


void ssi_wizard_type_switch( GtkWidget *widget )
{
	GladeXML *xml;
	gint page;

	xml = glade_get_widget_tree( widget );
	widget = GTK_OPTION_MENU( widget )->menu_item;
	page = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget ),
						   "type" ) );

	widget = glade_xml_get_widget( xml, "directives" );
	gtk_notebook_set_current_page( GTK_NOTEBOOK( widget ), page );
}

void ssi_wizard_clicked( GtkWidget *dialog, gint button, gpointer data )
{
	SsiWizard *wizard;

	wizard = (SsiWizard*)g_object_get_data( G_OBJECT( dialog ),
						"ssi_wizard" );
	

	if( button == GTK_RESPONSE_APPLY || button == GTK_RESPONSE_OK ) {
		GtkWidget *widget;
		gint page;
		const gchar *path;
		const gchar *mode;
		const gchar *name;
		const gchar *value;
		const gchar *msg;
		const gchar *timemsg;
		const gchar *sizefmt;
		const gchar *expr;
		gchar *text;
		GladeXML *xml;

		xml = wizard->xml;
		text = NULL;
		widget = glade_xml_get_widget( xml, "directives" );
		page = gtk_notebook_get_current_page( GTK_NOTEBOOK( widget ) );
		switch( page ) {
		case CONFIGURATION:
			widget = glade_xml_get_widget( xml, "error_message" );
			msg = gtk_entry_get_text( GTK_ENTRY( widget ) );

			widget = glade_xml_get_widget( xml, "time_format" );
			timemsg = gtk_entry_get_text( GTK_ENTRY( widget ) );

			widget = glade_xml_get_widget( xml, "config_size" );
			widget = GTK_OPTION_MENU( widget )->menu_item;
			sizefmt = (const gchar*)g_object_get_data( G_OBJECT( widget ),
								   "size" );

			text = g_strconcat( "<!--#config errmsg=\"", msg,
					    "\" sizefmt=\"", sizefmt,
					    "\" timefmt=\"", timemsg,
					    "\" -->", NULL );
			break;
		case VARIABLES:
			widget = glade_xml_get_widget( xml, "print_var" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#printenv -->" );
			} else {
				widget = glade_xml_get_widget( xml,
							       "var_name" );
				name = gtk_entry_get_text( GTK_ENTRY(widget) );
				widget = glade_xml_get_widget( xml,
							       "var_value" );
				value = gtk_entry_get_text( GTK_ENTRY(widget));
				text = g_strconcat( "<!--#set var=\"",
						    name,
						    " \" value=\"",
						    value,
						    "\" -->", NULL );
			}
			break;
		case EXECUTE:
			widget = glade_xml_get_widget( xml, "exec_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "execute_menu" );
			widget = GTK_OPTION_MENU( widget )->menu_item;
			mode = (const gchar*)g_object_get_data( G_OBJECT( widget ), "exec" );
			text = g_strconcat( "<!--#exec ", mode,
					    "=\"", path, "\" -->", NULL );
			break;
		case ECHO:
			widget = glade_xml_get_widget( xml, "echo_menu" );
			widget = GTK_OPTION_MENU( widget )->menu_item;
			mode = (const gchar*)g_object_get_data( G_OBJECT( widget ), "echo" );
			text = g_strconcat( "<!--#echo var=\" ", mode,
					    "\" -->", NULL );
			break;
		case INCLUDE:
			widget = glade_xml_get_widget( xml, "include_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			text = g_strconcat( "<!--#include virtual=\"",
					    path,
					    "\" -->", NULL );
			break;
		case FILE_INFORMATION:
			widget = glade_xml_get_widget( xml, "info_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "info_size" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				mode = "fsize";
			} else {
				mode = "flastmod";
			}
			text = g_strconcat( "<!--#", mode,
					    " virtual=\"", path, "\" -->",
					    NULL );
			break;
		case CONDITIONALS:
			widget = glade_xml_get_widget( xml, "endif" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#endif -->" );
			}
			widget = glade_xml_get_widget( xml, "else" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#else -->" );
			}
			widget = glade_xml_get_widget( xml, "expression" );
			expr = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "elif" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strconcat( "<!--#elif expr=\"",
						    expr, "\" -->", NULL );
			}
			widget = glade_xml_get_widget( xml, "if" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strconcat( "<!--#if expr=\"",
						    expr, "\" -->", NULL );
			}
			break;
		default:
			text = NULL;
			break;
		}
		if( text ) {
			gint pos;
			
			pos = screem_editor_get_pos( wizard->editor );
			screem_editor_insert( wizard->editor, pos, text );
			screem_editor_set_pos( wizard->editor, 
					       pos + strlen( text ) );
			g_free( text );
		}
	}
	if( button != GTK_RESPONSE_APPLY ) {
		gtk_widget_hide( dialog );
        }
}
