/*  Screem:  entityWizard.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <glib/gunicode.h>

#include <glade/glade.h>

#include <libgnomeui/gnome-stock-icons.h>
#include <libgnome/gnome-i18n.h>

#include <gtk/gtktoolbar.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtktreeselection.h>

#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "entities.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	GladeXML *xml;
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;

	GtkListStore *iso_store;
	GtkListStore *math_store;
	GtkListStore *markup_store;
} EntityWizard;


/* keep track of how many wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Entity Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

void entity_wizard_display( EggAction *action, gpointer user_data );
static void populate_lists( EntityWizard *entity_wizard );

void entity_row_selected( GtkTreeView *treeview,
			  GtkTreePath *path,
			  GtkTreeViewColumn *column,
			  gpointer data );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	EntityWizard *entity_wizard;

	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"EntityWizard\" verb=\"EntityWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"EntityWizard\" verb=\"EntityWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "EntityWizard", "label",
	  	"Screem_Entity", NULL, "tip",
	  	G_CALLBACK( entity_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;
	
	gchar *label;
	gchar *tip;

	entity_wizard = g_new0( EntityWizard, 1 );
	entity_wizard->window = SCREEM_WINDOW( window );
	entity_wizard->editor = SCREEM_EDITOR( editor );

	label = g_strdup( _( "Entity Wizard" ) );
	tip = g_strdup( _( "Select an Entity to Insert" ) );

	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = entity_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( entity_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( entity_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	
	g_free( label );
	g_free( tip );
	
	/* NOTE: make sure that the window being obtained isn't visible */
	entity_wizard->xml = glade_xml_new( GLADE_PATH"/entityWizard.glade",
					    "ent_wizard", NULL );
        entity_wizard->dialog = glade_xml_get_widget( entity_wizard->xml,
						      "ent_wizard" );
	populate_lists( entity_wizard );

        glade_xml_signal_autoconnect( entity_wizard->xml );

	g_object_set_data( G_OBJECT( entity_wizard->dialog ),
			   "entity_wizard", entity_wizard );

	wizards = g_list_append( wizards, entity_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	EntityWizard *entity_wizard;

	for( list = wizards; list; list = list->next ) {
		entity_wizard = (EntityWizard*)list->data;
		if( entity_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, entity_wizard );
	g_free( entity_wizard );
}

/* End of required section */

void entity_wizard_display( EggAction *action, gpointer user_data )
{
	EntityWizard *entity_wizard;
	ScreemPage *page;
 
	entity_wizard = (EntityWizard*)user_data;

	page = screem_window_get_document( entity_wizard->window );
        
	if( ! page )
		return;

	gtk_widget_show_all( entity_wizard->dialog );
	gdk_window_raise( entity_wizard->dialog->window );
}

static void populate_lists( EntityWizard *entity_wizard )
{
	GtkWidget *list;
	gint num;
	GladeXML *xml;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	
	xml = entity_wizard->xml;

	list = glade_xml_get_widget( xml, "iso_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	entity_wizard->iso_store = gtk_list_store_new( 2,
						       G_TYPE_STRING,
						       G_TYPE_STRING, NULL );
	for( num = 0; ISO_8859_1[ num ]; num ++ ) {
		GtkTreeIter it;

		/* ucs for ISO-8859-1 entity is 160 + num */
		gunichar c = (gunichar)( 160 + num );
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );
		
		gtk_list_store_append( entity_wizard->iso_store, &it );

		gtk_list_store_set( entity_wizard->iso_store, &it,
				    0, outbuf,
				    1, ISO_8859_1[ num ], -1 );
	}
	gtk_tree_view_set_model( GTK_TREE_VIEW( list ),
				 GTK_TREE_MODEL( entity_wizard->iso_store ) );


	list = glade_xml_get_widget( xml, "math_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	entity_wizard->math_store = gtk_list_store_new( 2,
							G_TYPE_STRING,
							G_TYPE_STRING, NULL );
	for( num = 0; MATH_GREEK_SYMBOLIC[ num ]; num ++ ) {
		GtkTreeIter it;
		gunichar c = MATH_GREEK_SYMBOLIC_ucs[ num ];
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );
		gtk_list_store_append( entity_wizard->math_store, &it );
		gtk_list_store_set( entity_wizard->math_store, &it,
				    0, outbuf,
				    1, MATH_GREEK_SYMBOLIC[ num ], -1 );
	}
	gtk_tree_view_set_model( GTK_TREE_VIEW( list ),
				 GTK_TREE_MODEL( entity_wizard->math_store ) );

	list = glade_xml_get_widget( xml, "markup_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	entity_wizard->markup_store = gtk_list_store_new( 2,
							  G_TYPE_STRING,
							  G_TYPE_STRING,NULL );
	for( num = 0; MARKUP_INTL[ num ]; num ++ ) {
		GtkTreeIter it;
		gunichar c = MARKUP_INTL_ucs[ num ];
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );

		gtk_list_store_append( entity_wizard->markup_store, &it );
		gtk_list_store_set( entity_wizard->markup_store, &it,
				    0, outbuf,
				    1, MARKUP_INTL[ num ], -1 );
	}
	gtk_tree_view_set_model(GTK_TREE_VIEW( list ),
				GTK_TREE_MODEL( entity_wizard->markup_store ));
}

void entity_row_selected( GtkTreeView *treeview,
			  GtkTreePath *path,
			  GtkTreeViewColumn *column,
			  gpointer data )
{
	GladeXML *xml;
	GtkWidget *dialog;
	EntityWizard *wizard;

	GtkTreeModel *model;
	GtkTreeSelection *sel;
	GtkTreeIter iter;
    
	xml = glade_get_widget_tree( GTK_WIDGET( treeview ) );
	dialog = glade_xml_get_widget( xml, "ent_wizard" );
	wizard = (EntityWizard*)g_object_get_data( G_OBJECT( dialog ),
						   "entity_wizard" );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( treeview ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( treeview ) );

	if( sel ) {
		const gchar *ent;
		GValue value = {0,};
		gint pos;
		gchar *text;
		
		gtk_tree_selection_get_selected( sel, &model, &iter );
		gtk_tree_model_get_value( model, &iter, 1, &value );
		ent = g_value_get_string( &value );
		pos = screem_editor_get_pos( wizard->editor );
		text = g_strdup_printf( "&%s;", ent );
		screem_editor_insert( wizard->editor, pos, text );
		screem_editor_set_pos( wizard->editor, pos + strlen( text ) );
		g_free( text );
		g_value_unset( &value );
	}
}
