/*  Screem:  screem-window.c
 *
 *  The ScreemCSSWindow widget
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-ui-component.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-popup-menu.h>

#include <libgnomevfs/gnome-vfs-uri.h>

#include <gtk/gtkhbox.h>
#include <gtk/gtknotebook.h>

#include <glade/glade.h>

#include <gtk/gtktreeview.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkhpaned.h>
#include <gtk/gtkvpaned.h>

#include <gconf/gconf-client.h>

#include "screem-window.h"
#include "screem-editor.h"

#include "css-window.h"
#include "css.h"

#include "fileops.h"
#include "support.h"

enum {
	ARG_0,
	ARG_APP_ID,
	ARG_APP
};

static void screem_css_window_initialise_class( ScreemCSSWindowClass *klass );
static void screem_css_window_initialise( ScreemCSSWindow *window );
static void screem_css_window_finalize( GObject *object );
static void screem_css_window_set_prop( GObject *object, guint prop_id, 
					const GValue *value, 
					GParamSpec *spec );
static void screem_css_window_get_prop( GObject *object, guint prop_id, 
					GValue *value, GParamSpec *spec );
static void screem_css_window_size_request( GtkWidget *widget, 
					    GtkRequisition *req );
static void screem_css_window_realize( GtkWidget *widget );
static void screem_css_window_show( GtkWidget *widget );



static void style_selected( GtkTreeSelection *selection );
static void recent_selected( GeditRecent *recent, const gchar *uri,
			     gpointer data );

ScreemCSSWindow *screem_css_window_new( void )
{
	ScreemCSSWindow *window;
	GType type;

	type = screem_css_window_get_type();

	window = SCREEM_CSS_WINDOW( g_object_new( type, NULL ) );

	window->preview = screem_preview_new();
	screem_preview_display( window->preview, "<html><body></body></html>",
				NULL, FALSE );
	
	return window;
}

void
screem_css_window_close( ScreemCSSWindow *window )
{
	g_return_if_fail( SCREEM_CSS_IS_WINDOW( window ) );

	gtk_widget_destroy( GTK_WIDGET( window ) );
}


Bonobo_UIContainer screem_css_window_get_ui_container(ScreemCSSWindow *window)
{
	g_return_val_if_fail( SCREEM_CSS_IS_WINDOW( window ),
			      CORBA_OBJECT_NIL );

	return bonobo_object_corba_objref(BONOBO_OBJECT(window->ui_container));
}

void screem_css_window_constructed( ScreemCSSWindow *csswindow,
				    ScreemWindow *window,
				    ScreemEditor *editor )
{
	CORBA_Environment ev;

	GtkWidget *content;
	GtkWidget *tree;
	GtkWidget *widget;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;

	csswindow->xml = glade_xml_new( GLADE_PATH"/css.glade", 
					"CaSSIUS", NULL );

	csswindow->window = SCREEM_WINDOW( window );
	csswindow->editor = SCREEM_EDITOR( editor );

	CORBA_exception_init( &ev );

	gtk_window_set_wmclass( GTK_WINDOW( csswindow ), "Screem", 
				"CSSWindow"  );
	/* Bonobo setup */
	csswindow->ui_container = bonobo_window_get_ui_container( BONOBO_WINDOW( csswindow ) );
	
	csswindow->ui = bonobo_ui_component_new( "Screem CSS Window" );
	bonobo_ui_component_set_container( csswindow->ui,
					   screem_css_window_get_ui_container(csswindow),
					   &ev);
	CORBA_exception_free( &ev );

	bonobo_ui_component_freeze( csswindow->ui, NULL );
	bonobo_ui_util_set_ui( csswindow->ui,
			       UIDATADIR,
			       "screem-css-window-ui.xml",
			       "screem", &ev );
	bonobo_ui_engine_config_set_path( bonobo_window_get_ui_engine( BONOBO_WINDOW( csswindow ) ), "/apps/screem/CSSUIConfig/kvps" );
	bonobo_ui_component_thaw( csswindow->ui, NULL );

	content = glade_xml_get_widget( csswindow->xml, 
					"CaSSIUS" );
	bonobo_window_set_contents( BONOBO_WINDOW( csswindow ), content );

	glade_xml_signal_autoconnect( csswindow->xml );

	csswindow->current_page = CSS_FONT;
	csswindow->closing = FALSE;

	csswindow->model = GTK_TREE_MODEL( gtk_tree_store_new( COL_MAX,
							       G_TYPE_STRING,
							       G_TYPE_POINTER,
							       NULL ) );
	tree = glade_xml_get_widget( csswindow->xml, "styles" );
	g_object_set_data( G_OBJECT( tree ), "window", csswindow );
       	gtk_tree_view_set_model( GTK_TREE_VIEW( tree ), csswindow->model );

	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Pattern" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( tree ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", COL_PATTERN,
					     NULL );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( style_selected ), NULL );

	widget = glade_xml_get_widget( csswindow->xml, "preview_placeholder" );
	gtk_container_remove( GTK_CONTAINER( content ), widget );
	gtk_paned_add2( GTK_PANED( content ), 
			GTK_WIDGET( csswindow->preview ) );
	gtk_widget_show( GTK_WIDGET( csswindow->preview ) );
	

	screem_css_window_initialise_menus( csswindow );
#if 0
	csswindow->recent_sheets = 
		gedit_recent_new_with_ui_component( "screem-styles", 4,
						    csswindow->ui,
						    "/menu/File/RecentSheets");
	g_signal_connect( G_OBJECT( csswindow->recent_sheets ),
			  "activate",
			  G_CALLBACK( recent_selected ), csswindow );
#endif
	g_signal_connect( G_OBJECT( csswindow ), "delete_event",
			  G_CALLBACK( gtk_widget_hide ), NULL );

	gtk_window_set_default_size( GTK_WINDOW( csswindow ),
				     500, 400 );
}

/* Static stuff */
static void style_selected( GtkTreeSelection *selection )
{
	GtkTreeView *tree;
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean selected;
	
	tree = gtk_tree_selection_get_tree_view( selection );
	selected = gtk_tree_selection_get_selected( selection, &model, &it );
	
	screem_css_style_selected( tree, selected );
}

static void recent_selected( GeditRecent *recent, const gchar *uri,
			     gpointer data )
{
	ScreemCSSWindow *window;
	GString *entireFile;

	window = SCREEM_CSS_WINDOW( data );
	
	entireFile = load_file( uri );
	if( window->filename )
		g_free( window->filename );
	window->filename = NULL;

	if( entireFile ) {
		gchar *title;
		
		css_parse( window, entireFile );
		g_string_free( entireFile, TRUE );
		title = g_strconcat( uri,
					" - Screem CSS Wizard", NULL );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );
	}
	window->filename = g_strdup( uri );
}

/* GObject Stuff */

#define PARENT_TYPE BONOBO_TYPE_WINDOW

static gpointer parent_class;

static void
screem_css_window_initialise_class( ScreemCSSWindowClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

       	object_class->finalize = screem_css_window_finalize;
	object_class->get_property = screem_css_window_get_prop;
	object_class->set_property = screem_css_window_set_prop;

	widget_class->show = screem_css_window_show;

	g_object_class_install_property(object_class,
					ARG_APP_ID,
					g_param_spec_string("app_id",
							    "Application ID",
							    "The ID",
							    "",
							    G_PARAM_READWRITE)
					);

	g_object_class_install_property(object_class,
					ARG_APP,
					g_param_spec_object("app",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);

	widget_class->realize = screem_css_window_realize;
	widget_class->size_request = screem_css_window_size_request;
}

static void
screem_css_window_initialise( ScreemCSSWindow *window )
{
}

static void
screem_css_window_set_prop( GObject *object, guint prop_id, 
			const GValue *value, GParamSpec *spec )
{
	ScreemCSSWindow *window;

	window = SCREEM_CSS_WINDOW( object );

	switch( prop_id ) {
	case ARG_APP_ID:
		break;
	case ARG_APP:
		break;
	}
}

static void
screem_css_window_get_prop( GObject *object, guint prop_id, 
			GValue *value, GParamSpec *spec )
{
	switch( prop_id ) {
	case ARG_APP_ID:
		break;
	case ARG_APP:
		break;
	}
}

static void
screem_css_window_finalize( GObject *object )
{
	ScreemCSSWindow *window;

	window = SCREEM_CSS_WINDOW( object );

	if( window->filename )
		g_free( window->filename );

	g_object_unref( window->recent_sheets );

	g_object_unref( window->xml );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}


static void
screem_css_window_realize( GtkWidget *widget )
{

	/* Create GdkWindow */
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void
screem_css_window_size_request( GtkWidget *widget, GtkRequisition *req )
{
	g_return_if_fail( SCREEM_CSS_IS_WINDOW( widget ) );
	g_return_if_fail( req != NULL );

	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void
screem_css_window_show( GtkWidget *widget )
{
	ScreemCSSWindow *window;

	window = SCREEM_CSS_WINDOW( widget );

	GTK_WIDGET_CLASS( parent_class )->show( widget );
}

GType screem_css_window_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemCSSWindowClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_css_window_initialise_class,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemCSSWindow ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_css_window_initialise
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemCSSWindow",
					       &info, 0 );
	}

	return type;
}
