"""The Ubuntu One Music Store Rhythmbox plugin."""
# Copyright (C) 2009 Canonical, Ltd.
#
# This library is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License
# version 3.0 as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License version 3.0 for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library. If not, see
# <http://www.gnu.org/licenses/>.
#
# Authored by Stuart Langridge <stuart.langridge@canonical.com>

# pylint: disable=W0201

# Import these here as gtk/gio import ordering seems to matter
import gobject
import gtk
import gio

import gconf
import gettext
import os
# pylint: disable=F0401
import rb as RB
import rhythmdb
# pylint: enable=F0401
import stat
import urllib
import urlparse
import xdg.BaseDirectory

from gettext import lgettext as _
from ubuntuone.gtkwidgets import MusicStore as U1MusicStore

gettext.bindtextdomain("rhythmbox-ubuntuone-music-store", "/usr/share/locale")
gettext.textdomain("rhythmbox-ubuntuone-music-store")

MUSIC_STORE_WIDGET = U1MusicStore()  # keep this around for later
U1LIBRARYPATH = MUSIC_STORE_WIDGET.get_library_location()
RB_LIBRARY_LOCATIONS = "/apps/rhythmbox/library_locations"
PARTNER_LIST = "canonical-partner-maverick.list"
SOURCES_DIR = "/etc/apt/sources.list.d/"
PLUGIN_PACKAGENAME = "gstreamer0.10-fluendo-plugins-mp3-partner"


class U1EntryType(rhythmdb.EntryType):
    """Entry type for the Ubuntu One Music Store source."""

    def __init__(self):
        rhythmdb.EntryType.__init__(self, name='ubuntuone')

    def do_can_sync_metadata(self, entry):
        """Not a real source, so we can't sync metadata."""
        return False

    def do_sync_metadata(self, entry, changes):
        """Do nothing."""
        return


class U1MusicStoreWidget(object):
    """The Ubuntu One Music Store."""
    def __init__(self, plugin):
        self.plugin = plugin
        self.entry_type = U1EntryType()

    def activate(self, shell):
        """Plugin startup."""
        self.db = shell.get_property("db")
        group = RB.rb_display_page_group_get_by_id("stores")

        icon = gtk.IconTheme().load_icon(
            "ubuntuone", gtk.icon_size_lookup(gtk.ICON_SIZE_MENU)[0], 0)

        self.db.register_entry_type(self.entry_type)

        self.source = gobject.new(U1Source,
                                  shell=shell,
                                  entry_type=self.entry_type,
                                  pixbuf=icon,
                                  plugin=self.plugin)
        shell.register_entry_type_for_source(self.source, self.entry_type)
        shell.append_display_page(self.source, group)

        self.shell = shell
        self.source.connect("preview-mp3", self.play_preview_mp3)
        self.source.connect("play-library", self.play_library)
        self.source.connect("download-finished", self.download_finished)
        self.source.connect("url-loaded", self.url_loaded)

        # Do these every time
        self.add_u1_library()

    def deactivate(self, shell):
        """Plugin shutdown."""
        # remove source
        self.source.delete_thyself()
        # remove the library, if it's empty
        try:
            filecount = len(os.listdir(self.u1_library_symlink))
        except OSError:
            # symlink is dangling
            # so they never downloaded anything
            filecount = 0
        if filecount == 0:
            client = gconf.client_get_default()
            libraries = client.get_list(RB_LIBRARY_LOCATIONS,
                                        gconf.VALUE_STRING)
            if self.u1_library_path_url in libraries:
                client.notify_remove(self.library_adder)
                libraries.remove(self.u1_library_path_url)
                client.set_list(RB_LIBRARY_LOCATIONS,
                                gconf.VALUE_STRING, libraries)
        # delete held references
        del self.db
        del self.source
        del self.shell

    def url_loaded(self, source, url):
        """A URL is loaded in the plugin"""
        print "URL loaded:", url
        if urlparse.urlparse(url)[2] == "https":
            pass
        else:
            pass

    def _udf_path_to_library_uri(self, path):
        """Calculate the path in the library.
        Since the library is accessed via the created symlink, but the path
        passed to us is to a file in the actual music store UDF, we need to
        work out the path inside the library, i.e., what the path to that file
        is via the symlink."""
        if path.startswith(U1LIBRARYPATH):
            subpath = path[len(U1LIBRARYPATH):]
        else:
            subpath = path
        if subpath.startswith("/"):
            subpath = subpath[1:]
        library_path = os.path.join(self.u1_library_symlink, subpath)
        # convert path to URI. Don't use urllib for this; Python and
        # glib escape URLs differently. gio does it the glib way.
        library_uri = gio.File(library_path).get_uri()
        return library_uri

    def download_finished(self, source, path):
        """A file is finished downloading"""
        library_uri = self._udf_path_to_library_uri(path)
        # Import the URI
        if not self.shell.props.db.entry_lookup_by_location(library_uri):
            self.db.add_uri(library_uri)

    def play_library(self, source, path):
        """Switch to and start playing a song from the library"""
        uri = self._udf_path_to_library_uri(path)
        entry = self.shell.props.db.entry_lookup_by_location(uri)
        if not entry:
            print "couldn't find entry", uri
            return
        libsrc = self.shell.props.library_source
        artist_view, album_view = libsrc.get_property_views()[0:2]
        song_view = libsrc.get_entry_view()
        artist = self.shell.props.db.entry_get(entry, rhythmdb.PROP_ARTIST)
        album = self.shell.props.db.entry_get(entry, rhythmdb.PROP_ALBUM)
        self.shell.props.display_page_tree.select(libsrc)
        artist_view.set_selection([artist])
        album_view.set_selection([album])
        song_view.scroll_to_entry(entry)
        player = self.shell.get_player()
        player.stop()
        player.play_entry(entry, libsrc)

    def play_preview_mp3(self, source, url, title):
        """Play a passed mp3; signal handler for preview-mp3 signal."""
        # create an entry, don't save it, and play it
        entry = self.shell.props.db.entry_lookup_by_location(url)
        if entry is None:
            entry = self.shell.props.db.entry_new(self.entry_type, url)
        self.shell.props.db.set(entry, rhythmdb.PROP_TITLE, title)
        player = self.shell.get_player()
        player.stop()
        player.play_entry(entry, self.source)
        # delete this entry when it finishes playing. Don't know how yet.

    def add_u1_library(self):
        """Add the U1 library if not listed in RB and re-add if changed."""
        u1_library_path_folder = xdg.BaseDirectory.save_data_path("ubuntuone")
        # Ensure that we can write to the folder, because syncdaemon creates it
        # with no write permissions
        os.chmod(u1_library_path_folder,
            os.stat(u1_library_path_folder)[stat.ST_MODE] | stat.S_IWUSR)
        # Translators: this is the name under Music for U1 music in Rhythmbox
        u1_library_path = os.path.join(u1_library_path_folder,
                                       _("Purchased from Ubuntu One"))
        if not os.path.islink(u1_library_path):
            if not os.path.exists(u1_library_path):
                print "Attempting to symlink %s to %s" % (U1LIBRARYPATH,
                                                          u1_library_path)
                os.symlink(U1LIBRARYPATH, u1_library_path)
            else:
                # something that isn't a symlink already exists. That's not
                # supposed to happen.
                # Write a warning and carry on.
                print ("Warning: library location %s existed. It should have "
                    "been a symlink to %s, and it wasn't. This isn't supposed "
                    "to happen. Carrying on anyway, on the assumption that "
                    "you know what you're doing. If this is a problem, then "
                    "delete or rename %s.") % (u1_library_path, U1LIBRARYPATH,
                    u1_library_path)
        self.u1_library_path_url = "file://%s" % urllib.quote(u1_library_path)
        self.u1_library_symlink = u1_library_path
        client = gconf.client_get_default()
        self._add_u1_library_if_not_present(client)
        self._remove_old_u1_library_if_present(client)
        # Watch for changes to the gconf key and re-add the library
        self.library_adder = client.notify_add(
            RB_LIBRARY_LOCATIONS,
            self._add_u1_library_if_not_present)

    def _add_u1_library_if_not_present(self, client, *args, **kwargs):
        """Check for the U1 library and add to libraries list."""
        libraries = client.get_list(RB_LIBRARY_LOCATIONS, gconf.VALUE_STRING)
        if self.u1_library_path_url not in libraries:
            libraries.append(self.u1_library_path_url)
            client.set_list(RB_LIBRARY_LOCATIONS,
                            gconf.VALUE_STRING, libraries)

    def _remove_old_u1_library_if_present(self, client, *args, **kwargs):
        """Check for the old U1 library and remove it from libraries list."""
        libraries = client.get_list(RB_LIBRARY_LOCATIONS, gconf.VALUE_STRING)
        removed = False
        # originally, this was the library path. Someone might have this.
        old_path = "file://%s" % os.path.expanduser("~/.ubuntuone/musicstore")
        if old_path in libraries:
            libraries.remove(old_path)
            removed = True
        # Then, this was the library path, which we put into gconf unescaped
        actual_udf_path_unescaped = "file://%s" % U1LIBRARYPATH
        if actual_udf_path_unescaped in libraries:
            libraries.remove(actual_udf_path_unescaped)
            removed = True
        # In theory, no-one should have the escaped path, but let's check
        actual_udf_path_escaped = "file://%s" % U1LIBRARYPATH
        if actual_udf_path_escaped in libraries:
            libraries.remove(actual_udf_path_escaped)
            removed = True
        # Also, remove any library which is in .local/share and *isn't*
        # the current library. This caters for people who got the library
        # created under one name (say, English) and then had it created
        # under another (say, after a translation to Dutch)
        u1_library_path_folder = xdg.BaseDirectory.save_data_path("ubuntuone")
        u1_library_path_folder_url = "file://%s" % urllib.quote(
            u1_library_path_folder)
        symlink_url = "file://" + urllib.quote(self.u1_library_symlink)
        for lib in libraries:
            if lib.startswith(u1_library_path_folder_url) and \
                    lib != symlink_url:
                libraries.remove(lib)
                # and delete the symlink itself
                symlink_to_remove = urllib.unquote(lib[7:])
                os.unlink(symlink_to_remove)
                removed = True
        if removed:
            client.set_list(RB_LIBRARY_LOCATIONS,
                            gconf.VALUE_STRING, libraries)


class U1Source(RB.Source, gobject.GObject):
    """A Rhythmbox source widget for the U1 Music Store."""
    # gproperties required so that rb.Source is instantiable
    __gproperties__ = {
      'plugin': (RB.Plugin, 'plugin', 'plugin',
                 gobject.PARAM_WRITABLE | gobject.PARAM_CONSTRUCT_ONLY),
      }
    # we have the preview-mp3 signal; we receive it from the widget, and
    # re-emit it so that the plugin gets it, because the plugin actually
    # plays the mp3
    __gsignals__ = {
      "preview-mp3": (gobject.SIGNAL_RUN_FIRST,
                      gobject.TYPE_NONE, (str, str)),
      "play-library": (gobject.SIGNAL_RUN_FIRST,
                       gobject.TYPE_NONE, (str,)),
      "download-finished": (gobject.SIGNAL_RUN_FIRST,
                            gobject.TYPE_NONE, (str,)),
      "url-loaded": (gobject.SIGNAL_RUN_FIRST,
                     gobject.TYPE_NONE, (str,)),
    }

    def __init__(self):
        RB.Source.__init__(self, name=_("Ubuntu One"))
        gobject.GObject.__init__(self)
        self.browser = MUSIC_STORE_WIDGET
        self.__activated = False
        self.add_music_store_widget()

    def do_impl_activate(self):
        """Source startup."""
        if self.__activated:
            return
        self.__activated = True
        RB.Source.do_impl_activate(self)

    def do_impl_want_uri(self, uri):
        """I want to handle u1ms URLs"""
        if uri.startswith("u1ms://"):
            return 100
        return 0

    def do_impl_add_uri(self, uri, title, genre):
        """Handle a u1ms URL"""
        if not uri.startswith("u1ms://"):
            return False
        uri_to_use = uri.replace("u1ms://", "http://")
        print "Calling u1musicstore plugin with %s" % uri_to_use
        shell = self.get_property("shell")
        shell.props.display_page_tree.select(self)
        self.browser.load_store_link(uri_to_use)
        return True

    def add_music_store_widget(self):
        """Display the music store widget in Rhythmbox."""
        self.add(self.browser)
        self.browser.show()
        self.show()
        self.browser.set_property("visible", True)
        self.browser.connect("preview-mp3",
                             self.re_emit_preview)
        self.browser.connect("play-library",
                             self.re_emit_playlibrary)
        self.browser.connect("download-finished",
                             self.re_emit_downloadfinished)
        self.browser.connect("url-loaded",
                             self.re_emit_urlloaded)

    def do_impl_can_pause(self):
        """Implementation can pause.
           If we don't handle this, Rhythmbox segfaults."""
        return True  # so we can pause, else we segfault

    def re_emit_preview(self, widget, url, title):
        """Handle the preview-mp3 signal and re-emit it to rb.Plugin."""
        self.emit("preview-mp3", url, title)

    def re_emit_playlibrary(self, widget, path):
        """Handle the play-library signal and re-emit it to rb.Plugin."""
        self.emit("play-library", path)

    def re_emit_downloadfinished(self, widget, path):
        """Handle the download-finished signal and re-emit it to rb.Plugin."""
        self.emit("download-finished", path)

    def re_emit_urlloaded(self, widget, url):
        """Handle the url-loaded signal and re-emit it to rb.Plugin."""
        self.emit("url-loaded", url)

    def do_set_property(self, prop, value):
        """Allow property settings to handle the plugin call."""
        if prop.name == 'plugin':
            self.__plugin = value
        else:
            raise AttributeError('unknown property %s' % prop.name)

