/* -*- Mode: c; c-basic-offset: 2 -*-
 *
 * rdf_log.h - RDF logging interfaces
 *
 * $Id: rdf_log.h,v 1.6 2004/11/01 20:43:33 cmdjb Exp $
 *
 * Copyright (C) 2004-2004, David Beckett http://purl.org/net/dajobe/
 * Institute for Learning and Research Technology http://www.ilrt.bristol.ac.uk/
 * University of Bristol, UK http://www.bristol.ac.uk/
 * 
 * This package is Free Software and part of Redland http://librdf.org/
 * 
 * It is licensed under the following three licenses as alternatives:
 *   1. GNU Lesser General Public License (LGPL) V2.1 or any newer version
 *   2. GNU General Public License (GPL) V2 or any newer version
 *   3. Apache License, V2.0 or any newer version
 * 
 * You may not use this file except in compliance with at least one of
 * the above three licenses.
 * 
 * See LICENSE.html or LICENSE.txt at the top of this package for the
 * complete terms and further detail along with the license texts for
 * the licenses in COPYING.LIB, COPYING and LICENSE-2.0.txt respectively.
 * 
 * 
 */


#ifndef LIBRDF_LOG_H
#define LIBRDF_LOG_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * librdf_log_level:
 *
 * Indicates the level of the log message
 */
typedef enum {
  LIBRDF_LOG_NONE = 0,
  LIBRDF_LOG_DEBUG,    /* debug message */
  LIBRDF_LOG_INFO,     /* information */
  LIBRDF_LOG_WARN,     /* warning */
  LIBRDF_LOG_ERROR,    /* recoverable error */
  LIBRDF_LOG_FATAL,     /* fatal error */
  LIBRDF_LOG_LAST=LIBRDF_LOG_FATAL
} librdf_log_level;
  

/**
 * librdf_log_facility:
 *
 * Indicates the part of the system that generated the log message
 */
typedef enum {
  LIBRDF_FROM_NONE = 0,
  LIBRDF_FROM_CONCEPTS,
  LIBRDF_FROM_DIGEST,
  LIBRDF_FROM_FILES,
  LIBRDF_FROM_HASH,
  LIBRDF_FROM_INIT,
  LIBRDF_FROM_ITERATOR,
  LIBRDF_FROM_LIST,
  LIBRDF_FROM_MODEL,
  LIBRDF_FROM_NODE,
  LIBRDF_FROM_PARSER,
  LIBRDF_FROM_QUERY,
  LIBRDF_FROM_SERIALIZER,
  LIBRDF_FROM_STATEMENT,
  LIBRDF_FROM_STORAGE,
  LIBRDF_FROM_STREAM,
  LIBRDF_FROM_URI,
  LIBRDF_FROM_UTF8,
  LIBRDF_FROM_LAST=LIBRDF_FROM_UTF8
} librdf_log_facility;
  

/**
 * librdf_log_message:
 *
 * Structure for storing parts of a log message generated by Redland.
 */
typedef struct 
{
  int code;                  /* The error code */
  librdf_log_level level;
  librdf_log_facility facility;
  const char *message;
  /* valid for certain facilities such as LIBRDF_FROM_PARSER */
  raptor_locator *locator;
} librdf_log_message;


/**
 * librdf_log_level_func:
 *
 * Handler for one log level, for the warning and error levels ONLY.
 * Used by librdf_world_set_warning and librdf_world_set_error.
 *
 * Return value: non-zero to indicate log message has been handled 
 */
typedef int (*librdf_log_level_func)(void *user_data, const char *message, va_list arguments);

/**
 * librdf_log_func:
 *
 * Handler for all log levels.
 *
 * Return value: non-zero to indicate log message has been handled 
 */
typedef int (*librdf_log_func)(void *user_data, librdf_log_message *message);

#ifdef LIBRDF_INTERNAL
void librdf_log_simple(librdf_world* world, int code, librdf_log_level level, librdf_log_facility facility, void *locator, const char *message);
void librdf_log(librdf_world* world, int code, librdf_log_level level, librdf_log_facility facility, void *locator, const char *message, ...);

void librdf_fatal(librdf_world* world, int facility, const char *file, int line, const char *function, const char *message);

void librdf_test_error(librdf_world* world, const char *message);
void librdf_test_warning(librdf_world* world, const char *message);

#endif

/* log message accessors */
REDLAND_API int librdf_log_message_code(librdf_log_message *message);
REDLAND_API librdf_log_level librdf_log_message_level(librdf_log_message *message);
REDLAND_API librdf_log_facility librdf_log_message_facility(librdf_log_message *message);
REDLAND_API const char * librdf_log_message_message(librdf_log_message *message);
REDLAND_API raptor_locator* librdf_log_message_locator(librdf_log_message *message);


#ifdef __cplusplus
}
#endif

#endif
