
import bughelper_error as Error
import utils
import libxml2
import urlparse
import warnings

warnings.warn("text_buglist API not stable yet, LP returns a list of all bugs ever reported", FutureWarning)
LPURLERROR = Error.LPUrlError


def sort(x,y,sort):
    if sort.startswith("-"):
        sort = sort.strip("-")
        order = -1
    else:
        order = 1
    if sort == "nr":
        return order * cmp(int(x.bugnumber), int(y.bugnumber))
    if sort == "importance":
        importance = ["Critical","High","Medium","Low","Wishlist","Undecided"]
        return order * cmp(importance.index(getattr(x,sort)), importance.index(getattr(y,sort)))
    if sort == "status":
        status = ["Fix Released","Fix Committed","In Progress",\
            "Triaged", "Confirmed","Won\'t Fix","Invalid","Incomplete", "New"]
        return order * cmp(status.index(getattr(x,sort)), status.index(getattr(y,sort)))


class BugInfo(object):
    # TODO: use same attribute names like Bug.Bug
    # object can be indetified by str(type(x)) == "BugInfo"
    def __init__(self, nr):
        self.bugnumber = nr
    def __getattr__(self,name):
        if name == "bugnumber":
            return self.bugnumber
        warnings.warn("unable to parse '%s' in text-mode" %name, FutureWarning)
        return ""
    def __int__(self):
        return int(self.bugnumber)
    def __repr__(self):
        return "<BugInfo %s>" %self.bugnumber
    def __str__(self):
        return "[Bug %s]" %self.bugnumber
    def __hash__(self):
        return int(self.bugnumber)
    def __eq__(self, other):
        return hash(self) == hash(other)

class BugPage(object):
    """
    grab content of a single bug-table    
    """
    def __init__(self, url, connection=None):
        assert connection, "Connection object needed"
        self.__connection = connection
        self.url = utils.valid_lp_url(url,utils.BUGPAGE)
        assert self.url, "Invalid launchpad url %s" %url
        url = urlparse.urlsplit(self.url)
        if url[3]:
            warnings.warn("LP does not allow url options in text mode, options will be ignored", FutureWarning)
        self.url = urlparse.urlunsplit((url[0],url[1],"%s-text" %url[2], url[3], url[4])) 
        self.following_page = None
        self.__bugs = set()
        self.pageBody = None
        
    def parse_bugs(self):
        lp_content = self.__connection.get(self.url)
            
        self.pageBody = lp_content.text
        self.__bugs = set([BugInfo(i) for i in self.pageBody.split("\n") if i])
        
    def get_bugs(self):
        if not self.pageBody:
            self.parse_bugs()
        return self.__bugs
    bugs = property(fget=get_bugs,doc="get bugs")



class BugList(object):
    """
    returns a SET of BugInfo objects
    searches baseurl and its following pages
    """
    def __init__(self, baseurl, run=False, bugs=set(), connection=None, filter=None):
        assert connection, "Connection object needed"
        self.__connection = connection
        
        self.__filter = filter
        
        self.baseurl = utils.valid_lp_url(baseurl,utils.BUGLIST)
        assert self.baseurl, "Invalid launchpad url %s" %baseurl
        self.__bugs = bugs
        if not run:
            self.add(self.baseurl)
        
    def __iter__(self):
        """ allow direct iteration over instance of BugList """
        for b in self.__bugs:
            yield(b)            
    
    def __iadd__(self, other):
        """ let l be an instance of BugList,
        l += BugList(<LP-URL>) will add bugs to l
        l.add(<LP-URL>) is still possible
        """
        __bugs = self.__bugs.union(other.bugs)
        return BugList(other.baseurl, run=True, bugs=__bugs, connection=self.__connection, filter=self.__filter)
            
    def filter(self,func=set()):
        """
        filters bugs-set by given func and returns a new BugList object
        each func has to return a tuple of sets (<remove>,<add>)
        """
        if type(func) != type(set()):
            func = set(func)
        func = func | self.__filter.functions
        __bugs = self.__bugs.copy()
        for f in set(func):
            __bugs = f(__bugs)
        return BugList(self.baseurl, run=True, bugs=__bugs, connection=self.__connection, filter=self.__filter)
    
    
    def sort(self, optsort):
        """ returns a LIST of bugs sorted by optsort """
        __bugs = list()
        __bugs = [x for x in self.__bugs]
        __bugs.sort(lambda x, y: sort(x,y,optsort))
        return __bugs
    
    def fetch(self, url):
        """
        searches given bugpage for bugs and 'next'-page and returns both values
        
        if calling <url> returns an LPURLERROR, this error will be ignored
        if there are still bugs in the buglist, otherwise raised again
        """
        __bugs = set()
        __follow = None
        try:
            __bp = BugPage(url, connection=self.__connection)
            __bugs = __bp.bugs
            __follow = __bp.following_page
        except LPURLERROR, e:
            if len(self.__bugs) == 0:
                raise LPURLERROR(text = str(e))
        return __bugs, __follow

    def add(self, url, follow=True):
        """ adds bugs to the buglist """
        __follow = True
        __url = url
        __bugs = set()
        while(__url and __follow):
            (__bugs,__url) = self.fetch(__url)
            __follow = follow
            self.__bugs = self.__bugs.union(__bugs)
        
    def get_bugs(self):
        return self.__bugs
    bugs = property(fget=get_bugs,doc="get list of bugs")

