#!/usr/bin/env ruby

Dir.chdir(File.dirname(__FILE__)) { (s = lambda { |f| File.exist?(f) ? require(f) : Dir.chdir("..") { s.call(f) } }).call("spec/spec_helper.rb") }


[:seluser, :selrole, :seltype, :selrange].each do |param|
    property = Puppet::Type.type(:file).attrclass(param)
    describe property do
        before do
            @resource = mock 'resource'
            @resource.stubs(:[]).with(:path).returns "/my/file"
            @sel = property.new :resource => @resource
        end

        it "retrieve on #{param} should return :absent if the file isn't statable" do
            @resource.expects(:stat).returns nil
            @sel.retrieve.should == :absent
        end

        it "should retrieve nil for #{param} if there is no SELinux support" do
            stat = stub 'stat', :ftype => "foo"
            @resource.expects(:stat).returns stat
            @sel.expects(:get_selinux_current_context).with("/my/file").returns nil
            @sel.retrieve.should be_nil
        end

        it "should retrieve #{param} if a SELinux context is found with a range" do
            stat = stub 'stat', :ftype => "foo"
            @resource.expects(:stat).returns stat
            @sel.expects(:get_selinux_current_context).with("/my/file").returns "user_u:role_r:type_t:s0"
            expectedresult = case param
                when :seluser; "user_u"
                when :selrole; "role_r"
                when :seltype; "type_t"
                when :selrange; "s0"
            end
            @sel.retrieve.should == expectedresult
        end

        it "should retrieve #{param} if a SELinux context is found without a range" do
            stat = stub 'stat', :ftype => "foo"
            @resource.expects(:stat).returns stat
            @sel.expects(:get_selinux_current_context).with("/my/file").returns "user_u:role_r:type_t"
            expectedresult = case param
                when :seluser; "user_u"
                when :selrole; "role_r"
                when :seltype; "type_t"
                when :selrange; nil
            end
            @sel.retrieve.should == expectedresult
        end

        it "should handle no default gracefully" do
            @sel.expects(:get_selinux_default_context).with("/my/file").returns nil
            @sel.default.must be_nil
        end

        it "should be able to detect matchpathcon defaults" do
            @sel.stubs(:debug)
            @sel.expects(:get_selinux_default_context).with("/my/file").returns "user_u:role_r:type_t:s0"
            expectedresult = case param
                when :seluser; "user_u"
                when :selrole; "role_r"
                when :seltype; "type_t"
                when :selrange; "s0"
            end
            @sel.default.must == expectedresult
        end

        it "should be able to set a new context" do
            stat = stub 'stat', :ftype => "foo"
            @sel.should = %w{newone}
            @sel.expects(:set_selinux_context).with("/my/file", ["newone"], param)
            @sel.sync
        end

        it "should do nothing for insync? if no SELinux support" do
            @sel.should = %{newcontext}
            @sel.expects(:selinux_support?).returns false
            @sel.insync?("oldcontext").should == true
        end
    end
end

