/*
 * access to SPU privileged registers
 */
#include <linux/module.h>

#include <asm/pgtable.h>
#include <asm/io.h>
#include <asm/spu.h>
#include <asm/lv1call.h>
#include "spu_priv.h"


/*  spu sysdev nodes  */

static ssize_t spu_show_lspu_id(struct sys_device *sysdev, char *buf)
{
	struct spu *spu = container_of(sysdev, struct spu, sysdev);
	return sprintf(buf, "%ld", spu->priv_data->lspu_id);

}
static SYSDEV_ATTR(lspu_id, 0400, spu_show_lspu_id, NULL);

void spu_create_sysdev_files(struct spu *spu)
{
	sysdev_create_file(&spu->sysdev, &attr_lspu_id);
}

void spu_destroy_sysdev_files(struct spu *spu)
{
	sysdev_remove_file(&spu->sysdev, &attr_lspu_id);
}


/*  spu setup funcs  */

int __init enum_and_create_spu(void)
{
	int i;
	int ret;

	ret = -ENODEV;
	for (i = 0; i < CONFIG_NR_SPUS; i++) {
		long lspu_id = spu_data[i].lspu_id;

		if (lspu_id == 0)
			continue;

		ret = create_spu(&spu_data[i]);
		if (ret) {
			printk(KERN_WARNING
			       "%s: Error initializing logical SPU %lu\n",
			       __FUNCTION__, lspu_id);
			break;
		}

		/* Make sure that spus are `execution' status */
		while (in_be64(&spu_data[i].shadow->spu_execution_status) != 0x3)
			;
	}
	return ret;
}
EXPORT_SYMBOL_GPL(enum_and_create_spu);

int __init spu_setup(struct spu *spu, void *spu_setup_data)
{
	spu->priv_data = spu_setup_data;

	spu->node = 0;
	spu->name = spu->priv_data->name;
	spu->local_store_phys.addr = spu->priv_data->ls_phys;
	spu->local_store_phys.size = LS_SIZE;
	spu->problem_phys.addr = spu->priv_data->problem_phys;
	spu->problem_phys.size = sizeof(struct spu_problem);
	spu->priv1_phys.addr = 0;
	spu->priv1_phys.size = 0;
	spu->priv2_phys.addr = spu->priv_data->priv2_phys;
	spu->priv2_phys.size = sizeof(struct spu_priv2);

	spu->priv_data->shadow = __ioremap(spu->priv_data->shadow_phys,
					   sizeof(struct spu_shadow),
					   PAGE_READONLY);
	return 0;
}
EXPORT_SYMBOL_GPL(spu_setup);

void spu_free_priv_data(struct spu *spu)
{
	if (spu->priv_data->shadow) {
		iounmap(spu->priv_data->shadow);
		spu->priv_data->shadow = 0;
	}
}
EXPORT_SYMBOL_GPL(spu_free_priv_data);

int spu_request_irqs(struct spu *spu,
	irqreturn_t (*cls0)(int, void *, struct pt_regs *),
	irqreturn_t (*cls1)(int, void *, struct pt_regs *),
	irqreturn_t (*cls2)(int, void *, struct pt_regs *))
{
	int ret;
	struct spu_priv_data *priv = spu->priv_data;

	snprintf(spu->irq_c0, sizeof(spu->irq_c0), "spe%02d.0", spu->number);
	ret = ps3pf_connect_spu_irq(priv->lspu_id, 0,
				  &priv->irq_outlet[0],
				  &priv->irq_no[0],
				  &priv->irq_cpu[0],
				  cls0, SA_INTERRUPT, spu->irq_c0, spu);
	if (ret)
		goto out;

	snprintf(spu->irq_c1, sizeof(spu->irq_c1), "spe%02d.1", spu->number);
	ret = ps3pf_connect_spu_irq(priv->lspu_id, 1,
				  &priv->irq_outlet[1],
				  &priv->irq_no[1],
				  &priv->irq_cpu[1],
				  cls1, SA_INTERRUPT, spu->irq_c1, spu);
	if (ret)
		goto out1;

	snprintf(spu->irq_c2, sizeof(spu->irq_c2), "spe%02d.2", spu->number);
	ret = ps3pf_connect_spu_irq(priv->lspu_id, 2,
				  &priv->irq_outlet[2],
				  &priv->irq_no[2],
				  &priv->irq_cpu[2],
				  cls2, SA_INTERRUPT, spu->irq_c2, spu);
	if (ret)
		goto out2;
	goto out;

out2:
	ps3pf_free_spu_irq(priv->irq_outlet[1],
			 priv->irq_no[1],
			 priv->irq_cpu[1],
			 spu);
out1:
	ps3pf_free_spu_irq(priv->irq_outlet[0],
			 priv->irq_no[0],
			 priv->irq_cpu[0],
			 spu);
out:
	return ret;
}
EXPORT_SYMBOL_GPL(spu_request_irqs);

void spu_free_irqs(struct spu *spu)
{
	struct spu_priv_data *priv = spu->priv_data;

	ps3pf_free_spu_irq(priv->irq_outlet[0],
			 priv->irq_no[0],
			 priv->irq_cpu[0],
			 spu);
	ps3pf_free_spu_irq(priv->irq_outlet[1],
			 priv->irq_no[1],
			 priv->irq_cpu[1],
			 spu);
	ps3pf_free_spu_irq(priv->irq_outlet[2],
			 priv->irq_no[2],
			 priv->irq_cpu[2],
			 spu);
}
EXPORT_SYMBOL_GPL(spu_free_irqs);


/*  spu priv1 access funcs.  */

#define spu_priv1_get64(spu, reg) \
	in_be64(&(spu)->priv_data->shadow->reg)

#define spu_priv1_set64(spu, reg, val) \
({ \
	lv1_set_spu_privilege_state_area_1_register((spu)->priv_data->lspu_id, \
						    offsetof(struct spu_priv1, reg), val); \
})

void spu_int_mask_and(struct spu *spu, int class, u64 mask)
{
	u64 old_mask;

	old_mask = spu_int_mask_get(spu, class);
	spu_int_mask_set(spu, class, old_mask & mask);
}
EXPORT_SYMBOL_GPL(spu_int_mask_and);

void spu_int_mask_or(struct spu *spu, int class, u64 mask)
{
	u64 old_mask;

	old_mask = spu_int_mask_get(spu, class);
	spu_int_mask_set(spu, class, old_mask | mask);
}
EXPORT_SYMBOL_GPL(spu_int_mask_or);

void spu_int_mask_set(struct spu *spu, int class, u64 mask)
{
	unsigned long __flags;

	spin_lock_irqsave(&spu->priv_data->regs_lock, __flags);
	lv1_set_spu_interrupt_mask(spu->priv_data->lspu_id, class, mask);
	spu->priv_data->irq_mask[class] = mask;
	spin_unlock_irqrestore(&spu->priv_data->regs_lock, __flags);
}
EXPORT_SYMBOL_GPL(spu_int_mask_set);

u64 spu_int_mask_get(struct spu *spu, int class)
{
	return spu->priv_data->irq_mask[class];
}
EXPORT_SYMBOL_GPL(spu_int_mask_get);

void spu_int_stat_clear(struct spu *spu, int class, u64 stat)
{
	/* notice: MFC_DSISR will be cleared,
	           when class1[Mf] was set. */
	lv1_clear_spu_interrupt_status(spu->priv_data->lspu_id,
				       class,
				       stat,
				       0);
}
EXPORT_SYMBOL_GPL(spu_int_stat_clear);

u64 spu_int_stat_get(struct spu *spu, int class)
{
	u64 stat;

	lv1_get_spu_interrupt_status(spu->priv_data->lspu_id, class, &stat);
	return stat;
}
EXPORT_SYMBOL_GPL(spu_int_stat_get);

void spu_cpu_affinity_set(struct spu *spu, int cpu)
{
	/* not supported */
}
EXPORT_SYMBOL_GPL(spu_cpu_affinity_set);

u64 spu_mfc_dar_get(struct spu *spu)
{
	return spu_priv1_get64(spu, mfc_dar_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_dar_get);

u64 spu_mfc_dsisr_get(struct spu *spu)
{
	return spu_priv1_get64(spu, mfc_dsisr_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_dsisr_get);

void spu_mfc_dsisr_set(struct spu *spu, u64 dsisr)
{
	/* it will clear at spu_int_stat_clear called. */
}
EXPORT_SYMBOL_GPL(spu_mfc_dsisr_set);

void spu_mfc_sdr_setup(struct spu *spu)
{
	/* nothing to do */
}
EXPORT_SYMBOL_GPL(spu_mfc_sdr_setup);

void spu_mfc_sr1_set(struct spu *spu, u64 sr1)
{
	unsigned long __flags;

	BUG_ON((sr1 & ~0x09) != (spu->priv_data->mfc_sr1 & ~0x09));

	spin_lock_irqsave(&spu->priv_data->regs_lock, __flags);
	spu_priv1_set64(spu, mfc_sr1_RW, sr1);
	spu->priv_data->mfc_sr1 = sr1;
	spin_unlock_irqrestore(&spu->priv_data->regs_lock, __flags);

}
EXPORT_SYMBOL_GPL(spu_mfc_sr1_set);

u64 spu_mfc_sr1_get(struct spu *spu)
{
	return spu->priv_data->mfc_sr1;
}
EXPORT_SYMBOL_GPL(spu_mfc_sr1_get);

void spu_mfc_tclass_id_set(struct spu *spu, u64 tclass_id)
{
	unsigned long __flags;

	spin_lock_irqsave(&spu->priv_data->regs_lock, __flags);
	spu_priv1_set64(spu, mfc_tclass_id_RW, tclass_id);
	spu->priv_data->mfc_tclass_id = tclass_id;
	spin_unlock_irqrestore(&spu->priv_data->regs_lock, __flags);
}
EXPORT_SYMBOL_GPL(spu_mfc_tclass_id_set);

u64 spu_mfc_tclass_id_get(struct spu *spu)
{
	return spu->priv_data->mfc_tclass_id;
}
EXPORT_SYMBOL_GPL(spu_mfc_tclass_id_get);

void spu_tlb_invalidate(struct spu *spu)
{
	/* nothing to do */
}
EXPORT_SYMBOL_GPL(spu_tlb_invalidate);

void spu_resource_allocation_groupID_set(struct spu *spu, u64 id)
{
	/* nothing to do */
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_groupID_set);

u64 spu_resource_allocation_groupID_get(struct spu *spu)
{
	/* nothing to do */
	return 0;
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_groupID_get);

void spu_resource_allocation_enable_set(struct spu *spu, u64 enable)
{
	/* nothing to do */
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_enable_set);

u64 spu_resource_allocation_enable_get(struct spu *spu)
{
	/* nothing to do */
	return 0;
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_enable_get);
