/*
 * pci.c
 *
 * Copyright (C) 2006  Sony Computer Entertainment Inc.
 *
 * PS3PF specific routines for PCI.
 * 
 * Based on code from iSeries_pci.c
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/kernel.h>
#include <linux/threads.h>
#include <linux/pci.h>
#include <linux/sysdev.h>
#include <asm/lv1call.h>
#include <asm/pci-bridge.h>

#include "pci_res.h"

#define PS3PF_GELIC_DEV_TYPE 3

extern uint32_t pci_dma_base;
extern unsigned long ps3pf_rm_limit, ps3pf_2nd_mem_base, ps3pf_2nd_mem_size,
		ps3pf_mem_total;
#define PS3PF_PCI_DMA_SIZE(mem) ((((mem -1) >> 27) + 1 ) << 27) /* 2^27=128M */

static int ps3pf_pci_usb_param = 1;	/* use internal usb */

#if defined(CONFIG_GELIC_NET) || defined(CONFIG_GELIC_NET_MODULE)
uint64_t gelic_irq_status = 0;
EXPORT_SYMBOL(gelic_irq_status);
#endif

/* dvice table */
struct ps3pf_dev_table {
	uint64_t bus_index; /* repository */
	uint64_t dev_index; /* repository */

	/* bus driver */
	uint64_t bus_id;
	uint64_t dev_id;
	uint64_t dev_type;

	uint64_t bus_pci_bus;
	uint64_t bus_pci_dev;
	uint64_t bus_pci_func;

	int pci_bus; /* pci */
	int pci_dev; /* pci */
	int pci_func; /* pci */

	int irq;
	int pci;
	int open;
	int region;
	uint64_t dma;

	uint64_t lpar[PS3PF_PCI_REGION_MAX];
};

struct ps3pf_pci {
	int bus_irqs[PS3PF_PCI_IRQ_MAX];
	struct ps3pf_dev_table dev[PS3PF_PCI_DEV_MAX];
	int dev_num;
	struct device_node dev_node[PS3PF_PCI_BUS_MAX];
} g_ps3pf_pci = {
	.dev_num = 0,
};

/* dummy pci config for internal USB */
static uint8_t ps3pf_ohci_config[] = {
0x2f,0x10,0xb6,0x01,0x00,0x00,0x18,0x00,
0x01,0x10,0x03,0x0c,0x00,0x00,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0xdc,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x70,0x25,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x01,0x00,0x0a,0x48,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};

static uint8_t ps3pf_ehci_config[] = {
0x2f,0x10,0xb5,0x01,0x00,0x00,0x00,0x00,
0x01,0x20,0x03,0x0c,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x68,0x25,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x20,0x20,0x07,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};

struct pci_dummy_config {
	uint64_t type; /* dev type */
	int pci_func;  /* func=1: ehci func:0: ohci*/
	uint8_t *regs;
};

static struct pci_dummy_config dummy_config[] = {
  { PS3PF_PCI_DEV_TYPE_UHC, 0, ps3pf_ohci_config },
  { PS3PF_PCI_DEV_TYPE_UHC, 1, ps3pf_ehci_config }
};

static uint8_t *get_config_reg(uint64_t type, int pci_func)
{
	int i;

	for (i = 0; i < sizeof(dummy_config) / sizeof(*dummy_config);
		i++) {
		if (dummy_config[i].type == type && 
			dummy_config[i].pci_func == pci_func) {
			return dummy_config[i].regs;
		}
	}
	return NULL;
}

static struct ps3pf_dev_table *get_dev_table(struct ps3pf_pci *pcibus,
					int pci_bus, int pci_dev, int pci_func)
{
	int i;
	struct ps3pf_dev_table *p = &pcibus->dev[0];

	for (i = 0; i < pcibus->dev_num; i++, p++) {
		if (p->pci_bus == pci_bus &&
			p->pci_dev == pci_dev &&
			p->pci_func == pci_func ) {
			return p;
		} 
	}
	return NULL;
}

static struct ps3pf_dev_table *get_dev_table_by_type(struct ps3pf_pci *pcibus,
						uint64_t type)
{
	int i;
	struct ps3pf_dev_table *p = &pcibus->dev[0];

	for (i = 0; i < pcibus->dev_num; i++, p++) {
		if (p->dev_type == type) {
			return p;
		} 
	}
	return NULL;
}

int ps3pf_get_dev_by_type( uint64_t type, uint64_t *bus_id,
			uint64_t *dev_id, uint64_t *irq)
{
	int i, num;
	struct ps3pf_dev_table *p;

	p = &g_ps3pf_pci.dev[0];
	num = g_ps3pf_pci.dev_num;
	for (i = 0; i < num; i++, p++) {
		if (p->dev_type == type ) {
			*bus_id = p->bus_id;
			*dev_id = p->dev_id;
			*irq = p->irq;
			return 0;
		} 
	}
	return -1;
}
EXPORT_SYMBOL(ps3pf_get_dev_by_type);

static int chk_dma_by_bus_id(struct ps3pf_pci *pcibus, int bus_id)
{
	int i, ret;
	struct ps3pf_dev_table *p = &pcibus->dev[0];

	ret = 0;
	for (i = 0; i < pcibus->dev_num; i++, p++) {
		if (p->bus_id == bus_id && p->dma) {
			ret = 1;
		} 
	}
	return ret;
}

static int chk_dma_by_dev_id(struct ps3pf_pci *pcibus, int dev_id)
{
	int i, ret;
	struct ps3pf_dev_table *p = &pcibus->dev[0];

	ret = 0;
	for (i = 0; i < pcibus->dev_num; i++, p++) {
		if (p->dev_id == dev_id && p->dma) {
			ret = 1;
		} 
	}
	return ret;
}

static int chk_dev_open(struct ps3pf_pci *pcibus, 
			uint64_t bus_id, uint64_t dev_id)
{
	int i, ret;
	struct ps3pf_dev_table *p = &pcibus->dev[0];

	ret = 0;
	for (i = 0; i < pcibus->dev_num; i++, p++) {
		if (p->bus_id == bus_id 
			&& p->dev_id == dev_id 
			&& p->open) {
			ret = 1;
		} 
	}
	return ret;
}

/* read/write internal PCI config */
static int
ps3pf_ipci_read_config(struct pci_bus *bus, unsigned int devfn,
			int offset, int size, uint32_t *val)
{
	uint8_t *regs;
	struct ps3pf_dev_table *p;
	struct ps3pf_pci *pcibus = &g_ps3pf_pci;

	p = get_dev_table(pcibus, bus->number, 
			PCI_SLOT(devfn), PCI_FUNC(devfn));
	if (p == NULL) {
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	if ((regs = get_config_reg(p->dev_type, p->pci_func)) == NULL) {
		DPRINTK("%s: get_config_reg failed \n", __FUNCTION__);
		return PCIBIOS_DEVICE_NOT_FOUND;
	}
	switch (size) {
	case 1:
		*val = regs[offset];
		break;
	case 2:
		*val = le16_to_cpu(*(u16*)&regs[offset]);
		break;
	case 4:
		*val = le32_to_cpu(*(uint32_t*)&regs[offset]);
		break;
	default:
		DPRINTK(" read PCI bus%d dev%02d: offset=0x%02x size=%d: "
			"not supported\n", bus->number, devfn, offset, size);
		return PCIBIOS_FUNC_NOT_SUPPORTED;
	}
	return 0;
}

static int
ps3pf_ipci_write_config(struct pci_bus *bus, unsigned int devfn,
				int offset, int size, uint32_t val)
{
	return 0;
}

/* read/write external PCI config */
static int
ps3pf_epci_read_config(struct pci_bus *bus, unsigned int devfn,
			int offset, int size, uint32_t *val)
{
	uint64_t status, data;
	struct ps3pf_dev_table *p;
	struct ps3pf_pci *pcibus = &g_ps3pf_pci;

	p = get_dev_table(pcibus, bus->number, 
			PCI_SLOT(devfn), PCI_FUNC(devfn));
	if (p == NULL) {
		return PCIBIOS_DEVICE_NOT_FOUND;
	}
	status = lv1_read_pci_config(p->bus_id, 
				p->bus_pci_bus, p->bus_pci_dev, p->bus_pci_func,
				(uint64_t)offset, size, &data);
	if(status) {
		printk("lv1_read_pci_config failed: status:%ld\n", status);
		return PCIBIOS_DEVICE_NOT_FOUND;
	}
	*val = data;
	return 0;
}

static int
ps3pf_epci_write_config(struct pci_bus *bus, unsigned int devfn,
					int offset, int size, uint32_t val)
{
	uint64_t status;
	struct ps3pf_dev_table *p;
	struct ps3pf_pci *pcibus = &g_ps3pf_pci;

	p = get_dev_table(pcibus, bus->number, 
			PCI_SLOT(devfn), PCI_FUNC(devfn));
	if (p == NULL) {
		return PCIBIOS_DEVICE_NOT_FOUND;
	}
	status = lv1_write_pci_config(p->bus_id, 
				p->bus_pci_bus, p->bus_pci_dev, p->bus_pci_func,
				(uint64_t)offset, size, (uint64_t)val);
	if(status) {
		/* lv1_write_pci_config can't write reg from 0x10 to 0x3f */
	}
	return 0;
}

/* pci device enumeration */
static int ps3pf_pci_enumerate_dev(struct ps3pf_pci *pcibus, uint64_t bus_index)
{
	int i, index, pci;
	uint64_t bus_id, num_of_dev;
	uint64_t device_id, device_type;
	uint64_t bus_pci_bus, bus_pci_dev, bus_pci_func;
	uint64_t pci_bus, pci_dev, pci_func;
	uint64_t irq_type, irq_number;

	bus_id = -1;
	num_of_dev = 0;

	if (ps3pf_pci_read_bus_id(bus_index, &bus_id)) {
		return -1;
	}
	if (ps3pf_pci_read_num_of_dev(bus_index, &num_of_dev)) {
		return -1;
	}

	/* scan all devs */
	for (index = 0; index < num_of_dev; index++) {
		pci = 0;
		device_id = device_type = -1;
		bus_pci_bus = bus_pci_dev = bus_pci_func = -1;
		irq_type = irq_number = -1;

		if (ps3pf_pci_read_device_id(bus_index, index, &device_id)) {
			device_id = -1;
		} else 
		if (ps3pf_pci_read_device_type(bus_index, index, &device_type)){
			device_type = -1;
		} else 
		if (ps3pf_pci_read_pci_bus(bus_index, index, &bus_pci_bus)) {
			bus_pci_bus = -1;
		} else 
		if (ps3pf_pci_read_pci_device(bus_index, index, &bus_pci_dev)) {
			bus_pci_dev = -1;
		} else 
		if (ps3pf_pci_read_pci_func(bus_index, index, &bus_pci_func)) {
			bus_pci_func = -1;
		} else {
			pci = 1;
		}

		/* fixup pci_dev id */
		if (device_type == PS3PF_PCI_DEV_TYPE_UHC) {
			pci_bus = 0; /* XXX */
			pci_dev = device_id;
			pci_func = 0;
		} else {
			pci_bus = bus_id - 1; /* XXX */
			pci_dev = bus_pci_dev;
			pci_func = bus_pci_func;
		}
		
		for (i = 0; i < PS3PF_PCI_INT_MAX; i++) {
			if(ps3pf_pci_read_interrupt(bus_index, index, 
						i, &irq_type, &irq_number)){
				if (bus_index == PS3PF_PCI_ID_STORAGE
					&& i == 0) { 
					/* one storage device */
				} else {
					break;
				}
			}
			pcibus->dev[pcibus->dev_num].bus_index = bus_index;
			pcibus->dev[pcibus->dev_num].dev_index = index;
			pcibus->dev[pcibus->dev_num].bus_id = bus_id;
			pcibus->dev[pcibus->dev_num].dev_id = device_id;
			pcibus->dev[pcibus->dev_num].dev_type = device_type;
			pcibus->dev[pcibus->dev_num].bus_pci_bus = bus_pci_bus;
			pcibus->dev[pcibus->dev_num].bus_pci_dev = bus_pci_dev;
			pcibus->dev[pcibus->dev_num].bus_pci_func = bus_pci_func;
			pcibus->dev[pcibus->dev_num].pci_bus = pci_bus;
			pcibus->dev[pcibus->dev_num].pci_dev = pci_dev;
			pcibus->dev[pcibus->dev_num].pci_func = pci_func + i;
			pcibus->dev[pcibus->dev_num].irq = irq_number;
			pcibus->dev[pcibus->dev_num].pci = pci;
			pcibus->dev[pcibus->dev_num].open = 0;
			pcibus->dev[pcibus->dev_num].region = 0;
			pcibus->dev[pcibus->dev_num].dma = 0;
			pcibus->dev_num++;
			if(pcibus->dev_num == PS3PF_PCI_DEV_MAX) {
				printk("error: PCI_DEV_MAX (%d)\n",
					pcibus->dev_num);
				goto full;
			}
		}
		pcibus->dev[pcibus->dev_num].bus_id = -1;
	}
full:
	return 0;
}

static void ps3pf_pci_fixup_enumeration(struct ps3pf_pci *pcibus)
{
	int i;

	if (!ps3pf_pci_usb_param) {
	/* not use usb */
		for (i = 0; i < pcibus->dev_num; i++) {
			if (pcibus->dev[i].dev_type == PS3PF_PCI_DEV_TYPE_UHC) {
				pcibus->dev[i].pci_bus = -1;
				pcibus->dev[i].pci_dev = 0;
				pcibus->dev[i].pci_func = 0;
			}
		}
	}
}

static int ps3pf_open_device(struct ps3pf_dev_table *p)
{
	uint64_t status;
	int res = -1;

	status = lv1_open_device(p->bus_id, p->dev_id, 0);
	if (status == -9) { /* XXX */ 
		DPRINTK("already opened: %ld:%ld\n", p->bus_id, p->dev_id);
		res = 0;
	} else if (status) {
		printk("lv1_open_device %ld:%ld faild, status=%ld\n", 
				status, p->bus_id, p->dev_id);
		res = -1;
	} else {
		DPRINTK("open_device: %lx:%lx\n", 
				p->bus_id, p->dev_id);
		p->open = 1; /* device opened */
		res = 0;
	}
	return res;
}


struct pci_ops ps3pf_pci_ipci_ops = {
	.read = ps3pf_ipci_read_config,
	.write = ps3pf_ipci_write_config
};

struct pci_ops ps3pf_pci_epci_ops = {
	.read = ps3pf_epci_read_config,
	.write = ps3pf_epci_write_config
};

void
ps3pf_pcibios_init(void)
{
	int j, busno;
	struct ps3pf_dev_table *table;
	struct ps3pf_pci *pcibus;
	struct pci_controller *phb;
	struct {
		struct pci_ops *ops;
	} phbs[] = {
		{ &ps3pf_pci_ipci_ops }, /* internal PCI */
#ifdef CONFIG_PPC_PS3PF_SB_EPCI
		{ &ps3pf_pci_epci_ops }, /* external PCI */
		{ &ps3pf_pci_epci_ops }  /* external PCIe */
#endif
	};

	busno = 0;
	pcibus = &g_ps3pf_pci;
	for (j = 0; j < sizeof(phbs)/sizeof(*phbs); j++) {
		ps3pf_pci_enumerate_dev(pcibus, j + 1);
		ps3pf_pci_fixup_enumeration(pcibus);

		phb = pcibios_alloc_controller(NULL);
		if (phb == NULL) {
			printk("%s: allocate pci_controller failed\n",
			       __FUNCTION__);
			return;
		}

		phb->first_busno =  busno++;
		phb->last_busno  =  phb->first_busno;
		phb->ops = phbs[j].ops;
		phb->buid = 0;
		phb->global_number = 0;
		phb->arch_data   = &pcibus->dev_node[j];

		pcibus->dev_node[j].data = (void *)phb;
	}

	/* special device */
	ps3pf_pci_enumerate_dev(pcibus, PS3PF_PCI_ID_STORAGE);

	/* gelic device */
	table = get_dev_table_by_type(pcibus, PS3PF_GELIC_DEV_TYPE);
	if (table) {
		ps3pf_open_device(table);
	}
	return;
}

uint64_t ps3pf_allocate_dma_region(uint64_t bus_id, uint64_t dev_id,
				int internal)
{
	uint64_t size, io_size, io_pagesize;
	uint64_t status , dma, flg = 0;
	
	io_size = PS3PF_PCI_DMA_SIZE(ps3pf_mem_total);
	io_pagesize = PS3PF_PCI_IO_PAGESIZE;
	if (internal) flg = 2; /* 8bit mode */
	status = lv1_allocate_device_dma_region(bus_id, dev_id,
					io_size,
					io_pagesize, 
					flg,
					&dma);
	if (status) {
		printk("lv1_allocate_device_dma_region faild, status=%ld\n",
			status);
		return 0;
	}

	size = ps3pf_rm_limit;
	status = lv1_map_device_dma_region(bus_id, dev_id,
					0, /* lpar addr */
					dma, /* I/O addr */
					size,
					0xf800000000000000UL  /* flags */);
	if(status) {
		printk("lv1_map_device_dma_region faild, status=%ld\n", status);
		return 0;
	}

	size = ps3pf_2nd_mem_size;
	status = lv1_map_device_dma_region(bus_id, dev_id, 
					ps3pf_2nd_mem_base,   /* lpar addr */
					ps3pf_rm_limit + dma, /* I/O addr */
					size,
					0xf800000000000000UL  /* flags */);

	if (status) {
		printk("lv1_map_device_dma_region faild, status=%ld\n", status);
		return 0;
	}
	return dma;
}
EXPORT_SYMBOL(ps3pf_allocate_dma_region);

static int ps3pf_pci_allocate_dma_region(struct ps3pf_pci *pcibus, 
			struct ps3pf_dev_table *p, int internal)
{
	uint64_t bus_id, dev_id;

	bus_id = p->bus_id;
	dev_id = p->dev_id;
	
	DPRINTK("allocate dma region(bus_id:%ld dev_id:%ld)\n", bus_id, dev_id);

	if (internal) {
		if (chk_dma_by_dev_id(pcibus, dev_id)) {
			DPRINTK("internal dev_id:%ld already dma mapped\n",
				dev_id);
			return 0;
		}
	} else {
		if (chk_dma_by_bus_id(pcibus, bus_id)) {
			DPRINTK("bus_id:%ld already dma mapped\n", bus_id);
			return 0;
		}
	}

	p->dma = ps3pf_allocate_dma_region(bus_id, dev_id, internal);
	pci_dma_base = p->dma;

	return 0;
}

uint64_t ps3pf_free_dma_region(uint64_t bus_id, uint64_t dev_id, uint64_t dma)
{
	uint64_t size, io_size, io_pagesize;
	uint64_t status;
	
	io_size = PS3PF_PCI_DMA_SIZE(ps3pf_mem_total);
	io_pagesize = PS3PF_PCI_IO_PAGESIZE;

	if (dma == 0) {
		return 0;
	}
	/* unmap dma_region */
	size = ps3pf_rm_limit;
	status = lv1_unmap_device_dma_region(bus_id, dev_id,
					dma, /* I/O addr */
					size);
	if(status) {
		printk("lv1_unmap_device_dma_region faild, status=%ld\n",
				status);
	}
	size = ps3pf_2nd_mem_size;
	status = lv1_unmap_device_dma_region(bus_id, dev_id, 
					ps3pf_rm_limit +  dma, /* I/O addr */
					size);

	if (status) {
		printk("lv1_unmap_device_dma_region faild, status=%ld\n",
				status);
	}

	/* free dma region */
	status = lv1_free_device_dma_region(bus_id, dev_id, dma);
	if (status) {
		printk("lv1_free_device_dma_region faild, status=%ld\n",
			status);
	}
	return 0;
}
EXPORT_SYMBOL(ps3pf_free_dma_region);

static int ps3pf_pci_free_dma_region(struct ps3pf_dev_table *p)
{
	ps3pf_free_dma_region(p->bus_id, p->dev_id, p->dma);
	p->dma = 0;

	return 0;
}

static void
ps3pf_fixup_device_resources_internal_usb(struct ps3pf_pci *pcibus, 
			struct pci_dev *dev, struct ps3pf_dev_table *p)
{
	uint64_t status, lpar_addr;
	uint64_t region_type, region_address, region_size;
	int res;

	/* open device */
	if (!chk_dev_open(pcibus, p->bus_id, p->dev_id)) { 
		res = ps3pf_open_device(p);
		if (res < 0) {
			printk("ps3pf_open_device faild\n");
			return;
		}
	}

	/* get region_address and map */
	region_type = region_address = region_size = lpar_addr = 0;
	if (ps3pf_pci_read_region_type(p->bus_index, p->dev_index, p->pci_func,
			&region_type)){
		goto end;
	} else
	if (ps3pf_pci_read_region_data(p->bus_index, p->dev_index, p->pci_func, 
			&region_address, &region_size)) {
			goto end;
	}
	status = lv1_map_device_mmio_region(p->bus_id, p->dev_id,
					region_address,
					region_size,
					PS3PF_PCI_MMIO_PAGESIZE,
					&lpar_addr);
	if (status) {
		printk("lv1_map_device_mmio_region failed, status=%ld\n",
			status);
		goto end;
	}
	p->lpar[p->pci_func] = lpar_addr;
	DPRINTK("%d:type:%ld region:%lx size:%lx > lpar:%lx-%lx\n",
			p->pci_func, region_type, 
			region_address, region_size, 
			lpar_addr, lpar_addr + region_size -1);
	/* set resource */
	if (region_type == PS3PF_PCI_REGION_TYPE_IO) {
		dev->resource[0].flags = IORESOURCE_IO;
	} else {
		dev->resource[0].flags = IORESOURCE_MEM;
	}
	dev->resource[0].start = lpar_addr;
	dev->resource[0].end = lpar_addr + region_size - 1;
	p->region = 1;
end:
	return ;
}

static void ps3pf_pci_unmap_mmio_region(struct ps3pf_dev_table *p)
{
	uint64_t status;
	int i;

	if (p->region == 0) {
		return ;
	}
	for (i = 0; i < PS3PF_PCI_REGION_MAX; i++) {
		if (p->lpar[i]) {
			status = lv1_unmap_device_mmio_region(p->bus_id,
							p->dev_id,
							p->lpar[i]);
			if (status) {
				printk("lv1_unmap_device_mmio_region failed, status=%ld\n",
					status);
				break;
			}
			p->lpar[i] = 0;
		}
	}
	p->region = 0;
}

static void
ps3pf_fixup_device_resources_pci(struct ps3pf_pci *pcibus, 
				struct pci_dev *dev, struct ps3pf_dev_table *p)
{
	uint64_t region_type, region_address, region_size;
	uint64_t status, lpar_addr;
	struct ps3pf_dev_table *regp;
	int res, i;

	/* open device */
	res = ps3pf_open_device(p);
	if (res < 0) {
		printk("ps3pf_open_device faild\n");
		return ;
	}
	/* get mmio region and map */
	for (i = 0; i < PS3PF_PCI_REGION_MAX; i++) {
		region_type = region_address = region_size = lpar_addr = 0;
		if (ps3pf_pci_read_region_type(p->bus_index, p->dev_index, i,
				&region_type)){
			break;
		} else
		if (ps3pf_pci_read_region_data(p->bus_index, p->dev_index, i, 
				&region_address, &region_size)) {
				break;
		}
		status = lv1_map_device_mmio_region(p->bus_id, p->dev_id,
						region_address,
						region_size,
						PS3PF_PCI_MMIO_PAGESIZE,
						&lpar_addr);
		if (status) {
			printk("lv1_map_device_mmio_region failed, status=%ld\n",
				status);
			break;
		}
		p->lpar[i] = lpar_addr;
		DPRINTK("%d:type:%ld region_addr:%lx size:%lx > lpar:%lx-%lx\n",
				i, region_type, 
				region_address, region_size, 
				lpar_addr, lpar_addr + region_size -1);
		/* set resource */
		if (region_type == PS3PF_PCI_REGION_TYPE_IO) {
			dev->resource[i].flags = IORESOURCE_IO;
		} else {
			dev->resource[i].flags = IORESOURCE_MEM;
		}
		dev->resource[i].start = lpar_addr;
		dev->resource[i].end = lpar_addr + region_size - 1;
		regp = get_dev_table(pcibus, p->pci_bus, p->pci_dev,
					p->pci_func);
		if (regp) {
			regp->region = 1;
		}
	}
}

static void
ps3pf_fixup_device_resources(struct ps3pf_pci *pcibus, struct pci_dev *dev)
{
	struct ps3pf_dev_table *p;
	uint32_t vendor_id;
	int ret;

	DPRINTK("%s bus:%d dev:%d func:%d\n", __FUNCTION__, 
		dev->bus->number, PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));
	DPRINTK(" %s:  IRQ %d\n", pci_name(dev), dev->irq);
	DPRINTK(" resource[0]:start:%lx end:%lx\n",
		dev->resource[0].start, dev->resource[0].end);

	p = get_dev_table(pcibus, dev->bus->number, 
				PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));
	if (p == NULL) {
		DPRINTK("no device in the table \n");
		return ;
	}

	/* check vendor id */
	if (p->pci_bus > 0) {
		ps3pf_epci_read_config(dev->bus, dev->devfn, 0, 4, &vendor_id);
		if (vendor_id == 0x81ff104d) {
			DPRINTK("%x:driver not supported \n", vendor_id);
			dev->resource[0].start = 0;
			dev->resource[0].end = 0;
			return;
		}
	}

	if (p->pci_bus > 0) { /* PCI or PCIe fixup irqs */
		p->irq += p->pci_func;
	}

	/* irq */
	ret = ps3pf_connect_io_irq(p->irq , &pcibus->bus_irqs[p->irq]);
	if (ret) {
		printk("ps3pf_connect_io_irq failed: ret:%d irq:%d\n",
			ret, p->irq);
		return ;
	}
	DPRINTK(" IRQ: intr=%2d IRQ=%d\n",
			p->irq, pcibus->bus_irqs[p->irq]);
	dev->irq = p->irq;

	switch(p->pci_bus) {
	case 0:		/* internl PCI */
		if(p->dev_type == PS3PF_PCI_DEV_TYPE_UHC) {
			ps3pf_fixup_device_resources_internal_usb(pcibus,
							dev, p);
		}
		ret = ps3pf_pci_allocate_dma_region(pcibus, p, 1);
		break;
	case 1:		/* external PCI */
	case 2:		/* external PCIe */
		ps3pf_fixup_device_resources_pci(pcibus, dev, p);
		ret = ps3pf_pci_allocate_dma_region(pcibus, p, 0);
		break;
	default:
		break;
	}
}

void
pcibios_fixup_bus(struct pci_bus *bus)
{
	struct ps3pf_pci *pcibus = &g_ps3pf_pci;
	struct list_head *l;

	for (l = bus->devices.next; l != &bus->devices; l = l->next) {
		struct pci_dev *dev = pci_dev_b(l);
		ps3pf_fixup_device_resources(pcibus, dev);
	}
}
EXPORT_SYMBOL(pcibios_fixup_bus);

/* free region and close device */
void
ps3pf_pci_shutdown(void)
{
	int i;
	uint64_t status, bus_id, dev_id;
	struct ps3pf_pci *pcibus;

	pcibus = &g_ps3pf_pci;

	for (i = 0; i < pcibus->dev_num; i++) {
		/* unmap dma region */
		ps3pf_pci_free_dma_region(&pcibus->dev[i]);
	}

	for (i = 0; i < pcibus->dev_num; i++) {
		/* unmap mmio region */
		ps3pf_pci_unmap_mmio_region(&pcibus->dev[i]);
	}

	for (i =0 ; i < pcibus->dev_num; i++) {
		/* close */
		bus_id = pcibus->dev[i].bus_id;
		dev_id = pcibus->dev[i].dev_id;
		if (pcibus->dev[i].open) {
			status = lv1_close_device(bus_id, dev_id);
			if (status) {
				DPRINTK("lv1_close_device failed, status=%ld\n", 
					status);
			}
			pcibus->dev[i].open = 0;
		}
	}
	pcibus->dev_num = 0;
}

static int __init early_param_sb_usb(char *p)
{
	if (!p) {
		ps3pf_pci_usb_param = 1;
		return 0;
	}
	if (strstr(p, "n"))
		ps3pf_pci_usb_param = 0;
	else
		ps3pf_pci_usb_param = 1;
	return 0;
}
early_param("sb_usb", early_param_sb_usb);
