/*
    tunefs.reiserfs.c -- reiserfs filesystem tuner
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include "getopt.h"

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <dal/file_dal.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/libprogs_tools.h>
#include <reiserfs/libprogs_gauge.h>

#if ENABLE_NLS
#	include <locale.h>
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

static void tunefs_print_usage(void) {
    fprintf(stderr, _("Usage: tunefs.reiserfs options device\n"));
    fprintf(stderr, _("Options:\n"
		"  -v | --version                      prints current version\n"
		"  -u | --usage                        prints program usage\n"
		"  -j FILE | --journal-device=FILE     device where journal is lies\n"
		"  -d FILE | --journal-new-device=FILE new journal device\n"
		"  -s N | --journal-size=N             journal size\n"
		"  -o N | --journal-offset=N           journal offset for relocated journal\n"
		"  -t N | --transaction-max-size=N     transaction max size\n"
		"  -n | --no-journal-available         no journal available now\n"));
    fprintf(stderr, _(
		"  -l LABEL | --label=LABEL            sets volume label\n"
		"  -i UUID | --uuid=UUID               sets given uuid to superblock\n"
		"  -q | --quiet                        non-interactive mode\n"));
}	

int main(int argc, char *argv[]) {
    int need_tuning = 1;
    int choice, quiet = 0, journal = 1;
	int new_journal_relocated, old_journal_relocated;
    
    long len = 0, start = 0;
    long max_trans_size = 0;

    reiserfs_fs_t *fs;
    reiserfs_gauge_t *gauge = NULL;
    
    dal_t *host_dal = NULL, *journal_dal = NULL, *new_journal_dal = NULL;
    char *host_dev = NULL, *journal_dev = NULL, *new_journal_dev = NULL;
    char *label = NULL, *uuid = NULL;
    
    static struct option long_options[] = {
		{"version", no_argument, NULL, 'v'},
		{"usage", no_argument, NULL, 'u'},
		{"journal-device", required_argument, NULL, 'j'},
		{"journal-new-device", required_argument, NULL, 'd'},
		{"journal-size", required_argument, NULL, 's'},
		{"journal-offset", required_argument, NULL, 'o'},
		{"transaction-max-size", required_argument, NULL, 't'},
		{"no-journal-available", required_argument, NULL, 'n'},
		{"label", required_argument, NULL, 'l'},
		{"uuid", required_argument, NULL, 'i'},
		{"quiet", no_argument, NULL, 'q'},
		{0, 0, 0, 0}
    };
    
#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif
    
    while ((choice = getopt_long_only(argc, argv, "uvj:d:s:o:t:nl:i:q", long_options, 
		(int *)0)) != EOF) 
    {
		switch (choice) {
			case 'u': {
				tunefs_print_usage();
				return 0;
			}
			case 'v': {
				printf("%s %s\n", argv[0], VERSION);
				return 0;
			}
			case 'n': {
				journal = 0;
				break;
			}
			case 'j': {
				if (!progs_check_dev(journal_dev = optarg)) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Device %s doesn't exists or invalid."), optarg);
					return 0xfe;
				}
				break;
			}
			case 'd': {
				if (!progs_check_dev(new_journal_dev = optarg)) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Device %s doesn't exists or invalid."), optarg);
					return 0xfe;
				}
				break;
			}
			case 's': {
				if ((len = progs_str2long(optarg, -1)) <= 0) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Invalid journal size (%s)."), optarg);
					return 0xfe;	
				}
				break;
			}
			case 'o': {
				if ((start = progs_str2long(optarg, -1)) == -1) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Invalid journal offset (%s)."), optarg);
					return 0xfe;	
				}
				break;
			}
			case 't': {
				if ((max_trans_size = progs_str2long(optarg, -1)) == -1) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Invalid transaction max size (%s)."), optarg);
					return 0xfe;	
				}
				break;
			}
			case 'i': {
				if (strlen(optarg) < 16) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						_("Invalid uuid (%s)."), optarg);
					return 0xfe;
				}
				uuid = optarg;
				break;
			}
			case 'l': {
				label = optarg;
				break;
			}
			case 'q': {
				quiet = 1;
				break;
			}
			case '?': {
				tunefs_print_usage();
				return 0xfe;
			}
		}
    }
    
    if (optind >= argc) {
		tunefs_print_usage();
		return 0xfe;
    }
    
    host_dev = argv[optind];
    
    /* Checking given device for validness */
    if (!progs_check_dev(host_dev)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Device %s doesn't exists or invalid."), host_dev);
		return 0xfe;
    }
    
    /* Creating device abstraction layer */
    if (!(host_dal = file_dal_open(host_dev, DEFAULT_BLOCK_SIZE, O_RDWR))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't create device abstraction handler for device %s."), 
			host_dev);
		goto error;
    }

    if (journal_dev) {
		if (!strcmp(journal_dev, host_dev))
			journal_dal = NULL;
		else {
			if (!(journal_dal = file_dal_open(journal_dev, dal_block_size(host_dal), 
				O_RDWR)))
			{
				libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					_("Couldn't create device abstraction handler for device %s."), 
					journal_dev);
				goto error_free_host_dal;
			}
		}    
    }

    if (new_journal_dev) {
		if ((journal_dev && !strcmp(new_journal_dev, journal_dev)) || 
				!strcmp(new_journal_dev, host_dev))
			new_journal_dal = NULL; 
		else {
			if (!(new_journal_dal = file_dal_open(new_journal_dev, 
				dal_block_size(host_dal), O_RDWR))) 
			{
				libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					_("Couldn't create device abstraction handler for device %s."), 
					new_journal_dev);
				goto error_free_journal_dal;
			}
		}
    }

    if (!(fs = reiserfs_fs_open(host_dal, (!journal ? NULL : 
		(journal_dal ? journal_dal : host_dal))))) 
    {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't open reiserfs on device %s."), host_dev);
		goto error_free_journal_dal;
    }
    
    old_journal_relocated = journal_dev && strcmp(journal_dev, host_dev);
    
    if (old_journal_relocated != reiserfs_fs_journal_relocated(fs)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Invalid journal location parameters detected. Filesystem "
			"has %s journal, but specified %s journal."), 
			reiserfs_fs_journal_kind_str(reiserfs_fs_journal_relocated(fs)), 
			reiserfs_fs_journal_kind_str(old_journal_relocated));
		goto error_free_journal_dal;
    }	
    
    new_journal_relocated = new_journal_dev && strcmp(new_journal_dev, host_dev);
    need_tuning = reiserfs_fs_journal_relocated(fs) != new_journal_relocated ||
		max_trans_size || start || len;
    
    if (!need_tuning && !label && !uuid) {
		libreiserfs_exception_throw(EXCEPTION_INFORMATION, EXCEPTION_CANCEL, 
			_("Filesystem doesn't needed tunning."));
		goto error_free_journal_dal;    
    }
	
	choice = 'y';
    if (!quiet) {
		if (!(choice = progs_user_choose("ynYN", _("Please select (y/n) "), 
				_("Are you ready (y/n) "))))
			goto error_free_journal_dal;
    }
    
    if (choice == 'n' || choice == 'N')
		goto error_free_journal_dal;

    if (!max_trans_size) {
        max_trans_size = JOURNAL_MAX_TRANS;
        if (reiserfs_fs_block_size(fs) < 4096)
            max_trans_size = JOURNAL_MAX_TRANS /(4096 / reiserfs_fs_block_size(fs));
    }

    if (new_journal_dal) {
    	if (!len)
	    	len = reiserfs_jr_max_len(new_journal_dal, start, 1);
    } else {
		blk_t max_journal_len;
		
		if (!start)
			start = reiserfs_fs_journal_offset(fs);
		
		if (!len) 
			len = dal_block_size(host_dal) == 1024 ? DEFAULT_JOURNAL_SIZE_BS1024 :
				DEFAULT_JOURNAL_SIZE_BS4096;

		if ((blk_t)len > (max_journal_len = reiserfs_jr_max_len(host_dal, start, 0)))
			len = max_journal_len;
    }

    fprintf(stderr, _("Tunning %s\n"), host_dev);
    fflush(stderr);

    if (need_tuning) {
		gauge = progs_gauge_create();
		
		if (!reiserfs_fs_journal_tune(fs, new_journal_dal ? new_journal_dal : host_dal, 
			start, len, max_trans_size, gauge)) 
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't tune filesystem."));
			goto error_free_fs;
		}
		
		if (gauge)
			progs_gauge_free(gauge);
    }
    
    if ((uuid || label) && reiserfs_fs_format(fs) == FS_FORMAT_3_5) {
		libreiserfs_exception_throw(EXCEPTION_INFORMATION, EXCEPTION_CANCEL, 
			_("Sorry, label and uuid not supported for reiserfs 3.5."));
		goto error_free_fs;
    }
    
    /* Updating uuid and label */
    if (uuid)
		reiserfs_fs_uuid_update(fs, uuid);
    
    if (label)
		reiserfs_fs_label_update(fs, label);
    
    reiserfs_fs_close(fs);
    
    fprintf(stderr, _("syncing..."));
    fflush(stderr);
    
    if (new_journal_dal) {
		dal_sync(new_journal_dal);
		file_dal_close(new_journal_dal);
    }
    
    if (journal_dal) {
		dal_sync(journal_dal);
		file_dal_close(journal_dal);
    }

    dal_sync(host_dal);
    file_dal_close(host_dal);

    fprintf(stderr, _("done\n"));
    fflush(stderr);

    return 0;
    
error_free_fs:
    if (gauge)
        progs_gauge_free(gauge);
    reiserfs_fs_close(fs);    
error_free_new_journal_dal:
    if (new_journal_dal)
		file_dal_close(new_journal_dal);    
error_free_journal_dal:
    if (journal_dal)
		file_dal_close(journal_dal);
error_free_host_dal:
    file_dal_close(host_dal);
error:
    return 0xff;
}

