/**
  @file bt-error.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/

#include <stdio.h>
#include <string.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>

#include "log.h"
#include "bt-dbus.h"
#include "bt-error.h"

typedef struct {
    gint code;
    const char *derr;
} BTErrorMap;

static BTErrorMap btsdp_map[] = {
    { BT_SDP_ERROR_BT_DISABLED,         BTSDP_ERROR_BT_DISABLED       },
    { BT_SDP_ERROR_CONNECT_FAILED,      BTSDP_ERROR_CONNECT_FAILED    },
    { BT_SDP_ERROR_IN_PROGRESS,         BTSDP_ERROR_QUERY_IN_PROGRESS },
    { BT_SDP_ERROR_FAILED,              BTSDP_ERROR_FAILED            },
    { 0, NULL }
};

static BTErrorMap btsearch_map[] = {
    { BT_SEARCH_ERROR_BT_DISABLED,      BTSEARCH_ERROR_BT_DISABLED        },
    { BT_SEARCH_ERROR_NO_SEARCH,        BTSEARCH_ERROR_NO_SEARCH          },
    { BT_SEARCH_ERROR_IN_PROGRESS,      BTSEARCH_ERROR_SEARCH_IN_PROGRESS },
    { BT_SEARCH_ERROR_FAILED,           BTSEARCH_ERROR_FAILED             },
    { 0, NULL }
};

static BTErrorMap btcond_map[] = {
    { BT_COND_ERROR_INVALID_SVC,        BTCOND_ERROR_INVALID_SVC     },
    { BT_COND_ERROR_INVALID_DEV,        BTCOND_ERROR_INVALID_DEV     },
    { BT_COND_ERROR_CONNECTED,          BTCOND_ERROR_CONNECTED       },
    { BT_COND_ERROR_BOUND,              BTCOND_ERROR_BOUND           },
    { BT_COND_ERROR_NOT_CONNECTED,      BTCOND_ERROR_NOT_CONNECTED   },
    { BT_COND_ERROR_NOT_BOUND,          BTCOND_ERROR_NOT_BOUND       },
    { BT_COND_ERROR_BT_DISABLED,        BTCOND_ERROR_BT_DISABLED     },
    { BT_COND_ERROR_CONNECT_FAILED,     BTCOND_ERROR_CONNECT_FAILED  },
    { BT_COND_ERROR_AUTH_FAILED,        BTCOND_ERROR_AUTH_FAILED     },
    { BT_COND_ERROR_SMALL_CRYPT_KEY,    BTCOND_ERROR_SMALL_CRYPT_KEY },
    { BT_COND_ERROR_CANCELLED,          BTCOND_ERROR_CANCELLED       },
    { BT_COND_ERROR_FAILED,             BTCOND_ERROR_INTERNAL        },
    { 0, NULL }
};

static const char *btsdp_dbus_error(GError *err)
{
    BTErrorMap *map;

    for (map = btsdp_map; map->derr != NULL; map++) {
        if (err->code == map->code)
            return map->derr;
    }

    error("Unknown btsdp error %d", err->code);
    return BTSDP_ERROR_FAILED;
}

static const char *btsearch_dbus_error(GError *err)
{
    BTErrorMap *map;

    for (map = btsearch_map; map->derr != NULL; map++) {
        if (err->code == map->code)
            return map->derr;
    }

    error("Unknown btsearch error %d", err->code);
    return BTSEARCH_ERROR_FAILED;
}

static const char *btcond_dbus_error(GError *err)
{
    BTErrorMap *map;

    for (map = btcond_map; map->derr != NULL; map++) {
        if (err->code == map->code)
            return map->derr;
    }

    error("Unknown btcond error %d", err->code);
    return BTCOND_ERROR_INTERNAL;
}

static const char *gerror_dbus_error(GError *err)
{
    static char name[128];
    if (sscanf(err->message, "D-BUS error %127s:", name) != 1) {
        error("Unrecognized gerror message: %s", err->message);
        return "com.nokia.bt.error.unknown";
    } 

    debug("Detected non-BT error: %s", name);

    return name;
}

static void btcond_set_g_error(GError **err, DBusError *derr)
{
    BTErrorMap *map;

    for (map = btcond_map; map->derr != NULL; map++) {
        if (strcmp(map->derr, derr->name) == 0) {
            g_set_error(err, BT_COND_ERROR,
                        map->code,
                        "%s", derr->message);
            return;
        }
    }

    error("Unknown btcond D-BUS error %s", derr->name);
    g_set_error(err, BT_COND_ERROR,
            BT_COND_ERROR_FAILED,
            "%s", derr->name);
}

static void btsearch_set_g_error(GError **err, DBusError *derr)
{
    BTErrorMap *map;

    for (map = btsearch_map; map->derr != NULL; map++) {
        if (strcmp(map->derr, derr->name) == 0) {
            g_set_error(err, BT_SEARCH_ERROR,
                        map->code,
                        "%s", derr->message);
            return;
        }
    }

    error("Unknown btsearch D-BUS error %s", derr->name);
    g_set_error(err, BT_SEARCH_ERROR,
            BT_SEARCH_ERROR_FAILED,
            "%s", derr->name);
}

static void btsdp_set_g_error(GError **err, DBusError *derr)
{
    BTErrorMap *map;

    for (map = btsdp_map; map->derr != NULL; map++) {
        if (strcmp(map->derr, derr->name) == 0) {
            g_set_error(err, BT_SDP_ERROR,
                        map->code,
                        "%s", derr->message);
            return;
        }
    }

    error("Unknown btsdp D-BUS error %s", derr->name);
    g_set_error(err, BT_SDP_ERROR,
            BT_SDP_ERROR_FAILED,
            "%s", derr->name);
}

const char *bt_dbus_error(GError *err)
{
    if (err->domain == BT_SDP_ERROR)
        return btsdp_dbus_error(err);
    if (err->domain == BT_COND_ERROR)
        return btcond_dbus_error(err);
    if (err->domain == BT_SEARCH_ERROR)
        return btsearch_dbus_error(err);
    if (err->domain == DBUS_GERROR)
        return gerror_dbus_error(err);

    error("Unknown error domain \"%s\" given to bt_dbus_error",
            g_quark_to_string(err->domain));
    return NULL;
}

gboolean bt_set_g_error(GError **err, DBusError *derr)
{
    if (g_str_has_prefix(derr->name, BTCOND_ERROR))
        btcond_set_g_error(err, derr);
    else if (g_str_has_prefix(derr->name, BTSDP_ERROR))
        btsdp_set_g_error(err, derr);
    else if (g_str_has_prefix(derr->name, BTSEARCH_ERROR))
        btsearch_set_g_error(err, derr);
    else
        return FALSE;

    return TRUE;
}

GQuark bt_cond_error_quark(void)
{
    static GQuark q = 0;
    if (q == 0)
        q = g_quark_from_static_string("bt-cond-error-quark");

    return q;
}

GQuark bt_sdp_error_quark(void)
{
    static GQuark q = 0;
    if (q == 0)
        q = g_quark_from_static_string("bt-sdp-error-quark");

    return q;
}

GQuark bt_search_error_quark(void)
{
    static GQuark q = 0;
    if (q == 0)
        q = g_quark_from_static_string("bt-search-error-quark");

    return q;
}


