/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: skeletonmaker.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 02:19:16 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "sal/config.h"

#include "codemaker/generatedtypeset.hxx"
#include "codemaker/commonjava.hxx"
#include "codemaker/options.hxx"
#include "codemaker/typemanager.hxx"
#include "codemaker/unotype.hxx"

#include "registry/reader.hxx"
#include "registry/types.h"
#include "rtl/strbuf.hxx"
#include "rtl/string.hxx"
#include "sal/types.h"

#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include <vector>

namespace {

struct ProgramOptions {
    ProgramOptions(): java15(true), all(false), dump(false),
                      language(0) {}

    bool java15;
    bool all;
    bool dump;
    // language specifier - is extendable
    // 1 = Java
    // 2 = C++
    short language;
    rtl::OString outputPath;
};


void printUsageAndExit(char* programname, char* version) {
    std::cerr << "\n using: " << programname
              << " dump [<options>] -L<typelibrary> ... -T<type> ...\n"
//              << "        " << programname << " component [<options>] -L<typelibrary> ... -N<name> -T<type> ...\n"
              << "        " << programname << " -V\n"
              << "        " << programname << " -h\n"
              <<
              ("\n sub-commands:\n"
               "    dump        dump declarations on stdout (e.g. constructors, methods, type mapping for properties)\n"
//               "    component   generates language specific code skeleton files using the implementation\n"
//               "                name as the file and class name\n"
               "\n options:\n"
               "    -a                list all interface methods, not only the direct ones\n"
//               "    -java4|java5|cpp  select the target language\n"
               "    -java4|java5      select the target language\n"
               "                      java4 generate output for Java 1.4 or earlier\n"
               "                      java5 generate output for Java 1.5 or later (is currently the default)\n"
//               "                      cpp generate output for C++\n"
               "    -B<name>          specify the base node within the binary type libraries (default is \"/\")\n"
//               "    -O<path>          path specifies an existing directory where the output files are generated\n"
               "    -L<file>          specifies a binary type library (can be used more than once)\n"
//               "    -N<name>          specifies an implementation name for the component (used as classname,\n"
//               "                      filename and package|namespace name)\n"
               "    -T<name>          specifies an UNOIDL type name, e.g. com.sun.star.text.XText (can be used more than once)\n"
               "    -V                print version number and exit\n"
               "    -h                print this help and exit\n\n"
               " Sun Microsystems (R) ")
              << programname << " Version " << version << "\n\n";
    exit(EXIT_FAILURE);
}

codemaker::UnoType::Sort decomposeResolveAndCheck(
    TypeManager const & manager, rtl::OString const & type,
    bool resolveTypedefs, bool allowVoid, bool allowExtraEntities,
    RTTypeClass * typeClass, rtl::OString * name, sal_Int32 * rank,
    std::vector< rtl::OString > * arguments)
{
    codemaker::UnoType::Sort sort = codemaker::decomposeAndResolve(
        manager, type, resolveTypedefs, allowVoid, allowExtraEntities,
        typeClass, name, rank, arguments);
    for (std::vector< rtl::OString >::iterator i(arguments->begin());
         i != arguments->end(); ++i)
    {
        RTTypeClass typeClass2;
        rtl::OString name2;
        sal_Int32 rank2;
        std::vector< rtl::OString > arguments2;
        decomposeResolveAndCheck(
            manager, *i, true, false, false, &typeClass2, &name2, &rank2,
            &arguments2);
    }
    return sort;
}

void printJavaType(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    rtl::OString const & type, bool referenceType);

void printJavaType(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    codemaker::UnoType::Sort sort, RTTypeClass typeClass,
    rtl::OString const & name, sal_Int32 rank,
    std::vector< rtl::OString > const & arguments, bool referenceType)
{
    o
        << (codemaker::java::translateUnoToJavaType(
                sort, typeClass, name, referenceType).
            replace('/', '.').getStr());
    if (!arguments.empty() && options.java15) {
        o << '<';
        for (std::vector< rtl::OString >::const_iterator i(arguments.begin());
             i != arguments.end(); ++i)
        {
            if (i != arguments.begin()) {
                o << ", ";
            }
            printJavaType(o, options, manager, *i, true);
        }
        o << '>';
    }
    for (sal_Int32 i = 0; i < rank; ++i) {
        o << "[]";
    }
}

void printJavaType(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    rtl::OString const & type, bool referenceType)
{
    RTTypeClass typeClass;
    rtl::OString name;
    sal_Int32 rank;
    std::vector< rtl::OString > arguments;
    codemaker::UnoType::Sort sort = codemaker::decomposeAndResolve(
        manager, type, true, true, true, &typeClass, &name, &rank, &arguments);
    printJavaType(o,
        options, manager, sort, typeClass, name, rank, arguments,
        referenceType);
}

bool printConstructorParameters(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader, typereg::Reader const & outerReader,
    std::vector< rtl::OString > const & arguments)
{
    bool previous = false;
    if (reader.getSuperTypeCount() != 0) {
        rtl::OString super(
            codemaker::convertString(reader.getSuperTypeName(0)));
        typereg::Reader superReader(manager.getTypeReader(super));
        if (!superReader.isValid()) {
            throw CannotDumpException("Bad type library entity " + super);
        }
        previous = printConstructorParameters(o,
            options, manager, superReader, outerReader, arguments);
    }
    for (sal_uInt16 i = 0; i < reader.getFieldCount(); ++i) {
        if (previous) {
            o << ", ";
        }
        previous = true;
        if ((reader.getFieldFlags(i) & RT_ACCESS_PARAMETERIZED_TYPE) == 0) {
            printJavaType(o,
                options, manager,
                codemaker::convertString(reader.getFieldTypeName(i)),
                false);
        } else if (arguments.empty()) {
            o << "java.lang.Object";
        } else {
            sal_uInt16 tparam = 0;
            while (outerReader.getReferenceTypeName(tparam)
                   != reader.getFieldTypeName(i))
            {
                ++tparam;
                OSL_ASSERT(tparam < outerReader.getReferenceCount());
            }
            // assume std::vector< rtl::OString >::size_type is at least as
            // large as sal_uInt16:
            printJavaType(o, options, manager, arguments[tparam], true);
        }
        o
            << ' '
            << (codemaker::java::translateUnoToJavaIdentifier(
                    codemaker::convertString(reader.getFieldName(i)),
                    "param").
                getStr());
    }
    return previous;
}

void printConstructor(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader,
    std::vector< rtl::OString > const & arguments)
{
    rtl::OString type(codemaker::convertString(reader.getTypeName()));
    o << "public " << type.copy(type.lastIndexOf('/') + 1) << '(';
    printConstructorParameters(o, options, manager, reader, reader, arguments);
    o << ");\n";
}

void printMethodParameters(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader, sal_uInt16 method, bool previous)
{
    for (sal_uInt16 i = 0; i < reader.getMethodParameterCount(method); ++i) {
        if (previous) {
            o << ", ";
        }
        previous = true;
        printJavaType(o,
            options, manager,
            codemaker::convertString(
                reader.getMethodParameterTypeName(method, i)),
            false);
        if (reader.getMethodParameterFlags(method, i) == RT_PARAM_OUT
            || reader.getMethodParameterFlags(method, i) == RT_PARAM_INOUT)
        {
            o << "[]";
        } else if ((reader.getMethodParameterFlags(method, i) & RT_PARAM_REST)
                   != 0)
        {
            o << (options.java15 ? "..." : "[]");
        }
        o
            << ' '
            << (codemaker::java::translateUnoToJavaIdentifier(
                    codemaker::convertString(
                        reader.getMethodParameterName(method, i)),
                    "param").
                getStr());
    }
}

void printExceptionSpecification(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader, sal_uInt16 method)
{
    if (reader.getMethodExceptionCount(method) > 0) {
        o << " throws ";
        for (sal_uInt16 i = 0; i < reader.getMethodExceptionCount(method); ++i)
        {
            if (i != 0) {
                o << ", ";
            }
            printJavaType(o,
                options, manager,
                codemaker::convertString(
                    reader.getMethodExceptionTypeName(method, i)),
                false);
        }
    }
}

void printMethods(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader, codemaker::GeneratedTypeSet & generated)
{
    rtl::OString type(codemaker::convertString(reader.getTypeName()));
    if (generated.contains(type) || type == "com/sun/star/uno/XInterface") {
        return;
    }
    generated.add(type);
    if (options.all) {
        for (sal_uInt16 i = 0; i < reader.getSuperTypeCount(); ++i) {
            typereg::Reader super(
                manager.getTypeReader(
                    codemaker::convertString(
                        reader.getSuperTypeName(i))));
            if (!super.isValid()) {
                throw CannotDumpException(
                    "Bad type library entity "
                    + codemaker::convertString(
                        reader.getSuperTypeName(i)));
            }
            printMethods(o, options, manager, super, generated);
        }
        if (reader.getFieldCount() > 0 || reader.getMethodCount() > 0) {
            o << "/* ";
            printJavaType(o, options, manager, type, false);
            o << ": */\n";
        }
    }
    sal_uInt16 method = 0;
    for (sal_uInt16 i = 0; i < reader.getFieldCount(); ++i) {
        o << "public ";
        printJavaType(o,
            options, manager,
            codemaker::convertString(reader.getFieldTypeName(i)), false);
        o
            << " get"
            << codemaker::convertString(reader.getFieldName(i)).getStr()
            << "()";
        if (method < reader.getMethodCount()
            && reader.getMethodFlags(method) == RT_MODE_ATTRIBUTE_GET
            && reader.getMethodName(method) == reader.getFieldName(i))
        {
            printExceptionSpecification(o, options, manager, reader, method++);
        }
        o << ";\n";
        if ((reader.getFieldFlags(i) & RT_ACCESS_READONLY) == 0) {
            o
                << "public void set"
                << (codemaker::convertString(reader.getFieldName(i)).
                    getStr())
                << '(';
            printJavaType(o, 
                options, manager,
                codemaker::convertString(reader.getFieldTypeName(i)),
                false);
            o << " the_value)";
            if (method < reader.getMethodCount()
                && reader.getMethodFlags(method) == RT_MODE_ATTRIBUTE_SET
                && reader.getMethodName(method) == reader.getFieldName(i))
            {
                printExceptionSpecification(o, options, manager, reader, method++);
            }
            o << ";\n";
        }
    }
    for (; method < reader.getMethodCount(); ++method) {
        o << "public ";
        printJavaType(o, 
            options, manager,
            codemaker::convertString(
                reader.getMethodReturnTypeName(method)),
            false);
        o
            << ' '
            << (codemaker::convertString(reader.getMethodName(method)).
                getStr())
            << '(';
        printMethodParameters(o, options, manager, reader, method, false);
        o << ')';
        printExceptionSpecification(o, options, manager, reader, method);
        o << ";\n";
    }
}

void printConstructionMethods(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader)
{
    for (sal_uInt16 i = 0; i < reader.getMethodCount(); ++i) {
        o << "public static ";
        printJavaType(o,
            options, manager,
            codemaker::convertString(reader.getSuperTypeName(0)), false);
        o << ' ';
        if (reader.getMethodName(i).getLength() == 0) {
            o << "create";
        } else {
            o
                << (codemaker::java::translateUnoToJavaIdentifier(
                        codemaker::convertString(reader.getMethodName(i)),
                        "method").
                    getStr());
        }
        o << '(';
        printMethodParameters(o, options, manager, reader, i, false);
        o << ')';
        printExceptionSpecification(o, options, manager, reader, i);
        o << ";\n";
    }
}

void generateDocumentation(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    rtl::OString const & type);

void printServiceMembers(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    typereg::Reader const & reader, rtl::OString const & type)
{
    for (sal_uInt16 i = 0; i < reader.getReferenceCount(); ++i) {
        rtl::OString referenceType(
            codemaker::convertString(
                reader.getReferenceTypeName(i)).replace('/', '.'));
        
        if ( reader.getReferenceSort(i) == RT_REF_SUPPORTS ) {
            o << "\n// supported interface " << referenceType.getStr() << "\n";
            generateDocumentation(o, options, manager, referenceType);
        } else if ( reader.getReferenceSort(i) == RT_REF_EXPORTS ) {
            o << "\n// exported service " << referenceType.getStr() << "\n";
            generateDocumentation(o, options, manager, referenceType);
        }
    }

    o << "\n// properties of service \""<< type.getStr() << "\"\n";
    for (sal_uInt16 i = 0; i < reader.getFieldCount(); ++i) {
        rtl::OString fieldName(
            codemaker::convertString(reader.getFieldName(i)));
        rtl::OString fieldType(
            codemaker::convertString(reader.getFieldTypeName(i)));

        o << "// private ";
        printJavaType(o, options, manager, fieldType, false);
        o << " "
          << codemaker::java::translateUnoToJavaIdentifier(
              fieldName, "property").getStr()
          << ";\n";        
    }
}

void printMapsToJavaType(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    codemaker::UnoType::Sort sort, RTTypeClass typeClass,
    rtl::OString const & name, sal_Int32 rank,
    std::vector< rtl::OString > const & arguments, const char * javaTypeSort)
{
    o << "maps to Java " << (options.java15 ? "1.5" : "1.4") << " ";
    if (javaTypeSort != 0) {
        o << javaTypeSort << ' ';
    }
    o << "type \"";
    if (rank == 0 && name == "com/sun/star/uno/XInterface") {
        o << "com.sun.star.uno.XInterface";
    } else {
        printJavaType(o, 
            options, manager, sort, typeClass, name, rank, arguments, false);
    }
    o << '"';
}

void generateDocumentation(std::ostream & o,
    ProgramOptions const & options, TypeManager const & manager,
    rtl::OString const & type)
{
    if (type.indexOf('/') >= 0) {
        throw CannotDumpException("Illegal type name " + type);
    }
    rtl::OString binType(type.replace('.', '/'));
    RTTypeClass typeClass;
    rtl::OString name;
    sal_Int32 rank;
    std::vector< rtl::OString > arguments;
    codemaker::UnoType::Sort sort = decomposeResolveAndCheck(
        manager, binType, false, true, true, &typeClass, &name, &rank,
        &arguments);
    o << "\n// UNO";
    if (rank > 0) {
        o << " sequence type";
    } else if (sort != codemaker::UnoType::SORT_COMPLEX) {
        o << " simple type";
    } else {
        typereg::Reader reader(manager.getTypeReader(name));
        if (!reader.isValid()) {
            throw CannotDumpException("Bad type library entity " + name);
        }
        switch (typeClass) {
        case RT_TYPE_INTERFACE:
            o << " interface type";
            break;

        case RT_TYPE_MODULE:
            o << "IDL module";
            break;

        case RT_TYPE_STRUCT:
            if (reader.getReferenceCount() == 0) {
                o << " simple struct type";
            } else if (arguments.empty()) {
                o << " polymorphic struct type template";
            } else {
                o << " instantiated polymorphic struct type";
            }
            break;

        case RT_TYPE_ENUM:
            o << " enum type";
            break;

        case RT_TYPE_EXCEPTION:
            o << " exception type";
            break;

        case RT_TYPE_TYPEDEF:
            o << "IDL typedef";
            break;

        case RT_TYPE_SERVICE:
            if (reader.getSuperTypeCount() > 0) {
                o << " single-inheritance--based service";
            } else {
                o << "IDL accumulation-based service";
            }
            break;

        case RT_TYPE_SINGLETON:
            if ((manager.getTypeReader(
                     codemaker::convertString(
                         reader.getSuperTypeName(0))).
                 getTypeClass())
                == RT_TYPE_INTERFACE)
            {
                o << " inheritance-based singleton";
            } else {
                o << "IDL service-based singleton";
            }
            break;

        case RT_TYPE_CONSTANTS:
            o << "IDL constant group";
            break;

        default:
            OSL_ASSERT(false);
            break;
        }
    }
    o << " \"" << type.getStr() << "\" ";
    sort = codemaker::decomposeAndResolve(
        manager, binType, true, true, true, &typeClass, &name, &rank,
        &arguments);
    if (rank > 0) {
        printMapsToJavaType(o, 
            options, manager, sort, typeClass, name, rank, arguments, "array");
        o << '\n';
    } else if (sort != codemaker::UnoType::SORT_COMPLEX) {
        printMapsToJavaType(o, 
            options, manager, sort, typeClass, name, rank, arguments, 0);
        o << '\n';
    } else {
        typereg::Reader reader(manager.getTypeReader(name));
        if (!reader.isValid()) {
            throw CannotDumpException("Bad type library entity " + name);
        }
        switch (typeClass) {
        case RT_TYPE_INTERFACE:
            printMapsToJavaType(o,
                options, manager, sort, typeClass, name, rank, arguments,
                "interface");
            if (name == "com/sun/star/uno/XInterface") {
                o << '\n';
            } else {
                o
                    << "; " << (options.all ? "all" : "direct")
                    << " methods:\n";
                codemaker::GeneratedTypeSet generated;
                printMethods(o, options, manager, reader, generated);
            }
            break;

        case RT_TYPE_MODULE:
            printMapsToJavaType(o, 
                options, manager, sort, typeClass, name, rank, arguments,
                "package");
            o << '\n';
            break;

        case RT_TYPE_STRUCT:
            if (reader.getReferenceCount() == 0) {
                printMapsToJavaType(o,
                    options, manager, sort, typeClass, name, rank, arguments,
                    "class");
            } else if (arguments.empty()) {
                printMapsToJavaType(o,
                    options, manager, sort, typeClass, name, rank, arguments,
                    options.java15 ? "generic class" : "class");
            } else {
                printMapsToJavaType(o,
                    options, manager, sort, typeClass, name, rank, arguments,
                    options.java15 ? "generic class instantiation" : "class");
            }
            o << "; full constructor:\n";
            printConstructor(o, options, manager, reader, arguments);
            break;

        case RT_TYPE_ENUM:
        case RT_TYPE_CONSTANTS:
            printMapsToJavaType(o,
                options, manager, sort, typeClass, name, rank, arguments,
                "class");
            o << '\n';
            break;

        case RT_TYPE_EXCEPTION:
            printMapsToJavaType(o,
                options, manager, sort, typeClass, name, rank, arguments,
                "exception class");
            o << "; full constructor:\n";
            printConstructor(o, options, manager, reader, arguments);
            break;

        case RT_TYPE_SERVICE:
            if (reader.getSuperTypeCount() > 0) {
                printMapsToJavaType(o,
                    options, manager, sort, typeClass, name, rank, arguments,
                    "class");
                o << "; construction methods:\n";
                printConstructionMethods(o, options, manager, reader);

                rtl::OString super(
                    codemaker::convertString(
                        reader.getSuperTypeName(0)).replace('/', '.'));

                generateDocumentation(o, options, manager, super);
            } else {
                o << ("does not map to Java\n"
                      "// the service members are generated instead\n");
                printServiceMembers(o, options, manager, reader, type);
            }
            break;

        case RT_TYPE_SINGLETON:
            if (reader.getSuperTypeCount() > 0) {
                printMapsToJavaType(o,
                    options, manager, sort, typeClass, name, rank, arguments,
                    "class");
                o << "; get method:\npublic static ";
                printJavaType(o,
                    options, manager,
                    codemaker::convertString(reader.getSuperTypeName(0)),
                    false);
                o
                    << " get(com.sun.star.uno.XComponentContext context);\n";
            } else {
                o << "does not map to Java\n";
            }
            break;

        default:
            OSL_ASSERT(false);
            break;
        }
    }
}

}

int
#if defined WNT
_cdecl
#endif
main(int argc, char ** argv) {

    char* version = "0.1";
    char* programname = "uno-skeletonmaker";
    ProgramOptions options;
    std::vector< rtl::OString > registries;
    std::vector< rtl::OString > types;
    char * base = 0;

    if (argc == 1)
        printUsageAndExit(programname, version);
    
    // check the first arg if it is a valid sub-command
    if (strcmp(argv[1], "dump") == 0) {
        options.dump = true;
    } else if (strcmp(argv[1], "component") == 0) {
        options.dump = false;
    } else if (strcmp(argv[1], "-V") == 0) {
        std::cerr << "\n Sun Microsystems (R) " << programname
                  << " Version " << version << "\n\n";
        exit(EXIT_FAILURE);
    } else if (strcmp(argv[1], "-h") == 0) {
        printUsageAndExit(programname, version);
    } else {
        std::cerr
            << "\nError: Unknwon command \"" << argv[1] << "\"\n\n";
        printUsageAndExit(programname, version);
    }
    
    for (int i = 2; i < argc; ++i) {
        if (argv[i][0] == '-') {
            if (strcmp(argv[i], "-a") == 0) {
                options.all = true;
            } else if (strcmp(argv[i], "-java4") == 0) {
                if (options.language != 0) {
                    std::cerr
                        << "\nError: language selected twice\n\n";
                    printUsageAndExit(programname, version);                    
                }
                options.java15 = false;
                options.language = 1;
            } else if (strcmp(argv[i], "-java5") == 0) {
                if (options.language != 0) {
                    std::cerr
                        << "\nError: language selected twice\n\n";
                    printUsageAndExit(programname, version);                    
                }
                options.language = 1;
            } else if (strcmp(argv[i], "-cpp") == 0) {
                if (options.language != 0) {
                    std::cerr
                        << "\nError: language selected twice\n\n";
                    printUsageAndExit(programname, version);                    
                }
                options.language = 2;
            } else if (argv[i][1] == 'B') {
                base = argv[i] + 2;
            } else if (argv[i][1] == 'O') {
                options.outputPath = rtl::OString(argv[i] + 2);
            } else if (argv[i][1] == 'L') {
                registries.push_back(argv[i] + 2);
            } else if (argv[i][1] == 'T') {
                types.push_back(argv[i] + 2);
            } else {
                if (strcmp(argv[i], "-h") != 0) {
                    std::cerr
                        << "\nError: Unknwon option \"" << argv[i] << "\"\n\n";
                }
                printUsageAndExit(programname, version);
            }
        } else {
            std::cerr
                << "\nError: Unknwon option \"" << argv[i] << "\"\n\n";
            printUsageAndExit(programname, version);
        }
    }

    if ( registries.empty() ) {
        std::cerr
            << ("\nError: no type library is specified, use the -L option "
                "at least once\n");
        printUsageAndExit(programname, version);        
    }
    if ( types.empty() ) {
        std::cerr
            << ("\nError: no type is specified, use the -T option at least once\n");
        printUsageAndExit(programname, version);        
    }
    
    RegistryTypeManager manager;
    if (!manager.init(registries)) {
        std::cerr
            << ("\nError: Using the binary type libraries failed, check the -L"
                " options\n");
        exit(EXIT_FAILURE);
    }
    if (base != 0) {
        manager.setBase(base);
    }
    std::vector< rtl::OString >::const_iterator iter = types.begin();
   	while (iter != types.end()) {
        try {
            std::cout << "\n/********************************************************************************/\n";
            generateDocumentation(std::cout, options, manager, *iter);
        } catch (CannotDumpException & e) {
            std::cout.flush();
            std::cerr << "\nError: " << e.m_message << std::endl;
        }
        ++iter;
    }

}
