/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: configimport.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: hr $ $Date: 2006/06/19 23:21:31 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <stdio.h>

#include "unomain.hxx"

#include <stdarg.h>
#include <osl/thread.h>

//=============================================================================
#define CONTEXT_BOOTSTRAP_PREFIX_ "/modules/com.sun.star.configuration/bootstrap/"
const char k_BootstrapSpoolDir [] = CONTEXT_BOOTSTRAP_PREFIX_ "SpoolDataUrl";
const char k_BootstrapModuleDir[] = CONTEXT_BOOTSTRAP_PREFIX_ "ModuleLayerUrl";
//=============================================================================
static inline rtl::OString narrow(rtl::OUString const & aString)
{
    return rtl::OUStringToOString( aString, osl_getThreadTextEncoding() );
}
//=============================================================================
static void usage()
{
    rtl::OString sApp = narrow( unoapp::getAppCommandPath() );

	fprintf(stderr, "%s - import files into the configuration database\n", sApp.getStr());
	fprintf(stderr, "\nusage :");
	fprintf(stderr, "\n      %s [-v|-q] [-a | -e <entity>] <file> ...\n", sApp.getStr());
	fprintf(stderr, "\n      %s [-v|-q] -s [<directory>]\n", sApp.getStr());
	fprintf(stderr, "\n      %s [-v|-q] -c\n", sApp.getStr());
	fprintf(stderr, "\n  Options:"
			        "\n       -v,--verbose   Report each file being imported to stderr"
			        "\n       -q,--quiet     Suppress all output (including error messages)"
			        "\n       -a,--admin     Import into the 'admin' entity of the default backend"
			        "\n       -e,--entity    Import into the given entity"
			        "\n       -s,--spool     Spool all files from a spool directory to the local module entity."
			        "\n       -c,--cleanup   Cleanup the local module entity, by removing all spooled files.\n");
	fprintf(stderr, "\n  <entity>    - An entity known to the backend"
                    "\n                into which the import should take place\n");
	fprintf(stderr, "\n  <file>      - The path or file-URLs of an XCU file. "
                    "\n                The configuration data of all listed files will be imported\n");
	fprintf(stderr, "\n  <directory> - The path or file-URLs of a directory. "
                    "\n                All files in this directory are imported.\n");
	fprintf(stderr, "\n\n  Optionally you can also use UNO bootstrap arguments "
                      "\n  to select the configuration backend. \n");
	fflush(stderr);
}

//---------------------------------------------------------------------------

#include <com/sun/star/configuration/backend/XLayerImporter.hpp>
#include <com/sun/star/configuration/backend/XLayer.hpp>
#include <com/sun/star/configuration/backend/XBackendEntities.hpp>
#include <vector>
namespace uno = ::com::sun::star::uno;
namespace backend = ::com::sun::star::configuration::backend;
using rtl::OUString;
//---------------------------------------------------------------------------

static uno::Reference< uno::XComponentContext >   getBootstrapContext(uno::Reference< uno::XComponentContext > const & xContext);
static uno::Reference< backend::XLayerImporter >  createImporter(uno::Reference< uno::XComponentContext > const & xContext);
static uno::Reference< backend::XLayer > createFileLayer(uno::Reference< uno::XComponentContext > const & xContext, OUString const & aURL);
static bool argToUrl(OUString const & aArgument, OUString & rURL);
static bool getUrlFromContext(uno::Reference< uno::XComponentContext > const & xContext, OUString const & aName, OUString & rURL);

typedef std::vector< OUString > Filelist;
static bool getArgFiles( uno::Sequence<OUString> const & aArgs, sal_Int32 nArgC, sal_Int32 & ix, Filelist& files);
static bool getDirFiles( OUString const & aBaseUrl, Filelist& files);
static bool clearModuleData( uno::Reference< uno::XComponentContext > const & xContext, OUString const & aBaseUrl );
static bool removeModuleTree( uno::Reference< uno::XComponentContext > const & xContext, OUString const & aBaseUrl );

static void verbose_msg(const char * fmt, ...);
static void error_msg(const char * fmt, ...);
// --------------------------------------------------------------------------

// global variables for output control
static bool g_bVerbose = false;
// bool unoapp::g_bQuiet = false;

static void verbose_msg(const char * fmt, ...)
{
    FILE * const msg_file = stderr;
    if (g_bVerbose)
    {
		va_list args;
		va_start(args, fmt);
		vfprintf(msg_file, fmt, args);
		va_end(args);
        fflush(msg_file);
    }
}
// --------------------------------------------------------------------------

static void error_msg(const char * fmt, ...)
{
    FILE * const msg_file = stderr;
    if (!unoapp::g_bQuiet)
    {
		va_list args;
		va_start(args, fmt);
		vfprintf(msg_file, fmt, args);
		va_end(args);
        fflush(msg_file);
    }
}
// --------------------------------------------------------------------------

static void check_msg_args(uno::Sequence< OUString > const & aArgs, sal_Int32 & ix)
{
    while (ix < aArgs.getLength())
    {
        if (aArgs[ix].equalsAscii("-v") || aArgs[ix].equalsAscii("--verbose"))
            g_bVerbose = true;
        
        else if (aArgs[ix].equalsAscii("-q") || aArgs[ix].equalsAscii("--quiet"))
            unoapp::g_bQuiet = true;

        else
            return; // stop the loop

        // we have found one - advance index and perform sanity check
        ++ix;
        if (g_bVerbose && unoapp::g_bQuiet)
            fprintf(stderr,"\nWarning: both --quiet and --verbose mode specified. Output may be inconsistent.\n");
    }
        
}
// --------------------------------------------------------------------------

int SAL_CALL unoapp::uno_main(uno::Reference< uno::XComponentContext > const & xContext)
        SAL_THROW( (uno::Exception) )
{
    // get args
    uno::Sequence< OUString > const aArgs( getAppCommandArgs() );

    sal_Int32 const nArgC = aArgs.getLength();

    if (nArgC < 1)
    {
        usage();
        return 1;
    }

    sal_Int32 ix = 0;

    // get entity if applicable
    OUString sEntity;
    bool bUseEntity = false;
    bool bUseAdminEntity = false;
    // spooling data
    OUString aBaseUrl;
    bool bSpool = false;
    bool bCleanup = false;

    check_msg_args(aArgs,ix);

    if (aArgs[ix].equalsAscii("-a") || aArgs[ix].equalsAscii("--admin"))
    {
        ix += 1;

        check_msg_args(aArgs,ix);
        
        if (nArgC <= ix)
        {
            usage();
            return 2;
        }

        bUseEntity = true;
        bUseAdminEntity = true;
    }
	else if (aArgs[ix].equalsAscii("-e") || aArgs[ix].equalsAscii("--entity"))
    {
        ix += 2;

        if (nArgC <= ix)
        {
            usage();
            return 3;
        }

        sEntity = aArgs[ix-1];
        bUseEntity = true;

        check_msg_args(aArgs,ix);
    }
	else if (aArgs[ix].equalsAscii("-s") || aArgs[ix].equalsAscii("--spool"))
    {
        ix += 1;

        check_msg_args(aArgs,ix);
        
        uno::Reference< uno::XComponentContext > xBootstrapContext = getBootstrapContext(xContext);
        OSL_ASSERT( xBootstrapContext.is());

        if (nArgC > ix)
        {
            if (!argToUrl(aArgs[ix] , aBaseUrl))
            {
                error_msg(" ERROR: invalid argument '%s'\n", narrow(aArgs[ix]).getStr() );
                return 20;
            }
            ix += 1;
        }
        else
        {
            if (!getUrlFromContext(xBootstrapContext,OUString::createFromAscii(k_BootstrapSpoolDir),aBaseUrl) )
            {
                error_msg(" ERROR: Cannot determine spool directory\n");
                return 10;
            }
        }
        bSpool = true;
            
        if (nArgC > ix) // spurious extra args
        {
            usage();
            return 4;
        }

        if (!getUrlFromContext(xBootstrapContext,OUString::createFromAscii(k_BootstrapModuleDir),sEntity) )
        {
            error_msg(" ERROR: Cannot determine module directory\n");
            return 11;
        }
        
        bUseEntity = true;
    }
	else if (aArgs[ix].equalsAscii("-c") || aArgs[ix].equalsAscii("--cleanup"))
    {
        ix += 1;

        check_msg_args(aArgs,ix);
        
        bCleanup = true;
            
        if (nArgC > ix) // spurious extra args
        {
            usage();
            return 5;
        }

        uno::Reference< uno::XComponentContext > xBootstrapContext = getBootstrapContext(xContext);
        OSL_ASSERT( xBootstrapContext.is());

        if (!getUrlFromContext(xBootstrapContext,OUString::createFromAscii(k_BootstrapModuleDir),sEntity) )
        {
            error_msg(" ERROR: Cannot determine module directory\n");
            return 11;
        }
        
        bUseEntity = true;
    }

    if (bCleanup)
    {
        if (!removeModuleTree(xContext,sEntity))
        {
            error_msg("\nConfiguration cleanup in module directory %s failed!\n", narrow(sEntity).getStr() );
			return 16;
        }
        verbose_msg("\nConfiguration cleanup in module directory %s completed successfully!\n", narrow(sEntity).getStr() );

        // preliminary cleanup - there is nothing to import
        return 0;
    }

    uno::Reference< backend::XLayerImporter > xImporter = createImporter(xContext);
    OSL_ASSERT( xImporter.is());

	if (bUseAdminEntity)
	{
		uno::Reference< backend::XBackendEntities > xBackend(xImporter->getTargetBackend(), uno::UNO_QUERY);
		if (!xBackend.is())
		{

            error_msg(" ERROR: Cannot determine admin entity - no backend\n" );
            return 12;
		}
		sEntity = xBackend->getAdminEntity();
		if (sEntity.getLength() == 0)
		{
            error_msg(" ERROR: Cannot determine admin entity - backend does not support admin access\n" );
			return 13;
		}	
	}
	
    Filelist aFiles;
    if (bSpool)
    {
        if (!getDirFiles(aBaseUrl,aFiles))
        {
            error_msg(" ERROR: Cannot enumerate spool files\n" );
			return 14;
        }
        if (!clearModuleData(xContext,sEntity))
        {
            error_msg(" ERROR: Cannot clear module directory\n" );
			return 15;
        }
    	verbose_msg("\nStarting configuration import into module directory %s ..\n", narrow(sEntity).getStr() );
    }
    else
    {
        if (!getArgFiles(aArgs,nArgC,ix,aFiles))
        {
            OSL_ASSERT(ix<nArgC);
            error_msg(" ERROR: invalid argument '%s'\n", narrow(aArgs[ix]).getStr() );
            return 21;
        }
        if (bUseEntity)
            verbose_msg("\nStarting configuration import into entity %s ..\n", narrow(sEntity).getStr() );
        else
            verbose_msg("\nStarting configuration import into user data ..\n" );
    }

    typedef Filelist::const_iterator Iter;
    for (Iter it=aFiles.begin(), end=aFiles.end(); it != end; ++it)
    {
        uno::Reference< backend::XLayer > xLayer = createFileLayer(xContext,*it);
        if (xLayer.is())
        {
            verbose_msg(".. importing layer %s", narrow(*it).getStr() );

            if (!bUseEntity)
                xImporter->importLayer(xLayer);

            else
                xImporter->importLayerForEntity(xLayer,sEntity);

            verbose_msg(" - succeeded\n" );
        }
        else
        {
            error_msg(" ERROR: layer '%s' not found\n", narrow(*it).getStr() );
            return 30;
        }
    }
    verbose_msg("Configuration import completed successfully!\n" );

    return 0;
}
// --------------------------------------------------------------------------
static bool getArgFiles( uno::Sequence<OUString> const & aArgs, sal_Int32 nArgC, sal_Int32 & ix, Filelist& files)
{
    files.reserve(nArgC-ix);
    for ( ; ix < nArgC; ++ix)
    {
        OUString aURL;
        if ( !argToUrl( aArgs[ix], aURL ) )
            return false;
        files.push_back(aURL);
    }
    return true;
}
// --------------------------------------------------------------------------
#include <osl/file.hxx>
#include <osl/process.h>

static OUString getCWD()
{
    OUString aCWD;
    OSL_VERIFY(! osl_getProcessWorkingDir(&aCWD.pData) );
    return aCWD;
}

static bool argToUrl(OUString const & aArgument, OUString & rURL)
{
    using osl::File;

    OUString aOther;

    if (File::getFileURLFromSystemPath(aArgument,aOther) == File::E_None)
    {
        rURL = aOther;
    }
    else if (File::getSystemPathFromFileURL(aArgument,aOther) == File::E_None)
    {
        rURL = aArgument;
    }
    else
        return false;

    static const OUString aCWD = getCWD();

    if (File::getAbsoluteFileURL(aCWD,rURL,aOther) == File::E_None)
    {
        rURL = aOther;
    }
    else
    {
        error_msg(" WARNING: Cannot get absolute path for %s\n",narrow(rURL).getStr());
    }
    
    return true;
}

static bool getUrlFromContext(uno::Reference< uno::XComponentContext > const & xContext, OUString const & aName, OUString & rURL)
{
    OUString aValue;
    if (xContext->getValueByName(aName) >>= aValue)
        return argToUrl(aValue,rURL);
    else
        return false;
}
// --------------------------------------------------------------------------

#include <com/sun/star/lang/ServiceNotRegisteredException.hpp>
#include <rtl/ustrbuf.hxx>
namespace lang = com::sun::star::lang;
using uno::UNO_QUERY;

const char k_BootstrapContextSingleton[] = "/singletons/com.sun.star.configuration.bootstrap.theBootstrapContext";
// --------------------------------------------------------------------------
static uno::Reference< uno::XComponentContext >   getBootstrapContext(uno::Reference< uno::XComponentContext > const & xContext)
{
    uno::Reference< uno::XComponentContext > xResult = xContext;
    if (xContext.is())
    {
        xContext->getValueByName(OUString::createFromAscii(k_BootstrapContextSingleton)) >>= xResult;
    }
    return xResult;
}

// --------------------------------------------------------------------------
static uno::Reference< uno::XInterface > createService(uno::Reference< uno::XComponentContext > const & xContext, OUString aService)
{

    uno::Reference< lang::XMultiComponentFactory > xFactory = xContext->getServiceManager();
    if (!xFactory.is())
    {
        rtl::OUStringBuffer sMsg;
        sMsg.appendAscii("Missing object ! ");
        sMsg.appendAscii("UNO context has no service manager.");

        throw uno::RuntimeException(sMsg.makeStringAndClear(),NULL);
    }

    uno::Reference< uno::XInterface > xInstance = xFactory->createInstanceWithContext(aService,xContext);
    if (!xInstance.is())
    {
        rtl::OUStringBuffer sMsg;
        sMsg.appendAscii("Missing service ! ");
        sMsg.appendAscii("Service manager can't instantiate service ");
        sMsg.append(aService).appendAscii(". ");

        throw lang::ServiceNotRegisteredException(sMsg.makeStringAndClear(),NULL);
    }
  
    return xInstance;
}

// --------------------------------------------------------------------------
static uno::Reference< backend::XLayerImporter >  createImporter(uno::Reference< uno::XComponentContext > const & xContext)
{
    static const rtl::OUString kImporter(
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.configuration.backend.MergeImporter")) ;

    uno::Reference< backend::XLayerImporter > xRet( createService(xContext,kImporter), UNO_QUERY );
    if (!xRet.is())
    {
        rtl::OUStringBuffer sMsg;
        sMsg.appendAscii("Missing interface ! Service ");
        sMsg.append(kImporter);
        sMsg.appendAscii(" does not support the required interface ");
        sMsg.appendAscii("com::sun::star::configuration::backend::XLayerImporter").appendAscii(".");

        throw uno::RuntimeException(sMsg.makeStringAndClear(),NULL);
    }

    return xRet;
}
// --------------------------------------------------------------------------

#include <com/sun/star/io/XInputStream.hpp>
#include <com/sun/star/io/XActiveDataSink.hpp>
namespace io = com::sun::star::io;

static uno::Reference< io::XInputStream >  createFileStream(uno::Reference< uno::XComponentContext > const & xContext, OUString const & aURL);
// --------------------------------------------------------------------------

static uno::Reference< backend::XLayer > createFileLayer(uno::Reference< uno::XComponentContext > const & xContext, OUString const & aURL)
{    
    static const OUString kXMLLayerParser( 
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.configuration.backend.xml.LayerParser")) ;
    
    uno::Reference< backend::XLayer > xLayerReader;

    uno::Reference< io::XInputStream > xFile = createFileStream(xContext,aURL);
    if (xFile.is())
    {
        xLayerReader.set( createService(xContext,kXMLLayerParser), UNO_QUERY );

        uno::Reference< io::XActiveDataSink > xAS(xLayerReader, UNO_QUERY);
        if (!xAS.is())
        {
            rtl::OUStringBuffer sMsg;
            sMsg.appendAscii("Missing interface ! Service ");
            sMsg.append(kXMLLayerParser);
            sMsg.appendAscii(" does not support one of the needed interfaces ");
            sMsg.appendAscii("com::sun::star::configuration::backend::XLayer").appendAscii(" or ");
            sMsg.appendAscii("com::sun::star::io::XActiveDataSink").appendAscii(".");

            throw uno::RuntimeException(sMsg.makeStringAndClear(),NULL);
        }

        xAS->setInputStream(xFile);
    }

    return xLayerReader;
}
// --------------------------------------------------------------------------

#include "oslstream.hxx"
#include "filehelper.hxx"
// --------------------------------------------------------------------------

static bool getDirFiles( OUString const & aBaseUrl, Filelist& files)
{
    using namespace ::osl;
    Directory aDirectory(aBaseUrl);
    Directory::RC result = aDirectory.open();
          
    DirectoryItem aItem;
    while (result == Directory::E_None)
    {
        result = aDirectory.getNextItem(aItem);
        if (result == Directory::E_None)
        {
            const sal_uInt32 k_Mask = FileStatusMask_Type | FileStatusMask_FileURL;
            FileStatus stat(k_Mask);

            result = aItem.getFileStatus(stat);

            if (result == Directory::E_None && stat.isValid(k_Mask))
            {
                if (stat.getFileType() == FileStatus::Regular)
                    files.push_back(stat.getFileURL());
            }
            else 
                error_msg( " ERROR: Getting URL of directory item failed\n");
        }
    }
    aDirectory.close();

    if (result != Directory::E_NOENT)
    {
        OUString const sErr = configmgr::FileHelper::createOSLErrorString(result);
        error_msg(" ERROR reading directory '%s': %s\n", narrow(aBaseUrl).getStr(), narrow(sErr).getStr() );
        return false;
    }

    return true;
}
// --------------------------------------------------------------------------

#include <com/sun/star/task/XJob.hpp>
// --------------------------------------------------------------------------

static bool getLayerFiles( uno::Reference< uno::XComponentContext > const & xContext, OUString const & aBaseUrl, uno::Sequence< OUString > & files)
{
    static const char k_BrowserSvc[] = "com.sun.star.configuration.backend.LocalHierarchyBrowser";
    namespace task = com::sun::star::task;
    namespace beans = com::sun::star::beans;

    uno::Reference< task::XJob > xBrowser( createService(xContext, OUString::createFromAscii(k_BrowserSvc)), uno::UNO_QUERY );
    if (!xBrowser.is()) return false;
    
    uno::Sequence< beans::NamedValue > aArgs(2);
    aArgs[0].Name = OUString( RTL_CONSTASCII_USTRINGPARAM("LayerDataUrl") );
    aArgs[0].Value <<= aBaseUrl;
    aArgs[1].Name = OUString( RTL_CONSTASCII_USTRINGPARAM("FetchComponentNames") );
    aArgs[1].Value <<= false;

    uno::Any aResult = xBrowser->execute(aArgs);

    return aResult >>= files;

}

static bool clearModuleData( uno::Reference< uno::XComponentContext > const & xContext, OUString const & aBaseUrl )
{
    uno::Sequence< OUString > aFiles;
    if (!getLayerFiles(xContext, aBaseUrl,aFiles))
        return false;

    bool result = true;
    for (sal_Int32 i = 0; i < aFiles.getLength(); ++i)
    {
        using namespace osl;
        File::RC rc = File::remove(aFiles[i]);

        if (rc != File::E_None)
        {
            OUString const sErr = configmgr::FileHelper::createOSLErrorString(rc);
            error_msg(" ERROR removing file '%s': %s\n", narrow(aFiles[i]).getStr(), narrow(sErr).getStr() );
            result = false;
        }
    }
    return result;
}

inline static bool isOK( osl::Directory::RC errcode) 
{ return errcode==osl::Directory::E_None; }

inline static bool isDone( osl::Directory::RC errcode) 
{ return errcode==osl::Directory::E_None || errcode == osl::Directory::E_NOENT; }

static bool removeEmptyDirectoryTree( OUString const & aDirectoryUrl )
{
    using namespace osl;

    Directory aDirectory(aDirectoryUrl);	
	    
    Directory::RC rc = aDirectory.open();
    if ( isOK(rc) )
    {
        DirectoryItem aItem;
        while( isOK( rc = aDirectory.getNextItem(aItem) ) )
        {
            sal_uInt32 const n_STATUS_FIELDS =  FileStatusMask_Type | FileStatusMask_FileURL;

            FileStatus aItemDescriptor( n_STATUS_FIELDS );
            rc = aItem.getFileStatus(aItemDescriptor);

            if ( !isOK(rc) )
            {
                OUString const sErr = configmgr::FileHelper::createOSLErrorString(rc);
                error_msg(" ERROR getting status of an item in '%s': %s\n", narrow(aDirectoryUrl).getStr(), narrow(sErr).getStr() );
                continue;
            }
                    
            OSL_ENSURE( aItemDescriptor.isValid(FileStatusMask_FileURL), "Could not get URL of subdirectory");
            OUString const aSubdirectoryUrl = aItemDescriptor.getFileURL();

            OSL_ENSURE( aItemDescriptor.isValid(FileStatusMask_Type), "Could not get type of directory item");
            if (aItemDescriptor.getFileType() != FileStatus::Directory)
            {
                error_msg(" ERROR removing directory tree: item '%s' is not a directory\n", narrow(aSubdirectoryUrl).getStr() );
                continue;
            }

            // recurse
            removeEmptyDirectoryTree(aSubdirectoryUrl);
        }
        aDirectory.close();
    }
    if ( !isDone(rc) )
    {
        OUString const sErr = configmgr::FileHelper::createOSLErrorString(rc);
        error_msg(" ERROR scanning directory '%s': %s\n", narrow(aDirectoryUrl).getStr(), narrow(sErr).getStr() );
    }

    rc = Directory::remove(aDirectoryUrl);
    if ( !isDone(rc) )
    {
        OUString const sErr = configmgr::FileHelper::createOSLErrorString(rc);
        error_msg(" ERROR removing directory '%s': %s\n", narrow(aDirectoryUrl).getStr(), narrow(sErr).getStr() );
    }
        
    return isDone(rc);
}

static bool removeModuleTree( uno::Reference< uno::XComponentContext > const & xContext, OUString const & aBaseUrl )
{
    bool result = clearModuleData(xContext,aBaseUrl);

    if (!removeEmptyDirectoryTree(aBaseUrl))
        result = false;

    return result;
}

namespace io = com::sun::star::io;
// --------------------------------------------------------------------------
static uno::Reference< io::XInputStream >  createFileStream(uno::Reference< uno::XComponentContext > const & /*xContext*/, OUString const & aURL)
{
    using osl::File;

    File * blobFile = new File(aURL);
    
    File::RC errorCode = blobFile->open(OpenFlag_Read);

    uno::Reference<io::XInputStream> xResult;
    switch (errorCode)
    { 
    case osl::File::E_None: // got it
        xResult.set( new configmgr::OSLInputStreamWrapper(blobFile,true) );
        break;

    case osl::File::E_NOENT: // no file => no stream
        delete blobFile;
        break;

    default:
        delete blobFile;
        {
            rtl::OUStringBuffer sMsg;
            sMsg.appendAscii("Cannot open output file \"");
            sMsg.append(aURL);
            sMsg.appendAscii("\" : ");
            sMsg.append(configmgr::FileHelper::createOSLErrorString(errorCode));

            throw io::IOException(sMsg.makeStringAndClear(),NULL);
        }   
    }
    return xResult;
}
// --------------------------------------------------------------------------

