/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: layer.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: kz $ $Date: 2006/12/13 15:16:51 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_slideshow.hxx"

// must be first
#include <canvas/debug.hxx>
#include <layer.hxx>

#include <basegfx/range/b2drange.hxx>
#include <basegfx/range/b1drange.hxx>
#include <basegfx/range/b2dmultirange.hxx>

#include <boost/bind.hpp>
#include <boost/utility.hpp>
#include <boost/mem_fn.hpp>
 

using namespace ::com::sun::star;

namespace slideshow
{
    namespace internal
    {
        Layer::Layer() :
            maShapeSet(),
            maViewLayers()
        { 
        }

        void Layer::addViewLayer( const ViewLayerSharedPtr& rNewLayer )
        {
            const ViewLayerVector::iterator aEnd( maViewLayers.end() );

            // already added?
            if( ::std::find( maViewLayers.begin(), 
                             aEnd, 
                             rNewLayer ) != aEnd )
            {
                return; // yes, nothing to do
            }

            // add to local list
            maViewLayers.push_back( rNewLayer );

            // TODO(Q2): Come up with something more intelligently
            // here. Basically, we're just lazily catching render
            // errors at a convenient high level, instead of
            // converting them to return values down in ViewShape (or
            // even in cppcanvas::Renderer). Then,
            // Shape::addViewLayer() simply has no return value, and
            // thus by definition never fails.
            try
            {
                // forward to all shape's addViewLayer method (which we request to
                // renders the Shape on the new ViewLayer. Since we add the shapes 
                // in the maShapeSet order (which is also the render order), this 
                // is equivalent to a subsequent render() call)
                ::std::for_each( maShapeSet.begin(),
                                 maShapeSet.end(),
                                 ::boost::bind( &Shape::addViewLayer,
                                                _1,
                                                ::boost::cref(rNewLayer),
                                                true ) );
            }
            catch( uno::Exception& )
            {
                // at least one shape could not be rendered - catch
                // error, view layer has nevertheless been added
                // successfully.
            }
        }

        bool Layer::removeViewLayer( const ViewLayerSharedPtr& rLayer )
        {
            const ViewLayerVector::iterator aEnd( maViewLayers.end() );

            OSL_ENSURE( ::std::count( maViewLayers.begin(),
                                      aEnd,
                                      rLayer ) < 2,
                        "Layer::removeViewLayer(): Duplicate ViewLayer entries!" );

            ViewLayerVector::iterator aIter;

            // remove from local list
            if( (aIter=::std::remove( maViewLayers.begin(),
                                      aEnd,
                                      rLayer )) == aEnd )
            {
                // ViewLayer was not added in the first place
                return false;
            }

            // actually perform the removal
            maViewLayers.erase( aIter, aEnd );

            // forward to all shape's removeViewLayer method
            if( ::std::count_if( maShapeSet.begin(),
                                 maShapeSet.end(),
                                 ::boost::bind<bool>( ::boost::mem_fn( &Shape::removeViewLayer ),
                                                      _1,
                                                      ::boost::cref(rLayer) ) ) 
                != static_cast<ShapeSet::difference_type>(maShapeSet.size()) )
            {
                // at least one of the Shape::removeViewLayer() calls did return 
                // false - removal failed on at least one Shape
                return false;
            }

            return true;
        }

        void Layer::viewLayerChanged( const ViewLayerSharedPtr& rChangedLayer )
        {
            // do we know the ViewLayer?
            OSL_ENSURE( ::std::find( maViewLayers.begin(), 
                                     maViewLayers.end(), 
                                     rChangedLayer ) != maViewLayers.end(),
                        "Layer::viewLayerChanged(): notified ViewLayer not found" );

            std::for_each( maShapeSet.begin(),
                           maShapeSet.end(),
                           ::boost::bind( &Shape::viewLayerChanged,
                                          _1,
                                          ::boost::cref(rChangedLayer)));
        }

        void Layer::addShape( const ShapeSharedPtr& rShape )
        {
            maShapeSet.insert( rShape );

            // add all view layers to the shape (and don't render.
            // Would most probably be out of order, anyway, and
            // overwrite underlying page content)
            ::std::for_each( maViewLayers.begin(),
                             maViewLayers.end(),
                             ::boost::bind( &Shape::addViewLayer,
                                            ::boost::ref(rShape),
                                            _1,
                                            false ) );
        }
            
        bool Layer::removeShape( const ShapeSharedPtr& rShape )
        {
            // removal was successful if we removed one or more elements
            const bool bWasRemoved( maShapeSet.erase( rShape ) > 0 );

            // remove all our view layers from the shape
            ::std::for_each( maViewLayers.begin(),
                             maViewLayers.end(),
                             ::boost::bind( &Shape::removeViewLayer,
                                            ::boost::ref(rShape),
                                            _1 ) );

            return bWasRemoved;
        }

        void Layer::enterAnimationMode( const AnimatableShapeSharedPtr& rShape )
        {
            rShape->enterAnimationMode();
        }

        void Layer::leaveAnimationMode( const AnimatableShapeSharedPtr& rShape )
        {
            rShape->leaveAnimationMode();
        }

        bool Layer::update( const ::basegfx::B2DMultiRange& rArea ) const
        {
            bool bRet( true );

            const ShapeSet::const_iterator 	aEnd( maShapeSet.end() );
            ShapeSet::const_iterator 		aCurr( maShapeSet.begin() );
            while( aCurr != aEnd )
            {
                try
                {
                    if( !(*aCurr)->isBackgroundDetached() )
                    {
                        // animated shapes -> sprites. Don't have to paint them 
                        // explicitely, the canvas handles compositing on screen

                        // now, check against intersection with update area
                        if( rArea.overlaps( (*aCurr)->getUpdateArea() ) )
                        {
                            if( !(*aCurr)->render() )
                            {
                                // render ultimately failed if one render failed.
                                // but don't bail out immediately, page might still
                                // come out sufficiently correct.
                                bRet = false;
                            }
                        }
                    }
                }
                catch( uno::Exception& )
                {
                    // shape could not be rendered - catch and
                    // delay error, we want all other shapes
                    // still to be rendered.
                    bRet = false;
                }

                ++aCurr;
            }

            return bRet;
        }

        bool Layer::render() const
        {
            bool bRet( true );

            // forward to each Shape's render() method
            const ShapeSet::const_iterator 	aEnd( maShapeSet.end() );
            ShapeSet::const_iterator 		aCurr( maShapeSet.begin() );
            while( aCurr != aEnd )
            {
                try
                {
                    if( !(*aCurr)->isBackgroundDetached() )
                    {
                        // animated shapes -> sprites. Don't have to paint them 
                        // explicitely, the canvas handles compositing on screen
                        if( !(*aCurr)->render() )
                        {
                            // render ultimately failed if one render failed.
                            // but don't bail out immediately, page might still
                            // come out sufficiently correct.
                            bRet = false;
                        }
                    }
                }
                catch( uno::Exception& )
                {
                    // shape could not be rendered - catch and
                    // delay error, we want all other shapes
                    // still to be rendered.
                    bRet = false;
                }

                ++aCurr;
            }

            return bRet;
        }

    }
}
