/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: TreeBuilder.java,v $
 *
 *  $Revision: 1.3.20.1 $
 *
 *  last change: $Author: rt $ $Date: 2007/01/16 08:18:52 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

package com.sun.xmlsearch.tree;

import java.io.PrintStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import java.net.URL;
import javax.swing.tree.TreeNode;
import org.xml.sax.HandlerBase;
import org.xml.sax.AttributeList;
import org.xml.sax.InputSource;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;
import com.sun.xmlsearch.util.IntegerArray;
import com.jclark.xsl.om.*;

public final class TreeBuilder extends HandlerBase implements NodeFactory, Names {
  
    private StringBuffer _textBuffer;
    private IntegerArray _offsets;
    private int _textCounter;
    private int _attributeCounter;

    private static final SafeNodeIterator EmptyIterator =
		new SafeNodeIterator() {
				public final Node next() { return null; }
			};
  
    private static final Enumeration EmptyEnumeration =
		new Enumeration() {
				public boolean hasMoreElements() { return false; }
				public Object nextElement() { return null; }
			};

    private abstract class NodeImpl implements Node2 {
		private Node2 _parent;
  
		public final Node getParent() {
			return _parent;
		}
  
		public int getIndex() {
			return 0;
		}
  
		public final Node getRoot() {
			Node node = this;
			for (Node parent = node.getParent();
				 parent != null;
				 node = parent, parent = parent.getParent())
				;
			return node;
		}
  
		public final void setParent(Node parent) {
			_parent = (Node2)parent;
		}
  
		/** if node is a child of this return its 0-based
			position among other children
		*/
		public int getChildIndex(Node node) {
			return -1;			// default
		}
  
		public void setChild(int n, Node node) {
			// by default don't do anything
		}

		// default implementations covering Text, Attribute
		public Node getAttribute(Name name) {
			return null;
		}
  
		public String getAttributeValue(Name name) {
			return null;
		}
  
		public Name getName() {
			return null;
		}
  
		public String getData() {
			return null;
		}
  
		public SafeNodeIterator getAttributes() {
			return EmptyIterator;
		}
  
		public SafeNodeIterator getNamespaces() {
			return EmptyIterator;
		}
  
		public SafeNodeIterator getChildren() {
			return EmptyIterator;
		}
  
		public SafeNodeIterator getFollowingSiblings() {
			return _parent != null
				? ((ElementImpl)_parent).getFollowingSiblings(this)
				: EmptyIterator;
		}
  
		public boolean isId(String name) {
			return false;
		}
  
		protected String getText(int index) {
			return ((NodeImpl)_parent).getText(index);
		}

		public URL getURL() { return null; }
		public int getLineNumber() { return -1; }
		public int getColumnNumber() { return -1; }
		public NamespacePrefixMap getNamespacePrefixMap() {
			return _parent.getNamespacePrefixMap();
		}
		public int compareTo(Node node) { return -1; }
		public Node getElementWithId(String id) { return null; }
		public String getGeneratedId() { return "getGeneratedId not implemented"; }
		public String getUnparsedEntityURI(String name) {
			return "getUnparsedEntityURI";
		}
		public String getPublicId() {
			return "getPublicId";
		}
		public String getSystemId() {
			return "getSystemId";
		}
  
		public boolean isLeaf() { return true; }
		public int getChildCount() { return 0; }
		public TreeNode getChildAt(int n) { return null; }
		public int getIndex(TreeNode node) { return -1; }
		public Enumeration children() { return EmptyEnumeration; }
		public void getSubelementsByTagName(Name name, Vector result) {}
		public boolean getAllowsChildren() { return false; }
		public abstract int getNodeType();
		public abstract String toString();
    } // end of NodeImpl

    private abstract class NamedNodeImpl extends NodeImpl {
		private final Name _name;
  
		protected NamedNodeImpl(Name name) {
			_name = name;
		}
    
		public final Name getName() {
			return _name;
		}
    } // end of NamedNodeImpl

    private final class AttributeImpl extends NamedNodeImpl {
		private final String _data;
  
		public AttributeImpl(Name name, String data) {
			super(name);
			_data = data;
		}

		public byte getType() {
			return Node.ATTRIBUTE;
		}
  
		public int getNodeType() {
			return XmlTreeNode.ATTRIBUTE;
		}
  
		public String getData() {
			return _data;
		}
  
		public String toString() {
			return "Attribute";
		}
    } // end of AttributeImpl

    private abstract class ElementImpl extends NamedNodeImpl {
		private NamespacePrefixMap _npm; // !!! should be final
		private final int[] _attributes;
  
		protected ElementImpl(Name name, int[] attributes) {
			super(name);
			_attributes = attributes;
		}
  
		public final void setNamespacePrefixMap(NamespacePrefixMap map) {
			_npm = map;
		}
  
		public final NamespacePrefixMap getNamespacePrefixMap() {
			return _npm;
		}
  
		public final byte getType() {
			return ELEMENT;
		}
  
		public final String getData() {
			return null;
		}
  
		public final Node getAttribute(Name name) {
			return _attributes != null
				? ((RootNodeImpl)this.getRoot()).getAttribute(name, _attributes)
				: null;
		}

		public final String getAttributeValue(Name name) {
			return _attributes != null
				? ((RootNodeImpl)this.getRoot()).getAttributeValue(name, _attributes)
				: null;
		}
  
		public final SafeNodeIterator getAttributes() {
			return _attributes != null
				? ((RootNodeImpl)this.getRoot()).getAttributes(_attributes)
				: EmptyIterator;
		}

		// default implementation
		public SafeNodeIterator getFollowingSiblings(NodeImpl node) {
			return EmptyIterator;
		}

		public final int getAttributesLength() {
			return _attributes != null ? _attributes.length : 0;
		}
  
		public final boolean getAllowsChildren() {
			return true;
		}
  
		public final int getNodeType() {
			return XmlTreeNode.ELEMENT;
		}
    
		public final String toString() {
			StringBuffer buffer = new StringBuffer(128);
			buffer.append(getName().getLocalPart());
			if (_attributes != null)
				for (int i = 0; i < _attributes.length; i++) {
					buffer.append(' ');
					buffer.append(_attributes[i]); // name 
					buffer.append("=\"");
					buffer.append(_attributes[i]); // value
					buffer.append('"');
				}
			return buffer.toString();
		}
  
		public final Vector getElementsByTagName(Name name) {
			Vector result = new Vector();
			getElementsByTagName(name, result);
			return result;
		}
  
		protected final void getElementsByTagName(Name name, Vector result) {
			if (getName() == name)
				result.addElement(this);
			// recurse to children
			getSubelementsByTagName(name, result);
		}
  
		public abstract void getSubelementsByTagName(Name name, Vector result);
    } // end of ElementImpl

    private final class TemporaryTextNodeImpl extends NodeImpl {
		private final String _data;

		public TemporaryTextNodeImpl(String text) {
			_data = text;
		}

		public final byte getType() {
			return TEXT;
		}
  
		public final int getNodeType() {
			return XmlTreeNode.TEXT_NODE;
		}
  
		public final String getData() {
			return _data;
		}
  
		public final String toString() {
			return _data;
		}
    } // end of TemporaryTextNodeImpl
  
    private final class TextNodeImpl extends NodeImpl {
		private final int _index;

		public TextNodeImpl(String text) {
			_index = storeText(text);
		}

		public final byte getType() {
			return TEXT;
		}
  
		public final int getNodeType() {
			return XmlTreeNode.TEXT_NODE;
		}
  
		public final String getData() {
			return getText(_index);
		}
  
		public final String toString() {
			return getData();
		}
    } // end of TextNodeImpl
  
    private final class RootNodeImpl extends NodeImpl {
		private ElementImpl _child;
		private char[] _texts;
		private int[] _offsets;
		private Name[] _attributeNames;
		private int[] _attributeValueIndexes;
    
		public RootNodeImpl(ElementImpl child,
							char[] texts,
							int[] offsets,
							Name[] attributeNames,
							int[] attributeValueIndexes) {
			(_child = child).setParent(this);
			_texts = texts;
			_offsets = offsets;
			_attributeNames = attributeNames;
			_attributeValueIndexes = attributeValueIndexes;
		}

		public final String getAttributeValue(Name name, int[] attrIndexes) {
			for (int i = 0; i < attrIndexes.length; i++)
				if (_attributeNames[attrIndexes[i]] == name)
					return getText(_attributeValueIndexes[attrIndexes[i]]);
			return null;
		}

		public final Node getAttribute(Name name, int[] attrIndexes) {
			for (int i = 0; i < attrIndexes.length; i++)
				if (_attributeNames[attrIndexes[i]] == name)
					return new AttributeImpl(name, getText(_attributeValueIndexes[attrIndexes[i]]));
			return null;
		}

		private final class AttributeIterator implements SafeNodeIterator {
			private int _current = 0;
			private final int[] _attrIndexes;
			private final int _length;

			public AttributeIterator(int[] attrIndexes) {
				_attrIndexes = attrIndexes;
				_length = attrIndexes.length;
			}

			public Node next() {
				return _current < _length
					? new AttributeImpl(_attributeNames[_attrIndexes[_current]],
										getText(_attributeValueIndexes[_attrIndexes[_current++]]))
						: null;
			}
		} // end of AttributeIterator

		public final SafeNodeIterator getAttributes(int[] attrIndexes) {
			return new AttributeIterator(attrIndexes);
		}

		public final byte getType() {
			return ROOT;
		}

		public int getNodeType() {
			return XmlTreeNode.ROOT;
		}
    
		public final NamespacePrefixMap getNamespacePrefixMap() {
			return _child.getNamespacePrefixMap(); // ugly!!!; should be Container
		}
    
		public SafeNodeIterator getChildren() {
			return new SafeNodeIterator() {
					private boolean _used = false;
					public Node next() {
						if (_used)
							return null;
						else {
							_used = true;
							return _child;
						}
					}
				};
		}
    
		public final String toString() {
			return "root";
		}
    
		public String getText(int index) {
			final int offset = _offsets[index];
			return new String(_texts, offset, _offsets[index + 1] - offset);
		}
    } // end of RootNodeImpl

    // empty element
    private final class XmlTreeElement0 extends ElementImpl {
		public XmlTreeElement0(Name name, int[] attributes) {
			super(name, attributes);
		}

		public final void getSubelementsByTagName(Name name, Vector result) {
			// kindly do nothing
		}
    } // end of XmlTreeElement0

    // element with a single child
    private final class XmlTreeElement1 extends ElementImpl {
		private NodeImpl _child;
  
		public XmlTreeElement1(Name name, int[] attributes, NodeImpl child) {
			super(name, attributes);
			_child = child;
		}

		private final NodeImpl child() {
			return _child;
		}
  
		public SafeNodeIterator getChildren() {
			return new SafeNodeIterator() {
					private boolean _used = false;
					public Node next() {
						if (_used)
							return null;
						else {
							_used = true;
							return child();
						}
					}
				};
		}

		public int getChildIndex(Node node) {
			return _child == node ? 0 : -1;
		}

		public void setChild(int n, Node node) {
			if (n == 0)
				_child = (NodeImpl)node;
		}

		public final boolean isLeaf() {
			return false;
		}
  
		public final int getChildCount() {
			return 1;
		}
  
		public final TreeNode getChildAt(int n) {
			return n == 0 ? XmlTreeNodeImpl.makeNode(child()) : null;
		}
  
		public final Enumeration children() {
			return new Enumeration() {
					private boolean _used;
					public boolean hasMoreElements() {
						return _used == false;
					}
					public Object nextElement() {
						_used = true;
						return XmlTreeNodeImpl.makeNode(child());
					}
				};
		}
  
		public final int getIndex(TreeNode node) {
			return ((XmlTreeNodeImpl)node).getRepresentation() == _child ? 0 : -1;
		}
  
		public final void getSubelementsByTagName(Name name, Vector result) {
			if (_child instanceof ElementImpl)
				((ElementImpl)_child).getElementsByTagName(name, result);
		}
    } // end of class XmlTreeElement1

    // element with multiple children
    private class XmlTreeElement2 extends ElementImpl {
		private Node2[] _children;

		public XmlTreeElement2(Name name, int[] attributes, Node2[] children) {
			super(name, attributes);
			_children = children;
		}

		private final NodeImpl child(int i) {
			return (NodeImpl)_children[i];
		}
  
		public int getChildIndex(Node node) {
			for (int i = 0; i < _children.length; i++)
				if (_children[i] == node)
					return i;
			return -1;
		}

		public void setChild(int n, Node node) {
			_children[n] = (Node2)node;
		}
    
		private final class NodeIterator implements SafeNodeIterator {
			private int _index;
			public NodeIterator(int startingIndex) {
				_index = startingIndex;
			}
			public Node next() {
				return _index < _children.length ? child(_index++) : null;
			}
		}

		public final SafeNodeIterator getChildren() {
			return new NodeIterator(0);
		}
  
		public final SafeNodeIterator getFollowingSiblings(final NodeImpl node) {
			int i = 0;
			while (_children[i++] != node && i < _children.length)
				;
			return i < _children.length ? new NodeIterator(i) : EmptyIterator;
		}

		public final boolean isLeaf() {
			return false;
		}
  
		public final int getChildCount() {
			return _children.length;
		}

		public final TreeNode getChildAt(int n) {
			return XmlTreeNodeImpl.makeNode(child(n));
		}

		public final Enumeration children() {
			return new Enumeration() {
					private int _index;
					public boolean hasMoreElements() {
						return _index < _children.length;
					}
					public Object nextElement() {
						return XmlTreeNodeImpl.makeNode(child(_index++));
					}
				};
		}

		public final int getIndex(TreeNode node) {
			return getChildIndex(((XmlTreeNodeImpl)node).getRepresentation());
		}
  
		// name is interned in the caller
		public final void getSubelementsByTagName(Name name, Vector result) {
			for (int i = 0; i < _children.length; i++)
				if (_children[i] instanceof ElementImpl)
					((ElementImpl)_children[i]).getElementsByTagName(name, result);
		}
    } // end of class XmlTreeElement2

    private final class SubstitutedElement extends XmlTreeElement2 {
		private final Node _originalNode;
    
		public SubstitutedElement(Name name, int[] attributes, Node2[] children,
								  Node originalNode) {
			super(name, attributes, children);
			_originalNode = originalNode;
		}
    
		public void revertToOriginal() {
			NodeImpl parent = (NodeImpl)getParent();
			int index = parent.getChildIndex(this);
			parent.setChild(index, _originalNode);
		}
    } // end of SubstitutedElement

    private SAXParser getParser() throws org.xml.sax.SAXException, javax.xml.parsers.ParserConfigurationException
    {
        SAXParserFactory spf = SAXParserFactory.newInstance();
        spf.setValidating(false);
        return spf.newSAXParser();
    }

    private static final int InitStackSize = 256;
    private static final Name EmptyName = null;
    private SAXParser _parser = null;
    private Hashtable _names = new Hashtable(256);
  
    private NameTable _nameTable;
    private NamespacePrefixMap _nsMap;
  
    public TreeBuilder() throws org.xml.sax.SAXException, javax.xml.parsers.ParserConfigurationException {
		this(new NameTableImpl());
    }

    public TreeBuilder(NameTable nameTable) throws org.xml.sax.SAXException, javax.xml.parsers.ParserConfigurationException {
		_parser = getParser();
		_nameTable = nameTable;
		_nsMap = _nameTable.getEmptyNamespacePrefixMap();
    }

    public Node makeSubstituteElement(Name name, Node2[] children, Node original) {
		return new SubstitutedElement(name, null, children, original);
    }

    public Node makeEmptyElement(Name name) {
		return new XmlTreeElement0(name, null);
    }

    public Node makeTextNode(String text) {
		return new TemporaryTextNodeImpl(text);
    }

    public void revertToOriginal(Node substitutedNode) {
		((SubstitutedElement)substitutedNode).revertToOriginal();
    }

    /*
      public static void main(String[] args) {
      TreeBuilder treeBuilder = new TreeBuilder();
    
      try {
      int times = 10;
      Node root = null;
      while (times-- > 0) {
      long start = System.currentTimeMillis();
      root = treeBuilder.getRoot(args[0]);
      System.out.println((System.currentTimeMillis()-start)
      +" msec parse");
      }
      //      TreeStatictics.report((XmlTreeNode)root);
      }
      catch (Exception e) {
      e.printStackTrace();
      }
      }
    */

    // if stack non-empty -- indexes top stack element
    private int _stackPointer;
    private NodeImpl[] _stack = new NodeImpl[InitStackSize];
    private Object[] _openNames = new Object[InitStackSize];
    private int[][] _attributes = new int[InitStackSize][];
    // used to concatenate consecutive 'characters' calls
    private String _currentText;
    /*
      this is a separate stack of NamespacePrefixMap-s
      Clark keeps a pointer to a NamespacePrefixMap in each Element node
      to keep track of namespace modifications
      typically children nodes inherit the parent's map, unless xmlns attribute
      creates a new namespace

      in my case (JRA)
      1) I don't want to keep these pointers in the DOM
      2) I wait w/ actual Node creation till its children are known
    */
    private NamespacePrefixMap[] _nsMapStack = new NamespacePrefixMap[InitStackSize];
    private int _nsMapStackPointer;
  
    private Hashtable    _attrValTable;
    private Vector       _attributeNames;
    private IntegerArray _attributeValues;
    private Hashtable    _tables;

    public void startDocument() {
		_stackPointer = -1;
		_nsMapStack[_nsMapStackPointer = 0] = _nsMap;
		_currentText = null;
		_textBuffer = new StringBuffer(4096);
		_offsets = new IntegerArray(512);
		_textCounter = 0;
		_attributeCounter = 0;
		_tables = new Hashtable();
		_attrValTable = new Hashtable(512);
		_attributeNames = new Vector(512);
		_attributeValues = new IntegerArray(512);
		_tables = new Hashtable();
    }

    public void endDocument() {
		final int textLength = _textBuffer.length();
		final char[] texts = new char[textLength];
		storeText("");
		_textBuffer.getChars(0, textLength, texts, 0);
		Name[] attributeNames = new Name[_attributeNames.size()];
		_attributeNames.toArray(attributeNames);
		_stack[0] = new RootNodeImpl((ElementImpl)_stack[0],
									 texts, _offsets.toIntArray(),
									 attributeNames, _attributeValues.toIntArray());
		_textCounter = 0;
		_attributeCounter = 0;
		_offsets = null;
		_textBuffer = null;
		_tables = null;
		_attributeValues = null;
		_attributeNames = null;
		_tables = null;
		_attrValTable = null;
    }

    private int storeText(String text) {
		_offsets.add(_textBuffer.length());
		_textBuffer.append(text);
		return _textCounter++;
    }

    private void push(NodeImpl node, Object name, int[] attributes) {
		if (++_stackPointer == _stack.length) {
			final int currentSize = _stack.length;
			final int newSize = currentSize << 1;
			NodeImpl[] newStack = new NodeImpl[newSize];
			System.arraycopy(_stack, 0, newStack, 0, currentSize);
			_stack = newStack;
			Object[] newNames = new Object[newSize];
			System.arraycopy(_openNames, 0, newNames, 0, currentSize);
			_openNames = newNames;
			int[][] newAttributes = new int[newSize][];
			System.arraycopy(_attributes, 0, newAttributes, 0, currentSize);
			_attributes = newAttributes;
		}
		_stack[_stackPointer] = node;
		_openNames[_stackPointer] = name;
		_attributes[_stackPointer] = attributes;
    }

    private void maybeCreateTextNode() {
		if (_currentText != null) {	// sth has accumulated here
			push(new TextNodeImpl(_currentText), null, null);
			_currentText = null;	// taken care of
		}
    }

    public final Name getElementName(String name) throws XSLException {
		// here names are created before NodeImpls so we don't have node
		// to give to expand; JJC uses it only in creating an Exception message
		return _nsMap.expandElementTypeName(name, null);
    }
			  
    public final Name getAttributeName(String name) throws XSLException {
		return _nsMap.expandAttributeName(name, null);
    }

    private int internAttribute(Name name, String value) {
		Hashtable table = (Hashtable)_tables.get(name);
		if (table == null) {
			_tables.put(name, table = new Hashtable());
			return allocateAttribute(name, value, table);
		}
		else {
			Integer indexOfAttribute = (Integer)table.get(value);
			return indexOfAttribute != null
				? indexOfAttribute.intValue()
				: allocateAttribute(name, value, table);
		}
    }

    private int allocateAttribute(Name name, String value, Hashtable table) {
		// intern value
		int indexOfValue;		// index of value string
		// _attrValTable stores indexes of attribute values
		Integer integer = (Integer)_attrValTable.get(value);
		if (integer != null)
			indexOfValue = integer.intValue();
		else
			_attrValTable.put(value, new Integer(indexOfValue = storeText(value)));
		// now allocate the attribute
		final int newAttributeIndex = _attributeCounter++;
		table.put(value, new Integer(newAttributeIndex));
		_attributeNames.addElement(name);
		_attributeValues.add(indexOfValue);
		return newAttributeIndex;
    }
			  
    public void startElement(String name, AttributeList attList) {
		maybeCreateTextNode();
		try {
			int[] attributeArray = null;
			// access parent's prefix map
			NamespacePrefixMap nsMap = _nsMapStack[_nsMapStackPointer];
			// code from James Clark's XSLT
			int nAtts = attList.getLength();
			if (nAtts > 0) {
				int nNsAtts = 0;
				for (int i = 0; i < nAtts; i++) {
					String tem = attList.getName(i);
					if (tem.startsWith("xmlns")) {
						nNsAtts++;
						if (tem.length() == 5) {
							String ns = attList.getValue(i);
							if (ns.length() == 0)
								nsMap = nsMap.unbindDefault();
							else
								nsMap = nsMap.bindDefault(ns);
						}
						else if (tem.charAt(5) == ':')
							nsMap = nsMap.bind(tem.substring(6),
											   attList.getValue(i));
					}
				}
				int n = nAtts - nNsAtts;
				if (n > 0) {
					int[] vec = new int[n];
					int j = 0;
					for (int i = 0; i < nAtts; i++) {
						final String tem = attList.getName(i);
						if (!tem.startsWith("xmlns")) {
							// FIXME resolve relative URL
							vec[j++] = internAttribute(nsMap.expandAttributeName(tem, null),
													   attList.getValue(i));
						}
						/* !!! IDs
						   if (attList.getType(i).length() == 2)
						   parent.addId(attList.getValue(i), null);
						*/
					}
					// Assign here to avoid inconsistent state if exception
					// is thrown.
					attributeArray = vec;
				}
			}
			// push possibly different nsMap for the subtree starting here
			_nsMapStack[++_nsMapStackPointer] = nsMap;

			Name nameObject  = nsMap.expandElementTypeName(name, null);

			Object internedName = _names.get(name);
			if (internedName == null)	// first encounter
				_names.put(name, internedName = name);
			push(null, internedName, attributeArray);
		}
		catch (XSLException e) {
			e.printStackTrace();
		}
    }
  
    public void endElement(String elementName) {
		maybeCreateTextNode();
		try {
			final Object internedName = _names.get(elementName);
			for (int i = _stackPointer; i >= 0; i--)
				if (_openNames[i] == internedName) {// look for an element to close
					NamespacePrefixMap nsMap = _nsMapStack[_nsMapStackPointer--];
					Name name = nsMap.expandElementTypeName(elementName, null);
					ElementImpl element;
					if (i < _stackPointer) {	// children exist
						if (_stackPointer - i == 1) {	// single child
							element = new XmlTreeElement1(name, _attributes[i], _stack[i + 1]);
							if (_stack[i + 1] instanceof NodeImpl)
								((NodeImpl)_stack[i+1]).setParent(element);
						}
						else {
							NodeImpl[] children = new NodeImpl[_stackPointer - i];
							System.arraycopy(_stack, i + 1, children, 0, _stackPointer - i);
							element = new XmlTreeElement2(name, _attributes[i], children);
							for (int j = 0; j < children.length; j++)
								children[j].setParent(element);
						}
						_stackPointer = i;	// all the children popped
					}
					else			// empty element
						element = new XmlTreeElement0(name, _attributes[i]);
	  
					element.setNamespacePrefixMap(nsMap);
					_stack[i] = element;
					_openNames[i] = null; // no longer open
					break;
				}
		}
		catch (XSLException e) {
			e.printStackTrace();
		}
    }

    public void characters(char[] ch, int start, int length) {
		_currentText = _currentText == null
			? new String(ch, start, length)
				: _currentText.concat(new String(ch, start, length));
    }
    
    // alternative method that is more efficient to work w/ JJC's XSLT
    public void characters(final String str) {
		_currentText = _currentText==null ? str : _currentText.concat(str);
    }

    public void charData(char[] ch, int start, int length) {
		characters(ch, start, length);
    }
    
    public synchronized Node getRoot(InputSource input) {
		try {
			// long start = System.currentTimeMillis();
			_parser.parse(input, this);
			/*
			  System.out.println((System.currentTimeMillis()-start)
			  +" msec parse");
			*/
			//      TreeStatictics.report((XmlTreeNode)_stack[0]);
			return getRoot();
		}
		catch (Exception e) {
			e.printStackTrace();
			return null;
		}
    }
  
    public synchronized Node getRoot() {
		return _stack[0];
    }
  
    public synchronized Node getRoot(URL docUrl) throws IOException {
		return getRoot(new InputSource(docUrl.toString()));
    }

    /*
      class TreeStatictics {
      private int _nodeCount;
      private int _elementCount;
      private int _textCount;
      private int _textLength;
      private int _allAttribs;
      private int _noAttribs;
      private int _oneAttrib;
      private int _twoAttribs;
      private int _moreThanTwoAttribs;
      private int _noChildren;
      private int _oneChild;
      private int _twoChildren;
      private int _moreThanTwoChildren;

      public static void report(Node node) {
      TreeStatictics instance = new TreeStatictics();
      instance.visit(node);
      instance.printReport(System.out);
      }

      private void printReport(PrintStream out) {
      out.println("*************************************************************");
      out.println("nodes\t\t\t\t" + _nodeCount);
      out.println("elements\t\t\t" + _elementCount);
      out.println("text nodes\t\t\t" + _textCount);
      out.println("elements with no children\t" + _noChildren
      + "\t" + (100*_noChildren/_elementCount) + '%');
      out.println("elements with one child\t\t" + _oneChild
      + "\t" + (100*_oneChild/_elementCount) + '%');
      out.println("elements with two children\t" + _twoChildren
      + "\t" + (100*_twoChildren/_elementCount) + '%');
      out.println("elements with more children\t" + _moreThanTwoChildren
      + "\t" + (100*_moreThanTwoChildren/_elementCount) + '%');
      out.println("elements with no attributes\t" + _noAttribs
      + "\t" + (100*_noAttribs/_elementCount) + '%');
      out.println("elements with one attribute\t" + _oneAttrib
      + "\t" + (100*_oneAttrib/_elementCount) + '%');
      out.println("elements with two attributes\t" + _twoAttribs
      + "\t" + (100*_twoAttribs/_elementCount) + '%');
      out.println("elements with more attributes\t" + _moreThanTwoAttribs
      + "\t" + (100*_moreThanTwoAttribs/_elementCount) + '%');
      out.println("total text length\t\t" + _textLength);
      out.println("total attr length\t\t" + _allAttribs);
      out.println("*************************************************************");
      }

      private void visit(Node node) {
      _nodeCount++;
      if (node instanceof ElementImpl) {
      ElementImpl el = (ElementImpl) node;
      
      switch (el.getChildCount()) {
      case 0:  _noChildren++;          break;
      case 1:  _oneChild++;            break;
      case 2:  _twoChildren++;         break;
      default: _moreThanTwoChildren++; break;
      }
      
      int nAttribs = el.getAttributesLength();
      _allAttribs += nAttribs;
      switch (nAttribs) {
      case 0:  _noAttribs++;          break;
      case 1:  _oneAttrib++;          break;
      case 2:  _twoAttribs++;         break;
      default: _moreThanTwoAttribs++; break;
      }
      
      Enumeration children = el.children();
      while (children.hasMoreElements())
      visit((Node)children.nextElement());
      _elementCount++;
      }
      else {			// text node
      _textCount++;
      _textLength += node.toString().length();
      }
      }
      }
    */
}
