/* PluginAppletSecurityContext -- execute plugin JNI messages
   Copyright (C) 2008  Red Hat

This file is part of IcedTea.

IcedTea is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

IcedTea is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with IcedTea; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

package sun.applet;

import java.io.File;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessControlContext;
import java.security.AccessControlException;
import java.security.AccessController;
import java.security.AllPermission;
import java.security.BasicPermission;
import java.security.CodeSource;
import java.security.Permissions;
import java.security.PrivilegedAction;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import net.sourceforge.jnlp.runtime.JNLPRuntime;
import netscape.javascript.JSObjectCreatePermission;



class Signature {
	private String signature;
	private int currentIndex;
	private List<Class> typeList;
	private static final char ARRAY = '[';
	private static final char OBJECT = 'L';
	private static final char SIGNATURE_ENDCLASS = ';';
	private static final char SIGNATURE_FUNC = '(';
	private static final char SIGNATURE_ENDFUNC = ')';
	private static final char VOID = 'V';
	private static final char BOOLEAN = 'Z';
	private static final char BYTE = 'B';
	private static final char CHARACTER = 'C';
	private static final char SHORT = 'S';
	private static final char INTEGER = 'I';
	private static final char LONG = 'J';
	private static final char FLOAT = 'F';
	private static final char DOUBLE = 'D';

	private String nextTypeName() {
		char key = signature.charAt(currentIndex++);

		switch (key) {
		case ARRAY:
			return nextTypeName() + "[]";

		case OBJECT:
			int endClass = signature.indexOf(SIGNATURE_ENDCLASS, currentIndex);
			String retVal = signature.substring(currentIndex, endClass);
			retVal = retVal.replace('/', '.');
			currentIndex = endClass + 1;
			return retVal;

			// FIXME: generated bytecode with classes named after
			// primitives will not work in this scheme -- those
			// classes will be incorrectly treated as primitive
			// types.
		case VOID:
			return "void";
		case BOOLEAN:
			return "boolean";
		case BYTE:
			return "byte";
		case CHARACTER:
			return "char";
		case SHORT:
			return "short";
		case INTEGER:
			return "int";
		case LONG:
			return "long";
		case FLOAT:
			return "float";
		case DOUBLE:
			return "double";

		case SIGNATURE_ENDFUNC:
		case SIGNATURE_FUNC:
			return nextTypeName();

		default:
			throw new IllegalArgumentException(
					"Invalid JNI signature character '" + key + "'");
		}
	}

	public Signature(String signature, ClassLoader cl) {
		this.signature = signature;
		currentIndex = 0;
		typeList = new ArrayList<Class>(10);

		String elem;
		while (currentIndex < signature.length()) {
			elem = nextTypeName();
			// System.out.println ("NEXT TYPE: " + elem);
			Class primitive = primitiveNameToType(elem);
			if (primitive != null)
				typeList.add(primitive);
			else {
				// System.out.println ("HERE1");
				int dimsize = 0;
				int n = elem.indexOf('[');
				if (n != -1) {
					// System.out.println ("HERE2");
					String arrayType = elem.substring(0, n);
					dimsize++;
					n = elem.indexOf('[', n + 1);
					// System.out.println ("HERE2.5");
					while (n != -1) {
						dimsize++;
						n = elem.indexOf('[', n + 1);
						// System.out.println ("HERE2.8");
					}
					int[] dims = new int[dimsize];
					primitive = primitiveNameToType(arrayType);
					// System.out.println ("HERE3");
					if (primitive != null) {
						typeList.add(Array.newInstance(primitive, dims)
								.getClass());
						// System.out.println ("HERE4");
					} else
						typeList.add(Array.newInstance(
								getClass(arrayType, cl), dims).getClass());
				} else {
					typeList.add(getClass(elem, cl));
				}
			}
		}
		if (typeList.size() == 0) {
			throw new IllegalArgumentException("Invalid JNI signature '"
					+ signature + "'");
		}
	}

	public static Class getClass(String name, ClassLoader cl) {

		Class c = null;
		
		try {
			c = Class.forName(name);
		} catch (ClassNotFoundException cnfe) {
			
			PluginDebug.debug("Class " + name + " not found in primordial loader. Looking in " + cl);
			try {
				c = cl.loadClass(name);
			} catch (ClassNotFoundException e) {
				throw (new RuntimeException(new ClassNotFoundException("Unable to find class " + name)));
			}
		}

		return c;
	}
	
	public static Class primitiveNameToType(String name) {
		if (name.equals("void"))
			return Void.TYPE;
		else if (name.equals("boolean"))
			return Boolean.TYPE;
		else if (name.equals("byte"))
			return Byte.TYPE;
		else if (name.equals("char"))
			return Character.TYPE;
		else if (name.equals("short"))
			return Short.TYPE;
		else if (name.equals("int"))
			return Integer.TYPE;
		else if (name.equals("long"))
			return Long.TYPE;
		else if (name.equals("float"))
			return Float.TYPE;
		else if (name.equals("double"))
			return Double.TYPE;
		else
			return null;
	}

	public Class[] getClassArray() {
		return typeList.subList(0, typeList.size() - 1).toArray(new Class[] {});
	}
}

public class PluginAppletSecurityContext {
	
	public static Hashtable<ClassLoader, String> classLoaders = new Hashtable<ClassLoader, String>();

	// FIXME: make private
	public PluginObjectStore store = new PluginObjectStore();
	private Throwable throwable = null;
	private ClassLoader liveconnectLoader = ClassLoader.getSystemClassLoader();
	int identifier = 0;
	
	public static PluginStreamHandler streamhandler;
	
	long startTime = 0;

	public PluginAppletSecurityContext(int identifier) {
		this.identifier = identifier;
		
		// also, override the basedir, use a different one for the plugin
		File f = new File(System.getProperty("user.home") + "/.icedteaplugin/");
		f.mkdir();
		JNLPRuntime.setBaseDir(f);

		// We need a security manager.. and since there is a good chance that 
		// an applet will be loaded at some point, we should make it the SM 
		// that JNLPRuntime will try to install
		if (System.getSecurityManager() == null) {
			JNLPRuntime.initialize(/* isApplication */ false);
		}

		JNLPRuntime.disableExit();

		this.classLoaders.put(liveconnectLoader, "file://");
	}

	private static <V> V parseCall(String s, ClassLoader cl, Class<V> c) {
		if (c == Integer.class)
			return (V) new Integer(s);
		else if (c == String.class)
			return (V) new String(s);
		else if (c == Signature.class)
			return (V) new Signature(s, cl);
		else
			throw new RuntimeException("Unexpected call value.");
	}

	private Object parseArgs(String s, Class c) {
		if (c == Boolean.TYPE || c == Boolean.class)
			return new Boolean(s);
		else if (c == Byte.TYPE || c == Byte.class)
			return new Byte(s);
		else if (c == Character.TYPE || c == Character.class) {
			String[] bytes = s.split("_");
			int low = Integer.parseInt(bytes[0]);
			int high = Integer.parseInt(bytes[1]);
			int full = ((high << 8) & 0x0ff00) | (low & 0x0ff);
			return new Character((char) full);
		} else if (c == Short.TYPE || c == Short.class)
			return new Short(s);
		else if (c == Integer.TYPE || c == Integer.class)
			return new Integer(s);
		else if (c == Long.TYPE || c == Long.class)
			return new Long(s);
		else if (c == Float.TYPE || c == Float.class)
			return new Float(s);
		else if (c == Double.TYPE || c == Double.class)
			return new Double(s);
		else
			return store.getObject(new Integer(s));
	}

	public void associateSrc(ClassLoader cl, String src) {
		PluginDebug.debug("Associating " + cl + " with " + src);
		this.classLoaders.put(cl, src);
	}

	public static void setStreamhandler(PluginStreamHandler sh) {
		streamhandler = sh;
	}

	public void handleMessage(int reference, String src, AccessControlContext callContext, String message) {

		startTime = new java.util.Date().getTime();

		try {
			if (message.startsWith("FindClass")) {
				ClassLoader cl = null;
				Class c = null;
				cl = liveconnectLoader;
				String className = message.substring("FindClass".length() + 1)
						.replace('/', '.');

				try {
					c = cl.loadClass(className);
					store.reference(c);
					write(reference, "FindClass " + store.getIdentifier(c));
				} catch (ClassNotFoundException cnfe) {
					write(reference, "FindClass 0");
				}

			} else if (message.startsWith("GetStaticMethodID")
					|| message.startsWith("GetMethodID")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				String methodName = parseCall(args[2], null, String.class);
				Signature signature = parseCall(args[3], ((Class) store.getObject(classID)).getClassLoader(), Signature.class);
				Object[] a = signature.getClassArray();

				Class c = (Class) store.getObject(classID);
				Method m = null;
				Constructor cs = null;
				Object o = null;
				if (methodName.equals("<init>")
						|| methodName.equals("<clinit>")) {
					o = cs = c.getConstructor(signature.getClassArray());
					store.reference(cs);
				} else {
					o = m = c.getMethod(methodName, signature.getClassArray());
					store.reference(m);
				}
				PluginDebug.debug(o + " has id " + store.getIdentifier(o));
				write(reference, args[0] + " " + store.getIdentifier(o));
			} else if (message.startsWith("GetStaticFieldID")
					|| message.startsWith("GetFieldID")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				String fieldName = parseCall(args[2], null, String.class);
				Signature signature = parseCall(args[3], ((Class) store.getObject(classID)).getClassLoader(), Signature.class);

				Class c = (Class) store.getObject(classID);
				Field f = null;
				f = c.getField(fieldName);

				store.reference(f);

				write(reference, "GetStaticFieldID " + store.getIdentifier(f));
			} else if (message.startsWith("GetStaticField")) {
				String[] args = message.split(" ");
				String type = parseCall(args[1], null, String.class);
				Integer classID = parseCall(args[1], null, Integer.class);
				Integer fieldID = parseCall(args[2], null, Integer.class);

				final Class c = (Class) store.getObject(classID);
				final Field f = (Field) store.getObject(fieldID);

				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				checkPermission(src, c, acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							return f.get(c);
						} catch (Throwable t) {
							return t;
						}
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				// System.out.println ("FIELD VALUE: " + ret);
				if (ret == null) {
					write(reference, "GetStaticField 0");
				} else if (f.getType() == Boolean.TYPE
						|| f.getType() == Byte.TYPE
						|| f.getType() == Character.TYPE
						|| f.getType() == Short.TYPE
						|| f.getType() == Integer.TYPE
						|| f.getType() == Long.TYPE
						|| f.getType() == Float.TYPE
						|| f.getType() == Double.TYPE) {
					write(reference, "GetStaticField " + ret);
				} else {
					// Track returned object.
					store.reference(ret);
					write(reference, "GetStaticField "
							+ store.getIdentifier(ret));
				}
			} else if (message.startsWith("SetStaticField")) {
				String[] args = message.split(" ");
				String type = parseCall(args[1], null, String.class);
				Integer classID = parseCall(args[2], null, Integer.class);
				Integer fieldID = parseCall(args[3], null, Integer.class);

				Object value = null;
				if (Signature.primitiveNameToType(type) != null) {
					value = parseArgs(args[4], Signature
							.primitiveNameToType(type));
					// System.out.println ("HERE1: " + value);
				} else {
					value = parseArgs(args[3], Object.class);
					// System.out.println ("HERE2: " + value);
				}

				final Class c = (Class) store.getObject(classID);
				final Field f = (Field) store.getObject(fieldID);

				final Object fValue = value;
				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				checkPermission(src, c, acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							f.set(c, fValue);
						} catch (Throwable t) {
							return t;
						}
						
						return null;
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				write(reference, "SetStaticField");
			} else if (message.startsWith("SetField")) {
				String[] args = message.split(" ");
				String type = parseCall(args[1], null, String.class);
				Integer objectID = parseCall(args[2], null, Integer.class);
				Integer fieldID = parseCall(args[3], null, Integer.class);

				Object value = null;
				if (Signature.primitiveNameToType(type) != null) {
					value = parseArgs(args[4], Signature
							.primitiveNameToType(type));
					// System.out.println ("HERE1: " + value);
				} else {
					value = parseArgs(args[3], Object.class);
					// System.out.println ("HERE2: " + value);
				}

				final Object o = (Object) store.getObject(objectID);
				final Field f = (Field) store.getObject(fieldID);

				final Object fValue = value;
				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				checkPermission(src, o.getClass(), acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							f.set(o, fValue);
						} catch (Throwable t) {
							return t;
						}
						
						return null;
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				write(reference, "SetField");
			} else if (message.startsWith("GetObjectArrayElement")) {
				String[] args = message.split(" ");
				Integer arrayID = parseCall(args[1], null, Integer.class);
				Integer index = parseCall(args[2], null, Integer.class);

				Object[] o = (Object[]) store.getObject(arrayID);
				Object ret = null;

				ret = o[index];

				// Track returned object.
				store.reference(ret);
				// System.out.println ("array element: " + index + " " + ret);
				write(reference, "GetObjectArrayElement "
						+ store.getIdentifier(ret));
			} else if (message.startsWith("SetObjectArrayElement")) {
				String[] args = message.split(" ");
				Integer arrayID = parseCall(args[1], null, Integer.class);
				Integer index = parseCall(args[2], null, Integer.class);
				Integer objectID = parseCall(args[3], null, Integer.class);

				Object[] o = (Object[]) store.getObject(arrayID);
				Object toSet = (Object) store.getObject(objectID);

				o[index] = toSet;

				write(reference, "SetObjectArrayElement");
			} else if (message.startsWith("GetArrayLength")) {
				String[] args = message.split(" ");
				Integer arrayID = parseCall(args[1], null, Integer.class);

				//System.out.println("ARRAYID: " + arrayID);
				Object o = (Object) store.getObject(arrayID);
				int len = 0;
				len = Array.getLength(o);
				// System.out.println ("Returning array length: " + len);

				// System.out.println ("array length: " + o + " " + len);
				write(reference, "GetArrayLength " + Array.getLength(o));
			} else if (message.startsWith("GetField")) {
				String[] args = message.split(" ");
				String type = parseCall(args[1], null, String.class);
				Integer objectID = parseCall(args[1], null, Integer.class);
				Integer fieldID = parseCall(args[2], null, Integer.class);

				final Object o = (Object) store.getObject(objectID);
				final Field f = (Field) store.getObject(fieldID);

				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				checkPermission(src, o.getClass(), acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							return f.get(o);
						} catch (Throwable t) {
							return t;
						}
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				// System.out.println ("FIELD VALUE: " + ret);
				if (ret == null) {
					write(reference, "GetField 0");
				} else if (f.getType() == Boolean.TYPE
						|| f.getType() == Byte.TYPE
						|| f.getType() == Character.TYPE
						|| f.getType() == Short.TYPE
						|| f.getType() == Integer.TYPE
						|| f.getType() == Long.TYPE
						|| f.getType() == Float.TYPE
						|| f.getType() == Double.TYPE) {
					write(reference, "GetField " + ret);
				} else {
					// Track returned object.
					store.reference(ret);
					write(reference, "GetField " + store.getIdentifier(ret));
				}
			} else if (message.startsWith("GetObjectClass")) {
				int oid = Integer.parseInt(message.substring("GetObjectClass"
						.length() + 1));
				// System.out.println ("GETTING CLASS FOR: " + oid);
				Class c = store.getObject(oid).getClass();
				// System.out.println (" OBJ: " + store.getObject(oid));
				// System.out.println (" CLS: " + c);
				store.reference(c);

				write(reference, "GetObjectClass " + store.getIdentifier(c));
			} else if (message.startsWith("CallStaticMethod")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				Integer methodID = parseCall(args[2], null, Integer.class);

				PluginDebug.debug("GETTING: " + methodID);
				final Method m = (Method) store.getObject(methodID);
				PluginDebug.debug("GOT: " + m);
				Class[] argTypes = m.getParameterTypes();

				Object[] arguments = new Object[argTypes.length];
				for (int i = 0; i < argTypes.length; i++) {
					arguments[i] = parseArgs(args[3 + i], argTypes[i]);
					// System.out.println ("GOT ARG: " + argTypes[i] + " " +
					// arguments[i]);
				}

				// System.out.println ("Calling " + m);

				final Object[] fArguments = arguments;
				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				Class c = (Class) store.getObject(classID);
				checkPermission(src, c, acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							return m.invoke(null, fArguments);
						} catch (Throwable t) {
							return t;
						}
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				// if (ret != null)
				// System.out.println ("RETURN VALUE: " + ret + " " +
				// ret.getClass());
				// else
				// System.out.println ("RETURN VALUE: " + ret);
				if (ret == null) {
					write(reference, "CallStaticMethod void");
				} else if (m.getReturnType() == Boolean.TYPE
						|| m.getReturnType() == Byte.TYPE
						|| m.getReturnType() == Short.TYPE
						|| m.getReturnType() == Integer.TYPE
						|| m.getReturnType() == Long.TYPE
						|| m.getReturnType() == Float.TYPE
						|| m.getReturnType() == Double.TYPE) {
					write(reference, "CallStaticMethod " + ret);
				} else if (m.getReturnType() == Character.TYPE) {
					char ch = (Character) ret;
					int high = (((int) ch) >> 8) & 0x0ff;
					int low = ((int) ch) & 0x0ff;
					write(reference, "CallStaticMethod " + low + "_" + high);
				} else {
					// Track returned object.
					store.reference(ret);
					write(reference, "CallStaticMethod "
							+ store.getIdentifier(ret));
				}
			} else if (message.startsWith("CallMethod")) {
				String[] args = message.split(" ");
				Integer objectID = parseCall(args[1], null, Integer.class);
				Integer methodID = parseCall(args[2], null, Integer.class);

				final Object o = (Object) store.getObject(objectID);
				final Method m = (Method) store.getObject(methodID);
				Class[] argTypes = m.getParameterTypes();

				Object[] arguments = new Object[argTypes.length];
				for (int i = 0; i < argTypes.length; i++) {
					arguments[i] = parseArgs(args[3 + i], argTypes[i]);
					PluginDebug.debug("GOT ARG: " + argTypes[i] + " "
							+ arguments[i]);
				}

				String collapsedArgs = "";
				for (String s : args) {
					collapsedArgs += " " + s;
				}

				PluginDebug.debug("Calling method " + m + " on object " + o
						+ " with " + arguments);

				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();
				checkPermission(src, o.getClass(), acc);

				final Object[] fArguments = arguments;
				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							return m.invoke(o, fArguments);
						} catch (Throwable t) {
							return t;
						}
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				String retO;
				if (ret == null) {
					retO = "null";
				} else {
					retO = ret.getClass().toString();
				}

				PluginDebug.debug("Calling " + m + " on " + o + " with "
						+ collapsedArgs + " and that returned: " + ret
						+ " of type " + retO);

				if (ret == null) {
					write(reference, "CallMethod void");
				} else if (m.getReturnType() == Boolean.TYPE
						|| m.getReturnType() == Byte.TYPE
						|| m.getReturnType() == Short.TYPE
						|| m.getReturnType() == Integer.TYPE
						|| m.getReturnType() == Long.TYPE
						|| m.getReturnType() == Float.TYPE
						|| m.getReturnType() == Double.TYPE) {
					write(reference, "CallMethod " + ret);
				} else if (m.getReturnType() == Character.TYPE) {
					char ch = (Character) ret;
					int high = (((int) ch) >> 8) & 0x0ff;
					int low = ((int) ch) & 0x0ff;
					write(reference, "CallMethod " + low + "_" + high);
				} else {
					// Track returned object.
					store.reference(ret);
					write(reference, "CallMethod " + store.getIdentifier(ret));
				}
			} else if (message.startsWith("GetSuperclass")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				Class c = null;
				Class ret = null;

				c = (Class) store.getObject(classID);
				ret = c.getSuperclass();
				store.reference(ret);

				write(reference, "GetSuperclass " + store.getIdentifier(ret));
			} else if (message.startsWith("IsAssignableFrom")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				Integer superclassID = parseCall(args[2], null, Integer.class);

				boolean result = false;
				Class clz = (Class) store.getObject(classID);
				Class sup = (Class) store.getObject(superclassID);

				result = sup.isAssignableFrom(clz);

				write(reference, "IsAssignableFrom " + (result ? "1" : "0"));
			} else if (message.startsWith("IsInstanceOf")) {
				String[] args = message.split(" ");
				Integer objectID = parseCall(args[1], null, Integer.class);
				Integer classID = parseCall(args[2], null, Integer.class);

				boolean result = false;
				Object o = (Object) store.getObject(objectID);
				Class c = (Class) store.getObject(classID);

				result = c.isInstance(o);

				write(reference, "IsInstanceOf " + (result ? "1" : "0"));
			} else if (message.startsWith("GetStringUTFLength")) {
				String[] args = message.split(" ");
				Integer stringID = parseCall(args[1], null, Integer.class);

				String o = null;
				byte[] b = null;
				o = (String) store.getObject(stringID);
				b = o.getBytes("UTF-8");
				// System.out.println ("STRING UTF-8 LENGTH: " + o + " " +
				// b.length);

				write(reference, "GetStringUTFLength " + o.length());
			} else if (message.startsWith("GetStringLength")) {
				String[] args = message.split(" ");
				Integer stringID = parseCall(args[1], null, Integer.class);

				String o = null;
				byte[] b = null;
				o = (String) store.getObject(stringID);
				b = o.getBytes("UTF-16LE");
				// System.out.println ("STRING UTF-16 LENGTH: " + o + " " +
				// b.length);

				// System.out.println ("Java: GetStringLength " + b.length);
				write(reference, "GetStringLength " + o.length());
			} else if (message.startsWith("GetStringUTFChars")) {
				String[] args = message.split(" ");
				Integer stringID = parseCall(args[1], null, Integer.class);

				String o = null;
				byte[] b = null;
				StringBuffer buf = null;
				o = (String) store.getObject(stringID);
				b = o.getBytes("UTF-8");
				buf = new StringBuffer(b.length * 2);
				buf.append(b.length);
				for (int i = 0; i < b.length; i++)
					buf
							.append(" "
									+ Integer
											.toString(((int) b[i]) & 0x0ff, 16));

				// System.out.println ("Java: GetStringUTFChars: " + o);
				// //System.out.println ("String UTF BYTES: " + buf);
				write(reference, "GetStringUTFChars " + buf);
			} else if (message.startsWith("GetStringChars")) {
				String[] args = message.split(" ");
				Integer stringID = parseCall(args[1], null, Integer.class);

				String o = null;
				byte[] b = null;
				StringBuffer buf = null;
				o = (String) store.getObject(stringID);
				// FIXME: LiveConnect uses UCS-2.
				b = o.getBytes("UTF-16LE");
				buf = new StringBuffer(b.length * 2);
				buf.append(b.length);
				for (int i = 0; i < b.length; i++)
					buf
							.append(" "
									+ Integer
											.toString(((int) b[i]) & 0x0ff, 16));

				PluginDebug.debug("Java: GetStringChars: " + o);
				PluginDebug.debug("  String BYTES: " + buf);
				write(reference, "GetStringChars " + buf);
			} else if (message.startsWith("NewArray")) {
				String[] args = message.split(" ");
				String type = parseCall(args[1], null, String.class);
				Integer length = parseCall(args[2], null, Integer.class);

				// System.out.println ("CALLING: NewArray: " + type + " " +
				// length + " "
				// + Signature.primitiveNameToType(type));

				Object newArray = null;
				newArray = Array.newInstance(Signature
						.primitiveNameToType(type), length);

				store.reference(newArray);
				write(reference, "NewArray " + store.getIdentifier(newArray));
			} else if (message.startsWith("NewObjectArray")) {
				String[] args = message.split(" ");
				Integer length = parseCall(args[1], null, Integer.class);
				Integer classID = parseCall(args[2], null, Integer.class);
				Integer objectID = parseCall(args[3], null, Integer.class);

				// System.out.println ("CALLING: NewObjectArray: " +
				// classID + " " + length + " "
				// + objectID);

				Object newArray = null;
				newArray = Array.newInstance((Class) store.getObject(classID),
						length);

				Object[] array = (Object[]) newArray;
				for (int i = 0; i < array.length; i++)
					array[i] = store.getObject(objectID);
				store.reference(newArray);
				write(reference, "NewObjectArray "
						+ store.getIdentifier(newArray));
			} else if (message.startsWith("NewObject")) {
				String[] args = message.split(" ");
				Integer classID = parseCall(args[1], null, Integer.class);
				Integer methodID = parseCall(args[2], null, Integer.class);

				final Constructor m = (Constructor) store.getObject(methodID);
				Class[] argTypes = m.getParameterTypes();

				// System.out.println ("NEWOBJ: HERE1");
				Object[] arguments = new Object[argTypes.length];
				// System.out.println ("NEWOBJ: HERE2");
				for (int i = 0; i < argTypes.length; i++) {
					arguments[i] = parseArgs(args[3 + i], argTypes[i]);
					// System.out.println ("NEWOBJ: GOT ARG: " + arguments[i]);
				}

				final Object[] fArguments = arguments;
				AccessControlContext acc = callContext != null ? callContext : getClosedAccessControlContext();

				Class c = (Class) store.getObject(classID);
				checkPermission(src, c, acc);

				Object ret = AccessController.doPrivileged(new PrivilegedAction<Object> () {
					public Object run() {
						try {
							return m.newInstance(fArguments);
						} catch (Throwable t) {
							return t;
						}
					}
				}, acc);

				if (ret instanceof Throwable)
					throw (Throwable) ret;

				store.reference(ret);

				write(reference, "NewObject " + store.getIdentifier(ret));

			} else if (message.startsWith("NewString")) {
				PluginDebug.debug("MESSAGE: " + message);
				String[] args = message.split(" ");
				Integer strlength = parseCall(args[1], null, Integer.class);
				int bytelength = 2 * strlength;
				byte[] byteArray = new byte[bytelength];
				String ret = null;
				for (int i = 0; i < strlength; i++) {
					int c = parseCall(args[2 + i], null, Integer.class);
					PluginDebug.debug("char " + i + " " + c);
					// Low.
					byteArray[2 * i] = (byte) (c & 0x0ff);
					// High.
					byteArray[2 * i + 1] = (byte) ((c >> 8) & 0x0ff);
				}
				ret = new String(byteArray, 0, bytelength, "UTF-16LE");
				PluginDebug.debug("NEWSTRING: " + ret);

				// System.out.println ("NEWOBJ: CALLED: " + ret);
				// System.out.println ("NEWOBJ: CALLED: " +
				// store.getObject(ret));
				store.reference(ret);
				write(reference, "NewString " + store.getIdentifier(ret));
			} else if (message.startsWith("NewStringUTF")) {
				PluginDebug.debug("MESSAGE: " + message);
				String[] args = message.split(" ");
				byte[] byteArray = new byte[60];
				String ret = null;
				int i = 0;
				int c = 0;
				while (((byte) c) != 0) {
					c = parseCall(args[1 + i], null, Integer.class);
					byteArray[i] = (byte) c;
					i++;
					if (i == byteArray.length) {
						byte[] newByteArray = new byte[2 * byteArray.length];
						System.arraycopy(byteArray, 0, newByteArray, 0,
								byteArray.length);
						byteArray = newByteArray;
					}
				}
				byteArray[i] = (byte) 0;
				ret = new String(byteArray, "UTF-8");
				PluginDebug.debug("NEWSTRINGUTF: " + ret);

				store.reference(ret);
				write(reference, "NewStringUTF " + store.getIdentifier(ret));
			} else if (message.startsWith("ExceptionOccurred")) {
				PluginDebug.debug("EXCEPTION: " + throwable);
				if (throwable != null)
					store.reference(throwable);
				write(reference, "ExceptionOccurred "
						+ store.getIdentifier(throwable));
			} else if (message.startsWith("ExceptionClear")) {
				if (throwable != null && store.contains(throwable))
					store.unreference(store.getIdentifier(throwable));
				throwable = null;
				write(reference, "ExceptionClear");
			} else if (message.startsWith("DeleteGlobalRef")) {
				String[] args = message.split(" ");
				Integer id = parseCall(args[1], null, Integer.class);
				store.unreference(id);
				write(reference, "DeleteGlobalRef");
			} else if (message.startsWith("DeleteLocalRef")) {
				String[] args = message.split(" ");
				Integer id = parseCall(args[1], null, Integer.class);
				store.unreference(id);
				write(reference, "DeleteLocalRef");
			} else if (message.startsWith("NewGlobalRef")) {
				String[] args = message.split(" ");
				Integer id = parseCall(args[1], null, Integer.class);
				store.reference(store.getObject(id));
				write(reference, "NewGlobalRef " + id);
			}
		} catch (Throwable t) {
			t.printStackTrace();
			String msg = t.getCause() != null ? t.getCause().getMessage() : t.getMessage();

			// add an identifier string to let javaside know of the type of error
			// check for cause as well, since the top level exception will be InvocationTargetException in most cases
			if (t instanceof AccessControlException || t.getCause() instanceof AccessControlException) {
				msg = "LiveConnectPermissionNeeded " + msg;
			}

			this.streamhandler.write("instance " + identifier + " reference " + reference + " Error " + msg);

			// ExceptionOccured is only called after Callmethod() by mozilla. So
			// for exceptions that are not related to CallMethod, we need a way
			// to log them. This is how we do it.. send an error message to the
			// c++ side to let it know that something went wrong, and it will do
			// the right thing to let mozilla know

			// Store the cause as the actual exception. This is needed because 
			// the exception we get here will always be an 
			// "InvocationTargetException" due to the use of reflection above
			if (message.startsWith("CallMethod") || message.startsWith("CallStaticMethod"))
				throwable = t.getCause();
		}

	}

	/**
	 * Checks if the calling script is allowed to access the specified class
	 *  
	 * See http://java.sun.com/j2se/1.3/docs/guide/plugin/security.html#liveconnect for details
	 *  
	 * @param jsSrc The source of the script
	 * @param target The target class that the script is trying to access
	 * @param acc AccessControlContext for this execution
	 * @throws AccessControlException If the script has insufficient permissions
	 */
	public void checkPermission(String jsSrc, Class target, AccessControlContext acc) throws AccessControlException {

		// target classloader == null => primordial loader. Allow this.
		if (target.getClassLoader() == null)
			return;

		String classSrc = this.classLoaders.get(target.getClassLoader());

		PluginDebug.debug("target = " + target + " jsSrc=" + jsSrc + " classSrc=" + classSrc);
		
		// if src is not a file and class loader does not map to the same base, UniversalBrowserRead (BrowserReadPermission) must be set
		if (!jsSrc.equals("file://") && !jsSrc.equals("[System]") && !classSrc.equals(jsSrc)) {
			acc.checkPermission(new BrowserReadPermission());
		}
	}

	private void write(int reference, String message) {
		PluginDebug.debug("appletviewer writing " + message);
		streamhandler.write("context " + identifier + " reference " + reference
				+ " " + message);
	}
	
	public void prePopulateLCClasses() {
		
		int classID;
		
		prepopulateClass("netscape/javascript/JSObject");
		classID = prepopulateClass("netscape/javascript/JSException");
		prepopulateMethod(classID, "<init>", "(Ljava/lang/String;Ljava/lang/String;ILjava/lang/String;I)V");
		prepopulateMethod(classID, "<init>", "(ILjava/lang/Object;)V");
		prepopulateField(classID, "lineno");
		prepopulateField(classID, "tokenIndex");
		prepopulateField(classID, "source");
		prepopulateField(classID, "filename");
		prepopulateField(classID, "wrappedExceptionType");
		prepopulateField(classID, "wrappedException");
		
		classID = prepopulateClass("netscape/javascript/JSUtil");
		prepopulateMethod(classID, "getStackTrace", "(Ljava/lang/Throwable;)Ljava/lang/String;");

		prepopulateClass("java/lang/Object");
		classID = prepopulateClass("java/lang/Class");
		prepopulateMethod(classID, "getMethods", "()[Ljava/lang/reflect/Method;");
		prepopulateMethod(classID, "getConstructors", "()[Ljava/lang/reflect/Constructor;");
		prepopulateMethod(classID, "getFields", "()[Ljava/lang/reflect/Field;");
		prepopulateMethod(classID, "getName", "()Ljava/lang/String;");
		prepopulateMethod(classID, "isArray", "()Z");
		prepopulateMethod(classID, "getComponentType", "()Ljava/lang/Class;");
		prepopulateMethod(classID, "getModifiers", "()I");
		

		classID = prepopulateClass("java/lang/reflect/Method");
		prepopulateMethod(classID, "getName", "()Ljava/lang/String;");
		prepopulateMethod(classID, "getParameterTypes", "()[Ljava/lang/Class;");
		prepopulateMethod(classID, "getReturnType", "()Ljava/lang/Class;");
		prepopulateMethod(classID, "getModifiers", "()I");

		classID = prepopulateClass("java/lang/reflect/Constructor");
		prepopulateMethod(classID, "getParameterTypes", "()[Ljava/lang/Class;");
		prepopulateMethod(classID, "getModifiers", "()I");
		
		classID = prepopulateClass("java/lang/reflect/Field");
		prepopulateMethod(classID, "getName", "()Ljava/lang/String;");
		prepopulateMethod(classID, "getType", "()Ljava/lang/Class;");
		prepopulateMethod(classID, "getModifiers", "()I");
		
		classID = prepopulateClass("java/lang/reflect/Array");
		prepopulateMethod(classID, "newInstance", "(Ljava/lang/Class;I)Ljava/lang/Object;");
		
		classID = prepopulateClass("java/lang/Throwable");
		prepopulateMethod(classID, "toString", "()Ljava/lang/String;");
		prepopulateMethod(classID, "getMessage", "()Ljava/lang/String;");
		
		classID = prepopulateClass("java/lang/System");
		prepopulateMethod(classID, "identityHashCode", "(Ljava/lang/Object;)I");
		
		classID = prepopulateClass("java/lang/Boolean");
		prepopulateMethod(classID, "booleanValue", "()D");
		prepopulateMethod(classID, "<init>", "(Z)V");

		classID = prepopulateClass("java/lang/Double");
		prepopulateMethod(classID, "doubleValue", "()D");
		prepopulateMethod(classID, "<init>", "(D)V");

		classID = prepopulateClass("java/lang/Void");
		prepopulateField(classID, "TYPE");

		prepopulateClass("java/lang/String");		
		prepopulateClass("java/applet/Applet");
	}

	private int prepopulateClass(String name) {
		name = name.replace('/', '.');
		ClassLoader cl = liveconnectLoader;
		Class c = null;

		try {
			c = cl.loadClass(name);
			store.reference(c);
		} catch (ClassNotFoundException cnfe) {
			// do nothing ... this should never happen
			cnfe.printStackTrace();
		}

		return store.getIdentifier(c);
	}
	
	private int prepopulateMethod(int classID, String methodName, String signatureStr) {
		Signature signature = parseCall(signatureStr, ((Class) store.getObject(classID)).getClassLoader(), Signature.class);
		Object[] a = signature.getClassArray();

		Class c = (Class) store.getObject(classID);
		Method m = null;
		Constructor cs = null;
		Object o = null;
		
		try {
			if (methodName.equals("<init>")
					|| methodName.equals("<clinit>")) {
				o = cs = c.getConstructor(signature.getClassArray());
				store.reference(cs);
			} else {
				o = m = c.getMethod(methodName, signature.getClassArray());
				store.reference(m);
			}
		} catch (NoSuchMethodException e) {
			// should never happen
			e.printStackTrace();
		}
		
		return store.getIdentifier(m);
	}
	
	private int prepopulateField(int classID, String fieldName) {

		Class c = (Class) store.getObject(classID);
		Field f = null;
		try {
			f = c.getField(fieldName);
		} catch (SecurityException e) {
			// should never happen
			e.printStackTrace();
		} catch (NoSuchFieldException e) {
			// should never happen			
			e.printStackTrace();
		}

		store.reference(f);
		return store.getIdentifier(f);
	}

	public void dumpStore() {
		store.dump();
	}

	public Object getObject(int identifier) {
		return store.getObject(identifier);		
	}

	public int getIdentifier(Object o) {
		return store.getIdentifier(o);
	}

	public void store(Object o) {
		store.reference(o);
	}

	/**
	 * Returns a "closed" AccessControlContext i.e. no permissions to get out of sandbox.
	 */
	public AccessControlContext getClosedAccessControlContext() {
		// Deny everything
		Permissions p = new Permissions();
		ProtectionDomain pd = new ProtectionDomain(null, p);
		return new AccessControlContext(new ProtectionDomain[] {pd});
	}

	public AccessControlContext getAccessControlContext(String[] nsPrivilegeList, String src) {

/*
		for (int i=0; i < nsPrivilegeList.length; i++) {
			String privilege = nsPrivilegeList[i];

			if (privilege.equals("UniversalAccept")) {
				SocketPermission sp = new SocketPermission("*", "accept,resolve");
				grantedPermissions.add(sp);
			} else if (privilege.equals("UniversalAwtEventQueueAccess")) {
				AWTPermission awtp = new AWTPermission("accessEventQueue");
				grantedPermissions.add(awtp);
			} else if (privilege.equals("UniversalConnect")) {
				SocketPermission sp = new SocketPermission("*", "connect,resolve");
				grantedPermissions.add(sp);
			} else if (privilege.equals("UniversalListen")) {
				SocketPermission sp = new SocketPermission("*", "listen,resolve");
				grantedPermissions.add(sp);
			} else if (privilege.equals("UniversalExecAccess")) {
				FilePermission fp = new FilePermission("<<ALL FILES>>", "execute");
				RuntimePermission rtp = new RuntimePermission("setIO");
				grantedPermissions.add(fp);
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalExitAccess")) {
				// Doesn't matter what the permissions are. Do not allow VM to exit.. we 
				// use a single VM for the entire browser lifecycle once invoked, we 
				// cannot let it exit

				//RuntimePermission rtp = new RuntimePermission("exitVM.*");
				//grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalFileDelete")) {
				FilePermission fp = new FilePermission("<<ALL FILES>>", "delete");
				grantedPermissions.add(fp);
			} else if (privilege.equals("UniversalFileRead")) {
				FilePermission fp = new FilePermission("<<ALL FILES>>", "read");
				grantedPermissions.add(fp);
			} else if (privilege.equals("UniversalFileWrite")) {
				FilePermission fp = new FilePermission("<<ALL FILES>>", "write");
				grantedPermissions.add(fp);
			}  else if (privilege.equals("UniversalFdRead")) {
				RuntimePermission rtp = new RuntimePermission("readFileDescriptor");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalFdWrite")) {
				RuntimePermission rtp = new RuntimePermission("writeFileDescriptor");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalLinkAccess")) {
				RuntimePermission rtp = new RuntimePermission("loadLibrary.*");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalListen")) {
				SocketPermission sp = new SocketPermission("*", "listen");
				grantedPermissions.add(sp);
			} else if (privilege.equals("UniversalMulticast")) {
				SocketPermission sp = new SocketPermission("*", "accept,connect,resolve");
				grantedPermissions.add(sp);
			} else if (privilege.equals("UniversalPackageAccess")) {
				RuntimePermission rtp = new RuntimePermission("defineClassInPackage.*");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalPackageDefinition")) {
				RuntimePermission rtp = new RuntimePermission("accessClassInPackage.*");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalPrintJobAccess")) {
				RuntimePermission rtp = new RuntimePermission("queuePrintJob");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalPropertyRead")) {
				PropertyPermission pp = new PropertyPermission("*", "read");
				grantedPermissions.add(pp);
			} else if (privilege.equals("UniversalPropertyWrite")) {
				PropertyPermission pp = new PropertyPermission("*", "write");
				grantedPermissions.add(pp);
			} else if (privilege.equals("UniversalSetFactory")) {
				RuntimePermission rtp = new RuntimePermission("setFactory");
				grantedPermissions.add(rtp);
			} else if (privilege.equals("UniversalSystemClipboardAccess")) {
				AWTPermission awtp = new AWTPermission("accessClipboard");
				grantedPermissions.add(awtp);
			} else if (privilege.equals("UniversalThreadAccess")) {
				RuntimePermission rtp1 = new RuntimePermission("modifyThread");
				RuntimePermission rtp2 = new RuntimePermission("stopThread");
				grantedPermissions.add(rtp1);
				grantedPermissions.add(rtp2);
			} else if (privilege.equals("UniversalThreadGroupAccess")) {
				RuntimePermission rtp1 = new RuntimePermission("modifyThreadGroup");
				RuntimePermission rtp2 = new RuntimePermission("modifyThread");
				RuntimePermission rtp3 = new RuntimePermission("stopThread");
				grantedPermissions.add(rtp1);
				grantedPermissions.add(rtp2);
				grantedPermissions.add(rtp3);
			} else if (privilege.equals("UniversalTopLevelWindow")) {
				AWTPermission awtp = new AWTPermission("topLevelWindow");
				grantedPermissions.add(awtp);
			} else if (privilege.equals("UniversalBrowserRead")) {
				BrowserReadPermission bp = new BrowserReadPermission();
				grantedPermissions.add(bp);
			} else if (privilege.equals("UniversalJavaPermissions")) {
				AllPermission ap = new AllPermission();
				grantedPermissions.add(ap);
			}
		}
		
		// what to do with these is unknown: UniversalConnectWithRedirect, UniversalDialogModality, UniversalSendMail, LimitedInstall, FullInstall, SilentInstall
*/

		Permissions grantedPermissions = new Permissions();

		for (int i=0; i < nsPrivilegeList.length; i++) {
			String privilege = nsPrivilegeList[i];

			if (privilege.equals("UniversalBrowserRead")) {
				BrowserReadPermission bp = new BrowserReadPermission();
				grantedPermissions.add(bp);
			} else if (privilege.equals("UniversalJavaPermission")) {
				AllPermission ap = new AllPermission();
				grantedPermissions.add(ap);
			}
		}

		CodeSource cs = new CodeSource((URL) null, (java.security.cert.Certificate  [])null);
		
		if (src != null && src.length() > 0) {
			try {
				cs = new CodeSource(new URL(src + "/"), (java.security.cert.Certificate[]) null);
			} catch (MalformedURLException mfue) {
				// do nothing
			}
		} else {
		    JSObjectCreatePermission perm = new JSObjectCreatePermission();
		    grantedPermissions.add(perm);
		}

		ProtectionDomain pd = new ProtectionDomain(cs, grantedPermissions, null, null);

		// Add to hashmap
		return new AccessControlContext(new ProtectionDomain[] {pd});
	}

	class BrowserReadPermission extends BasicPermission {
		public BrowserReadPermission() {
			super("browserRead");
		}
	}
}
