/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VirtualMachine_p.h"

#include "Debug.h"
#include "GTLCore/PrimitiveTypesTraits_p.h"
#include "GTLCore/Value.h"
#include <llvm/Function.h>
#include <llvm/ExecutionEngine/GenericValue.h>
#include <llvm/ExecutionEngine/ExecutionEngine.h>

#include <llvm/DerivedTypes.h> // <- I don't understand why I need to include that file to be able to display llvm::Type on the standard output

/**
 * This function is used as a fall back when everything else has fail.
 * It can be very slow.
 */
class FunctionCallerFallBack : public FunctionCaller {
  public:
    FunctionCallerFallBack(llvm::Function* llvmFunction, const Function* function) : FunctionCaller( llvmFunction, function)
    {
    }
    virtual GTLCore::Value call(const std::vector<GTLCore::Value>& arguments )
    {
      GTL_ASSERT(function());
      if(not arguments.empty())
      {
        GTL_WARNING("Calling function using the fall back, it can be extremely slow. Report it including the function signature: " << *(function()->getType()));
      }
      // Create the vector of arguments
      std::vector<llvm::GenericValue> args;
      for( std::vector< GTLCore::Value>::const_iterator it = arguments.begin();
          it != arguments.end(); ++it)
      {
        llvm::GenericValue v;
        v.IntVal = llvm::APInt(32, it->asInt32());
        args.push_back( v );
      }
      GTL_ASSERT(VirtualMachine::instance()->executionEngine());
      llvm::GenericValue gv = VirtualMachine::instance()->executionEngine()->runFunction( function(), args);
      return GTLCore::Value( (unsigned int)gv.IntVal.getLimitedValue() );
    }
};

#if 0

template<typename _TYPE_RETURN_, typename _TYPE_1_, typename _TYPE_2_, typename _TYPE_3_, typename _TYPE_4_>
class FunctionCaller4 : public FunctionCaller {
  public:
    FunctionCaller4(llvm::Function* _llvmFunction, const Function* _function) : FunctionCaller( _llvmFunction, _function)
    {
      m_funcPtr = VirtualMachine::instance()->getPointerToFunction(function());
    }
    virtual GTLCore::Value call(const std::vector<GTLCore::Value>& arguments )
    {
      OCTL_ASSERT(function());
      OCTL_ASSERT(arguments.size() == 4);
      _TYPE_1_ arg1 = GTLCore::PrimitiveTypeTrait<_TYPE_1_>::convert( arguments[0] );
      _TYPE_2_ arg2 = GTLCore::PrimitiveTypeTrait<_TYPE_2_>::convert( arguments[1] );
      _TYPE_3_ arg3 = GTLCore::PrimitiveTypeTrait<_TYPE_3_>::convert( arguments[2] );
      _TYPE_4_ arg4 = GTLCore::PrimitiveTypeTrait<_TYPE_4_>::convert( arguments[3] );
      _TYPE_RETURN_ (*PF)(_TYPE_1_, _TYPE_2_, _TYPE_3_, _TYPE_4_) = (_TYPE_RETURN_(*)(_TYPE_1_, _TYPE_2_, _TYPE_3_, _TYPE_4_)) m_funcPtr;
      
      return GTLCore::Value( PF(arg1, arg2, arg3, arg4) );
    }
  private:
    void* m_funcPtr;
};


#define PREPARE_FUNCTION_CALLER_1(_TYPE_RETURN_, _TYPE_1_) \
      OCTL_ASSERT(function()); \
      OCTL_ASSERT(arguments.size() == 0); \
      _TYPE_1_ arg1 = GTLCore::PrimitiveTypeTrait<_TYPE_1_>::convert( *arguments.begin() ); \
      _TYPE_RETURN_ (*PF)(_TYPE_1_) = (_TYPE_RETURN_(*)(_TYPE_1_)) VirtualMachine::instance()->getPointerToFunction(function());

template<typename _TYPE_RETURN_, typename _TYPE_1_>
class FunctionCaller1 : public FunctionCaller {
  public:
    FunctionCaller1(llvm::Function* llvmFunction, const Function* function) : FunctionCaller( llvmFunction, function)
    {
    }
    virtual GTLCore::Value call(const std::vector<GTLCore::Value>& arguments )
    {
      PREPARE_FUNCTION_CALLER_1(_TYPE_RETURN_, _TYPE_1_);
      return GTLCore::Value( PF(arg1) );
    }
  public:
    static FunctionCaller* create(llvm::Function* llvmFunction, const Function* function)
    {
      if(function->arguments().size() == 1)
      {
        return new FunctionCaller1<_TYPE_RETURN_, _TYPE_1_>(llvmFunction, function);
      } else {
#if 0
        switch( function->arguments()[0].type()->dataType() )
        {
          case GTLCore::Type::BOOLEAN:
            return FunctionCaller1<_TYPE_RETURN_, bool>::create(llvmFunction, function);
            break;
          case GTLCore::Type::INTEGER:
            return FunctionCaller1<_TYPE_RETURN_, int>::create(llvmFunction, function);
            break;
          case GTLCore::Type::UNSIGNED_INTEGER:
            return FunctionCaller1<_TYPE_RETURN_, unsigned int>::create(llvmFunction, function);
            break;
          case GTLCore::Type::HALF:
          case GTLCore::Type::FLOAT:
            return FunctionCaller1<_TYPE_RETURN_, float>::create(llvmFunction, function);
            break;
          default:
            break;
        };
#endif
      }
      return 0;
    }
};

#if 1
template<typename _TYPE_1_>
class FunctionCaller1<void, _TYPE_1_> : public FunctionCaller {
  public:
    static FunctionCaller* create(llvm::Function* llvmFunction, const Function* function)
    {
    }
};
    
    
/*template<typename _TYPE_1_>
GTLCore::Value FunctionCaller1<void, _TYPE_1_>::call(const std::vector<GTLCore::Value>& arguments )
{
  PREPARE_FUNCTION_CALLER_1(void, _TYPE_1_);
  PF(arg1);
  return GTLCore::Value( );
}*/
#endif

#define PREPARE_FUNCTION_CALLER_0(_TYPE_RETURN_) \
      OCTL_ASSERT(function()); \
      OCTL_ASSERT(arguments.size() == 0); \
      _TYPE_RETURN_ (*PF)() = (_TYPE_RETURN_(*)()) VirtualMachine::instance()->getPointerToFunction(function());

template<typename _TYPE_RETURN_>
class FunctionCaller0 : public FunctionCaller {
  public:
    FunctionCaller0(llvm::Function* llvmFunction, const Function* function) : FunctionCaller( llvmFunction, function)
    {
    }
    virtual GTLCore::Value call(const std::vector<GTLCore::Value>& arguments )
    {
      PREPARE_FUNCTION_CALLER_0(_TYPE_RETURN_);
      return GTLCore::Value( PF() );
    }
  public:
    static FunctionCaller* create(llvm::Function* llvmFunction, const Function* function)
    {
      if(function->arguments().size() == 0)
      {
        return new FunctionCaller0<_TYPE_RETURN_>(llvmFunction, function);
      } else {
        switch( function->arguments()[0].type()->dataType() )
        {
          case GTLCore::Type::BOOLEAN:
            return FunctionCaller1<_TYPE_RETURN_, bool>::create(llvmFunction, function);
            break;
          case GTLCore::Type::INTEGER:
            return FunctionCaller1<_TYPE_RETURN_, int>::create(llvmFunction, function);
            break;
          case GTLCore::Type::UNSIGNED_INTEGER:
            return FunctionCaller1<_TYPE_RETURN_, unsigned int>::create(llvmFunction, function);
            break;
          case GTLCore::Type::HALF:
          case GTLCore::Type::FLOAT:
            return FunctionCaller1<_TYPE_RETURN_, float>::create(llvmFunction, function);
            break;
          default:
            break;
        };
      }
      return 0;
    }
};

template<>
GTLCore::Value FunctionCaller0<void>::call(const std::vector<GTLCore::Value>& arguments )
{
  PREPARE_FUNCTION_CALLER_0(void);
  PF();
  return GTLCore::Value( );
}

#endif

#include "FunctionCaller_generated_p.h"
