/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* 
 * Copyright (C) 2009,2010  Alexander Sack <asac@jwsdot.com>
 *
 * This file is part of:
 *     ntrack - Network Status Tracking for Desktop Applications
 *              http://launchpad.net/ntrack
 *
 *  ntrack is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as
 *  published by the Free Software Foundation, either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  ntrack is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with ntrack.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "config.h"

#include "ntrackbase.h"
#include "ntrackarch.h"

#include <dirent.h>
#include <dlfcn.h>
#include <error.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#ifndef MODULES_DIR
#define MODULES_DIR "/usr/lib/ntrack/modules/"
#endif

typedef ntrack_monitor_t* (*_new_f) (ntrack_monitor_callback_f callback, ntrackpointer user_data);
typedef int* (*_get_rfds_f) (ntrack_monitor_t *self);
typedef int (*_process_data_f) (ntrackpointer self, int* fds);
typedef void (*_arch_free_f) (ntrackpointer self);

struct funcs_t {
	union {
		void * ptr;
		_new_f func;
	} arch_new;
	union {
		void *ptr;
		_get_rfds_f func;
	} get_rfds;
	union {
		void *ptr;
		_process_data_f func;
	} process_data;
	union {
		void *ptr;
		_arch_free_f func;
	} arch_free;
};

static struct funcs_t ftbl;

static char buf[4][PATH_MAX] = {
	MODULES_DIR,
	"",
	"",
	""
};

void
ntrack_init (int *argc, char ***argv)
{
	char *dirn = dirname ((*argv)[0]);
	int i=0;
#ifdef DEVELOPER_BUILD
	sprintf (buf[i++], "%s/../../modules/.libs/", dirn);
	sprintf (buf[i++], "%s/../../../modules/.libs/", dirn);
#endif
	dirn = MODULES_DIR;
	sprintf (buf[i++], "%s", dirn);
	buf[i++][0] = 0;
}

ntrack_monitor_t*
ntrack_arch_new (ntrack_monitor_callback_f callback, ntrackpointer user_data)
{
	DIR *dir;
	struct dirent *dirent;
	void *dlhandle;
	const char *dirpath;
	static int ftbl_initialized = 0;
	ntrack_monitor_t *self = 0;
	int i;

	if (ftbl_initialized)
		return (*ftbl.arch_new.func) (callback, user_data);

	memset (&ftbl, '\0', sizeof (struct funcs_t));

	for (i=0; !self && buf[i] && buf[i][0]; i++) {
		dir = opendir (buf[i]);
		dirpath = buf[i];
		
		while (!self && dir && (dirent = readdir (dir))) {
			char *filepath = malloc (sizeof(char) *
			                         (strlen(dirent->d_name) + 4096 + 2));
			sprintf (filepath, "%s%s", dirpath, dirent->d_name);
			dlhandle = dlopen (filepath, RTLD_LAZY | RTLD_LOCAL);
			if (!dlhandle) {
				continue;
			}
			free(filepath);

			ftbl.arch_new.ptr =
			    dlsym (dlhandle, "_ntrack_arch_new");
			ftbl.get_rfds.ptr =
			    dlsym (dlhandle, "_ntrack_arch_get_rfds");
			ftbl.process_data.ptr =
			    dlsym (dlhandle, "_ntrack_arch_process_data");
			ftbl.arch_free.ptr =
			    dlsym (dlhandle, "_ntrack_arch_free");

			if (ftbl.arch_new.func &&
			    ftbl.get_rfds.func &&
			    ftbl.process_data.func &&
			    ftbl.arch_free.func)
			{
				self = (*ftbl.arch_new.func) (callback, user_data);
				if (self) {
					printf ("ntrack backend selected and created: %s\n", dirent->d_name);
					ftbl_initialized = 1;
				}
			} else {
				dlclose (dlhandle);
			}
		}
		if (dir)
			closedir (dir);
	}

	if (!self)
		printf ("no backend found finale.\n");
	return self;
}

int*
ntrack_arch_get_rfds (ntrack_monitor_t *self)
{
	if (ftbl.get_rfds.func)
		return (*ftbl.get_rfds.func) (self);
	else
		printf ("no rfds func found, but called\n");

	return 0;
}

int
ntrack_arch_process_data (ntrackpointer self, int* fds)
{
	if (ftbl.process_data.func)
		return (*ftbl.process_data.func) (self, fds);
	else
		printf ("no process_data_func found, but called\n");

	return 0;
}

void
ntrack_arch_free (ntrackpointer self)
{
	if (ftbl.arch_free.func)
		(*ftbl.arch_free.func) (self);
	else
		printf ("no process_data_func found, but called\n");
	return;
}

