/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "LoadScene.h"

#include <glm/gtx/transform.hpp>
#include <boost/foreach.hpp>

namespace libgltf
{

Parser::Parser()
    : ptParse()
    , pScene(0)
{
}

Parser::~Parser()
{
}

void Parser::clearPropertyTree()
{
    ptParse.clear();
}

bool Parser::parseJsonFile(const std::string& jsonfile)
{
    try
    {
        boost::property_tree::json_parser::read_json(jsonfile, ptParse);
    }
    catch (boost::property_tree::ptree_error& e)
    {
        fprintf(stderr, "%s\n", e.what());
        return false;
    }
    catch (boost::exception const&)
    {
        fprintf(stderr, "%s\n", "unknown boost error");
        return false;
    }

    return true;
}

void Parser::getFileNamesInJson(std::vector<glTFFile>& o_glTFFiles)
{
    boost::property_tree::ptree pImagesTree;
    boost::property_tree::ptree pBuffersTree;
    boost::property_tree::ptree pShadersTree;

    if (isFoundPtree(ptParse, "images"))
    {
        pImagesTree = ptParse.get_child("images");
    }
    if (isFoundPtree(ptParse, "buffers"))
    {
        pBuffersTree = ptParse.get_child("buffers");
    }
    if (isFoundPtree(ptParse, "shaders"))
    {
        pShadersTree = ptParse.get_child("shaders");
    }

    o_glTFFiles.clear();
    o_glTFFiles.resize( pImagesTree.size() + pBuffersTree.size() +
                        pShadersTree.size() );

    unsigned int index = 0;
    for (boost::property_tree::ptree::iterator it = pImagesTree.begin();
         it != pImagesTree.end(); ++it)
    {
        boost::property_tree::ptree pImageTree = it->second;
        o_glTFFiles[index].type = GLTF_IMAGE;
        o_glTFFiles[index].filename = pImageTree.get<std::string>("path");
        ++index;
    }

    for (boost::property_tree::ptree::iterator it = pBuffersTree.begin();
         it != pBuffersTree.end(); ++it)
    {
        boost::property_tree::ptree pBufferTree = it->second;
        o_glTFFiles[index].type = GLTF_BINARY;
        o_glTFFiles[index].filename = pBufferTree.get<std::string>("path");
        ++index;
    }

    for (boost::property_tree::ptree::iterator it = pShadersTree.begin();
         it != pShadersTree.end(); ++it)
    {
        boost::property_tree::ptree pShaderTree = it->second;
        o_glTFFiles[index].type = GLTF_GLSL;
        o_glTFFiles[index].filename = pShaderTree.get<std::string>("path");
        ++index;
    }
}

void Parser::setScene(Scene* pscene)
{
    pScene = pscene;
}

int Parser::parseScene(const std::vector<glTFFile>& inputFiles)
{
    int status;
    status = readBuffers(inputFiles);
    if (status < 0)
    {
        return status;
    }

    if (ptParse.find("cameras") != ptParse.not_found())
    {
        if (!parseCameras())
        {
            return LIBGLTF_PARSE_CAMERA_ERROR;
        }
    }

    if (ptParse.find("lights") != ptParse.not_found())
    {
        if (!parseLights())
        {
            return LIBGLTF_PARSE_LIGHT_ERROR;
        }
    }

    if (!parseNodes())
    {
        return LIBGLTF_PARSE_NODE_ERROR;
    }

    if (!parseMeshs())
    {
        return LIBGLTF_PARSE_MESH_ERROR;
    }

    status = parseMaterials(inputFiles);
    if (status != LIBGLTF_SUCCESS)
    {
        return status;
    }

    if (!parseAttributes())
    {
        return LIBGLTF_PARSE_ATTRIBUTE_ERROR;
    }

    status = parseTechniques(inputFiles);
    if (status < 0)
    {
        return status;
    }

    if (ptParse.find("skins") != ptParse.not_found())
    {
        if (!parseSkins())
        {
            return LIBGLTF_PARSE_SKIN_ERROR;
        }
    }

    if (ptParse.find("animations") != ptParse.not_found())
    {
        if (!parseAnim())
        {
            return LIBGLTF_PARSE_ANIMATION_ERROR;
        }
    }

    pScene->removeBuffer();
    clearPropertyTree();
    return LIBGLTF_SUCCESS;
}

bool Parser::parseNodes()
{
    Node* pRootNode = new Node();
    const std::string& sceneId = ptParse.get<std::string>("scene");
    std::string pathNode = "scenes*" + sceneId + "*nodes";
    boost::property_tree::ptree& pNodeArray =
        ptParse.get_child(
        boost::property_tree::ptree::path_type(pathNode.c_str(), '*'));
    for (boost::property_tree::ptree::const_iterator it = pNodeArray.begin();
        it != pNodeArray.end(); ++it)
    {
        parseNode(it->second.data(), pRootNode,
                        pRootNode->getGlobalMatrix());
    }
    pScene->setRootNode(pRootNode);
    pNodeArray.clear();
    return true;
}

void Parser::parseNode(const std::string& id, Node* pParentNode,
                       const glm::mat4& parentMatrix)
{
    Node* pNode = new Node();
    const boost::property_tree::ptree& pNodeAtrr = ptParse.get_child(
        boost::property_tree::ptree::path_type(
        std::string("nodes*"+id).c_str(), '*'));
    boost::property_tree::ptree::const_assoc_iterator itAttr;

    pNode->setNodeName(id);
    itAttr = pNodeAtrr.find("jointId");
    if (itAttr != pNodeAtrr.not_found())
    {
        const std::string& jointId =
            itAttr->second.get_value<std::string>("jointId");
        pNode->setJointId(jointId);
        pNode->setJointFlag(true);
    }

    boost::property_tree::ptree::const_iterator it;
    boost::property_tree::ptree::const_iterator itEnd;
    itAttr = pNodeAtrr.find("meshes");
    if (itAttr != pNodeAtrr.not_found())
    {
        pNode->setNodeType(NodeType_Mesh);
        const boost::property_tree::ptree& pMeshes = itAttr->second;
        itEnd = pMeshes.end();
        for (it = pMeshes.begin(); it != itEnd; ++it)
        {
            pNode->pushMeshIndex(it->second.data());
        }
    }
    else if ((itAttr = pNodeAtrr.find("light")) != pNodeAtrr.not_found())
    {
        pNode->setLightIndex(itAttr->second.get_value<std::string>());
        pNode->setNodeType(NodeType_Light);
        pScene->insertLightNodeMap(id, pNode);
    }
    else if ((itAttr = pNodeAtrr.find("camera")) != pNodeAtrr.not_found())
    {
        pNode->setCameraIndex(itAttr->second.get_value<std::string>());
        pNode->setNodeType(NodeType_Camera);
        ParseCamera* pCamera =
            pScene->findCamera(itAttr->second.get_value<std::string>());
        if (pCamera != 0)
        {
            pCamera->setCameraNode(pNode);
        }
    }
    else if ((itAttr = pNodeAtrr.find("instanceSkin")) !=
             pNodeAtrr.not_found())
    {
        const boost::property_tree::ptree& pInstSkin = itAttr->second;
        const std::string& skinName = pInstSkin.get<std::string>("skin");
        pNode->setSkinIndex(skinName);

        const boost::property_tree::ptree& pSkeleTree =
            pInstSkin.get_child("skeletons");
        //All the "id" in pNodes are the same, so we only use the first one.
        it = pSkeleTree.begin();
        if (pSkeleTree.end() != it)
        {
            const std::string& skeletonId  = it->second.data();
            pNode->setSkeleIndex(skeletonId);
        }

        const boost::property_tree::ptree& pSources =
            pInstSkin.get_child("sources");
        itEnd =  pSources.end();
        for (it = pSources.begin(); it != itEnd; ++it)
        {
            pNode->pushMeshIndex(it->second.data());
        }
    }

    float buffer[16] = {0};
    glm::mat4 matrix(1.0f);
    itAttr = pNodeAtrr.find("matrix");
    if (itAttr != pNodeAtrr.not_found())
    {
        const boost::property_tree::ptree& pMatrix = itAttr->second;
        unsigned int index;
        itEnd = pMatrix.end();
        for (index = 0, it = pMatrix.begin(); it != itEnd; ++it)
        {
            buffer[index++] = (it->second).get_value<float>();
        }
        memcpy(&matrix, buffer, sizeof(glm::mat4));
    }
    else
    {
        unsigned int index;
        const boost::property_tree::ptree& pTrans =
            pNodeAtrr.get_child("translation");
        itEnd =  pTrans.end();
        for (index = 0, it = pTrans.begin(); it != itEnd; ++it)
        {
            buffer[index++] = (it->second).get_value<float>();
        }
        pNode->setTranslate(buffer);

        const boost::property_tree::ptree& pRotate =
            pNodeAtrr.get_child("rotation");
        itEnd = pRotate.end();
        for (index = 0, it = pRotate.begin(); it != itEnd; ++it)
        {
            buffer[index++] = (it->second).get_value<float>();
        }
        pNode->setRotate(buffer);

        const boost::property_tree::ptree& pScale =
            pNodeAtrr.get_child("scale");
        itEnd = pScale.end();
        for (index = 0, it = pScale.begin(); it != itEnd; ++it)
        {
            buffer[index++] = (it->second).get_value<float>();
        }
        pNode->setScale(buffer);

        matrix = pNode->getTranslate() * pNode->getRotate() *
                 pNode->getScale();
        pNode->setMatrixFlag(false);
    }
    pNode->setLocalMatrix(matrix);
    matrix = parentMatrix * matrix;
    pNode->setGlobalMatrix(matrix);

    pParentNode->pushChildNode(pNode);
    pNode->setParentNode(pParentNode);
    pScene->pushNode(pNode);

    itAttr = pNodeAtrr.find("children");

    if (itAttr != pNodeAtrr.not_found() && 0 !=
        pNodeAtrr.get_child("children").size())
    {
        pNode->setNodeType(NodeType_Node);
        const boost::property_tree::ptree& pChildren = itAttr->second;
        itEnd = pChildren.end();
        for (it = pChildren.begin(); it != itEnd; ++it)
        {
            parseNode(it->second.data(), pNode, matrix);
        }
    }
}

bool Parser::parseSkins()
{
    boost::property_tree::ptree& pSkinsTree = ptParse.get_child("skins");
    boost::property_tree::ptree::const_iterator itor = pSkinsTree.begin();
    boost::property_tree::ptree::const_iterator itorEnd =  pSkinsTree.end();
    for (; itor != itorEnd; ++itor)
    {
        Skin* pSkin = new Skin();
        pSkin->setSkinName(itor->first);
        const boost::property_tree::ptree& pSkinTree = itor->second;
        const boost::property_tree::ptree& pShapeTree =
            pSkinTree.get_child("bindShapeMatrix");
        float buffer[16];
        unsigned int i = 0;
        boost::property_tree::ptree::const_iterator it;
        boost::property_tree::ptree::const_iterator itEnd = pShapeTree.end();
        for (it = pShapeTree.begin(); it != itEnd; ++it)
        {
            buffer[i++] = (it->second).get_value<float>();
        }

        glm::mat4 shape(1.0);
        memcpy(&shape, buffer, sizeof(glm::mat4));

        const boost::property_tree::ptree& pBind =
            pSkinTree.get_child("inverseBindMatrices");
        std::string bufferView =
            "bufferViews." + pBind.get<std::string>("bufferView");
        const boost::property_tree::ptree& pBufferView =
            ptParse.get_child(bufferView.c_str());
        char* pbuffer = pScene->getBuffer()
            + pBufferView.get<unsigned int>("byteOffset")
            + pBind.get<unsigned int>("byteOffset");
        unsigned int count = pBind.get<unsigned int>("count");
        glm::mat4* pBindArray = new glm::mat4[count];
        for (unsigned int j = 0; j < count; j++)
        {
            memcpy(&pBindArray[j], pbuffer + j * 64, sizeof(glm::mat4));
            pBindArray[j] = pBindArray[j] * shape;
        }
        pSkin->setBindMatrix(pBindArray);
        pSkin->setBindMatrixCount(count);

        const boost::property_tree::ptree& pJoint =
            pSkinTree.get_child("joints");
        itEnd = pJoint.end();
        for (it = pJoint.begin(); it != itEnd; ++it)
        {
            pSkin->pushBoneId((it->second).get_value<std::string>());
        }
        pScene->pushSkin(pSkin);
    }
    pSkinsTree.clear();
    return true;
}

bool Parser::parseMeshs()
{
    boost::property_tree::ptree& pMeshsTree =
        ptParse.get_child("meshes");
    boost::property_tree::ptree::const_iterator it = pMeshsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd =  pMeshsTree.end();
    for (;it != itEnd; ++it)
    {
        Mesh* pMesh = new Mesh();
        const boost::property_tree::ptree& pMeshTree = it->second;
        pMesh->setMeshName(pMeshTree.get<std::string>("name"));
        parsePrimitive(pMeshTree.get_child("primitives"), pMesh);

        pScene->insertMeshMap(it->first, pMesh);
    }
    pMeshsTree.clear();
    return true;
}

bool Parser::parsePrimitive(const boost::property_tree::ptree& pTree,
                            Mesh* pMesh)
{
    boost::property_tree::ptree::const_iterator it = pTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pTree.end();
    for (; it != itEnd; ++it)
    {
        Primitives* pPrim = new Primitives();

        const boost::property_tree::ptree& pPrimTree = it->second;
        const boost::property_tree::ptree& pPrimValueTree =
          pPrimTree.get_child("attributes");
        boost::property_tree::ptree::const_assoc_iterator itPVT =
          pPrimValueTree.find("NORMAL");
        if (itPVT != pPrimValueTree.not_found())
        {
            pPrim->insertAttribute("NORMAL",
                                   itPVT->second.get_value<std::string>());
        }
        itPVT = pPrimValueTree.find("POSITION");
        if (itPVT != pPrimValueTree.not_found())
        {
            pPrim->insertAttribute("POSITION",
                                   itPVT->second.get_value<std::string>());
        }
        itPVT = pPrimValueTree.find("TEXCOORD_0");
        if (itPVT != pPrimValueTree.not_found())
        {
            pPrim->insertAttribute("TEXCOORD_0",
                                   itPVT->second.get_value<std::string>());
        }
        itPVT = pPrimValueTree.find("JOINT");
        if (itPVT != pPrimValueTree.not_found())
        {
            pPrim->insertAttribute("JOINT",
                                   itPVT->second.get_value<std::string>());
        }
        itPVT = pPrimValueTree.find("WEIGHT");
        if (itPVT != pPrimValueTree.not_found())
        {
            pPrim->insertAttribute("WEIGHT",
                                   itPVT->second.get_value<std::string>());
        }
        pPrim->setIndicesIndex(pPrimTree.get<std::string>("indices"));
        pPrim->setMaterialIndex(pPrimTree.get<std::string>("material"));
        pMesh->setPrimitiveVec(pPrim);
    }
    return true;
}

int Parser::parseMaterials(const std::vector<glTFFile>& inputFiles)
{
    boost::property_tree::ptree& pMaterialsTree =
        ptParse.get_child("materials");
    boost::property_tree::ptree::const_iterator it = pMaterialsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pMaterialsTree.end();
    for (; it != itEnd; ++it)
    {
        Material* pMaterial = new Material();

        const boost::property_tree::ptree& pMaterialTree = it->second;
        const std::string& techniqueId =
            pMaterialTree.get<std::string>("instanceTechnique.technique");
        int nRet = parseMaterialProper(
            pMaterialTree.get_child("instanceTechnique.values"),
            pMaterial, techniqueId, inputFiles);

        if( nRet != LIBGLTF_SUCCESS )
            return nRet;

        pMaterial->setTechniqueId(techniqueId);
        pScene->insertMaterialMap(it->first, pMaterial);
    }
    pMaterialsTree.clear();
    return LIBGLTF_SUCCESS;
}

int Parser::parseMaterialProper(const boost::property_tree::ptree& pTree,
                                 Material* pMaterial,
                                 const std::string& techniqueId,
                                 const std::vector<glTFFile>& inputFiles)
{
    std::string properName;
    boost::property_tree::ptree::const_iterator it = pTree.begin();
    boost::property_tree::ptree::const_iterator itEnd =  pTree.end();
    for (; it != itEnd; ++it)
    {
        MaterialProperty* pMaterialProper = new MaterialProperty();

        std::string typePath = "techniques*" + techniqueId +
                               "*parameters*" + it->first + "*type";
        unsigned int type = ptParse.get<unsigned int>(
            boost::property_tree::ptree::path_type(
            typePath.c_str(), '*'));
        properName = "u_" + it->first;
        pMaterialProper->setPropertyName(properName);
        switch (type)
        {
            case DataType_SAMPLER_2D:
                {
                    const std::string& textureId = it->second.data();
                    std::string textureSource =
                        "textures*" + textureId + "*source";
                    const std::string& imageId =
                        ptParse.get<std::string>(
                        boost::property_tree::ptree::path_type(
                        textureSource.c_str(), '*'));
                    std::string imagePath = "images*" + imageId + "*path";
                    pMaterialProper->setImagePath(
                        ptParse.get<std::string>(
                        boost::property_tree::ptree::path_type(
                        imagePath.c_str(), '*')));
                    pMaterialProper->setDataLength(0);
                    pMaterialProper->setDataType(DataType_SAMPLER_2D);
                    int nRet = pScene->loadTexture(pMaterialProper->getImagePath(), inputFiles);
                    if( nRet != LIBGLTF_SUCCESS )
                        return nRet;
                }
                break;
            case DataType_FLOAT:
                {
                    pMaterialProper->setDataLength(sizeof(float));
                    // todo=qulei design cause maybe memory leak;
                    float data = 0;
                    data = (it->second).get_value<float>();
                    pMaterialProper->setPropertyData((char*)&data,
                                                     sizeof(float));
                    pMaterialProper->setDataType(DataType_FLOAT);
                }
                break;
            case DataType_FLOAT_VEC4:
                {
                    pMaterialProper->setDataLength(sizeof(float) * 4);
                    const boost::property_tree::ptree& pValue = it->second;
                    float buffer[4] = {0};
                    boost::property_tree::ptree::const_iterator iter;
                    iter = pValue.begin();
                    boost::property_tree::ptree::const_iterator iterEnd;
                    iterEnd = pValue.end();
                    for (unsigned int i = 0; (i < 4) && (iter != iterEnd);
                         ++iter, ++i)
                    {
                        buffer[i] = (iter->second).get_value<float>();
                    }
                    pMaterialProper->setPropertyData((char*)buffer,
                                                     4*sizeof(float));
                    pMaterialProper->setDataType(DataType_FLOAT_VEC4);
                }
                break;
            default:
                break;
        }
        pMaterial->pushMaterialProper(pMaterialProper);
    }

    return LIBGLTF_SUCCESS;
}

bool Parser::parseCameras()
{
    boost::property_tree::ptree& pCamerasTree =
        ptParse.get_child("cameras");
    boost::property_tree::ptree::const_iterator it = pCamerasTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pCamerasTree.end();
    for (; it != itEnd; ++it)
    {
        ParseCamera* pCamera = new ParseCamera();

        const boost::property_tree::ptree& pCameraTree = it->second;
        const std::string& type = pCameraTree.get<std::string>("type");
        const boost::property_tree::ptree& pCameraAttrTree =
            pCameraTree.get_child(type.c_str());
        boost::property_tree::ptree::const_assoc_iterator itAttr;
        itAttr = pCameraAttrTree.find("aspect_ratio");
        if (pCameraAttrTree.find("aspect_ratio") ==
            pCameraAttrTree.not_found())
        {
            if (pCameraAttrTree.find("xfov") != pCameraAttrTree.not_found())
            {
                pCamera->setXFov(pCameraAttrTree.get<float>("xfov"));
            }
            if (pCameraAttrTree.find("yfov") != pCameraAttrTree.not_found())
            {
                pCamera->setYFov(pCameraAttrTree.get<float>("yfov"));
            }
        }
        else
        {
            pCamera->setAspectRatio(pCameraAttrTree.get<float>("aspect_ratio"));
            if (pCameraAttrTree.find("xfov") != pCameraAttrTree.not_found())
            {
                pCamera->setXFov(pCameraAttrTree.get<float>("xfov"));
                pCamera->setYFov(pCamera->getXFov() *
                                 pCamera->getAspectRatio());
            }
            if (pCameraAttrTree.find("yfov") != pCameraAttrTree.not_found())
            {
                pCamera->setYFov(pCameraAttrTree.get<float>("yfov"));
                pCamera->setXFov(pCamera->getYFov() /
                                 pCamera->getAspectRatio());
            }
        }
        pCamera->setFar(pCameraAttrTree.get<float>("zfar"));
        pCamera->setNear(pCameraAttrTree.get<float>("znear"));
        pScene->insertCameraMap(it->first, pCamera);
        pScene->setUseCameraInJson(true);
    }
    pCamerasTree.clear();
    return true;
}

Light* GetParseLight(const boost::property_tree::ptree&  pParseLight,
                     LightSourceType NodeType)
{
    unsigned int i;
    float fLight[3] = {0};
    Light* pLight = new Light();
    pLight->setType(NodeType);
    boost::property_tree::ptree::const_iterator it;
    boost::property_tree::ptree::const_iterator itEnd;
    boost::property_tree::ptree::const_assoc_iterator itL =
        pParseLight.find("color");
    if (itL != pParseLight.not_found())
    {
        i = 0;
        const boost::property_tree::ptree& color = itL->second;
        it = color.begin();
        itEnd = color.end();
        for (; it != itEnd; ++it, ++i)
        {
            fLight[i] = (it->second).get_value<float>();
        }
        const glm::vec3 lightcolor(fLight[0],fLight[1], fLight[2]);
        pLight->setColor(lightcolor);
    }
    if ((itL = pParseLight.find("constantAttenuation")) !=
        pParseLight.not_found())
    {
        pLight->setAttenuationConstant((itL->second).get_value<float>());
    }
    if ((itL = pParseLight.find("linearAttenuation")) !=
        pParseLight.not_found())
    {
        pLight->setAttenuationLinear((itL->second).get_value<float>());
    }
    if ((itL = pParseLight.find("quadraticAttenuation")) !=
        pParseLight.not_found())
    {
        pLight->setAttenuationQuadratic((itL->second).get_value<float>());
    }
    return pLight;
}

bool Parser::parseLights()
{
    boost::property_tree::ptree& pLightsTree =
        ptParse.get_child("lights");
    boost::property_tree::ptree::const_iterator it = pLightsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd =  pLightsTree.end();
    for (; it != itEnd; ++it)
    {
        Light* pLight = 0;
        const boost::property_tree::ptree& pLightTree = it->second;
        const std::string strLightName = it->first;
        boost::property_tree::ptree::const_assoc_iterator itL =
            pLightTree.find("type");
        if (itL != pLightTree.not_found())
        {
            std::string strType = pLightTree.get<std::string>("type");
            const boost::property_tree::ptree& tree =
                pLightTree.get_child(strType);
            if (strType == "point")
            {
                pLight = GetParseLight(tree, LightSource_POINT);
            }
            else if (strType == "directional")
            {
                pLight = GetParseLight(tree, LightSource_DIRECTIONAL);
            }
            else if (strType == "ambient")
            {
                pLight = GetParseLight(tree, LightSource_AMBIET);
            }
            else if (strType == "spot")
            {
                pLight = GetParseLight(tree, LightSource_SPOT);
            }
            else if (strType == "undefined")
            {
                pLight = GetParseLight(tree, LightSource_UNDEFINED);
            }
        }
        if (pLight)
        {
            pLight->setLightName(strLightName);
            pScene->insertLightMap(it->first, pLight);
        }
        pScene->insertLightMap(it->first, pLight);
    }
    pLightsTree.clear();
    return true;
}

bool Parser::parseAttributes()
{
    boost::property_tree::ptree& pAttrsTree =
        ptParse.get_child("accessors");
    boost::property_tree::ptree::const_iterator it = pAttrsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pAttrsTree.end();
    for (; it != itEnd; ++it)
    {
        Attribute* pAttribute = new Attribute();
        const boost::property_tree::ptree& pAttrTree = it->second;
        unsigned int type = pAttrTree.get<unsigned int>("type");
        pAttribute->setDataType((enum DataType)type);

        unsigned int byteStride;
        switch (type)
        {
            case DataType_UNSIGNED_SHORT:
                byteStride = 2;
                break;
            case DataType_FLOAT:
                byteStride = 4;
                break;
            case DataType_FLOAT_VEC2:
                byteStride = 8;
                break;
            case DataType_FLOAT_VEC3:
                byteStride = 12;
                break;
            case DataType_FLOAT_VEC4:
                byteStride = 16;
                break;
            default:
                byteStride = 4;
                break;
        }
        pAttribute->setByteStride(byteStride);

        unsigned int count = pAttrTree.get<unsigned int>("count");
        pAttribute->setDataCount(count);

        std::string bufferView = "bufferViews*" +
                                 pAttrTree.get<std::string>("bufferView");
        const boost::property_tree::ptree& pBufferView =
            ptParse.get_child(
            boost::property_tree::ptree::path_type(bufferView.c_str(), '*'));
        char* pbuffer = pScene->getBuffer()
                        + pBufferView.get<unsigned int>("byteOffset")
                        + pAttrTree.get<unsigned int>("byteOffset");
        pAttribute->setAttributeData(pbuffer, count * byteStride);

        if (DataType_FLOAT_VEC3 == type)
        {
           boost::property_tree::ptree::const_assoc_iterator itAttr;
           itAttr = pAttrTree.find("max");
           if (itAttr != pAttrTree.not_found())
           {
              float max[3] = {0.f, 0.f, 0.f};
              const boost::property_tree::ptree& pMaxTree = itAttr->second;
              boost::property_tree::ptree::const_iterator iter;
              iter = pMaxTree.begin();
              boost::property_tree::ptree::const_iterator iterEnd;
              iterEnd = pMaxTree.end();
              for (unsigned int i = 0; (iter != iterEnd) && (i<3); ++iter, ++i)
              {
                  max[i] = (iter->second).get_value<float>();
              }
              pScene->setVertexMax(max[0], max[1], max[2]);
          }
          itAttr = pAttrTree.find("min");
          if (itAttr != pAttrTree.not_found())
          {
              float min[3] = {0.f, 0.f, 0.f};
              const boost::property_tree::ptree& pMinTree = itAttr->second;
              boost::property_tree::ptree::const_iterator iter;
              iter = pMinTree.begin();
              boost::property_tree::ptree::const_iterator iterEnd;
              iterEnd = pMinTree.end();
              for (unsigned int i = 0; (iter != iterEnd) && (i<3); ++iter, ++i)
              {
                  min[i++] = (iter->second).get_value<float>();
              }
              pScene->setVertexMin(min[0], min[1], min[2]);
          }
        }
        pScene->insertAttributeMap(it->first, pAttribute);
    }
    pAttrsTree.clear();
    return true;
}

int Parser::parseTechniques(const std::vector<glTFFile>& inputFiles)
{
    boost::property_tree::ptree& pTechniquesTree =
        ptParse.get_child("techniques");
    boost::property_tree::ptree::const_iterator it = pTechniquesTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pTechniquesTree.end();
    for (; it != itEnd; ++it)
    {
        Technique* pTechnique = new Technique();

        const boost::property_tree::ptree& pTechniqueTree = it->second;
        const std::string& passId = pTechniqueTree.get<std::string>("pass");
        std::string passPath = "passes*" + passId + "*instanceProgram";
        const boost::property_tree::ptree& pInstanceTree =
            pTechniqueTree.get_child(
            boost::property_tree::ptree::path_type(passPath.c_str(), '*'));
        const boost::property_tree::ptree& pParameterTree =
            pTechniqueTree.get_child("parameters");
        parseTechniqueLight(pInstanceTree, pParameterTree, pTechnique);
        int status = parseTechniqueProgram(pInstanceTree,
            pParameterTree, pTechnique, inputFiles);
        if (status != LIBGLTF_SUCCESS)
            return status;
        std::string statePath = "passes*" + passId + "*states";
        boost::property_tree::ptree pStateTree =
            pTechniqueTree.get_child(
            boost::property_tree::ptree::path_type(statePath.c_str(), '*'));
        parseTechniqueState(pStateTree, pTechnique);
        pTechnique->setTechId(it->first);
        pScene->pushTechnique(pTechnique);
    }
    pTechniquesTree.clear();
    return LIBGLTF_SUCCESS;
}

bool Parser::parseTechniqueState(boost::property_tree::ptree& pInstanceTree,
                                 Technique* pTechnique)
{
    for (boost::property_tree::ptree::iterator it = pInstanceTree.begin();
         it != pInstanceTree.end(); ++it)
    {
        if ("blendFunc" == it->first)
        {
            boost::property_tree::ptree dfactorNode =
                pInstanceTree.get_child("blendFunc.dfactor");
            boost::property_tree::ptree sfactorNode =
                pInstanceTree.get_child("blendFunc.sfactor");
            pTechnique->getTechState()->blendFuncDfactor =
                dfactorNode.get_value<unsigned int>();
            pTechnique->getTechState()->blendFuncSfactor =
                sfactorNode.get_value<unsigned int>();
            continue;
        }
        if ("blendEnable" == it->first)
        {
            pTechnique->getTechState()->blendEnable =
                it->second.get_value<unsigned int>();
            continue;
        }
        if ("blendEquation" == it->first)
        {
            pTechnique->getTechState()->blendEquation =
                it->second.get_value<unsigned int>();
            continue;
        }
        if ("cullFaceEnable" == it->first)
        {
            pTechnique->getTechState()->cullFaceEnable =
                it->second.get_value<unsigned int>();
            continue;
        }
        if ("depthMask" == it->first)
        {
            pTechnique->getTechState()->depthMask =
                it->second.get_value<unsigned int>();
            continue;
        }
        if ("depthTestEnable" == it->first)
        {
            pTechnique->getTechState()->depthTestEnable =
                it->second.get_value<unsigned int>();
            continue;
        }
    }
    return true;
}

int Parser::parseTechniqueProgram(
    const boost::property_tree::ptree& pInstanceTree,
    const boost::property_tree::ptree& pParameterTree,
    Technique* pTechnique, const std::vector<glTFFile>& inputFiles)
{
    boost::property_tree::ptree pAttrsTree =
        pInstanceTree.get_child("attributes");
    boost::property_tree::ptree::const_iterator it = pAttrsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pAttrsTree.end();
    for (; it != itEnd; ++it)
    {
        const std::string& parameterId = it->second.data();
        const boost::property_tree::ptree& pParaTree =
            pParameterTree.get_child(parameterId.c_str());
        boost::property_tree::ptree::const_assoc_iterator itPT;
        itPT = pParaTree.find("semantic");
        if (itPT != pParaTree.not_found())
        {
            TechAttribute* pTechAttribute = new TechAttribute();
            pTechAttribute->setAttributeIndex(
               itPT->second.get_value<std::string>());
            pTechAttribute->setAttributeName(it->first);
            pTechnique->insertTechAttribute(it->first, pTechAttribute);
        }
    }
    const boost::property_tree::ptree& pUniformsTree =
        pInstanceTree.get_child("uniforms");
    it = pUniformsTree.begin();
    itEnd = pUniformsTree.end();
    for (; it != itEnd; ++it)
    {
        TechUniform* pTechUniform = new TechUniform();
        pTechUniform->setUniformIndex(it->second.data());
        pTechUniform->setUniformName(it->first);
        pTechnique->pushTechUniform(pTechUniform);
    }
    std::string programId = "programs*" +
                            pInstanceTree.get<std::string>("program");

    const boost::property_tree::ptree& pProgramTree = ptParse.get_child(
        boost::property_tree::ptree::path_type(programId.c_str(), '*'));
    std::string verId = "shaders*" +
                        pProgramTree.get<std::string>("vertexShader") +
                        "*path";
    std::string fraId = "shaders*" +
                        pProgramTree.get<std::string>("fragmentShader") +
                        "*path";
    pTechnique->setVertexShader(ptParse.get<std::string>(
        boost::property_tree::ptree::path_type(verId.c_str(), '*')));
    pTechnique->setFragmentShader(ptParse.get<std::string>(
        boost::property_tree::ptree::path_type(fraId.c_str(), '*')));
    return pTechnique->initTechnique(inputFiles);
}

bool Parser:: parseTechniqueLight(
    const boost::property_tree::ptree& pInstanceTree,
    const boost::property_tree::ptree& pParameterTree,
    Technique* pTechnique)
{
    const boost::property_tree::ptree& pUniformsTree =
        pInstanceTree.get_child("uniforms");
    boost::property_tree::ptree::const_iterator it = pUniformsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pUniformsTree.end();
    for (; it != itEnd; ++it)
    {
        const std::string& parameterId = it->second.data();
        const boost::property_tree::ptree& pParaTree =
            pParameterTree.get_child(parameterId.c_str());
        boost::property_tree::ptree::const_assoc_iterator itPT;
        itPT = pParaTree.find("source");
        if (itPT != pParaTree.not_found())
        {
            techLight *ptLight = new techLight();
            ptLight->mName = it->first;
            ptLight->mSource = itPT->second.get_value<std::string>();
            pTechnique->pushTLight(ptLight);
        }
        itPT = pParaTree.find("value");
        if (itPT != pParaTree.not_found())
        {
            unsigned int type = pParaTree.get<unsigned int>("type");
            if (type == DataType_FLOAT_VEC3)
            {
                int i=0;
                float fValue[3] = {0};
                techLight *ptLight = new techLight();
                ptLight->mName = it->first;
                const boost::property_tree::ptree pValueTree = itPT->second;
                boost::property_tree::ptree::const_iterator it2 = pValueTree.begin();
                boost::property_tree::ptree::const_iterator itEnd2 = pValueTree.end();
                for (; it2 != itEnd2; ++it2, ++i)
                    fValue[i] = (it2->second).get_value<float>();
                const glm::vec3 vValue(fValue[0], fValue[1], fValue[2]);
                ptLight->vecValue = vValue;
                ptLight->type = DataType_FLOAT_VEC3;
                pTechnique->pushTLight(ptLight);
            }
            if (type == DataType_FLOAT)
            {
                techLight *ptLight = new techLight();
                ptLight->mName = it->first;
                ptLight->type = DataType_FLOAT;
                ptLight->floatValue = pParaTree.get<float>("value");
                pTechnique->pushTLight(ptLight);
            }
        }
    }
    return true;
}

bool Parser::parseAnim()
{
    boost::property_tree::ptree pAnimsTree =
        ptParse.get_child("animations");
    boost::property_tree::ptree::const_iterator it = pAnimsTree.begin();
    boost::property_tree::ptree::const_iterator itEnd = pAnimsTree.end();
    std::string nodeId;
    for (; it != itEnd; ++it)
    {
        boost::property_tree::ptree pAnimTree = it->second;
        nodeId = parseChannel(pAnimTree);
        if (nodeId.empty())
        {
            continue;
        }
        Animation* pAnimation = new Animation();
        parsePara(pAnimTree, pAnimation, nodeId);
        pScene->insertAnimMap(nodeId, pAnimation);
    }
    pAnimsTree.clear();
    return true;
}

std::string Parser::parseChannel(const boost::property_tree::ptree& pTree)
{
    std::string id;
    const boost::property_tree::ptree& pChannsTree =
        pTree.get_child("channels");
    //All the "id" in pNodes are the same, so we only use the first one.
    boost::property_tree::ptree::const_iterator it = pChannsTree.begin();
    if (pChannsTree.end() != it)
    {
        const boost::property_tree::ptree pChannTree = it->second;
        id = pChannTree.get<std::string>("target.id");
    }
    return id;
}

int Parser::parsePara(const boost::property_tree::ptree& pTree,
                      Animation* pAnimation, const std::string& nodeId)
{
    unsigned int count = pTree.get<unsigned int>("count");
    const std::string& timeIndex = pTree.get<std::string>("parameters.TIME");
    const Attribute* pTimeAttr = pScene->findAttribute(timeIndex);
    if (0 == pTimeAttr)
    {
        return -1;
    }
    const float* pTimeBuf = (const float*)pTimeAttr->getAttributeData();

    std::string nodePath = "nodes*" + nodeId;
    boost::property_tree::ptree pNode = ptParse.get_child(
        boost::property_tree::ptree::path_type(nodePath.c_str(), '*'));
    double duration = pTimeBuf[count-1]-pTimeBuf[0];
    pAnimation->setDuration(duration);
    if (pNode.find("jointId") != pNode.not_found() &&
        duration > pScene->getDuration())
    {
        pScene->setDuration(duration);
    }

    const boost::property_tree::ptree pParaTree =
        pTree.get_child("parameters");

    glm::mat4 dMatrix = glm::mat4(1.0);
    for (unsigned int index = 0; index < count; ++index)
    {
        pAnimation->pushTimeValue(pTimeBuf[index], dMatrix);
    }

    boost::property_tree::ptree::const_assoc_iterator itAttr;
    itAttr = pParaTree.find("translation");
    if (itAttr != pParaTree.not_found())
    {
        const std::string& transIndex =
            (itAttr->second).get_value<std::string>("translation");
        const Attribute* pTransAttr = pScene->findAttribute(transIndex);
        if (0 != pTransAttr)
        {
            glm::vec3 value;
            const char* pTransBuf  = pTransAttr->getAttributeData();
            for (unsigned int index = 0, sizeVec3 = sizeof(glm::vec3);
                 index < count; ++index)
            {
                memcpy(&value, pTransBuf+index * sizeVec3, sizeVec3);
                dMatrix = pAnimation->getTimeValue(index) *
                          glm::translate(value);
                pAnimation->setTimeValue(dMatrix, index);
            }
            pAnimation->setChannelBits(TRANS_CHANNEL);
        }
    }
    itAttr = pParaTree.find("rotation");
    if (itAttr != pParaTree.not_found())
    {
        const std::string& rotaIndex =
            (itAttr->second).get_value<std::string>("rotation");
        const Attribute* pRotaAttr = pScene->findAttribute(rotaIndex);
        if (0 != pRotaAttr)
        {
            glm::vec4 value;
            const char* pRotaBuf  = pRotaAttr->getAttributeData();
            for (unsigned int index = 0, sizeVec4 = sizeof(glm::vec4);
                 index < count; ++index)
            {
                memcpy(&value, pRotaBuf + index * sizeVec4, sizeVec4);
                dMatrix = pAnimation->getTimeValue(index) *
                          glm::rotate(value.w * 57, glm::vec3(value));
                pAnimation->setTimeValue(dMatrix, index);
            }
            pAnimation->setChannelBits(ROTATE_CHANNEL);
        }
    }
    itAttr = pParaTree.find("scale");
    if (itAttr != pParaTree.not_found())
    {
        const std::string& scaleIndex =
            (itAttr->second).get_value<std::string>("scale");
        const Attribute* pScaleAttr = pScene->findAttribute(scaleIndex);
        if (0 != pScaleAttr)
        {
            glm::vec3 value;
            const char* pScaleBuf  = pScaleAttr->getAttributeData();
            for (unsigned int index = 0, sizeVec3 = sizeof(glm::vec3);
                 index < count; ++index)
            {
                memcpy(&value, pScaleBuf + index * sizeVec3, sizeVec3);
                dMatrix = pAnimation->getTimeValue(index) * glm::scale(value);
                pAnimation->setTimeValue(dMatrix, index);
            }
            pAnimation->setChannelBits(SCALE_CHANNEL);
        }
    }
    return 0;
}

int Parser::readBuffers(const std::vector<glTFFile>& inputFiles)
{
    unsigned int length = 0;
    std::string binName;
    const  boost::property_tree::ptree& pBuffers =
        ptParse.get_child("buffers");
    boost::property_tree::ptree::const_iterator it = pBuffers.begin();
    boost::property_tree::ptree::const_iterator itEnd = pBuffers.end();
    for (; it != itEnd; ++it)
    {
        const boost::property_tree::ptree& pbuffer = it->second;
        binName = pbuffer.get<std::string>("path");
        length = pbuffer.get<unsigned int>("byteLength");
    }
    return pScene->setBuffer(binName, length, inputFiles);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
