/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
* This file is part of the libgltf project.
*
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

#include "Common.h"
#include <cstring>
#include <glm/gtx/transform.hpp>

namespace libgltf
{

int gltf_get_file_index_by_name(const std::vector<glTFFile>& inputFiles, const std::string& name)
{
    for (unsigned int i = 0; i < inputFiles.size(); i++)
    {
        if ( name == inputFiles[i].filename && inputFiles[i].size > 0 )
        {
            return (int)i;
        }
    }
    return LIBGLTF_FILE_NOT_LOAD;
}

/* -- Attribute -- */
Attribute::Attribute()
    : mByteStride(0)
    , mDataType(DataType_UNKNOW)
    , mCount(0)
    , pData(0)
{
}

Attribute::~Attribute()
{
    delete[] pData;
}

void Attribute::setDataCount(unsigned int count)
{
    mCount = count;
}

unsigned int Attribute::getDataCount() const
{
    return mCount;
}

void Attribute::setByteStride(unsigned int byteStride)
{
    mByteStride = byteStride;
}

unsigned int Attribute::getByteStride() const
{
    return mByteStride;
}

void Attribute::setDataType(DataType type)
{
    mDataType = type;
}

DataType Attribute::getDataType() const
{
    return mDataType;
}

void Attribute::setAttributeData(const char* pValue, unsigned int length)
{
    if (0 == pData)
    {
        pData = new char[length];
    }
    memcpy(pData, pValue, length);
}

const char* Attribute::getAttributeData() const
{
    return pData;
}

/* -- Light -- */
Light::Light()
    : mName("light0Color")
    , mType(LightSource_UNDEFINED)
    , mAttenuationConstant(0)
    , mAttenuationLinear(0)
    , mAttenuationQuadratic(0)
    , mColor(1.0f, 1.0f, 1.0f)
{

}

Light::Light(const Light& light)
    : mName(light.mName)
    , mType(light.mType)
    , mAttenuationConstant(light.mAttenuationConstant)
    , mAttenuationLinear(light.mAttenuationLinear)
    , mAttenuationQuadratic(light.mAttenuationQuadratic)
    , mColor(light.mColor)
{
}

Light::~Light()
{

}

void Light::setLightName(std::string strName)
{
    mName = strName;
}

std::string Light::getLightName() const
{
    return mName;
}

void Light::setType(LightSourceType Type)
{
    mType = Type;
}

void Light::setColor(glm::vec3 Color)
{
    mColor = Color;
}

glm::vec3 Light::getColor() const
{
    return mColor;
}

void Light::setAttenuationConstant(float AttenuationConstant)
{
    mAttenuationConstant = AttenuationConstant;
}

float Light::getAttenuationConstant() const
{
    return mAttenuationConstant;
}

void Light::setAttenuationLinear(float AttenuationLinear)
{
    mAttenuationLinear = AttenuationLinear;
}

float Light::getmAttenuationLinear() const
{
    return mAttenuationLinear;
}

void Light::setAttenuationQuadratic(float AttenuationQuadratic)
{
    mAttenuationQuadratic = AttenuationQuadratic;
}

float Light::getAttenuationQuadratic() const
{
    return mAttenuationQuadratic;
}

/* -- Camera -- */
ParseCamera::ParseCamera()
    : mName()
    , mXFov(37.8492f)
    , mYFov(1.0f)
    , mNear(1.0f)
    , mFar(500000.0f)
    , mAspectRatio(1.5f)
    , cameraNode(0)
{
}

ParseCamera::~ParseCamera()
{
}

void ParseCamera::setXFov(float xFov)
{
    mXFov = xFov;
}

float ParseCamera::getXFov()
{
    return mXFov;
}

void ParseCamera::setYFov(float yFov)
{
    mYFov = yFov;
}

float ParseCamera::getYFov()
{
    return mYFov;
}

void ParseCamera::setFar(float far)
{
    mFar = far;
}

float ParseCamera::getFar()
{
    return mFar;
}

void ParseCamera::setNear(float near)
{
    mNear = near;
}

float ParseCamera::getNear()
{
    return mNear;
}

void ParseCamera::setAspectRatio(float aspectRatio)
{
    mAspectRatio = aspectRatio;
}

float ParseCamera::getAspectRatio()
{
    return mAspectRatio;
}

void ParseCamera::setCameraNode(Node* pNode)
{
    cameraNode = pNode;
}

Node* ParseCamera::getCameraNode()
{
    return cameraNode;
}

/* -- TechAttribute -- */
TechAttribute::TechAttribute()
    : mAttributeName()
    , mAttributeIndex()
{
}

TechAttribute::~TechAttribute()
{
}

void TechAttribute::setAttributeName(const std::string& name)
{
    mAttributeName = name;
}

const std::string TechAttribute::getAttributeName()
{
    return mAttributeName;
}

void TechAttribute::setAttributeIndex(const std::string& index)
{
    mAttributeIndex = index;
}

const std::string TechAttribute::getAttributeIndex()
{
    return mAttributeIndex;
}

/* -- TechUniform -- */
TechUniform::TechUniform()
    : mUniformName()
    , mUniformIndex()
{
}

TechUniform::~TechUniform()
{
}

void TechUniform::setUniformName(const std::string& name)
{
    mUniformName = name;
}

void TechUniform::setUniformIndex(const std::string& index)
{
    mUniformIndex = index;
}

/* -- Technique -- */
Technique::Technique()
    : mShaderProg()
    , mTechAttrMap()
    , mTechUnifVec()
    , mTLight()
    , mProgramId(0)
    , mTechniqueId()
    , mVShaderName()
    , mFShaderName()
    , mTechniqueState(false)
    , pState(new TechniqueState())
{
}

Technique::~Technique()
{
    std::vector<TechUniform*>::iterator vit;
    for (vit = mTechUnifVec.begin(); vit != mTechUnifVec.end(); ++vit)
    {
        delete(*vit);
    }

    std::map<std::string, TechAttribute*>::iterator mit;
    for (mit = mTechAttrMap.begin(); mit != mTechAttrMap.end(); ++mit)
    {
        delete mit->second;
    }
    mTechAttrMap.clear();

    std::vector<techLight*>::iterator vlit;
    for (vlit = mTLight.begin(); vlit != mTLight.end(); ++vlit)
    {
        delete(*vlit);
    }
    mTLight.clear();

    if (mTechniqueState)
    {
        freeTechnique();
    }
    delete pState;
}

void Technique::setProgramState(bool flag)
{
    mTechniqueState = flag;
}

bool Technique::getProgramState() const
{
    return mTechniqueState;
}

unsigned int Technique::getProgramId() const
{
    return mProgramId;
}

void Technique::setTechId(std::string techId)
{
    mTechniqueId = techId;
}

const std::string& Technique::getTechId()
{
    return mTechniqueId;
}

void Technique::setVertexShader(const std::string& shaderName)
{
    mVShaderName = shaderName;
}

void Technique::setFragmentShader(const std::string& shaderName)
{
    mFShaderName = shaderName;
}

void Technique::insertTechAttribute(const std::string& key,
                                    TechAttribute* pTechAttr)
{
    mTechAttrMap.insert(
        std::map<std::string, TechAttribute*>::value_type(key, pTechAttr));
}

TechAttribute* Technique::getTechAttribute(const std::string& key)
{
    std::map<std::string, TechAttribute*>::const_iterator it;
    it = mTechAttrMap.find(key);
    if (it != mTechAttrMap.end())
    {
        return it->second;
    }
    return 0;
}

void Technique::pushTechUniform(TechUniform* pUniform)
{
    (mTechUnifVec).push_back(pUniform);
}

void Technique::pushTLight(techLight* ptLight)
{
    (mTLight).push_back(ptLight);
}

const std::vector<techLight*> Technique::poptLight()
{
    return mTLight;
}

TechniqueState* Technique::getTechState()
{
    return pState;
}

int Technique::initTechnique(const std::vector<glTFFile>& inputFiles)
{
    int vShaderIdx = gltf_get_file_index_by_name(inputFiles,
                                                 mVShaderName);
    int fShaderIdx = gltf_get_file_index_by_name(inputFiles,
                                                 mFShaderName);

    if( vShaderIdx == LIBGLTF_FILE_NOT_LOAD || fShaderIdx == LIBGLTF_FILE_NOT_LOAD )
        return LIBGLTF_FILE_NOT_LOAD;

    const char* pvShader = inputFiles[vShaderIdx].buffer;
    size_t ivShaderSize = inputFiles[vShaderIdx].size;
    const char* pfShader = inputFiles[fShaderIdx].buffer;
    size_t ifShaderSize = inputFiles[fShaderIdx].size;
    mProgramId = mShaderProg.createProgram(pvShader, ivShaderSize,
                                            pfShader, ifShaderSize);
    if (0 != mProgramId)
    {
        setProgramState(true);
        return LIBGLTF_SUCCESS;
    }
    else
    {
        return LIBGLTF_SHADER_ERROR;
    }
}

bool Technique::useTechnique()
{
    if (mTechniqueState)
    {
        mShaderProg.useProgram(0);
        mShaderProg.useProgram(mProgramId);
    }
    return mTechniqueState;
}

void Technique::freeTechnique()
{
    mShaderProg.deleteProgram(mProgramId);
    setProgramState(false);
}

/* -- MaterialProperty -- */
MaterialProperty::MaterialProperty()
    : mPropertyName()
    , mTextureType(TextureType_NONE)
    , mImagePath()
    , mDataType(DataType_UNKNOW)
    , mDataLength(0)
    , pData(0)
{
}

MaterialProperty::~MaterialProperty()
{
    delete [] pData;
}

void MaterialProperty::setPropertyName(const std::string& name)
{
    mPropertyName = name;
}

const std::string& MaterialProperty::getPropertyName() const
{
    return mPropertyName;
}

void MaterialProperty::setImagePath(const std::string& index)
{
    mImagePath = index;
}

const std::string& MaterialProperty::getImagePath() const
{
    return mImagePath;
}

void MaterialProperty::setDataLength(unsigned int length)
{
    mDataLength = length;
}

void MaterialProperty::setDataType(DataType type)
{
    mDataType = type;
}

DataType MaterialProperty::getDataType() const
{
    return mDataType;
}

void MaterialProperty::setPropertyData(char* pValue, unsigned int size)
{
    delete [] pData;
    pData = new char[size];
    memcpy(pData, pValue, size);
}

const char* MaterialProperty::getPropertyData() const
{
    return pData;
}

/* -- Material -- */
Material::Material()
    : mTechniqueId()
    , mPropertyVec()
{
}

Material::~Material()
{
    std::vector<MaterialProperty*>::iterator it;
    for (it = mPropertyVec.begin(); it != mPropertyVec.end(); ++it)
    {
        delete(*it);
    }
    mPropertyVec.clear();
}

void Material::pushMaterialProper(MaterialProperty* pProperty)
{
    mPropertyVec.push_back(pProperty);
}

const MaterialProperty* Material::getMaterialProper(unsigned int index)
{
    if (mPropertyVec.size() <= index)
    {
        return 0;
    }
    return mPropertyVec[index];
}

unsigned int Material::getMaterialProperSize()
{
    return mPropertyVec.size();
}

void Material::setTechniqueId(const std::string& techniqueId)
{
    mTechniqueId = techniqueId;
}

const std::string& Material::getTechniqueId() const
{
    return mTechniqueId;
}

/* -- Primitives -- */
Primitives::Primitives()
    : mAttributeMap()
    , mIndicesIndex()
    , mMaterialIndex()
{
}

Primitives::~Primitives()
{
    mAttributeMap.clear();
}

void Primitives::setMaterialIndex(std::string materialIndex)
{
    mMaterialIndex = materialIndex;
}

const std::string Primitives::getMaterialIndex() const
{
    return mMaterialIndex;
}

void Primitives::setIndicesIndex(std::string indicesIndex)
{
    mIndicesIndex = indicesIndex;
}

const std::string Primitives::getIndicesIndex() const
{
    return mIndicesIndex;
}

void Primitives::insertAttribute(const std::string& key,
                                 const std::string& value)
{
    mAttributeMap.insert(
        std::map<std::string, std::string>::value_type(key, value));
}

const std::string Primitives::getAttributeIndex(const std::string& key) const
{
    std::map<std::string, std::string>::const_iterator it;
    it = mAttributeMap.find(key);
    if (it != mAttributeMap.end())
    {
        return it->second;
    }

    return "";
}

/* -- Mesh -- */
Mesh::Mesh()
    : mName()
    , mPrimitiveVec()
{
}

Mesh::~Mesh()
{
    std::vector<Primitives*>::iterator it;
    for (it = mPrimitiveVec.begin(); it != mPrimitiveVec.end(); ++it)
    {
        delete(*it);
    }
    mPrimitiveVec.clear();
}

void Mesh::setMeshName(const std::string& sName)
{
    mName = sName;
}

void Mesh::setPrimitiveVec(Primitives* pPrim)
{
    (mPrimitiveVec).push_back(pPrim);
}

const Primitives* Mesh::getPrimitiveVec(unsigned int index) const
{
    if (mPrimitiveVec.size() <= index)
    {
        return 0;
    }
    return mPrimitiveVec[index];
}

unsigned int Mesh::getPrimitiveVecSize() const
{
    return mPrimitiveVec.size();
}

/* -- Node -- */
Node::Node()
    : mNodeType(NodeType_Init)
    , mName()
    , mLocalMatrix(1.0)
    , mGlobalMatrix(1.0)
    , mScaleMatrix(1.0)
    , mTransMatrix(1.0)
    , mRotateMatrix(1.0)
    , pParent(0)
    , mChildrenVec()
    , mMesheIndexVec()
    , mCameraIndex()
    , mLightIndex()
    , mJointIndex()
    , mSkinIndex()
    , mpSkin(0)
    , mBoneVec()
    , pAnimation(0)
    , mSkeleIndex()
    , bJointFlag(false)
    , bMatrixFlag(true)
    , bUpdateFlag(false)
{
}

Node::~Node()
{
    std::vector<Node*>::iterator it;
    for (it = mChildrenVec.begin(); it != mChildrenVec.end(); ++it)
    {
        delete(*it);
    }
    mChildrenVec.clear();

    mBoneVec.clear();
}

void Node::setGlobalMatrix(const glm::mat4& matrix)
{
    mGlobalMatrix = matrix;
}

glm::mat4& Node::getGlobalMatrix()
{
    return mGlobalMatrix;
}

void Node::setLocalMatrix(const glm::mat4& matrix)
{
    mLocalMatrix = matrix;
}

glm::mat4& Node::getLocalMatrix()
{
    return mLocalMatrix;
}

void Node::setRotate(const glm::mat4& rotate)
{
    mRotateMatrix = rotate;
}

void Node::setRotate(const float* pbuf)
{
    glm::vec4 rotate;
    memcpy(&rotate, pbuf, sizeof(glm::vec4));
    mRotateMatrix = glm::rotate(rotate.w*57, glm::vec3(rotate));
}

const glm::mat4& Node::getRotate()
{
    return mRotateMatrix;
}

void Node::setScale(const glm::mat4& scale)
{
    mScaleMatrix = scale;
}

void Node::setScale(const float* pbuf)
{
    glm::vec3 scale;
    memcpy(&scale, pbuf, sizeof(glm::vec3));
    mScaleMatrix = glm::scale(scale);
}

const glm::mat4& Node::getScale()
{
    return mScaleMatrix;
}

void Node::setTranslate(const glm::mat4& trans)
{
    mTransMatrix = trans;
}

void Node::setTranslate(const float* pbuf)
{
    glm::vec3 trans;
    memcpy(&trans, pbuf, sizeof(glm::vec3));
    mTransMatrix = glm::translate(trans);
}

const glm::mat4& Node::getTranslate()
{
    return mTransMatrix;
}

void Node::setNodeType(NodeType _type)
{
    mNodeType = _type;
}

void Node::setNodeName(const std::string& name)
{
    mName = name;
}

const std::string& Node::getNodeName() const
{
    return mName;
}

void Node::setJointId(const std::string& jointId)
{
    mJointIndex = jointId;
}

const std::string& Node::getJointId() const
{
    return mJointIndex;
}

void Node::setLightIndex(const std::string& lightIndex)
{
    mLightIndex = lightIndex;
}

const std::string Node::getLightIndex() const
{
    return mLightIndex;
}

void Node::setSkinIndex(const std::string& index)
{
    mSkinIndex = index;
}

const std::string& Node::getSkinIndex() const
{
    return mSkinIndex;
}

void Node::setSkinPoint(Skin* pSkin)
{
    mpSkin = pSkin;
}

Skin* Node::getSkinPoint()
{
    return mpSkin;
}

void Node::setAnimPoint(Animation* pAnima)
{
    pAnimation = pAnima;
}

Animation* Node::getAnimPoint()
{
    return pAnimation;
}

void Node::setSkeleIndex(const std::string& index)
{
    mSkeleIndex = index;
}

const std::string& Node::getSkeleIndex() const
{
    return mSkeleIndex;
}

void Node::pushBoneNode(Node* pBone)
{
    (mBoneVec).push_back(pBone);
}

Node* Node::getBoneNode(unsigned int index)
{
    if (mBoneVec.size() <= index)
    {
        return 0;
    }
    return mBoneVec[index];
}

unsigned int Node::getBoneNodeSize()
{
    return (mBoneVec).size();
}

void Node::setCameraIndex(std::string cameraIndex)
{
    mCameraIndex = cameraIndex;
}

const std::string Node::getCameraIndex()
{
    return mCameraIndex;
}

void Node::pushMeshIndex(const std::string& _index)
{
    (mMesheIndexVec).push_back(_index);
}

const std::string Node::getMeshIndex(unsigned int index) const
{
    if (mMesheIndexVec.size() <= index)
    {
        return 0;
    }
    return mMesheIndexVec[index];
}

unsigned int Node::getMeshIndexSize() const
{
    return mMesheIndexVec.size();
}

void Node::pushChildNode(Node* pNode)
{
    (mChildrenVec).push_back(pNode);
}

Node* Node::getChildNode(unsigned int index)
{
    if (mChildrenVec.size() <= index)
    {
        return 0;
    }
    return mChildrenVec[index];
}

unsigned int Node::getChildNodeSize() const
{
    return mChildrenVec.size();
}

void Node::setParentNode(Node* pParentNode)
{
    pParent = pParentNode;
}

Node* Node::getParentNode()
{
    return pParent;
}

void Node::setJointFlag(bool flag)
{
    bJointFlag = flag;
}

bool Node::getJointFlag()
{
    return bJointFlag;
}

void Node::setMatrixFlag(bool flag)
{
    bMatrixFlag = flag;
}

bool Node::getMatrixFlag()
{
    return bMatrixFlag;
}

void Node::setUpdateFlag(bool flag)
{
    bUpdateFlag = flag;
}

bool Node::getUpdateFlag()
{
    return bUpdateFlag;
}



/* -- Animation -- */
Animation::Animation()
    : mBoneId()
    , mDuration(0)
    , mChannelBits(0)
    , mTimeValueVec()
{
}

Animation::~Animation()
{
    mTimeValueVec.clear();
}

void Animation::setDuration(double duration)
{
    mDuration = duration;
}

double Animation::getDuration()
{
    return mDuration;
}

const glm::mat4& Animation::findTimeValue(double time)
{
    std::vector<QuatKey>::const_iterator locIt;
    locIt = std::lower_bound(mTimeValueVec.begin(), mTimeValueVec.end(),
                             time, Animation::compareQuatKey);
    return locIt->mValue;
}

void Animation::pushTimeValue(double time, const glm::mat4& matrix)
{
    QuatKey quatKey;
    quatKey.mTime = time;
    quatKey.mValue = matrix;
    mTimeValueVec.push_back(quatKey);
}

void Animation::setTimeValue(const glm::mat4& matrix, unsigned int index)
{
    mTimeValueVec[index].mValue = matrix;
}

const glm::mat4& Animation::getTimeValue(unsigned int index)
{
    return mTimeValueVec[index].mValue;
}

void Animation::setBoneId(std::string boneId)
{
    mBoneId = boneId;
}

const std::string& Animation::getBoneId()
{
    return mBoneId;
}

/* -- Skin -- */
Skin::Skin()
    : mName()
    , pBindMatrix(0)
    , mBindMatrixCount(0)
    , mBoneIdVec()
{
}

Skin::~Skin()
{
    delete[] pBindMatrix;
    mBoneIdVec.clear();
}

void Skin::setSkinName(const std::string& name)
{
    mName = name;
}

const std::string& Skin::getSkinName()
{
    return mName;
}

void Skin::setBindMatrix(glm::mat4* matrix)
{
    pBindMatrix = matrix;
}

glm::mat4* Skin::getBindMatrix()
{
    return pBindMatrix;
}

void Skin::setBindMatrixCount(unsigned int count)
{
    mBindMatrixCount = count;
}

unsigned int Skin::getBindMatrixCount()
{
    return mBindMatrixCount;
}

void Skin::pushBoneId(std::string boneId)
{
    mBoneIdVec.push_back(boneId);
}

const std::string Skin::getBoneId(unsigned int index)
{
    if (mBoneIdVec.size() <= index)
    {
        return 0;
    }
    return mBoneIdVec[index];
}

unsigned int Skin::getBoneIdSize()
{
    return mBoneIdVec.size();
}

/* -- Scene -- */
Scene::Scene()
    : mAnimaMap()
    , mDuration(0.0)
    , mLightNodeMap()
    , mSkinVec()
    , pRootNode(0)
    , mNodeVec()
    , mMeshMap()
    , mMaterialMap()
    , mLightMap()
    , mCameraMap()
    , mVertexMax(-1e10f, -1e10f, -1e10f)
    , mVertexMin(1e10f, 1e10f, 1e10f)
    , mAttributeMap()
    , mTechniqueVec()
    , pBuffer(0)
    , bUseCameraInJson(false)
    , pGltfHandle(0)
    , mTexturesMap()
{
}

Scene::~Scene()
{
    freeMap(mMeshMap);
    freeMap(mMaterialMap);
    freeMap(mLightMap);
    freeMap(mCameraMap);
    freeMap(mTexturesMap);
    freeMap(mAnimaMap);
    delete pRootNode;
    mNodeVec.clear();
    mLightNodeMap.clear();

    std::vector<Technique*>::iterator it;
    for (it = mTechniqueVec.begin(); it != mTechniqueVec.end(); ++it)
    {
        delete *it;
    }
    mTechniqueVec.clear();

    std::vector<Skin*>::iterator itSkin;
    for (itSkin = mSkinVec.begin(); itSkin != mSkinVec.end(); ++itSkin)
    {
        delete *itSkin;
    }
    mSkinVec.clear();
}

template<typename M> void Scene::freeMap(M& aMap)
{
    for (typename M::iterator it = aMap.begin(); it != aMap.end(); ++it)
    {
        delete it->second;
    }
    aMap.clear();
}

Animation* Scene::findAnimation(const std::string& nodeId)
{
    std::map<std::string, Animation*>::iterator it;
    it = mAnimaMap.find(nodeId);
    if (it != mAnimaMap.end())
    {
        return it->second;
    }
    return 0;
}

void Scene::insertAnimMap(const std::string& key, Animation* pAnim)
{
    mAnimaMap.insert(
        std::map<std::string, Animation*>::value_type(key, pAnim));
}

unsigned int Scene::getAnimationCount()
{
    return mAnimaMap.size();
}

void Scene::insertLightNodeMap(const std::string& LightIndex, Node* pNode)
{
    mLightNodeMap.insert(
        std::map<std::string, Node*>::value_type(LightIndex, pNode));
}

Node* Scene::findLightNodeMap(std::string key)
{
    std::map<std::string, Node*>::iterator it;
    it = mLightNodeMap.find(key);
    if (it != mLightNodeMap.end())
    {
        return it->second;
    }
    return NULL;
}

Mesh* Scene::findMesh(const std::string& key)
{
    std::map<std::string, Mesh*>::iterator it;
    it = mMeshMap.find(key);
    if (it != mMeshMap.end())
    {
        return it->second;
    }
    return 0;
}

void Scene::insertMeshMap(const std::string& key, Mesh* pMesh)
{
    mMeshMap.insert(
        std::map<std::string, Mesh*>::value_type(key, pMesh));
}

Material* Scene::findMaterial(const std::string& key)
{
    std::map<std::string, Material*>::iterator it;
    it = mMaterialMap.find(key);
    if (it != mMaterialMap.end())
    {
        return it->second;
    }
    return 0;
}

void Scene::insertMaterialMap(const std::string& key, Material* pMaterial)
{
    mMaterialMap.insert(
        std::map<std::string, Material*>::value_type(key, pMaterial));
}

ParseCamera* Scene::findCamera(const std::string& key)
{
    std::map<std::string, ParseCamera*>::iterator it;
    it = mCameraMap.find(key);
    if (it != mCameraMap.end())
    {
        return it->second;
    }
    return 0;
}

void Scene::insertCameraMap(const std::string& key, ParseCamera* pCamera)
{
    mCameraMap.insert(
        std::map<std::string, ParseCamera*>::value_type(key, pCamera));
}

std::string Scene::getCameraIndex()
{
    return "";
}

Light* Scene::findLight(const std::string& key)
{
    std::map<std::string, Light*>::iterator it;
    it = mLightMap.find(key);
    if (it != mLightMap.end())
    {
        return it->second;
    }
    return 0;
}

const Attribute* Scene::findAttribute(const std::string& key)
{
    std::map<std::string, Attribute*>::iterator it;
    it = mAttributeMap.find(key);
    if (it != mAttributeMap.end())
    {
        return it->second;
    }
    return 0;
}

void Scene::insertAttributeMap(std::string key, Attribute* pAttribute)
{
    mAttributeMap.insert(
        std::map<std::string, Attribute*>::value_type(key, pAttribute));
}

void Scene::clearAttributeMap()
{
    freeMap(mAttributeMap);
}

void Scene::insertLightMap(const std::string& key, Light* pLight)
{
    mLightMap.insert(
        std::map<std::string, Light*>::value_type(key, pLight));
}

Node* Scene::getRootNode()
{
    return pRootNode;
}

void Scene::setRootNode(Node* pNode)
{
    pRootNode = pNode;
}

void Scene::pushNode(Node* pNode)
{
    mNodeVec.push_back(pNode);
}

Node* Scene::getNode(unsigned int index)
{
    if (mNodeVec.size() <= index)
    {
        return 0;
    }
    return mNodeVec[index];
}

unsigned int Scene::getNodeSize()
{
    return mNodeVec.size();
}

void Scene::pushSkin(Skin* pSkin)
{
    mSkinVec.push_back(pSkin);
}

Skin* Scene::getSkin(unsigned int index)
{
    if (mSkinVec.size() <= index)
    {
        return 0;
    }
    return mSkinVec[index];
}

unsigned int Scene::getSkinSize()
{
    return mSkinVec.size();
}

char* Scene::getBuffer()
{
    return pBuffer;
}

int Scene::setBuffer(const std::string& binName, const unsigned int length, const std::vector<glTFFile>& inputFiles)
{
    int idx = gltf_get_file_index_by_name(inputFiles, binName);
    if (idx != LIBGLTF_FILE_NOT_LOAD)
    {
        if( inputFiles[idx].size == length )
            pBuffer = inputFiles[idx].buffer;
        else
            return LIBGLTF_FILE_NOT_LOAD;
    }
    return idx;
}

void Scene::removeBuffer()
{
    pBuffer = 0;
}

void Scene::setVertexMax(float x, float y, float z)
{
    mVertexMax.x = std::max(x, mVertexMax.x);
    mVertexMax.y = std::max(y, mVertexMax.y);
    mVertexMax.z = std::max(z, mVertexMax.z);
}

glm::vec3& Scene::getVertexMax()
{
    return mVertexMax;
}

void Scene::setVertexMin(float x, float y, float z)
{
    mVertexMin.x = std::min(x, mVertexMin.x);
    mVertexMin.y = std::min(y, mVertexMin.y);
    mVertexMin.z = std::min(z, mVertexMin.z);
}

glm::vec3& Scene::getVertexMin()
{
    return mVertexMin;
}

void Scene::pushTechnique(class Technique* pTechnique)
{
    mTechniqueVec.push_back(pTechnique);
}

class Technique* Scene::getTechnique(unsigned int index)
{
    if (mTechniqueVec.size() <= index)
    {
        return 0;
    }
    return mTechniqueVec[index];
}

unsigned int Scene::getTechSize()
{
    return mTechniqueVec.size();
}

int Scene::loadTexture(const std::string& imagePath, const std::vector<glTFFile>& inputFiles)
{
    if (findTexture(imagePath) == NULL)
    {
        Texture* oneTexture = new Texture();

        const glTFFile* pglTFFile = getGltfFileByFileName(imagePath, inputFiles);
        if( !pglTFFile || pglTFFile->imagewidth == 0 || pglTFFile->imageheight == 0 )
            return LIBGLTF_FILE_NOT_LOAD;
        oneTexture->loadTexture2DFromBuffer((unsigned char*)
                                            (pglTFFile->buffer),
                                            pglTFFile->imagewidth,
                                            pglTFFile->imageheight);
        oneTexture->setFiltering(TEXTURE_FILTER_MAG_NEAREST,
                                 TEXTURE_FILTER_MIN_NEAREST);
        insertTextureMap(imagePath, oneTexture);
    }
    return LIBGLTF_SUCCESS;
}

Texture* Scene::findTexture(const std::string& key)
{
    std::map<std::string, Texture*>::iterator it;
    it = mTexturesMap.find(key);
    if (it != mTexturesMap.end())
    {
        return it->second;
    }
    return NULL;
}

void Scene::insertTextureMap(const std::string& key, Texture* pTexture)
{
    mTexturesMap.insert(
        std::map<std::string, Texture*>::value_type(key, pTexture));
}

double Scene::getDuration()
{
    return mDuration;
}

void Scene::setDuration(double duration)
{
    mDuration = duration;
}

void Scene::setGltfHandle(glTFHandle* handle)
{
    if (handle != NULL)
    {
        pGltfHandle = handle;
    }
}

glTFHandle* Scene::getGltfHandle()
{
    return pGltfHandle;
}

const glTFFile* Scene::getGltfFileByFileName(const std::string& fileName, const std::vector<glTFFile>& inputFiles)
{
    for (unsigned int i = 0; i < inputFiles.size(); i++)
    {
        if ( inputFiles[i].filename == fileName )
        {
            return &inputFiles[i];
        }
    }
    return NULL;
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
