/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * DivideOperator.java
 * ------------
 */
package org.pentaho.reporting.libraries.formula.operators;

import java.math.BigDecimal;

import org.pentaho.reporting.libraries.formula.EvaluationException;
import org.pentaho.reporting.libraries.formula.LibFormulaErrorValue;
import org.pentaho.reporting.libraries.formula.LibFormulaBoot;
import org.pentaho.reporting.libraries.formula.util.NumberUtil;

/**
 * A division operation. This operation expects two valid numbers.
 *
 *
 * @author Thomas Morgner
 */
public class DivideOperator extends AbstractNumericOperator
{
  private static final long serialVersionUID = 3298154333839229191L;

  public DivideOperator()
  {
  }

  public Number evaluate(final Number number1, final Number number2) throws EvaluationException
  {
    final BigDecimal bd1 = NumberUtil.getAsBigDecimal(number1);
    final BigDecimal bd2 = NumberUtil.getAsBigDecimal(number2);
    return divide(bd1, bd2);
  }

  public static BigDecimal divide(final BigDecimal bd1, final BigDecimal bd2)
      throws EvaluationException
  {
    if (bd2.signum() == 0)
    {
      // prevent a division by zero ..
      throw new EvaluationException(LibFormulaErrorValue.ERROR_ARITHMETIC_VALUE);
    }
    final BigDecimal divide = bd1.divide(bd2, LibFormulaBoot.GLOBAL_SCALE, BigDecimal.ROUND_HALF_UP);
    return NumberUtil.removeTrailingZeros(divide);
  }

  public int getLevel()
  {
    return 100;
  }


  public String toString()
  {
    return "/";
  }

  public boolean isLeftOperation()
  {
    return true;
  }

  /**
   * Defines, whether the operation is associative. For associative operations,
   * the evaluation order does not matter, if the operation appears more than
   * once in an expression, and therefore we can optimize them a lot better than
   * non-associative operations (ie. merge constant parts and precompute them
   * once).
   *
   * @return true, if the operation is associative, false otherwise
   */
  public boolean isAssociative()
  {
    return false;
  }

}
