// Author: Enrico Zini
// License: GPL


#include <config.h>
#include <apt-front/init.h>
#include <apt-front/cache/component/packages.h>
#include <apt-front/cache/component/tags.h>
#include <apt-front/cache/component/debtags/pkgidx.h>
#include <apt-front/cache/component/debtags/update.h>
#include <apt-front/utils/paths.h>
#include "AIWOptions.h"

#include <apt-pkg/configuration.h>
#include <iostream>
#include <syslog.h>
#include <unistd.h>
#include <errno.h>

using namespace aptFront;
using namespace cache;
using namespace component;
using namespace std;

bool needsUpdate()
{
	vector<debtags::source> sources = debtags::readSources();
	time_t ts, ts1;

	if ((ts = debtags::PkgIdx::computeTimestamp()) == 0)
		return true;
	if (Packages::computeTimestamp() > ts)
		return true;

	if ((ts = utils::Path::timestamp(utils::Path::tagdbIndex())) == 0)
		return true;

	if ((ts1 = utils::Path::timestamp(utils::Path::vocabulary())) == 0)
		return true;
	if (ts1 > ts) ts = ts1;

	if ((ts1 = utils::Path::timestamp(utils::Path::vocabularyIndex())) == 0)
		return true;
	if (ts1 > ts) ts = ts1;

	if (utils::Path::timestamp(utils::Path::defaultVocabulary()) > ts)
		return true;

	for (vector<debtags::source>::const_iterator i = sources.begin();
			i != sources.end(); i++)
		if (i->canBeUsed() && i->timestamp() > ts)
			return true;

	return false;
}

int main(int argc, const char* argv[])
{
	Tagcoll::commandline::AIWOptions opts;
	bool do_syslog = false;
	int res = 0;

	try {
		// Install the handler for unexpected exceptions
		Tagcoll::InstallUnexpected installUnexpected;

		opts.parse(argc, argv);
		if (!opts.lastCommand())
			throw Tagcoll::commandline::BadOption("could not understand the command to execute");

		do_syslog = opts.outputGroup.syslog->boolValue();
		if (do_syslog)
			openlog("apt-index-watcher", LOG_PID, LOG_DAEMON);

		// Perform the correct operation
		if (opts.helpGroup.help->boolValue())
		{
			// Provide help as requested
			Tagcoll::commandline::Help help("apt-index-watcher", VERSION);
			Tagcoll::commandline::OptionParser* o = opts.lastCommand();

			if (o && !o->name().empty())
				// Help on a specific command
				help.outputHelp(cout, *o);
			else
				// General help
				help.outputHelp(cout, opts);
		}
		else if (opts.helpGroup.version->boolValue())
		{
			// Print the program version
			Tagcoll::commandline::Help help("apt-index-watcher", VERSION);
			help.outputVersion(cout);
		}
		else if (opts.lastCommand() == &opts.generic)
		{
			Tagcoll::commandline::Help help("apt-index-watcher", VERSION);
			help.outputHelp(cout, opts);
		}
		else if (opts.lastCommand() == &opts.help)
		{
			Tagcoll::commandline::Help help("apt-index-watcher", VERSION);
			Tagcoll::commandline::OptionParser* o = 0;
			if (opts.hasNext())
				o = opts.command(opts.next());

			if (o)
				// Help on a specific command
				help.outputHelp(cout, *o);
			else
				// General help
				help.outputHelp(cout, opts);
		}
		else if (opts.lastCommand() == &opts.run)
		{
			aptFront::init();

			while (needsUpdate())
			{
			        // enforce that the cache uses memory only
			        _config->Set("Dir::Cache","/var/cache/apt-index-watcher");
				Cache& c = Global::get();
				c.open(cache::Cache::OpenDefault
						| cache::Cache::OpenReadOnly);
				c.close();
				//cout << "Indexes updated." << endl;
			}
		}
		else if (opts.lastCommand() == &opts.watch)
		{
			aptFront::init();

			while (1)
			{
				if (needsUpdate())
				{
					string cmd = string(argv[0]) + " run";
					if (do_syslog)
						cmd += " --syslog";
					int rcode = system(cmd.c_str());
					if (do_syslog)
					{
						if (rcode == -1)
							syslog(LOG_ERR, "Error when trying to run '%s': %m", cmd.c_str());
						else if (res != 0)
							syslog(LOG_ERR, "Child process '%s' returned with error code %d", cmd.c_str(), rcode);
					}
					else
					{
						if (rcode == -1)
							cerr << "Error when trying to run '" << cmd << "': " << strerror(errno) << endl;
						else if (res != 0)
							cerr << "Child process '" << cmd << "' returned with error code " << rcode << endl;
					}
				}
				sleep(5);
			}

			res = 1;
		}
		else
			throw Tagcoll::commandline::BadOption(string("unhandled command ") +
						(opts.lastCommand() ? opts.lastCommand()->name() : "(null)"));
	} catch (Tagcoll::commandline::BadOption& e) {
		cerr << e.desc() << endl;
		Tagcoll::commandline::Help help("apt-index-watcher", VERSION);
		if (opts.lastCommand())
		{
			help.outputHelp(cerr, *opts.lastCommand());
		} else {
			help.outputHelp(cerr, opts);
		}
		res = 1;
	} catch (Tagcoll::Exception& e) {
		if (do_syslog)
			syslog(LOG_ERR, "%s: %s", e.type(), e.desc().c_str());
		else
			cerr << e.type() << ": " << e.desc() << endl;
		res = 1;
	} catch (std::exception& e) {
		if (do_syslog)
			syslog(LOG_ERR, "%s", e.what());
		else
			cerr << e.what() << endl;
		res = 1;
	} catch (...) {
		if (do_syslog)
			syslog(LOG_ERR, "Caught unknown exception.");
		else
			cerr << "Caught unknown exception.";
		res = 1;
	}

	if (do_syslog)
		closelog();

	return res;
}

// vim:set ts=4 sw=4:
