/** -*- C++ -*-
    @file cache/entity/relation.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/component/packages.h>
#include <stdexcept>
#include <string>
#include <vector>

#ifndef APTFRONT_CACHE_ENTITY_RELATION_H
#define APTFRONT_CACHE_ENTITY_RELATION_H

namespace aptFront {
namespace cache {
namespace entity {

/**
   @brief Iterate over relations of a given package with others.

   This is your interface to the package relationships stored within apt's
   cache. This includes dependencies, reverse dependencies, conflicts,
   reverse conflicts. You can get the relevant relation iterators using
   the relevant PackageIterator accessors.
*/
class Relation : public Implementation<Relation, Base> {
public:
    enum Type { Dependency, ReverseDependency, Conflict, ReverseConflict };
    Relation() {}
    Relation( const Entity &i ) {
        initFromBase( i.impl() );
    }

    /* Interface safety. */
    bool valid() const { // XXX
        return m_cache;
    }

    /* RelationIterator functionality. */
    bool operator==( const Relation &d ) const {
        return d.m_cache == m_cache;
    }

    bool operator<( const Relation &d ) const {
        return false; // grr
    }

    /**
       @brief Get a sensible name of the relation.

       This will give you a reasonably useful name of the relation's declaration.
       It does NOT include the type! For a given package/version/type combination,
       this should also be fairly unique. It roughly corresponds how apt-cache
       formats the Depends: et al fields when showing package details.
    */
    std::string name() const { return ""; };

    /**
       @brief Get list of possible targets for this relation.

       This method will return you a list of all VersionIterator's that are
       fit for satisfying this relation declaration. For depends, this means
       list of packages that would satisfy the dependency (you need to have
       only once of those installed to satisfy it!), for conflicts, this means
       list of all packages conflicting with owner of this relation (you
       need to uninstall all of those to be able to install owner!).
    */
    std::vector<Version> targetList() const;

    /// Get the type of this relation.
    Type type() const;

    /// Get the owner PackageIterator (that is, the one from which we were obtained).
    Package ownerPackage() const;

    /// Get the owner VersionIterator (that is the one from which we were obtained).
    Version ownerVersion() const;

    /**
       @brief Get a "straight" version of this RelationIterator.
       
       This will return a straight (as opposed to reverse) version of this
       very same iterator. If it was already straight, you will get a copy
       of it. If it was reverse, the straight variant will be looked up in the
       cache and returned.
    */
    Relation straight() const;

protected:
    // XXX data
};

}
}
}
#endif
