/**
   @file cache/component/state.cpp
   @author Peter Rockai <me@mornfall.net>
*/

#include <apt-front/cache/component/state.h>
#include <apt-front/cache/entity/version.h>
#include <apt-front/error.h>
#include <iostream>

using namespace aptFront::cache;
using namespace aptFront::cache::entity;
using namespace aptFront;
using namespace aptFront::cache::component;

bool State::isInstalled( const entity::Package &P )
{
    if (P.pkgPtr()->CurrentState == pkgCache::State::ConfigFiles) return false;
    if (P.pkgPtr()->CurrentState == pkgCache::State::NotInstalled) return false;
    return true;
}

Package::State State::packageState( const entity::Package &P )
{
    entity::Package::State s = 0;
    if (isInstalled (P))
        s |= entity::Package::State::Installed;
    if (isInstalled( P ) && P.candidateVersion() != P.installedVersion())
        s |= entity::Package::State::Upgradable;
    pkgDepCache::StateCache S = operator[]( P );
    if (S.Install())
        s |= entity::Package::State::Install;
    if (S.Keep())
        s |= entity::Package::State::Keep;
    if (S.Delete())
        s |= entity::Package::State::Remove;
    if (S.NowBroken())
        s |= entity::Package::State::NowBroken;
    if (S.InstBroken())
        s |= entity::Package::State::WillBreak;
    return s;
}

State::State( const State &s )
    : pkgDepCache( s ), Implementation<State>( s ), m_policy( s.m_policy )
{
    Cache = s.Cache;
    if (Cache) {
        // std::cerr << "deep-copying State!" << std::endl;
        // std::cerr << "Cache = " << Cache << std::endl;
        PkgState = new StateCache[Head().PackageCount];
        DepState = new unsigned char[Head().DependsCount];
        std::copy( s.PkgState, s.PkgState+
                   Head().PackageCount, PkgState );
        std::copy( s.DepState, s.DepState+
                   Head().DependsCount, DepState );
        LocalPolicy = &policy();
        delLocalPolicy = 0;
    }
}

State::State (cache::Cache *cache)
    : pkgDepCache (&( cache->packages().baseReference() ), 0),
      m_policy( cache )
{
    delete delLocalPolicy;
    delLocalPolicy = 0;
    LocalPolicy = &policy();
    exception::checkGlobal( "error constructing package policy" );
    if (ReadPinFile(policy()) == false)
        throw exception::Error( "error reading pin file" );
}

State::~State ()
{
}

void State::updateCounts() {
    Packages::iterator i;
    m_availableCount = m_removeCount = m_newInstallCount = m_upgradeCount
                     = m_installedCount = m_upgradableCount = 0;
    for (i = ownerCache()->packages().packagesBegin();
         i != ownerCache()->packages().packagesEnd();
         ++ i) {
        if (i->hasVersion())
            ++ m_availableCount;
        if (i->isUpgradable())
            ++ m_upgradableCount;
        if (i->isInstalled())
            ++ m_installedCount;
        if (i->markedNewInstall())
            ++ m_newInstallCount;
        if (i->markedRemove())
            ++ m_removeCount;
        if (i->markedUpgrade())
            ++ m_upgradeCount;
    }
}

bool State::changed() {
    return (removeCount() || newInstallCount() || upgradeCount());
}

void State::notifyPostChange() {
    updateCounts();
    Implementation<State>::notifyPostChange();
}

#ifdef COMPILE_TESTSUITE
#include "test-utils.h"

namespace tut {

struct cache_component_state_shar {
    cache_component_state_shar() {
        aptInit();
        c.open( Cache::OpenDefault | Cache::OpenReadOnly );
    }
    Cache c;
};

TESTGRP( cache_component_state );

template<> template<>
void to::test<1>()
{
    State *s1 = &c.state();
    State *s2 = new State( *s1 );
}

template<> template<>
void to::test<2>()
{
    entity::Package gcide = c.packages().packageByName("dict-gcide"),
                    dictd = c.packages().packageByName("dictd");
    ensure( dictd.isInstalled() );
    ensure( gcide.isInstalled() );
    dictd.markRemove();
    ensure( dictd.markedRemove() );
    ensure( gcide.markedRemove() );
}

template<> template<>
void to::test<3>()
{
    entity::Package gcide = c.packages().packageByName("dict-gcide"),
                    dictd = c.packages().packageByName("dictd");
    ensure( dictd.isInstalled() );
    ensure( gcide.isInstalled() );
    dictd.markRemove();
    ensure( dictd.markedRemove() );
    ensure( gcide.markedRemove() );
    gcide.markKeep();
    ensure( gcide.markedKeep() );
    ensure( dictd.markedKeep() );
}

}

#endif
