/*
 * mod_security (apache 2.x), http://www.modsecurity.org/
 * Copyright (c) 2002-2004 Ivan Ristic <ivanr@webkreator.com>
 *
 * $Id: mod_security.c,v 1.137 2004/07/28 15:56:08 ivanr Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
 
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#ifdef WIN32
#include <direct.h>
#endif

#if !(defined(WIN32) || defined(NETWARE))
#include <sys/types.h>
#include <unistd.h>
#endif

#if !defined(OS2) && !defined(WIN32) && !defined(BEOS) && !defined(NETWARE)
#include "unixd.h"
#define __SET_MUTEX_PERMS
#endif

#include "ap_config.h"
#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "util_script.h"
#include "ap_mpm.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_user.h"
#include "apr_lib.h"
#include "apr_signal.h"
#include "apr_global_mutex.h"

module AP_MODULE_DECLARE_DATA security_module;

static apr_global_mutex_t *modsec_debuglog_lock = NULL;
static apr_global_mutex_t *modsec_auditlog_lock = NULL;

static ap_filter_rec_t *global_sec_filter_in;
static ap_filter_rec_t *global_sec_filter_out;

#define MODULE_RELEASE "1.8.4"

#define CREATEMODE ( APR_UREAD | APR_UWRITE )

#if defined(NETWARE)
#define CREATEMODE_UNISTD ( S_IREAD | S_IWRITE )
#elif defined(WIN32)
#define CREATEMODE_UNISTD ( _S_IREAD | _S_IWRITE )
#else
#define CREATEMODE_UNISTD ( S_IRUSR | S_IWUSR )
#endif

#define MODSEC_SKIP                         -2000

#define UNICODE_ERROR_CHARACTERS_MISSING    -1
#define UNICODE_ERROR_INVALID_ENCODING      -2
#define UNICODE_ERROR_OVERLONG_CHARACTER    -3

#define NOT_SET                         -1
#define NOT_SET_P                       (void *)-1

#define FILTERING_OFF                   0
#define FILTERING_ON                    1
#define FILTERING_DYNAMIC_ONLY          2

#define AUDITLOG_OFF                    0
#define AUDITLOG_ON                     1
#define AUDITLOG_DYNAMIC_OR_RELEVANT    2
#define AUDITLOG_RELEVANT_ONLY          3

#define ACTION_NONE                 0
#define ACTION_DENY                 1
#define ACTION_REDIRECT             2
#define ACTION_ALLOW                3
#define ACTION_SKIP                 4

#define VAR_ACTION_ALLOW            1
#define VAR_ACTION_DENY             0

#define VAR_UNKNOWN                 0
#define VAR_CUSTOM                  1
#define VAR_HEADER                  2
#define VAR_ENV                     3
#define VAR_ARGS                    4
#define VAR_POST_PAYLOAD            5
#define VAR_ARGS_NAMES              6
#define VAR_ARGS_VALUES             7
#define VAR_ARGS_SELECTIVE          8
#define VAR_OUTPUT                  9
#define VAR_COOKIES_NAMES           10
#define VAR_COOKIES_VALUES          11
#define VAR_COOKIE                  12
#define VAR_RESERVED_5              13
#define VAR_RESERVED_6              14
#define VAR_RESERVED_7              15
#define VAR_RESERVED_8              16
#define VAR_RESERVED_9              17
#define VAR_RESERVED_10             18
#define VAR_RESERVED_11             19
#define VAR_RESERVED_12             20

#define VAR_REMOTE_ADDR             21
#define VAR_REMOTE_HOST             22
#define VAR_REMOTE_USER             23   
#define VAR_REMOTE_IDENT            24
#define VAR_REQUEST_METHOD          25
#define VAR_SCRIPT_FILENAME         26
#define VAR_PATH_INFO               27
#define VAR_QUERY_STRING            28
#define VAR_AUTH_TYPE               29
#define VAR_DOCUMENT_ROOT           30
#define VAR_SERVER_ADMIN            31
#define VAR_SERVER_NAME             32
#define VAR_SERVER_ADDR             33
#define VAR_SERVER_PORT             34
#define VAR_SERVER_PROTOCOL         35
#define VAR_SERVER_SOFTWARE         36
#define VAR_TIME_YEAR               37
#define VAR_TIME_MON                38
#define VAR_TIME_DAY                39
#define VAR_TIME_HOUR               40
#define VAR_TIME_MIN                41
#define VAR_TIME_SEC                42
#define VAR_TIME_WDAY               43
#define VAR_TIME                    44
#define VAR_API_VERSION             45
#define VAR_THE_REQUEST             46
#define VAR_REQUEST_URI             47
#define VAR_REQUEST_FILENAME        48
#define VAR_IS_SUBREQ               49
#define VAR_HANDLER                 50

#define MULTIPART_BUF_SIZE              1024

#define MULTIPART_TMP_FILE_NONE         0
#define MULTIPART_TMP_FILE_CREATE       1
#define MULTIPART_TMP_FILE_CREATE_LEAVE 2

#define MULTIPART_FORMDATA              1
#define MULTIPART_FILE                  2

#define POST_ON_DISK                    1
#define POST_IN_MEMORY                  2

#define NOTE                        "mod_security-note"
#define NOTE_DYNAMIC                "mod_security-dynamic"
#define NOTE_MESSAGE                "mod_security-message"
#define NOTE_NOAUDITLOG             "mod_security-noauditlog"
#define NOTE_EXECUTED               "mod_security-executed"
#define NOTE_ACTION                 "mod_security-action"
#define NOTE_MSR                    "mod_security-msr"
#define NOTE_ACTED                  "mod_security-relevant"

#define FATAL_ERROR                 "Unable to allocate memory"

#define UNKNOWN_CSID    0
#define MB_CSID         800         /* First multibyte character set */
#define UNI3_CSID       873         /* Unicode 3.x character set ID  */
#define SJIS1_CSID      832         /* SJIS character set ID         */
#define SJIS2_CSID      834         /* SJIS+YEN character set ID     */
#define BIG5_CSID       865         /* BIG5 character set ID         */
#define GBK_CSID        852         /* GBK character set ID          */
#define GB2312_CSID     850         /* GB2312 character set ID       */
#define ZHT32EUC_CSID   860         /* Chinese 4-byte character set  */
#define JEUC1_CSID      830         /* JEUC character set ID         */
#define JEUC2_CSID      831         /* JEUC+YEN character set ID     */
#define JA16VMS_CSID    829         /* VMS 2-byte Japanese           */

static char *all_variables[] = {
    "UNKOWN",
    "CUSTOM",
    "HEADER",
    "ENV",
    "ARGS",
    "POST_PAYLOAD",
    "ARGS_NAMES",
    "ARGS_VALUES",
    "ARGS_SELECTIVE",
    "OUTPUT",
    "COOKIES_NAMES",    /* 10 */
    "COOKIES_VALUES",
    "COOKIE",
    "RESERVED_5",
    "RESERVED_6",
    "RESERVED_7",
    "RESERVED_8",
    "RESERVED_9",
    "RESERVED_10",
    "RESERVED_11",
    "RESERVED_12",      /* 20 */
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_USER",
    "REMOTE_IDENT",
    "REQUEST_METHOD",
    "SCRIPT_FILENAME",
    "PATH_INFO",
    "QUERY_STRING",
    "AUTH_TYPE",
    "DOCUMENT_ROOT",    /* 30 */
    "SERVER_ADMIN",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "TIME_YEAR",
    "TIME_MON",
    "TIME_DAY",
    "TIME_HOUR",        /* 40 */
    "TIME_MIN",
    "TIME_SEC",
    "TIME_WDAY",
    "TIME",
    "API_VERSION",
    "THE_REQUEST",
    "REQUEST_URI",
    "REQUEST_FILENAME",
    "IS_SUBREQ",
    "HANDLER",          /* 50 */
    NULL
};

typedef struct {
    char *name;
    int type;
    int action;
} variable;

typedef struct {
    request_rec *r;
    char *command;
    char *args;
} exec_data;

typedef struct {
    int log;
    int action;
    int status;
    int pause;
    int skip_count;
    int is_chained;
    char *redirect_url;
    int exec;
    char *exec_string;
    char *id;
    char *msg;
} actionset_t;

typedef struct {
    actionset_t *actionset;
    char *pattern;
    regex_t *regex;
    int is_selective;
    int is_negative;
    int is_allow;
    int is_output;
    int requires_parsed_args;
    apr_array_header_t *variables;
} signature;

typedef struct {
    int filter_engine;
    int configuration_helper;
    int scan_post;
    int scan_output;
    actionset_t *action;
    apr_array_header_t *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    apr_file_t *auditlog_fd;
    int filter_debug_level;
    char *debuglog_name;
    apr_file_t *debuglog_fd;
    int filters_clear;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
    char *scan_output_mimetypes;
    char *upload_dir;
    int upload_keep_files;
    char *upload_approve_script;
    int upload_in_memory_limit;
    int normalize_cookies;
    int check_cookie_format;
    int charset_id;
    int multibyte_replacement_byte;
    apr_pool_t *p;
} sec_dir_config;

typedef struct {
    int server_response_token;
    char *chroot_dir;
    int chroot_completed;
    char *chroot_lock;
    char *server_signature;
} sec_srv_config;

typedef struct sec_filter_in_ctx_ {
    char *buffer;
    int type; /* POST_ON_DISK or POST_IN_MEMORY */
    int is_multipart;
    unsigned long int buflen; /* max. size of the buffer */
    unsigned long int bufleft; /* space left in buffer */
    unsigned long int sofar; /* data read sofar */
    int access_check_performed;
    apr_bucket_brigade *pbbTmp;
    char *output_ptr; 
    unsigned long int output_sent;
    int done_reading;
    int done_writing;
    char *tmp_file_name;
    int tmp_file_fd;
} sec_filter_in_ctx;

typedef struct sec_filter_out_ctx_ {
    char *buffer;
    unsigned long int buflen;
    unsigned long int bufused;
    char *output_ptr; 
    unsigned long int output_sent;
    char *input_ptr;
    int done_reading;
    int done_writing;
} sec_filter_out_ctx;



typedef struct {
    /* part type, can be MULTIPART_FORMDATA or MULTIPART_FILE */
    int type;
    /* the name */
    char *name;
    
    /* variables only, variable the value */
    char *value;
    /* files only, the content type (where available) */
    char *content_type;
    
    /* files only, the name of the temporary file holding data */
    char *tmp_file_name;
    int tmp_file_fd;
    unsigned tmp_file_size;
    /* files only, filename as supplied by the browser */
    char *filename;
} multipart_part;

typedef struct {
    request_rec *r;
    sec_dir_config *dcfg;
    apr_pool_t *p;    
    
    /* this array keeps parts */
    apr_array_header_t *parts;
    
    /* do we need to make a copy of the request
     * in a temporary file
     */
    int create_tmp_file;
    char *tmp_file_name;
    int tmp_file_fd;
    
    /* mime boundary used to detect when
     * parts end and new begin
     */
    char *boundary;
    
    /* internal buffer and other variables
     * used while parsing
     */
    char buf[MULTIPART_BUF_SIZE + 2];
    int buf_contains_line;
    char *bufptr;
    int bufleft;    
    
    /* pointer that keeps track of a part while
     * it is being built
     */
    multipart_part *mpp;
    
     
    /* part parsing state; 0 means we are reading
     * headers, 1 means we are collecting data
     */
    int mpp_state;
    
    /* because of the way this parsing algorithm
     * works we hold back the last two bytes of
     * each data chunk so that we can discard it
     * later if the next data chunk proves to be
     * a boundary; the first byte is an indicator
     * 0 - no content, 1 - two data bytes available
     */
    char reserve[4];
} multipart_data;

typedef struct {
    request_rec *r;
    char *_the_request;
    char *_post_payload;
    char *_fake_post_payload;
    int should_body_exist;
    int is_body_read;
    unsigned long _post_len;
    int post_payload_dynamic_off;
    sec_dir_config *dcfg;
    sec_srv_config *scfg;
    apr_table_t *parsed_args;
    apr_table_t *parsed_cookies;
    char *tmp_message;
    char *tmp_redirect_url;
    int tmp_log_message;
    multipart_data *mpd;
} modsec_rec;

static int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type);
static void sec_debug_log(request_rec *r, int level, const char *text, ...);

static char *normalise_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg);
static char *normalise_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_relaxed(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg);
static char *normalise_other_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_urlencoding_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_urlencoding_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);

static int check_single_signature(modsec_rec *msr, signature *sig);
static int parse_cookies(request_rec *r, apr_table_t *parsed_cookies, char **error_msg);
static const char *get_env_var(request_rec *r, char *name);
static const char *get_variable(request_rec *r, variable *v, apr_table_t *parsed_args);
static request_rec *find_last_request(request_rec *r);
static unsigned char x2c(unsigned char *what);
static char *get_temp_folder(void);
static char *remove_binary_content(request_rec *r, char *data, long size);
static int parse_arguments(char *s, apr_table_t *parsed_args, request_rec *r, sec_dir_config *dcfg, char **error_msg);
static void send_error_bucket(ap_filter_t *f, int status);
static char *strtolower(char *str);
static char *parse_action(char *p2, actionset_t *actionset, apr_pool_t *_pool);
static void sec_set_dir_defaults(sec_dir_config *dcfg);
static int is_filtering_on_here(request_rec *r, sec_dir_config *dcfg);
static int detect_unicode_character(request_rec *r, unsigned char *p_read);

static int sec_initialize(modsec_rec *msr);
static int sec_check_access(request_rec *r);
static int sec_check_all_signatures(modsec_rec *msr, int is_output);

static int multipart_init(multipart_data *mpd, request_rec *r);
static apr_status_t multipart_finish(multipart_data *mpd);
static apr_status_t multipart_cleanup(void *data);
static int multipart_process_chunk(multipart_data *mpd, const char *buf, unsigned int size);
static char *multipart_construct_filename(multipart_data *mpd);
static int multipart_process_data_chunk(multipart_data *mpd);
static int multipart_process_boundary(multipart_data *mpd);
static int verify_uploaded_files(request_rec *r, multipart_data *mpd, char *approver_script, char **error_msg);
static int multipart_get_variables(multipart_data *mpd, apr_table_t *parsed_args, sec_dir_config *dcfg, char **error_msg);
static int multipart_contains_files(multipart_data *mpd);
static int create_chroot_lock(server_rec *s, apr_pool_t *p, char *lockfilename);
static int is_time_to_chroot(server_rec *s, apr_pool_t *p);
static int change_server_signature(server_rec *s, sec_srv_config *scfg);
static int sec_mkstemp(char *template);
static char *debuglog_escape(apr_pool_t *p, char *text);
static char *real_debuglog_escape(apr_pool_t *p, char *text);

static int convert_charset_to_id(char *name);
static char *filter_multibyte_inplace(int cs_id, char replacement_byte, char *outbuf);
static char *filter_multibyte_other(int charset_id, char replacement_byte, char *inptr);
static char *filter_multibyte_unicode(int charset_id, char replacement_byte, char *inptr);

static int sec_exec_child(char *command, const char *argv[], request_rec *r, char **output);

static int perform_action(modsec_rec *msr, actionset_t *actionset);
char *construct_fake_urlencoded(modsec_rec *msr, apr_table_t *args);

/* ----------------------------------------------------------------------------- */

char *construct_fake_urlencoded(modsec_rec *msr, apr_table_t *args) {
    apr_table_entry_t *te;
    const apr_array_header_t *arr;
    int k;
    char *body;
    unsigned int body_len;
    
    if (args == NULL) return NULL;
    
    /* calculate buffer size */
    body_len = 1;
    arr = apr_table_elts(args);
    te = (apr_table_entry_t *)arr->elts;
    for(k = 0; k < arr->nelts; k++) {
        body_len += 4;
        body_len += strlen(te[k].key);
        body_len += strlen(te[k].val);
    }
    
    /* allocate the buffer */
    body = apr_palloc(msr->r->pool, body_len + 1);
    if (body == NULL) return NULL;
    *body = 0;
    
    /* loop through remaining variables
     * and create a single string out of them
     */
    arr = apr_table_elts(args);
    te = (apr_table_entry_t *)arr->elts;
    for(k = 0; k < arr->nelts; k++) {
        if (*body != 0) {
            strncat(body, "&", body_len - strlen(body));
        }
        strncat(body, te[k].key, body_len - strlen(body));
        strncat(body, "=", body_len - strlen(body));
        strncat(body, te[k].val, body_len - strlen(body));
    }
    
    return body;
}

int perform_action(modsec_rec *msr, actionset_t *actionset) {
    char *message = NULL;
    request_rec *r = msr->r;
    int log_level = 1;
    int rc = DECLINED;
    
    if (msr->tmp_message == NULL) {
        msr->tmp_message = "Unknown error";
    }
    
    if (actionset->log == 0) {
        apr_table_setn(msr->r->notes, NOTE_NOAUDITLOG, "noauditlog");
    
        /* log level 2 will log to mod_security log but not
         * to the Apache error log
         */
        log_level = 2;
    }

    /* perform action */
    switch(actionset->action) {
        
        case ACTION_ALLOW :
            message = apr_psprintf(r->pool, "Access allowed. %s", msr->tmp_message);
            rc = DECLINED;
            break;
        
        case ACTION_DENY :
            rc = actionset->status;
            message = apr_psprintf(r->pool, "Access denied with code %i. %s", rc, msr->tmp_message);
            break;
                
        case ACTION_REDIRECT :
            message = apr_psprintf(r->pool, "Access denied with redirect to [%s]. %s", actionset->redirect_url, msr->tmp_message);
            apr_table_setn(r->headers_out, "Location", actionset->redirect_url);
            rc = HTTP_MOVED_TEMPORARILY;
            break;
                
        default :
            message = apr_psprintf(r->pool, "Warning. %s", msr->tmp_message);
            rc = DECLINED;
            break;
    }
    
    sec_debug_log(r, log_level, "%s", message);
    apr_table_setn(r->headers_in, NOTE_MESSAGE, msr->tmp_message);
    
    if ((rc != DECLINED)&&(rc != MODSEC_SKIP)) {
        char *action = apr_psprintf(msr->r->pool, "%i", rc);
        apr_table_setn(r->headers_in, NOTE_ACTION, action);
    }
        
    /* execute the external script */
    if (actionset->exec) {
        sec_debug_log(r, log_level, "Executing command \"%s\"", debuglog_escape(r->pool, actionset->exec_string));
        apr_table_setn(r->headers_in, NOTE_EXECUTED, actionset->exec_string);
        sec_exec_child(actionset->exec_string, NULL, r, NULL);
    }
        
    if (actionset->pause != 0) {
        /* TODO add uri information to the message, it won't have enough
         * information when it is in the error log
         */
        sec_debug_log(r, log_level, "Pausing for %i ms", actionset->pause);
        /* apr_sleep accepts microseconds */
        apr_sleep(actionset->pause * 1000);
    }

    msr->tmp_message = NULL;
    return rc;
}

int sec_mkstemp(char *template) {

    #if !(defined(WIN32)||!defined(NETWARE))
    return mkstemp(template);
    #else
    
    if (mktemp(template) == NULL) return -1;
    return open(template, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE_UNISTD);
    
    #endif
}

#ifdef WIN32

char *strtok_r(char *s, const char *sep, char **lasts) {
    char *sbegin, *send;
    
    /* These two parameters must always be valid */
    if ((sep == NULL)||(lasts == NULL)) return NULL;
    
    /* Either of the following two must not be NULL */
    if ((s == NULL)&&(*lasts == NULL)) return NULL;
    
    sbegin = s ? s : *lasts;
    
    /* Advance through the separator at the beginning */
    sbegin += strspn(sbegin, sep);
    if (*sbegin == '\0') {
        *lasts = NULL;
        return NULL;
    }

    /* Find the next separator */    
    send = strpbrk(sbegin, sep);
    if (send != NULL) *send++ = 0;
    *lasts = send;
    
    return sbegin;
}

#endif

/*
 * Change the signature of the server if change
 * was requested in configuration. We do this by
 * locating the signature in server memory and
 * writing over it.
 */
int change_server_signature(server_rec *s, sec_srv_config *scfg) {
    char *server_version;
    
    if (scfg->server_signature == NULL) return 0;
    
    server_version = (char *)ap_get_server_version();
    if (server_version == NULL) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "SecServerSignature: ap_get_server_version returned NULL");
        return -1;
    }
    
    if (strlen(server_version) >= strlen(scfg->server_signature)) {
        strcpy(server_version, scfg->server_signature);
        return 1;
    }
    else {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "SecServerSignature: not enough space to copy new signature");
        return -1;
    }
}

int convert_charset_to_id(char *name) {
    if (strcasecmp(name, "utf-8") == 0) return UNI3_CSID;
    if (strcasecmp(name, "shift-jis") == 0) return SJIS1_CSID;
    if (strcasecmp(name, "shift_jis") == 0) return SJIS2_CSID;
    if (strcasecmp(name, "big5") == 0) return BIG5_CSID;
    if (strcasecmp(name, "gbk") == 0) return GBK_CSID;
    if (strcasecmp(name, "gb2312") == 0) return GB2312_CSID;
    if (strcasecmp(name, "euc-tw") == 0) return ZHT32EUC_CSID;
    if (strcasecmp(name, "euc-jp") == 0) return JEUC1_CSID;
    if (strcasecmp(name, "eucjis") == 0) return JEUC2_CSID;
    if (strcasecmp(name, "jis0208") == 0) return JA16VMS_CSID;
    return -1;
}

char *filter_multibyte_unicode(int charset_id, char replacement_byte, char *inptr) {
    char *outptr = inptr;
    int i, j, k, n;
    
    i = strlen(inptr);
    j = 0;
    
    /* Unicode */
    while(j < i) {
        k = inptr[j] & 0xFF;
        if (k < 0x80) {
            j++;
            *outptr++ = (char)k;
        }
        else if (k < 0xC0) {
            j++;
            *outptr++ = replacement_byte;
        }
        else {
            if (k < 0xE0) n = 2;
            else if (k < 0xF0) n = 3;
            else if (k < 0xF8) n = 4;
            else if (k < 0xFC) n = 5;
            else if (k < 0xFE) n = 6;
            else n = 1;

            if (i - j >= n) {
                j += n;
            }
            else {
                i = j;
            }

            *outptr++ = replacement_byte;
        }
    }
    
    *outptr = 0;
    return inptr;
}

char *filter_multibyte_other(int charset_id, char replacement_byte, char *inptr) {
    char *outptr = inptr;
    int i, j, k, n;
    
    i = strlen(inptr);
    j = 0;
    
    while(j < i) {
        k = inptr[j] & 0xFF;
        if (k < 0x80) {
            j++;
            *outptr++ = (char)k;
        }
        else {
            n = 2;
                
            if ((k == 0x8E)&&(charset_id == ZHT32EUC_CSID)) {
                n = 4;
            }
            else if ((k == 0x8F)&&((charset_id == JEUC1_CSID)||(charset_id == JEUC2_CSID))) {
                n = 3;
            }
            else if ( ((k == 0x80)||(k == 0xFF))
                    && ((charset_id == BIG5_CSID)||(charset_id == GBK_CSID)||(charset_id == GB2312_CSID)) ) {
                n = 1;
            }
            else if ( ((k == 0x80)||((k >= 0xA0) && (k < 0xE0)))
                    && ((charset_id == SJIS1_CSID)||(charset_id == SJIS2_CSID)) ) {
                n = 1;
            }

            if (i - j >= n) {
                j += n;
            }
            else {
                i = j;
            }

            *outptr++ = (n == 1) ? (char)k : replacement_byte;
        }
    }
    
    *outptr = 0;
    return inptr;
}

char *filter_multibyte_inplace(int charset_id, char replacement_byte, char *inptr) {
    if (charset_id < MB_CSID) return inptr; /* not multibyte charset */
    if (charset_id == UNI3_CSID) return filter_multibyte_unicode(charset_id, replacement_byte, inptr);
    else return filter_multibyte_other(charset_id, replacement_byte, inptr);
}

static char *current_logtime(request_rec *r) {
    apr_time_exp_t t;
    char tstr[100];
    apr_size_t len;
            
    apr_time_exp_lt(&t, apr_time_now());
                
    apr_strftime(tstr, &len, 80, "%d/%b/%Y:%H:%M:%S ", &t);
    apr_snprintf(tstr + strlen(tstr), 80 - strlen(tstr), "%c%.2d%.2d",
        t.tm_gmtoff < 0 ? '-' : '+',
        t.tm_gmtoff / (60 * 60), t.tm_gmtoff % (60 * 60));
    return apr_pstrdup(r->pool, tstr);
}

static char *current_filetime(request_rec *r) {
    apr_time_exp_t t;
    char tstr[100];
    apr_size_t len;
            
    apr_time_exp_lt(&t, apr_time_now());
                
    apr_strftime(tstr, &len, 80, "%Y%m%d-%H%M%S", &t);
    return apr_pstrdup(r->pool, tstr);
}

int parse_cookies(request_rec *r, apr_table_t *parsed_cookies, char **error_msg) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    char *header, *header_copy, *attr_name, *attr_value, *p;
    char *prev_attr_name = NULL;
    int cookie_count = 0;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
    
    header = (char *)apr_table_get(r->headers_in, "Cookie");
    /* Return here if no cookies found */
    if (header == NULL) return 0;
    
    header_copy = apr_pstrdup(r->pool, header);
    if (header_copy == NULL) return -1;
    
    sec_debug_log(r, 6, "Raw cookie header \"%s\"", debuglog_escape(r->pool, header));
    
    p = header_copy;
    while(*p != 0) {
        attr_name = NULL;
        attr_value = NULL;
    
        /* attribute name */
        while(isspace(*p)) p++;
        attr_name = p;
        while((*p != 0)&&(*p != '=')) p++;
     
        if (*p == 0) {   
            /* we've reached the end of the string,
             * and the attribute value is missing
             */
            *error_msg = apr_psprintf(r->pool, "Cookie value is missing #1");
            return -1;
        }

        /* terminate the attribute name,
         * writing over the = character
         */
        *p++ = 0;
        
        /* attribute value */
        while((isspace(*p))&&(*p != 0)) p++;
        if (*p == 0) {
            *error_msg = apr_psprintf(r->pool, "Cookie value is missing #2");
            return -1;
        }
        
        if (*p == '"') {
            char *s, *d;
            
            /* quoted string */
            if (*++p == 0) {
                /* invalid formating */
                *error_msg = apr_psprintf(r->pool, "Missing cookie value data");
                return -1;
            }
            
            attr_value = p;
            
            for(;;) {
                while((*p != 0)&&(*p != '"')) p++;
                if (*p == 0) break;
                
                /* we allow quotation marks to be escaped
                 * using two methods: double quotation mark
                 * and escaping with a slash
                 */
                 
                if (*(p - 1) == '\\') {
                    /* this quotation mark was escaped */
                    p++;
                    continue;
                }
                
                if (*(p + 1) == '"') {
                    /* the next character is also the quotation mark */
                    p += 2;
                    continue;
                }
            
                /* we've reached the end of the string */
                break;
            }
            
            if (*p == 0) {
                /* we are missing the quotation
                 * mark at the end of the string
                 */
                *error_msg = apr_psprintf(r->pool, "Missing quotation mark at the end");
                return -1;
            }
            
            /* terminate the attribute value
             * writing over the the quotation mark
             */
            *p++ = 0;
            
            while(isspace(*p)) p++;
            if ((*p != ',')&&(*p != ';')&&(*p != 0)) {
                *error_msg = apr_psprintf(r->pool, "Expected , or ; as boundary character [got %i]", *p);
                return -1;
            }

            if (*p != 0) {            
                /* move the pointer to the first character
                 * of the next attribute
                 */
                p++;
            }

            /* decode escaped quotation marks */            
            s = d = attr_value;
            while(*s != 0) {
                if ((*s == '"')&&(*(s + 1) == '"')) {
                    *d++ = '"';
                    s += 2;
                    continue;
                }
                
                if ((*s == '\\')&&(*(s + 1) == '"')) {
                    *d++ = '"';
                    s += 2;
                    continue;
                }
                
                *d++ = *s++;
            }
            *d = 0;
            
        } else {
            /* non-quoted string */
            attr_value = p;
            while((*p != 0)&&(*p != ',')&&(*p != ';')) p++;
            
            /* we only increase the pointer if we haven't
             * reached the end of the string, to allow the
             * loop to continue
             */
            if (*p != 0) {
                *p++ = 0;
            }
        }
        
        /* use the attr_name & attr_value */
        if (dcfg->normalize_cookies) {
            char *my_error_msg;
            
            if (normalise_inplace(r, dcfg, attr_name, &my_error_msg) == NULL) {
                *error_msg = apr_psprintf(r->pool, "Error normalizing cookie name: %s", my_error_msg);
                return -1;   
            }
            if (normalise_inplace(r, dcfg, attr_value, &my_error_msg) == NULL) {
                *error_msg = apr_psprintf(r->pool, "Error normalizing cookie value: %s", my_error_msg);
                return -1;   
            }
        }
        
        if (attr_name[0] == '$') {
            if (strlen(attr_name) == 1) {
                *error_msg = apr_psprintf(r->pool, "Cookie keyword name empty");
                return -1;
            }
        
            if (prev_attr_name != NULL) {
                /* cookie keyword, we change the name we use
                 * so they can have a unique name in the cookie table
                 */
                attr_name = apr_psprintf(r->pool, "$%s_%s", prev_attr_name, attr_name + 1);
            }
        }
        
        if (strlen(attr_name) == 0) {
            *error_msg = apr_psprintf(r->pool, "Cookie name empty");
            return -1;
        }
        
        sec_debug_log(r, 4, "Adding cookie \"%s\"=\"%s\"", debuglog_escape(r->pool, attr_name), debuglog_escape(r->pool, attr_value));
        apr_table_add(parsed_cookies, attr_name, attr_value);
        
        cookie_count++;
        
        /* only keep the cookie names for later */
        if (attr_name[0] != '$') prev_attr_name = attr_name;
    }    
    
    return cookie_count;
}

const char *get_env_var(request_rec *r, char *name) {
    const char *result = apr_table_get(r->notes, name);

    if (result == NULL) {
        result = apr_table_get(r->subprocess_env, name);
    }

    if (result == NULL) {
        result = getenv(name);
    }
    
    return result;
}

const char *get_variable(request_rec *r, variable *v, apr_table_t *parsed_args) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    char *my_error_msg = NULL;
    const char *result = NULL;
    struct tm *tm;
    time_t tc;

    switch (v->type) {

        case VAR_CUSTOM:
            if (parsed_args != NULL) {
                /* we don't normalise parameter values becaue
                 * they were stored normalised
                 */
                result = apr_table_get(parsed_args, v->name);
            }
            else {
                sec_debug_log(r, 1, "get_variable: VAR_CUSTOM requested but parsed_args is NULL");
            }
            break;

        case VAR_HEADER:
            result = apr_table_get(r->headers_in, v->name);
            if (result != NULL) result = normalise_relaxed(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_ENV:
            result = apr_table_get(r->notes, v->name);

            if (result == NULL) {
                result = apr_table_get(r->subprocess_env, v->name);
            }

            if (result == NULL) {
                result = getenv(v->name);
            }
            break;

        case VAR_ARGS:
            /* this variable type should have been resolved
             * without calling this function
             */
            sec_debug_log(r, 1, "get_variable: internal error, VAR_ARGS should not be requested from this function");
            break;

        case VAR_REMOTE_ADDR:
            result = r->connection->remote_ip;
            break;

        case VAR_REMOTE_HOST:
            result = ap_get_remote_host(r->connection, r->per_dir_config, REMOTE_NAME, NULL);
            break;

        case VAR_REMOTE_USER:
            result = r->user;
            break;

        case VAR_REMOTE_IDENT:
            result = ap_get_remote_logname(r);
            break;

        case VAR_REQUEST_METHOD:
            result = r->method;
            break;

        case VAR_REQUEST_URI:
            result = r->unparsed_uri;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_AUTH_TYPE:
            result = r->ap_auth_type;
            break;

        case VAR_IS_SUBREQ:
            result = (r->main != NULL ? "true" : "false");
            break;

        case VAR_DOCUMENT_ROOT:
            result = ap_document_root(r);
            break;

        case VAR_SERVER_ADMIN:
            result = r->server->server_admin;
            break;

        case VAR_SERVER_NAME:
            result = ap_get_server_name(r);
            break;

        case VAR_SERVER_ADDR:
            result = r->connection->local_ip;
            break;

        case VAR_SERVER_PORT:
            result = apr_psprintf(r->pool, "%i", (int)ap_get_server_port(r));
            break;

        case VAR_SERVER_PROTOCOL:
            result = r->protocol;
            break;

        case VAR_SERVER_SOFTWARE:
            result = ap_get_server_version();
            break;

        case VAR_API_VERSION:
            result = apr_psprintf(r->pool, "%d:%d", MODULE_MAGIC_NUMBER_MAJOR, MODULE_MAGIC_NUMBER_MINOR);
            break;

        case VAR_TIME_YEAR:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d%02d", (tm->tm_year / 100) + 19, tm->tm_year % 100);
            break;

        case VAR_TIME:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d%02d%02d%02d%02d%02d%02d",
                     (tm->tm_year / 100) + 19, (tm->tm_year % 100),
                     tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
                     tm->tm_sec);
            break;

        case VAR_TIME_WDAY:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%d", tm->tm_wday);
            break;

        case VAR_TIME_SEC:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d", tm->tm_sec);
            break;

        case VAR_TIME_MIN:
            tc = time(NULL);       
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d", tm->tm_min);
            break;

        case VAR_TIME_HOUR:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d", tm->tm_hour);
            break;

        case VAR_TIME_MON:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d", tm->tm_mon + 1);
            break;

        case VAR_TIME_DAY:
            tc = time(NULL);
            tm = localtime(&tc);
            result = apr_psprintf(r->pool, "%02d", tm->tm_mday);
            break;

        case VAR_SCRIPT_FILENAME:
        case VAR_REQUEST_FILENAME:
            result = r->filename;
            break;

        case VAR_PATH_INFO:
            result = r->path_info;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_THE_REQUEST:
            result = r->the_request;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_QUERY_STRING:
            result = r->args;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;
            
        case VAR_HANDLER :
            result = r->handler;
            break;
            
        case VAR_COOKIE:
            /* cookies were escaped earlier */
            if (parsed_args == NULL) {
                sec_debug_log(r, 1, "get_variable: VAR_COOKIE requested but parsed_args is NULL");
            }
            else {
                result = apr_table_get(parsed_args, v->name);
            }
            break;
    }

    if (result == NULL) {
        result = "";
    }

    return result;
}

request_rec *find_last_request(request_rec *r) {
    request_rec *rlast = r;
    sec_debug_log(r, 9, "find_last_request: start with %x \"%s\"", rlast, debuglog_escape(r->pool, rlast->uri));
    while (rlast->next != NULL) {
        rlast = rlast->next;
        sec_debug_log(r, 9, "find_last_request: found next %x [%s]", rlast, debuglog_escape(r->pool, rlast->uri));
    }
    return rlast;
}

static void *sec_create_srv_config(apr_pool_t *p, server_rec *s) {
    sec_srv_config *scfg = (sec_srv_config *)apr_pcalloc(p, sizeof(*scfg));
    /* fprintf(stderr, "sec_create_srv_config: %s\n", s->server_hostname); */
    if (scfg == NULL) return NULL;

    scfg->server_response_token = 0;
    scfg->chroot_dir = NULL;
    scfg->server_signature = NULL;
    scfg->chroot_completed = 0;
    scfg->chroot_lock = ap_server_root_relative(p, "logs/modsec_chroot.lock");

    return scfg;
}

static void *sec_merge_srv_config(apr_pool_t *p, void *_parent, void *_child) {
    sec_srv_config *parent = (sec_srv_config *)_parent;
    sec_srv_config *new = (sec_srv_config *)apr_pcalloc(p, sizeof(sec_srv_config));
    if (new == NULL) return NULL;
    
    /* fprintf(stderr, "sec_merge_srv_config\n"); */
    new->server_signature = parent->server_signature;
    return new;
}

void sec_set_dir_defaults(sec_dir_config *dcfg) {
    if (dcfg == NULL) return;

    /* return immediatelly if we've already been here */
    if (dcfg->configuration_helper == 1) return;
    
    dcfg->configuration_helper = 1;
    if (dcfg->filter_engine == NOT_SET) dcfg->filter_engine = 0;
    if (dcfg->scan_output == NOT_SET) dcfg->scan_output = 0;
    if (dcfg->scan_output_mimetypes == NOT_SET_P) dcfg->scan_output_mimetypes = NULL;
    
    if (dcfg->scan_post == NOT_SET) dcfg->scan_post = 0;
    if (dcfg->auditlog_flag == NOT_SET) dcfg->auditlog_flag = 0;
    if (dcfg->filter_debug_level == NOT_SET) dcfg->filter_debug_level = 0;
    if (dcfg->filters_clear == NOT_SET) dcfg->filters_clear = 0;
    if (dcfg->action == NOT_SET_P) {
        dcfg->action = (actionset_t *)apr_pcalloc(dcfg->p, sizeof(actionset_t));
        dcfg->action->log = 1;
        dcfg->action->action = ACTION_DENY;
        dcfg->action->status = HTTP_FORBIDDEN;
        dcfg->action->skip_count = 1;
    }
    
    if (dcfg->auditlog_name == NOT_SET_P) dcfg->auditlog_name = NULL;
    if (dcfg->debuglog_name == NOT_SET_P) dcfg->debuglog_name = NULL;
    
    if (dcfg->range_start == NOT_SET) dcfg->range_start = 0;
    if (dcfg->range_end == NOT_SET) dcfg->range_end = 255;
    if (dcfg->check_encoding == NOT_SET) dcfg->check_encoding = 0;
    if (dcfg->check_unicode_encoding == NOT_SET) dcfg->check_unicode_encoding = 0;
    if (dcfg->upload_dir == NOT_SET_P) dcfg->upload_dir = NULL;
    if (dcfg->upload_keep_files == NOT_SET) dcfg->upload_keep_files = 0;
    if (dcfg->upload_approve_script == NOT_SET_P) dcfg->upload_approve_script = NULL;
    if (dcfg->upload_in_memory_limit == NOT_SET) dcfg->upload_in_memory_limit = 65535;
    
    if (dcfg->normalize_cookies == NOT_SET) dcfg->normalize_cookies = 1;
    if (dcfg->check_cookie_format == NOT_SET) dcfg->check_cookie_format = 0;
    
    if (dcfg->charset_id == NOT_SET) dcfg->charset_id = UNKNOWN_CSID;
    if (dcfg->multibyte_replacement_byte == NOT_SET) dcfg->multibyte_replacement_byte = 0x0A;
}

static void *sec_create_dir_config(apr_pool_t *p, char *path) {
    sec_dir_config *dcfg = (sec_dir_config *)apr_pcalloc(p, sizeof(*dcfg));
    if (dcfg == NULL) return NULL;
    
    /* fprintf(stderr, "sec_create_dir_config: %s\n", path); */
    
    dcfg->p = p;
    
    dcfg->configuration_helper = NOT_SET;
    dcfg->filter_engine = NOT_SET;
    dcfg->scan_post = NOT_SET;
    dcfg->scan_output = NOT_SET;
    dcfg->scan_output_mimetypes = NOT_SET_P;
    dcfg->action = NOT_SET_P;

    dcfg->signatures = apr_array_make(p, 10, sizeof(signature *));

    if (path == NULL) {
        dcfg->path = apr_pstrdup(p, "(null)");
    }
    else {
        dcfg->path = apr_pstrdup(p, path);
    }

    dcfg->auditlog_flag = NOT_SET;
    dcfg->auditlog_name = NOT_SET_P;
    dcfg->auditlog_fd = NOT_SET_P;
    
    dcfg->filter_debug_level = NOT_SET;
    dcfg->filters_clear = NOT_SET;
    dcfg->debuglog_name = NOT_SET_P;
    dcfg->debuglog_fd = NOT_SET_P;
    
    dcfg->range_start = NOT_SET;
    dcfg->range_end = NOT_SET;
    dcfg->check_encoding = NOT_SET;
    dcfg->check_unicode_encoding = NOT_SET;
    
    dcfg->upload_dir = NOT_SET_P;
    dcfg->upload_keep_files = NOT_SET;
    dcfg->upload_approve_script = NOT_SET_P;
    dcfg->upload_in_memory_limit = NOT_SET;
    
    dcfg->normalize_cookies = NOT_SET;
    dcfg->check_cookie_format = NOT_SET;
    
    dcfg->charset_id = NOT_SET;
    dcfg->multibyte_replacement_byte = NOT_SET;

    return dcfg;
}

static void *sec_merge_dir_config(apr_pool_t *p, void *_parent, void *_child) {
    sec_dir_config *parent = (sec_dir_config *)_parent;
    sec_dir_config *child = (sec_dir_config *)_child;
    
    /* merge child & parent into new */
    
    sec_dir_config *new = (sec_dir_config *)apr_pcalloc(p, sizeof(*new));
    if (new == NULL) return NULL;
    
    /* fprintf(stderr, "sec_merge_dir_config: parent=%s, child=%s\n", parent->path, child->path); */
    
    memcpy(new, child, sizeof(*child));

    new->filter_engine = (child->filter_engine == NOT_SET) ? parent->filter_engine : child->filter_engine;
    new->scan_post = (child->scan_post == NOT_SET) ? parent->scan_post : child->scan_post;
    new->action = (child->action == NOT_SET_P) ? parent->action : child->action;

    /* filters_clear is a special case, the value is not inherited from the
     * parent, they work only where explicitely used
     */
    new->filters_clear = child->filters_clear;
    
    new->signatures = apr_array_copy(p, child->signatures);
    
    /* we copy signatures from the parent only if not told not to in
     * the child configuration. new->filters_clear may be NOT_SET here
     * (defaults are configured on the fly, but this is one decision
     * we must make here.
     */
    if (new->filters_clear != 1) apr_array_cat(new->signatures, parent->signatures);
    
    new->auditlog_flag = (child->auditlog_flag == NOT_SET) ? parent->auditlog_flag : child->auditlog_flag;
    
    if (child->auditlog_fd == NOT_SET_P) {
        new->auditlog_fd = parent->auditlog_fd;
        new->auditlog_name = parent->auditlog_name;
    }
    else {
        new->auditlog_fd = child->auditlog_fd;
        new->auditlog_name = child->auditlog_name;
    }
    
    new->filter_debug_level = (child->filter_debug_level == NOT_SET) ? parent->filter_debug_level : child->filter_debug_level;
    
    if (child->debuglog_fd == NOT_SET_P) {
        new->debuglog_fd = parent->debuglog_fd;
        new->debuglog_name = parent->debuglog_name;
    }
    else {
        new->debuglog_fd = child->debuglog_fd;
        new->debuglog_name = child->debuglog_name;
    }
    
    new->range_start = (child->range_start == NOT_SET) ? parent->range_start : child->range_start;
    new->range_end = (child->range_end == NOT_SET) ? parent->range_end : child->range_end;
    new->check_encoding = (child->check_encoding == NOT_SET) ? parent->check_encoding : child->check_encoding;    
    new->check_unicode_encoding = (child->check_unicode_encoding == NOT_SET) ? parent->check_unicode_encoding : child->check_unicode_encoding;
    new->upload_dir = (child->upload_dir == NOT_SET_P) ? parent->upload_dir : child->upload_dir;
    new->upload_keep_files = (child->upload_keep_files == NOT_SET) ? parent->upload_keep_files : child->upload_keep_files;
    new->upload_approve_script = (child->upload_approve_script == NOT_SET_P) ? parent->upload_approve_script : child->upload_approve_script;
    
    new->normalize_cookies = (child->normalize_cookies == NOT_SET) ? parent->normalize_cookies : child->normalize_cookies;
    new->check_cookie_format = (child->check_cookie_format == NOT_SET) ? parent->check_cookie_format : child->check_cookie_format;
    new->charset_id = (child->charset_id == NOT_SET) ? parent->charset_id : child->charset_id;
    new->multibyte_replacement_byte = (child->multibyte_replacement_byte == NOT_SET) ? parent->multibyte_replacement_byte : child->multibyte_replacement_byte;
    
    return new;
}

unsigned char x2c(unsigned char *what) {
    register unsigned char digit;

    digit = (what[0] >= 'A' ? ((what[0] & 0xdf) - 'A') + 10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A') + 10 : (what[1] - '0'));
    
    return(digit);
}

int detect_unicode_character(request_rec *r, unsigned char *p_read) {
    int unicode_len = 0;
    unsigned int d = 0;
    unsigned char c;

    if (p_read == NULL) return 0;
    c = *p_read;
    if (c == 0) return 0;
    
    if ((c & 0xE0) == 0xC0) {
        /* two byte unicode */
        if (*(p_read + 1) == 0) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            unicode_len = 2;
            d = ((c & 0x1F) << 6) | (*(p_read + 1) & 0x3F);
        }
    }
    else if ((c & 0xF0) == 0xE0) {
        /* three byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            unicode_len = 3;
            d = ((c & 0x0F) << 12) | ((*(p_read + 1) & 0x3F) << 6) | (*(p_read + 2) & 0x3F);
        }
    }
    else if ((c & 0xF8) == 0xF0) {
        /* four byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x07) << 18) | ((*(p_read + 1) & 0x3F) << 12) | ((*(p_read + 2) & 0x3F) < 6) | (*(p_read + 3) & 0x3F);
            unicode_len = 4;
        }
    }
    else if ((c & 0xFC) == 0xF8) {
        /* five byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x03) << 24) | ((*(p_read + 1) & 0x3F) << 18) | ((*(p_read + 2) & 0x3F) << 12) | ((*(p_read + 3) & 0x3F) << 6) | (*(p_read + 4) & 0x3F);
            unicode_len = 5;
        }
    }
    else if ((c & 0xFE) == 0xFC) {
        /* six byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)||(*(p_read + 5) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 5)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x01) << 30) | ((*(p_read + 1) & 0x3F) << 24) | ((*(p_read + 2) & 0x3F) << 18) | ((*(p_read + 3) & 0x3F) << 12) | ((*(p_read + 4) & 0x3F) << 6) | (*(p_read + 5) & 0x3F);
            unicode_len = 6;
        }
    }
        
    if ((unicode_len > 1)&&((d & 0x7F) == d)) {
        unicode_len = UNICODE_ERROR_OVERLONG_CHARACTER;
    }
        
    return(unicode_len);
}

char *normalise_urlencoding_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write;
    unsigned char c;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    while ((c = *p_read) != 0) {
        
        if (c == '%') {
        
            /* see if there are enough bytes available */
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                c = 0;
            }
            else {
                /* here we only decode a %xx combo if it is a valid
                 * encoding, we leave it as is otherwise
                 */
                char c1 = *(p_read + 1), c2 = *(p_read + 2);
                
                if ( (((c1 >= '0')&&(c1 <= '9')) || ((c1 >= 'a')&&(c1 <= 'f')) || ((c1 >= 'A')&&(c1 <= 'F'))) 
                    && (((c2 >= '0')&&(c2 <= '9')) || ((c2 >= 'a')&&(c2 <= 'f')) || ((c2 >= 'A')&&(c2 <= 'F'))) ) {
                    
                    c = x2c(++p_read);
                    p_read++;
                }
            }
        } else {
            /* this check is performed only against the original data
             * and not against the decoded values (we want to
             * avoid false positives)
             */
            if ((c < dcfg->range_start)||(c > dcfg->range_end)) {
                *error_msg = apr_psprintf(r->pool, "Invalid character detected [%i]", c);
                return NULL;
            }
        }

        /* replace null bytes with whitespace */
        if (c == 0) c = 32;
        
        *p_write++ = c;
        p_read++;
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_urlencoding_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write;
    unsigned char c;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    while ((c = *p_read) != 0) {
        
        /* decode URL decoding */
        if (c == '+') c = 32;
        else
        if (c == '%') {
        
            /* see if there are enough bytes available */
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                if (dcfg->check_encoding) {
                    *error_msg = apr_psprintf(r->pool, "Invalid URL encoding detected: not enough characters");
                    return NULL;
                }
                else c = 0;
            }
            else {
        
                /* move onto the first hexadecimal letter */
                p_read++;
                
                c = *p_read;
                
                if (dcfg->check_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        *error_msg = apr_psprintf(r->pool, "Invalid URL encoding detected: invalid characters used");
                        return NULL;
                    }
                }
                
                c = *(p_read + 1);
                
                if (dcfg->check_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        *error_msg = apr_psprintf(r->pool, "Invalid URL encoding detected: invalid characters used");
                        return NULL;
                    }
                }
                
                /* decode two hexadecimal letters into a single byte */
                c = x2c(p_read);
                p_read++;
            }
        }

        if ((c < dcfg->range_start)||(c > dcfg->range_end)) {
            *error_msg = apr_psprintf(r->pool, "Invalid character detected [%i]", c);
            return NULL;
        }

        /* we replace null bytes with whitespace */        
        if (c == 0) c = 32;
        *p_write++ = c;
        p_read++;
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_other_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write, *p_slash;
    unsigned char c;
    int count;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    p_slash = NULL;
    count = 0;
    while (*p_read != 0) {
        c = *p_read;

        if (dcfg->check_unicode_encoding) {
            int urc = detect_unicode_character(r, (unsigned char *)p_read);
            
            switch(urc) {
                case UNICODE_ERROR_CHARACTERS_MISSING :
                    *error_msg = apr_psprintf(r->pool, "Invalid Unicode encoding: not enough bytes");
                    return NULL;
                    break;
                case UNICODE_ERROR_INVALID_ENCODING :
                    *error_msg = apr_psprintf(r->pool, "Invalid Unicode encoding: invalid byte value");
                    return NULL;
                    break;
                case UNICODE_ERROR_OVERLONG_CHARACTER :
                    *error_msg = apr_psprintf(r->pool, "Invalid Unicode encoding: overlong character");
                    return NULL;
                    break;
            }
            
        }
        
        switch (c) {
        
            #ifdef WIN32
            case '\\' :
            #endif
        
            case '/' :
                if (p_slash == NULL) {
    
                    /* remove the occurencies of "./" */
                    if ( (count > 1) && ((*(p_write - 1) == '.') && (*(p_write - 2) == '/')) ) {
                        count -= 2;
                        p_write -= 2;
                    }
                    
                    p_slash = p_read;
                    *p_write++ = '/';
                    p_read++;
                    count++;
                }
                else {
                    /* the previous character was a slash, we
                     * will ignore this one - just increment
                     * the read pointer
                     */
                    p_read++;
                }
                break;

            default:
                /* p_slash is used to detect more than one
                 * slash character in a row
                 */
                p_slash = NULL;
                *p_write++ = c;
                p_read++;
                count++;
                
                break;
            }
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;

    if (normalise_urlencoding_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }
    
    if (normalise_other_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }    
    
    return filter_multibyte_inplace(dcfg->charset_id, (char)dcfg->multibyte_replacement_byte, uri);
}

char *normalise_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;

    if (normalise_urlencoding_relaxed_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }
    
    if (normalise_other_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }    
    
    return filter_multibyte_inplace(dcfg->charset_id, (char)dcfg->multibyte_replacement_byte, uri);
}

char *normalise_relaxed(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    uri = apr_pstrdup(r->pool, _uri);
    if (uri == NULL) return NULL;
    
    return normalise_relaxed_inplace(r, dcfg, uri, error_msg);
}

char *normalise(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    uri = apr_pstrdup(r->pool, _uri);
    if (uri == NULL) return NULL;
    
    return normalise_inplace(r, dcfg, uri, error_msg);
}


int is_filtering_on_here(request_rec *r, sec_dir_config *dcfg) {
    
    /* refuse to work if the per dir config is null */
    if (dcfg == NULL) {
        sec_debug_log(r, 2, "Filtering off, dcfg is null");
        return 0;
    }
    
    /* refuse to work if not initialised properly */
    if (dcfg->filter_engine == NOT_SET) {
        return 0;
    }
    
    /* Since mod_security reads the request body during the
     * first request, it depends on having a filter in place
     * to feed the body back to the handler. Although we add
     * the filter to the input filter chain, Apache sometimes
     * (always?) loses the filter during redirects or
     * subrequests. So we have to search all around to see
     * if the context exists, and if it does add the filter
     * to the current request.
     */
    if (!ap_is_initial_req(r)) {
        void *ctx = NULL;

        /* look everywhere to find the main note */

        if (r->main) {
            ctx = (void *)apr_table_get(r->main->notes, NOTE);
            if (ctx != NULL) {
                sec_debug_log(r, 3, "Not initial request: found NOTE in r->main %x", r->main);
            }
        }

        if (ctx == NULL) {
            request_rec *rx = r;
            while((rx = rx->prev) != NULL) {
                ctx = (void *)apr_table_get(rx->notes, NOTE);
                if (ctx != NULL) {
                    sec_debug_log(r, 3, "Not initial request: found NOTE in r->prev %x", rx);
                    break;
                }
            }
        }

        if (ctx == NULL) {
            request_rec *rx = r;
            while((rx = rx->next) != NULL) {
                ctx = (void *)apr_table_get(rx->notes, NOTE);
                if (ctx != NULL) {
                    sec_debug_log(r, 3, "Not initial request: found NOTE in r->next %x", rx);
                    break;
                }
            }
        }
        
        /* if found, add the input filter to this request */

        if (ctx != NULL) {
            ap_filter_t *f = r->input_filters;
            int already_there = 0;

            while(f != NULL) {
                if (f->ctx == ctx) {
                    already_there = 1;
                    break;
                }
                f = f->next;
            }

            if (already_there == 0) {
                sec_debug_log(r, 3, "Not initial request: added sec_filter_in filter (ctx %x)", ctx);
                ap_add_input_filter_handle(global_sec_filter_in, ctx, r, r->connection);
            }
        }
    }

    /* refuse to work if this is a subrequest */
    if (!ap_is_initial_req(r)) {
        char *s = NULL;
        if ( ((r->main)&&((s = (char * )apr_table_get(r->main->notes, NOTE_DYNAMIC)) != NULL))
            || ((r->prev)&&((s = (char * )apr_table_get(r->prev->notes, NOTE_DYNAMIC)) != NULL)) ) {
            sec_debug_log(r, 2, "Looking into subrequest because initial request skipped because of DynamicOnly");
        }
        else {
            sec_debug_log(r, 2, "Filtering off for a subrequest");
            return 0;
        }
    }
    
    /* refuse to work if filtering is off */
    if (dcfg->filter_engine == FILTERING_OFF) {
        sec_debug_log(r, 2, "Filtering off, switched off for path \"%s\"", debuglog_escape(r->pool, dcfg->path));
        return 0;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(r->handler == NULL)) {
        apr_table_setn(r->notes, NOTE_DYNAMIC, "skipped");
        sec_debug_log(r, 2, "Filtering off for non-dynamic resources (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
        return 0;
    }

    return 1;
}

static int read_post_payload(modsec_rec *msr) {
    request_rec *r = msr->r;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    int is_multipart = FALSE;
    int put_where = POST_IN_MEMORY;
    long len;
    char *s;
    
    msr->_post_payload = NULL;
    msr->_post_len = 0;
        
    if ((r->method_number != M_POST)&&(strncmp(r->the_request, r->method, strlen(r->method)) == 0)) {
        sec_debug_log(r, 2, "read_post_payload: skipping a non-POST request");
        return 0;
    }
    
    msr->should_body_exist = 1;
    
    if (dcfg->scan_post == 0) {
        sec_debug_log(r, 2, "read_post_payload: POST scanning is off here");
        return 0;
    }

    s = (char *)get_env_var(r, "MODSEC_NOPOSTBUFFERING");
    if (s != NULL) {
        sec_debug_log(r, 2, "read_post_payload: POST scanning turned off dynamically (MODSEC_NOPOSTBUFFERING=%s)", debuglog_escape(r->pool, s));
        msr->post_payload_dynamic_off = 1;
        return 0;
    }

    /* figure out the Content-Length */
    s = (char *)apr_table_get(r->headers_in, "Content-Length");    
    if (s == NULL) {
        sec_debug_log(r, 1, "read_post_payload: Content-Length not available!");
        msr->_post_payload = NULL;
        return 0;
    }
    
    len = strtol(s, NULL, 10);
    if ((len < 0)||(len + 1 <= 0)) {
        msr->_post_payload = NULL;
        msr->tmp_message = apr_psprintf(r->pool, "Invalid Content-Length: %li", len);
        return -1;
    }    
    
    /* msr->_post_len is unsigned long int */
    msr->_post_len = len;
    
    /* test for the boundary case */
    if (msr->_post_len + 1 == 0) {
        msr->_post_payload = NULL;
        msr->tmp_message = apr_psprintf(r->pool, "Invalid Content-Length [%lu]", msr->_post_len);
        return -1;
    }
    
    /* figure out the content-type */
    s = (char *)apr_table_get(r->headers_in, "Content-Type");
    
    /*
     * if the encoding is multipart/form-data and if
     * the size of the upload is greater than the maximum allowed
     * size, redirect the payload to a temporary file on disk
     */
    if ((s != NULL)&&(strncmp(s, "multipart/form-data", 19) == 0)) {
        is_multipart = TRUE;
        if (msr->_post_len > (unsigned int)dcfg->upload_in_memory_limit) put_where = POST_ON_DISK;
    }

    {   
    sec_filter_in_ctx *ctx = NULL;
    apr_bucket_brigade *bb;
    int seen_eos = 0;
    apr_status_t rv;

    ctx = apr_pcalloc(r->pool, sizeof(*ctx));
    if (ctx == NULL) {
        msr->_post_payload = NULL;
        msr->tmp_message = apr_psprintf(r->pool, "Unable to allocate %i bytes", sizeof(*ctx));
        return -1;
    }
    ctx->type = put_where;
    ctx->is_multipart = is_multipart;
    ctx->buflen = msr->_post_len;
    
    ctx->sofar = 0;
    ctx->done_reading = 0;
    ctx->done_writing = 0;
    ctx->output_sent = 0;
    ctx->access_check_performed = 0;
    
    /* initialize multipart handling */
    if (is_multipart) {
        msr->mpd = (multipart_data *)apr_pcalloc(r->pool, sizeof(*(msr->mpd)));
        if (msr->mpd == NULL) {
            msr->_post_payload = NULL;
            msr->tmp_message = apr_psprintf(r->pool, "Unable to allocate %i bytes", sizeof(*(msr->mpd)));
            return -1;
        }
                
        /* always create file on disk, we may
         * need it for the audit log
         */
        msr->mpd->create_tmp_file = MULTIPART_TMP_FILE_CREATE;
        
        if (multipart_init(msr->mpd, r) < 0) {
            msr->_post_payload = NULL;
            msr->tmp_message = apr_psprintf(r->pool, "Invalid multipart/form-data format");
            return -1;
        }
    }
        
    if (put_where == POST_IN_MEMORY) {
        /* reserve a sufficient memory block and initialize variables */
        ctx->buffer = apr_palloc(r->pool, ctx->buflen + 1);
        if ((ctx->buffer == NULL)||(ctx->buflen + 1 == 0)) {
            msr->_post_payload = NULL;
            msr->tmp_message = apr_psprintf(r->pool, "Failed to allocate %lu bytes", ctx->buflen + 1);
            return -1;
        }
        ctx->buffer[ctx->buflen] = 0;
        ctx->bufleft = ctx->buflen;
        ctx->output_ptr = ctx->buffer;
    }
    else {
        ctx->bufleft = ctx->buflen;
    }
    
    bb = apr_brigade_create(r->pool, r->connection->bucket_alloc);
    do {
        apr_bucket *bucket = NULL;
        rv = ap_get_brigade(r->input_filters, bb, AP_MODE_READBYTES, APR_BLOCK_READ, HUGE_STRING_LEN);
        if (rv != APR_SUCCESS) {
            msr->_post_payload = NULL;
            msr->tmp_message = apr_psprintf(r->pool, "Error reading POST data, error_code=%i", rv);
            return -1;
        }

        while(!APR_BRIGADE_EMPTY(bb)) {
            const char *data;
            apr_size_t len;

            bucket = APR_BRIGADE_FIRST(bb);

            if (APR_BUCKET_IS_EOS(bucket)) {
                seen_eos = 1;
            }
            else if (APR_BUCKET_IS_FLUSH(bucket)) {
                /* do nothing */
            }
            else {
                rv = apr_bucket_read(bucket, &data, &len, APR_BLOCK_READ);
                if (rv != APR_SUCCESS) {
                    msr->_post_payload = NULL;
                    msr->tmp_message = apr_psprintf(r->pool, "Error reading from a bucket");
                    return -1;
                }
                sec_debug_log(r, 5, "read_post_payload: read %lu bytes", len);

                if (len <= ctx->bufleft) {
                    if (is_multipart) {
                        if (multipart_process_chunk(msr->mpd, data, len) < 0) {
                            msr->_post_payload = NULL;
                            msr->tmp_message = apr_psprintf(r->pool, "Error processing POST data");
                            return -1;
                        }
                    }
            
                    if (put_where == POST_IN_MEMORY) {
                        memcpy(ctx->buffer + ctx->sofar, data, len);
                        ctx->sofar += len;
                        ctx->bufleft -= len;
                    }
                    else {
                        ctx->sofar += len;
                    }
                }
                else {
                    sec_debug_log(r, 1, "read_post_payload: POST buffer overflow; %lu bytes in buffer, %lu bytes incoming", ctx->bufleft, len);
                    ctx->bufleft = 0;
                }
                
                apr_bucket_delete(bucket);
            }                
            
            if (seen_eos) break;
        }
        apr_brigade_cleanup(bb);
    } while(!seen_eos);

    ctx->done_reading = 1;    

    if (is_multipart) {
        multipart_finish(msr->mpd);
    }

    if (ctx->type == POST_ON_DISK) {
        ctx->tmp_file_name = msr->mpd->tmp_file_name;
    }

    /* add note (needed for audit logging) */
    apr_table_setn(r->notes, NOTE, (char *)ctx);
    sec_debug_log(r, 2, "read_post_payload: Added mod_security-note to %x", r);
    msr->is_body_read = 1;
        
    ap_add_input_filter_handle(global_sec_filter_in, ctx, r, r->connection);
    
    /* this is OK in all cases, ctx->buffer will
     * be NULL if the payload is not in memory
     */
    msr->_post_payload = ctx->buffer;
    msr->_post_len = ctx->buflen;
    }
        
    return 1;
}

int parse_arguments(char *s, apr_table_t *parsed_args, request_rec *r, sec_dir_config *dcfg, char **error_msg) {
    long inputlength, i, j;
    char *my_error_msg = NULL;
    char *value = NULL;
    char *buf;
    int status;

    if (error_msg == NULL) return -1;
    *error_msg = NULL;
    
    if (s == NULL) return -1;
    inputlength = strlen(s);
    if (inputlength == 0) return 1;
    if (inputlength + 1 <= 0) return -1;

    buf = (char *)malloc(inputlength + 1);
    if (buf == NULL) {
        *error_msg = apr_psprintf(r->pool, "Failed to allocate %li bytes", inputlength + 1);
        return -1;
    }

    i = 0;
    j = 0;
    status = 0;
    while (i < inputlength) {
        if (status == 0) {
            /* parameter name */
            while ((s[i] != '=') && (s[i] != '&') && (i < inputlength)) {
                buf[j] = s[i];
                j++;
                i++;
            }
            buf[j++] = 0;
        } else {
            /* parameter value */
            while ((s[i] != '&') && (i < inputlength)) {
                buf[j] = s[i];
                j++;
                i++;
            }
            buf[j++] = 0;
        }
        
        if (status == 0) {
            if (normalise_inplace(r, dcfg, buf, &my_error_msg) == NULL) {
                free(buf);
                *error_msg = apr_psprintf(r->pool, "Error normalizing parameter name: %s", my_error_msg);
                return -1;
            }
            
            if (s[i] == '&') {
                /* Empty parameter */
                sec_debug_log(r, 4, "Adding parameter: [%s][]", debuglog_escape(r->pool, buf));
                apr_table_add(parsed_args, buf, "");
                status = 0; /* unchanged */
                j = 0;
            } else {
                status = 1;
                value = &buf[j];
            }
        }
        else {
            if (normalise_inplace(r, dcfg, value, &my_error_msg) == NULL) {
                free(buf);
                *error_msg = apr_psprintf(r->pool, "Error normalizing parameter value: %s", my_error_msg);
                return -1;
            }
            sec_debug_log(r, 4, "Adding parameter: [%s][%s]", debuglog_escape(r->pool, buf), debuglog_escape(r->pool, value));
            apr_table_add(parsed_args, buf, value);
            status = 0;
            j = 0;
        }
        
        i++; /* skip over the separator */
    }

    /* last parameter was empty */
    if (status == 1) {
        sec_debug_log(r, 4, "Adding parameter: [%s][]", debuglog_escape(r->pool, buf));
        apr_table_add(parsed_args, buf, "");
    }

    free(buf);
    return 1;
}

char *remove_binary_content(request_rec *r, char *data, long size) {
    char *src, *dst, *newdata;
    
    if (data == NULL) return NULL;
    if ((size < 0)||(size + 1 <= 0)) return NULL;

    /* make a copy of the payload first */
    newdata = apr_palloc(r->pool, size + 1);
    if (newdata == NULL) {
        sec_debug_log(r, 1, "remove_binary_content: failed to allocate %li bytes", size + 1);
        return NULL;
    }
    
    /* remove zeros from the payload */
    src = data;
    dst = newdata;
    while(size--) {
        if (*src != 0) *dst++ = *src++;
        else src++;
    }
    *dst = 0;
    
    return newdata;
}

int check_single_signature(modsec_rec *msr, signature *sig) {
    int j, rs;
    
    /*
     * the idea behind non-selective filters is to apply them over
     * raw data, typically the complete first line of the request
     * and the complete POST payload
     */
        
    if (sig->is_selective == 0) {
        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at THE_REQUEST", debuglog_escape(msr->r->pool, sig->pattern));
            
        rs = check_sig_against_string(msr, sig, msr->_the_request, VAR_THE_REQUEST);
        if (rs != OK) return rs;
        
        if (msr->is_body_read) {
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", debuglog_escape(msr->r->pool, sig->pattern));
            
            if (msr->mpd != NULL) {
                /* multipart/form-data request */
                if (msr->_fake_post_payload == NULL) {
                    msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                    if (msr->_fake_post_payload == NULL) {
                        sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                        return DECLINED;
                    }
                }
                rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            } else {
                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            }
        }
    }
    else {
        variable **variables;
        const char *v;

        /* this is a selective signature, this means that we need to
         * check only one part of the request and leave the rest alone
         */
            
        /* selective signatures can be negative and non-negative;
         * non-negative signatures consist of a list of variables
         * that represent parts of the request that need to be
         * checked; negative signatures apply only to request
         * arguments, when you want to exclude an argument from
         * a check            
         */
         
        if (sig->is_negative == 0) {
                
            /* loop through signature variables and
             * check them
             */

            variables = (variable **)sig->variables->elts;
            for (j = 0; j < sig->variables->nelts; j++) {
            
                if (variables[j]->type == VAR_ARGS) {
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at QUERY_STRING", debuglog_escape(msr->r->pool, sig->pattern));
            
                    variables[j]->type = VAR_QUERY_STRING;
                    v = get_variable(msr->r, variables[j], msr->parsed_args);
                    variables[j]->type = VAR_ARGS;
                    
                    rs = check_sig_against_string(msr, sig, v, VAR_QUERY_STRING);
                    if (rs != OK) return rs;

                    if (msr->is_body_read) {
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", debuglog_escape(msr->r->pool, sig->pattern));
                  
                        if (msr->mpd != NULL) {
                            /* multipart/form-data request */
                            if (msr->_fake_post_payload == NULL) {
                                msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                                if (msr->_fake_post_payload == NULL) {
                                    sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                                    return DECLINED;
                                }
                            }
                            rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                            if (rs != OK) return rs;
                        } else {
                            rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                            if (rs != OK) return rs;
                        }
                    }
                }
                else if (variables[j]->type == VAR_POST_PAYLOAD) {
                    /* Ignore requests without bodies */
                    if (msr->should_body_exist) {
                        /* Note it can happen that a body is available but
                         * _post_payload is NULL.
                         */
                        if (msr->is_body_read == 0) {
                            /* Only complain if body is not available by configuration mistake */
                            if (msr->post_payload_dynamic_off == 0) {
                               sec_debug_log(msr->r, 1, "Filtering against POST payload requested but payload is not available");
                            }
                            return OK;
                        } else {
                            if (msr->mpd == NULL) {
                                if (msr->_post_payload == NULL) {
                                    sec_debug_log(msr->r, 1, "Expected POST payload but got NULL");
                                    return DECLINED;
                                }
                                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            } else {
                                /* multipart/form-data request */
                                if (msr->_fake_post_payload == NULL) {
                                    msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                                    if (msr->_fake_post_payload == NULL) {
                                        sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                                        return DECLINED;
                                    }
                                }
                                rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            }
                        }
                    }
                }
                else if (variables[j]->type == VAR_ARGS_NAMES) {
                    apr_table_entry_t *te;
                    const apr_array_header_t *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_NAMES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_args == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                        return DECLINED;
                    }
                        
                    arr = apr_table_elts(msr->parsed_args);
                    te = (apr_table_entry_t *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        rs = check_sig_against_string(msr, sig, te[k].key, VAR_ARGS_NAMES);
                        if (rs != OK) return rs;
                    }
                }
                else if (variables[j]->type == VAR_ARGS_VALUES) {
                    apr_table_entry_t *te;
                    const apr_array_header_t *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_VALUES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_args == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                        return DECLINED;
                    }
                    
                    arr = apr_table_elts(msr->parsed_args);
                    te = (apr_table_entry_t *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        rs = check_sig_against_string(msr, sig, te[k].val, VAR_ARGS_VALUES);
                        if (rs != OK) return rs;
                    }
                }
                else if(variables[j]->type == VAR_COOKIES_NAMES) {
                    apr_table_entry_t *te;
                    const apr_array_header_t *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_NAMES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_cookies == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                        return DECLINED;
                    }
                    
                    arr = apr_table_elts(msr->parsed_cookies);
                    te = (apr_table_entry_t *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        sec_debug_log(msr->r, 5, "Cookie \"%s\"=\"%s\"", debuglog_escape(msr->r->pool, te[k].key), debuglog_escape(msr->r->pool, te[k].val));
                        rs = check_sig_against_string(msr, sig, te[k].key, VAR_COOKIES_NAMES);
                        if (rs != OK) return rs;
                    }
                }
                else if(variables[j]->type == VAR_COOKIES_VALUES) {
                    apr_table_entry_t *te;
                    const apr_array_header_t *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_VALUES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_cookies == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                        return DECLINED;
                    }
                    
                    arr = apr_table_elts(msr->parsed_cookies);
                    te = (apr_table_entry_t *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        sec_debug_log(msr->r, 5, "Cookie \"%s\"=\"%s\"", debuglog_escape(msr->r->pool, te[k].key), debuglog_escape(msr->r->pool, te[k].val));
                        rs = check_sig_against_string(msr, sig, te[k].val, VAR_COOKIES_VALUES);
                        if (rs != OK) return rs;
                    }
                }
                else {
                    /* simple variable, get the value and check it */
                    
                    if (variables[j]->type == VAR_COOKIE) {
                        if (msr->parsed_cookies == NULL) {
                            sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                            return DECLINED;
                        }
                        v = get_variable(msr->r, variables[j], msr->parsed_cookies);
                    }
                    else {
                        if (msr->parsed_args == NULL) {
                            sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                            return DECLINED;
                        }
                        v = get_variable(msr->r, variables[j], msr->parsed_args);
                    }
                    
                    if (v != NULL) {
                        if (variables[j]->name != NULL) {
                            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s(%s)", debuglog_escape(msr->r->pool, sig->pattern), all_variables[variables[j]->type], variables[j]->name);
                        }
                        else {                        
                            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s", debuglog_escape(msr->r->pool, sig->pattern), all_variables[variables[j]->type]);
                        }
                            
                        /* sec_debug_log(msr->r, 5, "Variable value \"%s\"", v); */

                        rs = check_sig_against_string(msr, sig, (char *)v, variables[j]->type);
                        if (rs != OK) return rs;
                    }
                    else {
                        sec_debug_log(msr->r, 1, "Variable not found \"%s\"", variables[j]->name);
                    }
                }
            }
        }
        else {
            apr_table_t *our_parsed_args;
            char *fake_body = NULL;
            
            if (msr->parsed_args == NULL) {
                sec_debug_log(msr->r, 1, "Arguments are not parsed, internal error");
                return DECLINED;
            }

            our_parsed_args = apr_table_copy(msr->r->pool, msr->parsed_args);

            /* Find the unwanted variable names in the signature
             * data and remove them from the variable list.
             */
            variables = (variable **)sig->variables->elts;
            for (j = 0; j < sig->variables->nelts; j++) {
                if ((variables[j]->type == VAR_CUSTOM) && (variables[j]->action == VAR_ACTION_ALLOW)) {
                    apr_table_unset(our_parsed_args, variables[j]->name);
                }
            }
            
            fake_body = construct_fake_urlencoded(msr, our_parsed_args);
            if (fake_body == NULL) {
                sec_debug_log(msr->r, 1, "Failed with construct_fake_urlencoded");
                return DECLINED;
            }
                                                                    
            /* make the check against the compiled string */
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_SELECTIVE", debuglog_escape(msr->r->pool, sig->pattern));
            rs = check_sig_against_string(msr, sig, (char *)fake_body, VAR_ARGS_SELECTIVE);
            if (rs != OK) return rs;
        }
    }
    
    return OK;
}

int sec_check_all_signatures(modsec_rec *msr, int is_output) {
    request_rec *r = msr->r;
    signature **signatures;
    int i;
    int mode = 0;
    int skip_count = 0;
    int rc = DECLINED;
    
    /* loop through all signatures */
    signatures = (signature **)msr->dcfg->signatures->elts;
    for (i = 0; i < msr->dcfg->signatures->nelts; i++) {
    
        if (signatures[i]->actionset == NULL) {
            /* VERY IMPORTANT, configure rule action on-the-fly;
             * this cannot be done during the configuration phase
             * because per-dir configuration is also configured
             * on-the-fly
             */
            signatures[i]->actionset = msr->dcfg->action;
        }
        
        /* output rules are not processed here */
        if (signatures[i]->is_output != is_output) continue;
        
        /* check if we need to skip this rule */
        if (skip_count > 0) {
            skip_count--;
            continue;
        }
        
        /* just clear the flag, we had to use the flag
         * to detect a case when the last rule in the
         * rule chain was marked as chained            
         */
        if (mode == 2) mode = 0;
            
        /* in mode 1, we are looking for the next filter,
         * next chained that is, and then skip it to
         * execute the next filter in the chain
         */
        if (mode == 1) {
            if (signatures[i]->actionset->is_chained == 0) mode = 0;
            continue;
        }
        
        msr->tmp_message = NULL;
        msr->tmp_redirect_url = NULL;
        msr->tmp_log_message = 0;
        
        rc = check_single_signature(msr, signatures[i]);
        sec_debug_log(r, 9, "Signature check returned %i", rc);
        
        /* We won't print any messages if the rule belongs
         * to a chain - only the last rule counts
         */
        if (signatures[i]->actionset->is_chained == 0) {    
            if (msr->tmp_message != NULL) {
                apr_table_setn(r->headers_in, NOTE_MESSAGE, msr->tmp_message);
                if (msr->tmp_log_message) {
                    sec_debug_log(r, 1, "%s", msr->tmp_message);
                }
                else {
                    sec_debug_log(r, 2, "%s", msr->tmp_message);
                    apr_table_setn(r->notes, NOTE_NOAUDITLOG, "noauditlog");
                }
            } else {
                /* The final message is NULL, we unset any temporary
                 * messages that were present - it may happen when an
                 * external script is executed and rules are chained
                 */
                apr_table_unset(r->headers_in, NOTE_MESSAGE);
            }
        }
        
        /* DECLINED means that an allow action was called,
         * we need to pass the request through            
         */
        if (rc == DECLINED) {
            sec_debug_log(r, 9, "Allow request to pass through");
            return DECLINED;
        }
        
        /* OK means there was no filter match, we
         * switch to mode 1, processing will continue
         * with the next filter chain
         */
        if (rc == OK) {
            /* we go into mode 1 (looking for the last rule
             * in the chain) if this rule is not it
             */
            if (signatures[i]->actionset->is_chained == 1) {
                sec_debug_log(r, 9, "Chained rule and no match, find the next rule not in chain");
                mode = 1;
            }
            
            continue;
        }
        
        /* any status greater than zero means there
         * was a filter match, so we either stop execution
         * or proceed to the next rule if this rule was
         * chained
         */
        if (rc > 0) {
            if (signatures[i]->actionset->is_chained == 1) {
                mode = 2;
                sec_debug_log(r, 9, "Chained rule with match, continue in the loop");
                continue;
            }
            else {
rule_match:
                if (msr->tmp_redirect_url != NULL) {
                    apr_table_setn(msr->r->headers_out, "Location", msr->tmp_redirect_url);
                }
                
                sec_debug_log(r, 9, "Rule match, returning code %i", rc);
                return rc;
            }
        }
        
        /* if the return status is zero this
         * means skip some rules, so we skip
         */
        if (rc == MODSEC_SKIP) {
            skip_count = signatures[i]->actionset->skip_count;
            continue;
        }
            
        sec_debug_log(r, 1, "Unprocessed return code %i", rc);
        return DECLINED;
    }
        
    /* handle the case where there was a match on the
     * last filter so mode 2 check could not be done        
     * strickly speaking this should be a configuration error
     */
    if (mode == 2) {
        sec_debug_log(r, 1, "Last rule marked as chained - ignoring");
        goto rule_match;
    }

    return DECLINED;
}    
    
static int sec_exec_child(char *command, const char *argv[], request_rec *r, char **output) {
    apr_procattr_t *procattr;
    apr_proc_t *procnew;
    apr_status_t rc = APR_SUCCESS;
    const char * const *env;
    apr_file_t *script_out;
    char *exec_dir = NULL, *exec_command = NULL;
    
    ap_add_cgi_vars(r);
    ap_add_common_vars(r);
    
    /* PHP hack, getting around its security checks */
    apr_table_add(r->subprocess_env, "PATH_TRANSLATED", command);
    apr_table_add(r->subprocess_env, "REDIRECT_STATUS", "302");                                        
    
    env = (const char * const *)ap_create_environment(r->pool, r->subprocess_env);
    
    procnew = apr_pcalloc(r->pool, sizeof(*procnew));
    if (procnew == NULL) {
        sec_debug_log(r, 1, "sec_exec_child: Unable to allocate %i bytes", sizeof(*procnew));
        return DECLINED;
    }
    
    apr_procattr_create(&procattr, r->pool);
    apr_procattr_io_set(procattr, APR_NO_PIPE, APR_FULL_BLOCK, APR_NO_PIPE);
    
    #ifndef WIN32
    {
        char *p;
        
        /* Suexec will complain if the name of the
         * script starts with a slash. To work around
         * that we chdir to the folder, and then execute
         * the script giving a relative filename. We have
         * already forked so we can do that.
         */
        exec_dir = apr_pstrdup(r->pool, command);        
        p = strrchr(exec_dir, '/');
        if (p != NULL) {
            exec_command = p + 1;
            *p = 0;
            chdir(exec_dir);
        }
    }
    #else
        exec_command = command;
        exec_dir = "";
    #endif
    
    rc = ap_os_create_privileged_process(r, procnew, exec_command, argv, env, procattr, r->pool);
    if (rc != APR_SUCCESS) {
        sec_debug_log(r, 1, "Failed to execute: \"%s\" (rc=%d)", debuglog_escape(r->pool, command), rc);
        return rc;
    }
    
    apr_pool_note_subprocess(r->pool, procnew, APR_KILL_AFTER_TIMEOUT);
    
    script_out = procnew->out;
    if (!script_out) return APR_EBADF;
    apr_file_pipe_timeout_set(script_out, r->server->timeout);
    
    {
        char buf[260];
        char *p = buf;
        apr_size_t nbytes = 255;
        apr_status_t rc2;
        
        rc2 = apr_file_read(script_out, buf, &nbytes);
        if (rc2 == APR_SUCCESS) {
            buf[nbytes] = 0;
        
            /* if there is more than one line ignore them */
            while(*p != 0) {
                if (*p == 0x0a) *p = 0;
                p++;
            }
        
            sec_debug_log(r, 4, "sec_exec_child: First line from script output: \"%s\"", debuglog_escape(r->pool, buf));
        
            if (output != NULL) *output = apr_pstrdup(r->pool, buf);

            /* soak up the remaining data */
            nbytes = 255;
            while(apr_file_read(script_out, buf, &nbytes) == APR_SUCCESS) nbytes = 255;
        }
    }
    
    apr_proc_wait(procnew, NULL, NULL, APR_WAIT);
    
    return rc;
}

int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type) {
    request_rec *r = msr->r;
    int regex_result = 0;
    int rc = OK;
    
    if (_sig->regex == NULL) {
        sec_debug_log(r, 1, "Compiled regex for pattern \"%s\" is NULL!", debuglog_escape(r->pool, _sig->pattern));
        return OK;
    }
    
    sec_debug_log(r, 4, "Checking against \"%s\"", debuglog_escape(r->pool, (char *)s));

    regex_result = ap_regexec(_sig->regex, s, 0, NULL, 0);
    
    if ( ((regex_result == 0)&&(_sig->is_allow == 0)) || ((regex_result != 0)&&(_sig->is_allow == 1)) ) {
        actionset_t *actionset = _sig->actionset;
    
        /* perform action */
        switch(actionset->action) {
        
            case ACTION_SKIP :
                sec_debug_log(r, 2, "Skipping %i statements on pattern match \"%s\" at %s", actionset->skip_count, debuglog_escape(r->pool, _sig->pattern), all_variables[var_type]);
                rc = MODSEC_SKIP;
                break;
        
            case ACTION_ALLOW :
                msr->tmp_message = apr_psprintf(r->pool, "Access allowed based on pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                /* returning DECLINED will interrupt filter processing but
                 * it won't do anything the the request
                 */
                rc = DECLINED;
                break;
        
            case ACTION_DENY :
                msr->tmp_message = apr_psprintf(r->pool, "Access denied with code %i. Pattern match \"%s\" at %s", actionset->status, _sig->pattern, all_variables[var_type]);
                rc = actionset->status;
                break;
                
            case ACTION_REDIRECT :
                msr->tmp_message = apr_psprintf(r->pool, "Access denied with redirect to [%s]. Pattern match \"%s\" at %s", actionset->redirect_url, _sig->pattern, all_variables[var_type]);
                msr->tmp_redirect_url = actionset->redirect_url;
                rc = HTTP_MOVED_TEMPORARILY;
                break;
                
            default :
                msr->tmp_message = apr_psprintf(r->pool, "Warning. Pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                break;
        }
        
        if ((msr->tmp_message != NULL)&&(actionset->log)) msr->tmp_log_message = 1;
        
        /* execute the external script */
        if (actionset->exec) {
            sec_debug_log(r, 1, "Executing command \"%s\"", debuglog_escape(r->pool, actionset->exec_string));
            apr_table_setn(r->headers_in, NOTE_EXECUTED, actionset->exec_string);
            if (msr->tmp_message != NULL) {
                apr_table_setn(r->headers_in, NOTE_MESSAGE, msr->tmp_message);
            }
            if ((rc != DECLINED)&&(rc != MODSEC_SKIP)) {
                char *action = apr_psprintf(r->pool, "%i", rc);
                apr_table_setn(r->headers_in, NOTE_ACTION, action);
            }
            sec_exec_child(actionset->exec_string, NULL, r, NULL);
        }
        
        if (actionset->pause != 0) {
            sec_debug_log(r, 1, "Pausing [%s] for %i ms", debuglog_escape(r->pool, r->uri), actionset->pause);
            /* apr_sleep accepts microseconds */
            apr_sleep(actionset->pause * 1000);
        }
    }

    return rc;
}

char *parse_action(char *p2, actionset_t *actionset, apr_pool_t *_pool) {
    char *t = apr_pstrdup(_pool, p2);
    char *saveptr;
    char *p = strtok_r(t, ",", &saveptr);
    int found;
    
    actionset->skip_count = 1;
    actionset->action = ACTION_DENY;
    actionset->status = HTTP_FORBIDDEN;

    while (p != NULL) {
        found = 0;
        
        if (strcmp(p, "log") == 0) {
            found = 1;
            actionset->log = 1;
        }
        else if (strcmp(p, "nolog") == 0) {
            found = 1;
            actionset->log = 0;
        }
        else if (strncmp(p, "status", 6) == 0) {
            found = 1;
            actionset->action = ACTION_DENY;
            if (strlen(p) > 8) {
                actionset->status = atoi(p + 7);
            }
        }
        else if (strcmp(p, "chain") == 0) {
            found = 1;
            actionset->is_chained = 1;
        }
        else if (strcmp(p, "chained") == 0) {
            found = 1;
            actionset->is_chained = 1;
        }
        else if (strncmp(p, "skipnext", 8) == 0) {
            found = 1;
            actionset->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                actionset->skip_count = atoi(p + 9);
            }
        }
        else if (strncmp(p, "skip", 4) == 0) {
            found = 1;
            actionset->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                actionset->skip_count = atoi(p + 9);
            }
        }
        else if (strcmp(p, "deny") == 0) {
            found = 1;
            actionset->action = ACTION_DENY;
        }
        else if (strcmp(p, "allow") == 0) {
            found = 1;
            actionset->action = ACTION_ALLOW;
        }
        else if (strcmp(p, "pass") == 0) {
            found = 1;
            actionset->action = ACTION_NONE;
        }
        else if (strncmp(p, "exec", 4) == 0) {
            found = 1;
            actionset->exec = 1;
            if (strlen(p) > 6) {
                actionset->exec_string = apr_pstrdup(_pool, p + 5);
            }
        }
        else if (strncmp(p, "redirect", 8) == 0) {
            found = 1;
            actionset->action = ACTION_REDIRECT;
            if (strlen(p) > 10) {
                actionset->redirect_url = apr_pstrdup(_pool, p + 9);
            }
        }
        else if (strncmp(p, "msg", 3) == 0) {
            found = 1;
            if (strlen(p) > 5) {
                actionset->msg = apr_pstrdup(_pool, p + 4);
            }
        }
        else if (strncmp(p, "id", 2) == 0) {
            found = 1;
            if (strlen(p) > 4) {
                actionset->id = apr_pstrdup(_pool, p + 3);
            }
        } else if (strncmp(p, "pause", 5) == 0) {
            found = 1;
            if (strlen(p) > 7) {
                actionset->pause = atoi(p + 6);
            }
        }
        
        if (found == 0) {
            return apr_psprintf(_pool, "Unknown mod_security action \"%s\"", p);
        }

        p = strtok_r(NULL, ",", &saveptr);
    }
    
    /* Chained rules must always try to deny
     * access in order for chaining to work
     * properly
     */
    if (actionset->is_chained) {
        actionset->action = ACTION_DENY;
        actionset->status = HTTP_FORBIDDEN;
    }

    return NULL;
}

static const char *cmd_filter_check_encoding(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->check_encoding = flag;
    return NULL;
}

static const char *cmd_filter_check_unicode_encoding(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->check_unicode_encoding = flag;
    return NULL;
}

static const char *cmd_filter_force_byte_range(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->range_start = atoi(p1);
    dcfg->range_end = atoi(p2);
    return NULL;
}

static const char *cmd_filter_engine(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    
    if (strcasecmp(p1, "On") == 0) dcfg->filter_engine = FILTERING_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->filter_engine = FILTERING_OFF;
    else
    if (strcasecmp(p1, "DynamicOnly") == 0) dcfg->filter_engine = FILTERING_DYNAMIC_ONLY;
    else
    return (const char *)apr_psprintf(cmd->pool, "Unrecognized parameter value for SecFilterEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_filter_inheritance(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    if (flag) dcfg->filters_clear = 0;
    else dcfg->filters_clear = 1;
    return NULL;
}

static const char *cmd_server_response_token(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);

    if (cmd->server->is_virtual) {
        return "SecServerResponseToken not allowed in VirtualHost";
    }    
    
    scfg->server_response_token = flag;
    return NULL;
}

static const char *cmd_audit_engine(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    
    if (strcasecmp(p1, "On") == 0) dcfg->auditlog_flag = AUDITLOG_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->auditlog_flag = AUDITLOG_OFF;
    else
    if (strcasecmp(p1, "RelevantOnly") == 0) dcfg->auditlog_flag = AUDITLOG_RELEVANT_ONLY;
    else
    if (strcasecmp(p1, "DynamicOrRelevant") == 0) dcfg->auditlog_flag = AUDITLOG_DYNAMIC_OR_RELEVANT;
    else
    return (const char *)apr_psprintf(cmd->pool, "Unrecognized parameter value for SecAuditEngine: %s", p1);
                                    
    return NULL;
}

static const char *cmd_audit_log(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    int rc;

    dcfg->auditlog_name = ap_server_root_relative(cmd->pool, (char *)p1);
    
    rc = apr_file_open(&dcfg->auditlog_fd, dcfg->auditlog_name, 
                    APR_WRITE | APR_APPEND | APR_CREATE,
                   CREATEMODE, cmd->pool);

    if (rc != APR_SUCCESS) {
        return apr_psprintf(cmd->pool, "mod_security: Failed to open the audit log file: %s", dcfg->auditlog_name);
    }

    return NULL;
}

static const char *cmd_scan_post(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_post = flag;
    return NULL;
}

static const char *cmd_scan_output(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_output = flag;
    return NULL;
}

static const char *cmd_default_action(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->action = (actionset_t *)apr_pcalloc(cmd->pool, sizeof(actionset_t));
    return parse_action((char *)p1, dcfg->action, cmd->pool);
}

static const char *cmd_chroot_dir(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    char cwd[1025] = "";

    if (cmd->server->is_virtual) {
        return "SecChrootDir not allowed in VirtualHost";
    }    
    
    scfg->chroot_dir = (char *)p1;
    
    if (getcwd(cwd, 1024) == NULL) {
        return "SecChrootDir: failed to get the current working directory";
    }
    
    if (chdir(scfg->chroot_dir) < 0) {
        return apr_psprintf(cmd->pool, "SecChrootDir: failed to chdir to \"%s\", errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
    }
    
    if (chdir(cwd) < 0) {
        return apr_psprintf(cmd->pool, "SecChrootDir: failed to chdir to \"%s\", errno=%d(%s)", cwd, errno, strerror(errno));
    }
    
    return NULL;
}

static const char *cmd_chroot_lock(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    
    if (cmd->server->is_virtual) {
        return "SecChrootLock not allowed in VirtualHost";
    }

    scfg->chroot_lock = ap_server_root_relative(cmd->pool, p1);
    if (scfg->chroot_lock == NULL) {
        return "SecChrootLock: allocation failed";
    }
    
    return NULL;
}

static const char *cmd_server_signature(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    
    if (cmd->server->is_virtual) {
        return "SecServerSignature not allowed in VirtualHost";
    }
    
    scfg->server_signature = (char *)p1;        
    return NULL;
}            

static const char *cmd_upload_dir(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    if (strcasecmp(p1, "none") == 0) dcfg->upload_dir = NULL;
    else dcfg->upload_dir = ap_server_root_relative(cmd->pool, p1);
    return NULL;
}

static const char *cmd_upload_in_memory_limit(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->upload_in_memory_limit = atoi(p1);
    if (dcfg->upload_in_memory_limit < 0) return "Upload memory limit cannot be negative";
    return NULL;
}

static const char *cmd_upload_keep_files(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->upload_keep_files = flag;
    /* TODO does the directory exist? */
    return NULL;
}

static const char *cmd_upload_approve_script(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    if (strcasecmp(p1, "none") == 0) dcfg->upload_approve_script = NULL;
    else dcfg->upload_approve_script = (char *)p1;
    /* TODO does the script exist? */
    return NULL;
}

char *strtolower(char *str) {
    char *c = str;
    
    if (str == NULL) return NULL;
    
    while(*c != 0) {
        *c = tolower(*c);
        c++;
    }
    
    return str;
}

static const char *cmd_filter_output_mimetypes(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_output_mimetypes = apr_psprintf(cmd->pool, " %s ", p1);
    strtolower(dcfg->scan_output_mimetypes);
    return NULL;
}            

static const char *cmd_filter(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2) {
    sec_dir_config *dcfg = in_dcfg;
    signature *sig;

    sig = apr_pcalloc(cmd->pool, sizeof(*sig));
    if (sig == NULL) return FATAL_ERROR;
    
    /* p1 is the regular expression string */
    if (p1[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = (char *)p1;
        sig->regex = ap_pregcomp(cmd->pool, p1 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    else {
        sig->pattern = (char *)p1;
        sig->regex = ap_pregcomp(cmd->pool, p1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return apr_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    /* p2 is an optional action string */
    if (p2 != NULL) {
        char *rc;
        
        sig->actionset = (actionset_t *)apr_pcalloc(cmd->pool, sizeof(actionset_t));
        rc = parse_action((char *)p2, sig->actionset, cmd->pool);
        if (rc != NULL) return rc;
    }

    /* add the signature to the list of all signatures */
    *(signature **)apr_array_push(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_filter_debug_log(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    int rc;
    
    dcfg->debuglog_name = ap_server_root_relative(cmd->pool, p1);
   
    rc = apr_file_open(&dcfg->debuglog_fd, dcfg->debuglog_name,
                   APR_WRITE | APR_APPEND | APR_CREATE,
                   CREATEMODE, cmd->pool);

    if (rc != APR_SUCCESS) {
        return apr_psprintf(cmd->pool, "mod_security: Failed to open the debug log file: %s", dcfg->debuglog_name);
    }
    
    return NULL;
}

static const char *cmd_filter_debug_level(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->filter_debug_level = atoi(p1);
    return NULL;
}

static const char *cmd_filter_selective(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2, const char *p3) {
    sec_dir_config *dcfg = in_dcfg;
    char *p, *t, *saveptr;
    signature *sig;
    
    /* initialise the structure first */
    sig = apr_pcalloc(cmd->pool, sizeof(signature));
    if (sig == NULL) return FATAL_ERROR;
    
    sig->is_allow = 0;
    sig->is_selective = 1;
    sig->is_negative = 0;
    sig->requires_parsed_args = 0;
    sig->variables = apr_array_make(cmd->pool, 10, sizeof (variable *));
    
    if (p2[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = (char *)p2;
        sig->regex = ap_pregcomp(cmd->pool, p2 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    else {
        sig->pattern = (char *)p2;
        sig->regex = ap_pregcomp(cmd->pool, p2, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return apr_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    /* split parameter 1 apart and extract variable names */
    
    p = strdup(p1);
    t = strtok_r(p, "|", &saveptr);
    while (t != NULL) {
        char *x = t;

        /* add the token to the list */
        variable *v = (variable *)apr_pcalloc(cmd->pool, sizeof(variable));
        if (v == NULL) return FATAL_ERROR;
        v->type = VAR_UNKNOWN;
        v->name = NULL;

        /* when ! is the first character in the variable
         * name, that means that the restrictions need to be
         * relaxed for that variable (within the filter scope)
         */
        if (t[0] == '!') {
            v->action = VAR_ACTION_ALLOW;
            sig->is_negative = 1;
            sig->requires_parsed_args = 1;
            x++;
        }
        else {
            v->action = VAR_ACTION_DENY;
        }

        /* arguments */
        if (strncmp (x, "ARG_", 4) == 0) {
            v->type = VAR_CUSTOM;
            v->name = apr_pstrdup(cmd->pool, x + 4);
            sig->requires_parsed_args = 1;
        }
        /* HTTP headers */
        else if (strncmp (x, "HTTP_", 5) == 0) {
            char *px;

            v->type = VAR_HEADER;
            v->name = apr_pstrdup(cmd->pool, x + 5);
    
            /* replace all "_" with "-" */
            px = v->name;
            while (*px != 0) {
                if (*px == '_') *px = '-';
                px++;
            }
        }    
        /* COOKIES */
        else if (strncmp(x, "COOKIE_", 7) == 0) {
            v->type = VAR_COOKIE;
            v->name = apr_pstrdup(cmd->pool, x + 7);
        }
        /* environment variables */
        else if (strncmp (x, "ENV_", 4) == 0) {
            v->type = VAR_ENV;
            v->name = apr_pstrdup(cmd->pool, x + 4);
        }
        /* all arguments */
        else if (strcmp (x, "ARGS") == 0) {
            v->type = VAR_ARGS;
            v->name = apr_pstrdup(cmd->pool, x);
        }
        /* just the post payload */
        else if (strcmp(x, "POST_PAYLOAD") == 0) {
            v->type = VAR_POST_PAYLOAD;
            v->name = apr_pstrdup(cmd->pool, x);
        }
        else if (strcmp(x, "OUTPUT") == 0) {
            v->type = VAR_OUTPUT;
            v->name = apr_pstrdup(cmd->pool, x);
            sig->is_output = 1;
        }
        /* everything else */
        else {
            char **vl = all_variables;
            int i = 0;

            while (*vl != NULL) {
                if (strcmp(*vl, x) == 0) {
                    v->type = i;
                    /* v->name = apr_pstrdup(cmd->pool, x); */
                    break;
                }

                i++;
                vl++;
            }
        }

        if (v->type == VAR_UNKNOWN) {
            v->name = apr_pstrdup(cmd->pool, "UKNOWN");
            return apr_psprintf(cmd->pool, "Unknown variable name: %s", x);
        }

        if ((v->type == VAR_ARGS_NAMES)||(v->type == VAR_ARGS_VALUES)) sig->requires_parsed_args = 1;

        *(variable **)apr_array_push(sig->variables) = v;

        /* and proceed to the next token */
        t = strtok_r(NULL, "|", &saveptr);
    }

    free(p);

    /* is default action parameter present? */
    if (p3 != NULL) {
        char *rc;
        
        sig->actionset = (actionset_t *)apr_pcalloc(cmd->pool, sizeof(actionset_t));
        rc = parse_action((char *)p3, sig->actionset, cmd->pool);
        if (rc != NULL) return rc;
    }

    /* add the signature to the list of all signatures */
    *(signature **)apr_array_push(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_normalize_cookies(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->normalize_cookies = flag;
    return NULL;
}

static const char *cmd_check_cookie_format(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->check_cookie_format = flag;
    return NULL;
}

static const char *cmd_charset(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->charset_id = convert_charset_to_id((char *)p1);
    if (dcfg->charset_id == -1) {
        return apr_psprintf(cmd->pool, "Unknown charset: %s", p1);
    }
    return NULL;
}

static const command_rec sec_cmds[] = {

     AP_INIT_TAKE12 (
         "SecFilter",
         cmd_filter,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The filtering expression"
     ),
     
     AP_INIT_TAKE1 (
        "SecFilterDebugLog",
        cmd_filter_debug_log,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "The filename of the filter debugging log file"
     ),
          
     AP_INIT_TAKE1 (
         "SecFilterDebugLevel",
         cmd_filter_debug_level,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The level of the debugging log file verbosity"
     ),

     AP_INIT_TAKE23 (
         "SecFilterSelective",
         cmd_filter_selective,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The variable representing areas where filtering is wanted, the filtering regular expression and optional action to take on match"
     ),

     AP_INIT_TAKE1 (
         "SecFilterEngine",
         cmd_filter_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "On, Off, or DynamicOnly to determine when will request be filtered"
     ),
     
     AP_INIT_FLAG (
        "SecServerResponseToken",
        cmd_server_response_token,
        NULL,
        RSRC_CONF,
        "On or Off to set whether the mod_security token will appear in the server signature" 
     ),

     AP_INIT_FLAG (
         "SecFilterScanPOST",
         cmd_scan_post,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "On or Off to set whether a request body will be processed"
     ),
     
     AP_INIT_FLAG (
         "SecFilterScanOutput",
         cmd_scan_output,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "On or Off to set whether output will be scanned too"
     ),

     AP_INIT_TAKE1 (
         "SecFilterDefaultAction",
         cmd_default_action,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The default action to take on rule match"
     ),

     AP_INIT_TAKE1 (
         "SecAuditEngine",
         cmd_audit_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "On, Off, RelevantOnly or DynamicOrRelevent to determine the level of audit logging"
     ),

     AP_INIT_TAKE1 (
         "SecAuditLog",
         cmd_audit_log,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The filename of the audit log file"
     ),
     
     AP_INIT_TAKE1 (
         "SecChrootDir",
         cmd_chroot_dir,
         NULL,
         RSRC_CONF,
         "The path of the directory to which server will be chrooted"
     ),
     
     AP_INIT_TAKE1 (
         "SecChrootLock",
         cmd_chroot_lock,
         NULL,
         RSRC_CONF,
         "The filename of the lock file used during the chroot process. Defaults to \"logs/modsec_chroot.lock\""
     ),
     
     AP_INIT_TAKE1 (
         "SecServerSignature",
         cmd_server_signature,
         NULL,
         RSRC_CONF,
         "The new signature of the server"
     ),
     
     AP_INIT_TAKE1 (
         "SecFilterOutputMimeTypes",
         cmd_filter_output_mimetypes,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The list of mime types that will be scanned on output"
     ),
     
     AP_INIT_FLAG (
        "SecFilterInheritance",
        cmd_filter_inheritance,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "On or Off to set whether rules from the parent context will be inherited"
     ),
     
     AP_INIT_FLAG (
        "SecFilterCheckURLEncoding",
        cmd_filter_check_encoding,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "On or Off to set whether URL encoding validation will be performed"
     ),
     
     AP_INIT_FLAG (
        "SecFilterCheckUnicodeEncoding",
        cmd_filter_check_unicode_encoding,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "On or Off to set whether Unicode encoding validation will be performed"
     ),
     
     AP_INIT_TAKE2 (
         "SecFilterForceByteRange",
         cmd_filter_force_byte_range,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The first and the last byte value of the range that will be accepted"
     ),
     
     AP_INIT_TAKE1 (
         "SecUploadDir",
         cmd_upload_dir,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "The path to the directory where uploaded files should be stored"
     ),
     
     AP_INIT_TAKE1 (
         "SecUploadInMemoryLimit",
         cmd_upload_in_memory_limit,
         NULL,
         RSRC_CONF,
         "The maximal size of memory used for storing multipart/form-data requests"
     ),
     
     AP_INIT_FLAG (
         "SecUploadKeepFiles",
         cmd_upload_keep_files,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         "On or Off to choose whether to keep the uploaded files or not"
      ),     
       
     AP_INIT_TAKE1 (
          "SecUploadApproveScript",
          cmd_upload_approve_script,
          NULL,
          RSRC_CONF | OR_AUTHCFG,
          "The path to the script that will be called to approve every uploaded file"
     ),
     
     AP_INIT_FLAG (
        "SecFilterNormalizeCookies",
        cmd_normalize_cookies,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "On or Off to determine whether cookie values will be normalized for testing, defaults to On"
     ),
     
     AP_INIT_FLAG (
        "SecFilterCheckCookieFormat",
        cmd_check_cookie_format,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        "On or Off to determine whether cookie format will be checked, defaults to On"
     ),
     
     AP_INIT_TAKE1 (
          "SecCharset",
          cmd_charset,
          NULL,
          RSRC_CONF | OR_AUTHCFG,
          "Configures the charset"
     ),

     { NULL }
};


static int sec_logger(request_rec *_r) {
    unsigned int o1size = 0, o2size = 0;
    char *o1 = NULL, *o2 = NULL;
    request_rec *r = _r;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *modsec_message = NULL;
    const apr_array_header_t *arr;
    apr_table_entry_t *te;
    char *remote_user, *local_user, *t;
    char *escaped_the_request = real_debuglog_escape(r->pool, r->the_request);
    int i = 0;
    apr_size_t nbytes;
    apr_status_t rv;
    int is_post = 0;

    /* refuse to work if not initialised properly */
    if ((dcfg == NULL)||(dcfg->filter_engine == NOT_SET)) {
        return DECLINED;
    }
    
    if (escaped_the_request == NULL) {
        sec_debug_log(r, 1, "sec_logger: escaped_the_request is NULL");
        return DECLINED;
    }
    
    r = find_last_request(r);

    sec_debug_log(r, 2, "sec_logger: start");

    switch(dcfg->auditlog_flag) {
    
        case AUDITLOG_OFF :
            sec_debug_log(r, 2, "Audit log off here");
            return OK;
            break;
            
        case AUDITLOG_DYNAMIC_OR_RELEVANT :
            modsec_message = apr_table_get(r->headers_in, NOTE_MESSAGE);
            if (((modsec_message == NULL)&&(r->handler == NULL))||(apr_table_get(_r->notes, NOTE_NOAUDITLOG) != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-dynamic and non-relevant request (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
                return OK;
            }
            break;
            
        case AUDITLOG_RELEVANT_ONLY :
            modsec_message = apr_table_get(r->headers_in, NOTE_MESSAGE);
            if ((modsec_message == NULL)||(apr_table_get(_r->notes, NOTE_NOAUDITLOG) != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-relevant request (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
                return OK;
            }
            break;
    }
    
    /* return immediatelly if we don't have a file to write to */
    if (dcfg->auditlog_fd == NULL) {
        sec_debug_log(r, 1, "Audit log: not found, uri=\"%s\"", debuglog_escape(r->pool, r->uri));
        return OK;
    }
    
    /* ok, we now need to determine whether this is a POST request
     * or not; looking at r->method_number is not enough in cases
     * when ErrorDocument is used (it contains M_GET)
     */
    if (r->method_number == M_POST) is_post = 1;
    else
    if (strncmp(r->the_request, r->method, strlen(r->method)) != 0) is_post = 1;
    
    if (r->connection->remote_logname == NULL) {
        remote_user = "-";
    }
    else {
        remote_user = r->connection->remote_logname;
    }

    if (r->user == NULL) {
        local_user = "-";
    }
    else {
        local_user = r->user;
    }
    
    /* before allocating the first buffer, determine the size
     * of data; start with a reasonable number for the data we
     * ourselves produce and the overhead, add the_request twice,
     * and input headers
     */
    o1size = 1024 + strlen(remote_user) + strlen(local_user);
    o1size += strlen(escaped_the_request) * 3; /* the request line is used twice, once as is, once escaped */
    arr = apr_table_elts(r->headers_in);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o1size += strlen(te[i].key);
        o1size += strlen(te[i].val);
        o1size += 5;
    }
    
    o1 = apr_palloc(r->pool, o1size + 1);
    if ((o1 == NULL)||(o1size + 1 == 0)) {
        sec_debug_log(r, 1, "sec_logger: Could not allocate output buffer #1 [asked for %lu]", o1size + 1);
        return DECLINED;
    }
    
    strcpy(o1, "");
    strncat(o1, "========================================\n", o1size);
    
    t = (char *)get_env_var(r, "UNIQUE_ID");
    if (t != NULL) {
        t = apr_psprintf(r->pool, "UNIQUE_ID: %s\n", t);
        strncat(o1, t, o1size - strlen(o1));    
    }

    t = apr_psprintf(r->pool, "Request: %s %s %s [%s] \"%s\" %i %li\n",
                     r->connection->remote_ip, remote_user, local_user,
                     current_logtime(r), debuglog_escape(r->pool, escaped_the_request), r->status,
                     r->bytes_sent);
             
    strncat(o1, t, o1size - strlen(o1));

    t = apr_psprintf(r->pool, "Handler: %s\n", r->handler);
    strncat(o1, t, o1size - strlen(o1));
    
    /* see if there is an error message stored in notes */
    t = (char *)apr_table_get(r->notes, "error-notes");
    if (t != NULL) {
        t = apr_psprintf(r->pool, "Error: %s\n", t);
        strncat(o1, t, o1size - strlen(o1));
    }

    strncat(o1, "----------------------------------------\n", o1size - strlen(o1));

    /* the request line */
    t = apr_psprintf(r->pool, "%s\n", escaped_the_request);
    strncat(o1, t, o1size - strlen(o1));

    /* input headers */
    arr = apr_table_elts(r->headers_in);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = apr_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o1, t, o1size - strlen(o1));
    }

    strncat(o1, "\n", o1size - strlen(o1));
    
    /* determine the size of the second buffer */
    o2size = 1024;
    if (r->status_line != NULL) o2size += strlen(r->status_line);
    
    arr = apr_table_elts(r->headers_out);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o2size += strlen(te[i].key);
        o2size += strlen(te[i].val);
        o2size += 5;
    }
    
    arr = apr_table_elts(r->err_headers_out);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o2size += strlen(te[i].key);
        o2size += strlen(te[i].val);
        o2size += 5;
    }
    
    o2 = apr_palloc(r->pool, o2size + 1);
    if ((o2 == NULL)||(o2size + 1 == 0)) {
        sec_debug_log(r, 1, "sec_logger: Could not allocate output buffer #2 [asked for %lu]", o2size + 1);
        return DECLINED;
    }

    if (is_post) {
        strcpy(o2, "\n\n");
    }
    else {
        strcpy(o2, "");
    }

    /* status line; it is not available when the protocol used is HTTP 0.9 */
    if (r->status_line != NULL) t = apr_psprintf(r->pool, "%s %s\n", r->protocol, r->status_line);
    else t = apr_psprintf(r->pool, "%s\n", r->protocol);
    strncat(o2, t, o2size - strlen(o2));

    /* output headers */
    arr = apr_table_elts(r->headers_out);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = apr_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, o2size - strlen(o2));
    }
    
    arr = apr_table_elts(r->err_headers_out);
    te = (apr_table_entry_t *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = apr_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, o2size - strlen(o2));
    }

    /* write to the file */
    
    rv = apr_global_mutex_lock(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server, "mod_security: apr_global_mutex_lock(modsec_auditlog_lock) failed");
    }    
    
    nbytes = strlen(o1);
    apr_file_write(dcfg->auditlog_fd, o1, &nbytes);
    
    if (is_post) {
        sec_filter_in_ctx *ctx = (sec_filter_in_ctx *)apr_table_get(_r->notes, NOTE);
        modsec_rec *msr = (modsec_rec *)apr_table_get(_r->notes, NOTE_MSR);
        int body_action = 0; /* body not available by default */
        char *message = NULL, *filename = NULL;
        
        if (ctx != NULL) {
            if (ctx->is_multipart == 0) {
                if (ctx->buffer != NULL) body_action = 1; /* write from memory */
            } else {
                if ((ctx->type == POST_IN_MEMORY)&&(multipart_contains_files(msr->mpd) == 0)) {
                    /* there are no files in the payload, and we already
                     * have the payload in memory - we write it to the
                     * audit log for convenience
                     */
                    body_action = 1;
                }
                else if (msr->mpd->tmp_file_name != NULL) {
                    /* we tell the multipart cleaner not to erase the file
                     * and we reference the existing file in the audit log
                     */
                    sec_debug_log(r, 4, "sec_logger: telling multipart_cleanup not to delete the request file \"%s\"", debuglog_escape(r->pool, msr->mpd->tmp_file_name));
                    msr->mpd->create_tmp_file = MULTIPART_TMP_FILE_CREATE_LEAVE;
                    body_action = 2;
                    
                    /* Use only the base filename. If files
                     * get moved around, absolute references
                     * won't work.
                     */
                    filename = strrchr(msr->mpd->tmp_file_name, '/');
                    if (filename == NULL) filename = msr->mpd->tmp_file_name;
                    else filename = filename + 1;
                }
            }
        }
        
        message = NULL;
        switch(body_action) {
            case 0 :
                message = "[POST payload not available]";
                nbytes = strlen(message);
                break;
            case 1 :
                message = ctx->buffer;
                nbytes = ctx->buflen;
                break;
            case 2 :
                message = apr_psprintf(r->pool, "[@file:%s]", filename);
                nbytes = strlen(message);
                break;
        }
        
        if (message != NULL) {
            char *o3 = apr_psprintf(r->pool, "%lu\n", (unsigned long)nbytes);
            apr_size_t o3nbytes = strlen(o3);
            
            /* We first write out the size of the payload */
            apr_file_write(dcfg->auditlog_fd, o3, &o3nbytes);
            
            /* and then the payload itself */
            apr_file_write(dcfg->auditlog_fd, message, &nbytes);
        }
    }

    /* write the second part of the log */
    nbytes = strlen(o2);
    apr_file_write(dcfg->auditlog_fd, o2, &nbytes);
    
    rv = apr_global_mutex_unlock(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server, "mod_security: apr_global_mutex_unlock(modsec_auditlog_lock) failed");
    }    

    return OK;
}

int multipart_contains_files(multipart_data *mpd) {
    multipart_part **parts;
    int i, file_count = 0;
        
    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if ((parts[i]->type == MULTIPART_FILE)&&(parts[i]->tmp_file_size != 0)) file_count++;
    }
    
    return file_count;
}

static void sec_error_log(const char *file, int line, int level, apr_status_t status, const server_rec *s, const request_rec *r, apr_pool_t *pool, const char *fmt) {
    /* TODO this function gets called when something is written to the error log
     * if (r != NULL) sec_debug_log((request_rec *)r, 2, "HERE!");
     */
}

int sec_initialize(modsec_rec *msr) {
    char *my_error_msg = NULL;
    request_rec *r = msr->r;
    apr_status_t rc;
    
    /* Although we could use the contents of r->unparsed_uri
     * to populate the variable below I have found it to be
     * safer to do it this way.
     */    
    if ((r->args == NULL)&&(r->main != NULL)&&(r->main->args != NULL)) {
        /* I have found that the value of r->args is not populated
         * for subrequests although it is for for the main request; therefore
         * here we use the value available in the main request
         */
        msr->_the_request = apr_pstrcat(r->pool, r->uri, "?", r->main->args, NULL);
    }
    else if (r->args == NULL) {
        msr->_the_request = apr_pstrdup(r->pool, r->uri);
    }
    else {
        msr->_the_request = apr_pstrcat(r->pool, r->uri, "?", r->args, NULL);
    }
    
    msr->_the_request = normalise_inplace(r, msr->dcfg, msr->_the_request, &my_error_msg);
    if (msr->_the_request == NULL) {
        msr->tmp_message = apr_psprintf(r->pool, "Error normalizing REQUEST_URI: %s", my_error_msg);
        return perform_action(msr, msr->dcfg->action);
    }
    
    sec_debug_log(r, 4, "Normalised REQUEST_URI: \"%s\"", debuglog_escape(r->pool, msr->_the_request));
    sec_debug_log(r, 2, "Parsing arguments...");

    /* parse and validate GET parameters when available */    
    if (r->args != NULL) {
        if (parse_arguments(r->args, msr->parsed_args, r, msr->dcfg, &my_error_msg) < 0) {
            msr->tmp_message = apr_psprintf(r->pool, "Invalid parameters: %s", my_error_msg);
            return perform_action(msr, msr->dcfg->action);
        }
    }

    /* parse, optionally validate cookies */    
    if ((parse_cookies(r, msr->parsed_cookies, &my_error_msg) < 0)&&(msr->dcfg->check_cookie_format == 1)) {
        msr->tmp_message = apr_psprintf(r->pool, "Invalid cookie format: %s", my_error_msg);        
        return perform_action(msr, msr->dcfg->action);
    }

    if (msr->dcfg->scan_post) {
        char *content_type = (char *)apr_table_get(r->headers_in, "Content-Type");
        if (content_type != NULL) sec_debug_log(r, 3, "content-type = \"%s\"", debuglog_escape(r->pool, content_type));
        else sec_debug_log(r, 3, "content-type is NULL");

        rc = read_post_payload(msr);
        if (rc < 0) {
            /* the error message prepared by read_post_payload */
            return perform_action(msr, msr->dcfg->action);
        }
        
        if ((content_type != NULL) && (strcmp(content_type, "application/x-www-form-urlencoded") == 0)) {
        
            /* parse post payload */
            if (msr->_post_payload != NULL) {
                sec_debug_log(r, 3, "Parsing variables from POST payload");
                if (parse_arguments(msr->_post_payload, msr->parsed_args, r, msr->dcfg, &my_error_msg) < 0) {
                    msr->tmp_message = apr_psprintf(r->pool, "Error parsing POST parameters: %s", my_error_msg);
                    return perform_action(msr, msr->dcfg->action);
                }
                
                msr->_post_payload = normalise(r, msr->dcfg, msr->_post_payload, &my_error_msg);
                if (msr->_post_payload == NULL) {
                    msr->tmp_message = apr_psprintf(r->pool, "Error normalizing POST payload: %s", my_error_msg);
                    return perform_action(msr, msr->dcfg->action);
                }
            }
        }
        else if ((content_type != NULL) && (strncmp(content_type, "multipart/form-data", 19) == 0)) {
            
            /* get parsed variables */
            if (multipart_get_variables(msr->mpd, msr->parsed_args, msr->dcfg, &my_error_msg) < 0) {
                msr->tmp_message = apr_psprintf(r->pool, "Error parsing multipart parameters: %s", my_error_msg);
                return perform_action(msr, msr->dcfg->action);
            }
                    
            if (msr->dcfg->upload_approve_script != NULL) {
                /* we only accept 1 as a correct result; the function may also return
                 * 0 for verification failed and -1 for an error (e.g. the script cannot
                 * be executed)
                 */
                if (verify_uploaded_files(r, msr->mpd, msr->dcfg->upload_approve_script, &my_error_msg) != 1) {
                    msr->tmp_message = apr_psprintf(r->pool, "Error verifying files: %s", my_error_msg);
                    return perform_action(msr, msr->dcfg->action);
                }
            }
        }
        else if (msr->_post_payload != NULL) {
            msr->_post_payload = remove_binary_content(r, msr->_post_payload, msr->_post_len);
            if (msr->_post_payload == NULL) {
                msr->tmp_message = apr_psprintf(r->pool, "Error while removing binary content from POST");
                return perform_action(msr, msr->dcfg->action);
            }
        }
    }
    
    return DECLINED;
}

static int sec_check_access(request_rec *r) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    modsec_rec *msr;
    int rc;
    
    /* Finish the configuration process on the fly */
    sec_set_dir_defaults(dcfg);
    
    sec_debug_log(r, 2, "sec_check_access, path=%s", debuglog_escape(r->pool, dcfg->path));
    if (is_filtering_on_here(r, dcfg) == 0) return DECLINED;
    
    msr = (modsec_rec *)apr_pcalloc(r->pool, sizeof(*msr));
    if (msr == NULL) {
        sec_debug_log(r, 1, "sec_check_access: Unable to allocate %i bytes", sizeof(*msr));
        return DECLINED;
    }
    msr->r = r;
    msr->scfg = scfg;
    msr->dcfg = dcfg;
    msr->_the_request = NULL;
    msr->_post_payload = NULL;
    msr->parsed_args = apr_table_make(r->pool, 10);
    msr->parsed_cookies = apr_table_make(r->pool, 10);
    
    /* Store for sec_logger to find */
    apr_table_setn(r->notes, NOTE_MSR, (char *)msr);

    /* Initialize mod_security structures for this request */
    rc = sec_initialize(msr);
    
    /* Process rules only if there were no errors
     * in the initialization stage.
     */
    if (rc == DECLINED) {
        /* 0 means OUTPUT filters will be ignored */
        rc = sec_check_all_signatures(msr, 0);
    }        
    
    /* Make a note for the logger */
    if (rc != DECLINED) {
        char *note = apr_psprintf(r->pool, "%i", rc);
        apr_table_setn(r->headers_in, NOTE_ACTION, note);
        apr_table_setn(r->subprocess_env, NOTE_ACTED, "1");
    } else {
        apr_table_unset(r->headers_in, NOTE_ACTION);
    }
    
    return rc;
}

/*
 * This function only escapes the quotation
 * mark when it appears in the string. The name
 * of the function is not appropriate any more
 * and it will be changed in one of the future
 * releases.
 */
char *debuglog_escape(apr_pool_t *p, char *text) {
    const unsigned char *s = NULL;
    unsigned char *d = NULL;
    char *ret = NULL;
    
    if (text == NULL) return NULL;
    
    ret = apr_palloc(p, strlen(text) * 2 + 1);
    if (ret == NULL) return NULL;
    s = (const unsigned char *)text;
    d = (unsigned char *)ret;

    while(*s != 0) {
        if (*s == '"') {
            *d++ = '"';
            *d++ = '"';
        } else {
            *d++ = *s;
        }
        
        s++;
    }
    *d = 0;    
    
    return ret;
}

static const char c2x_table[] = "0123456789abcdef";

static unsigned char *c2x(unsigned what, unsigned char *where) {
    what = what & 0xff;
    *where++ = c2x_table[what >> 4];
    *where++ = c2x_table[what & 0xf];
    return where;
}

char *real_debuglog_escape(apr_pool_t *p, char *text) {
    const unsigned char *s = NULL;
    unsigned char *d = NULL;
    char *ret = NULL;
    
    if (text == NULL) return NULL;
    
    ret = apr_palloc(p, strlen(text) * 4 + 1);
    if (ret == NULL) return NULL;
    s = (const unsigned char *)text;
    d = (unsigned char *)ret;
    
    while(*s != 0) {
        if ((*s <= 0x1f)||(*s >= 0x7f)) {
            *d++ = '\\';
            *d++ = 'x';
            c2x(*s, d);
            d += 2;
        }
        else {
            switch(*s) {
                case '\b' :
                    *d++ = '\\';
                    *d++ = 'b';
                    break;
                case '\n' :
                    *d++ = '\\';
                    *d++ = 'n';
                    break;
                case '\r' :
                    *d++ = '\\';
                    *d++ = 'r';
                    break;
                case '\t' :
                    *d++ = '\\';
                    *d++ = 't';
                    break;
                case '\v' :
                    *d++ = '\\';
                    *d++ = 'v';
                    break;
                case '\\' :
                    *d++ = '\\';
                    *d++ = '\\';
                    break;
                default :
                    *d++ = *s;
                    break;
            }
        }
        
        s++;
    }
    *d = 0;
    
    return ret;
}
static void sec_debug_log(request_rec *r, int level, const char *text, ...) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    va_list ap;
    char str1[1024] = "";
    char str2[1256] = "";
    apr_size_t nbytes;
    apr_status_t rv;

    /* Return immediately if we don't have where to write
     * or if the log level of the message is higher than
     * wanted in the log.
     */
    if ((dcfg->debuglog_fd == NULL)||((level != 1)&&(level > dcfg->filter_debug_level))) return;
    
    va_start(ap, text);

    apr_vsnprintf(str1, sizeof(str1), text, ap);
    apr_snprintf(str2, sizeof(str2), "[%s] [%s/sid#%lx][rid#%lx][%s] %sN", current_logtime(r), ap_get_server_name(r), (unsigned long)(r->server), (unsigned long)r, r->uri, str1); 

    if (level <= dcfg->filter_debug_level) {
        char *escaped_str2 = real_debuglog_escape(r->pool, str2);
        if ((escaped_str2 != NULL)&&(strlen(escaped_str2) != 0)) {
            /* convert the last character in the string into a new line */
            escaped_str2[strlen(escaped_str2) - 1] = '\n';
            
            rv = apr_global_mutex_lock(modsec_debuglog_lock);
            if (rv != APR_SUCCESS) {
                ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server, "mod_security: apr_global_mutex_lock(modsec_debuglog_lock) failed");
            }    

            nbytes = strlen(escaped_str2);
            apr_file_write(dcfg->debuglog_fd, escaped_str2, &nbytes);
    
            rv = apr_global_mutex_unlock(modsec_debuglog_lock);
            if (rv != APR_SUCCESS) {
                ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server, "mod_security: apr_global_mutex_unlock(modsec_debuglog_lock) failed");
            }
        }
    }
    
    if (level == 1) {
        char *unique_id = (char *)get_env_var(r, "UNIQUE_ID");
        char *hostname = (char *)r->hostname;
        
        if (unique_id != NULL) unique_id = apr_psprintf(r->pool, " [unique_id %s]", unique_id);
        else unique_id = "";
        
        if (hostname != NULL) hostname = apr_psprintf(r->pool, " [hostname \"%s\"]", debuglog_escape(r->pool, hostname));
        else hostname = "";

        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r->server, "[client %s] mod_security: %s%s [uri \"%s\"]%s", r->connection->remote_ip, str1, hostname, debuglog_escape(r->pool, r->unparsed_uri), unique_id);
    }

    va_end(ap);
    return;
}

static apr_status_t locks_remove(void *data) {
    if (modsec_debuglog_lock != NULL) {
        apr_global_mutex_destroy(modsec_debuglog_lock);
        modsec_debuglog_lock = NULL;
    }
    if (modsec_auditlog_lock != NULL) {
        apr_global_mutex_destroy(modsec_auditlog_lock);
        modsec_auditlog_lock = NULL;
    }
    return 0;
}

#if !(defined(WIN32) || defined(NETWARE))

int create_chroot_lock(server_rec *s, apr_pool_t *p, char *lockfilename) {
    char buf[260] = "";
    int handle;
    
    handle = open(lockfilename, O_CREAT | O_TRUNC | O_RDWR, CREATEMODE_UNISTD);
    if (handle == -1) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: unable to create chroot lock \"%s\", errno=%d(%s)", lockfilename, errno, strerror(errno));
        return -1;
    }
    snprintf(buf, 255, "%i\n", getpid());
    if (write(handle, buf, strlen(buf)) != strlen(buf)) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: error writing to the chroot lock file: \"%s\"", lockfilename);
        close(handle);
        return -1;
    }
    close(handle);
    
    return 1;
}

int is_time_to_chroot(server_rec *s, apr_pool_t *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    int handle;
    char buf[260] = "";
    int ppid = getppid();

    /* always chroot if the ppid is 1 */
    if (ppid == 1) {
        unlink(scfg->chroot_lock);
        return 1;
    }    
    
    handle = open(scfg->chroot_lock, O_RDONLY);
    if (handle == -1) {
        /* the file does not exist, this is probably the
         * first time we are called
         */
        if (create_chroot_lock(s, p, scfg->chroot_lock) < 0) return -1;
        
        /* not yet the right time for chroot */
        return 0;
    } else {
        /* the file exists, read the pid to see if it is
         * the right time to perform chroot
         */
         int i = read(handle, buf, 255);
         if (i >= 0) {
             if (i > 255) i = 255;
             buf[i] = 0;
         }
         close(handle);

         if (atoi(buf) == ppid) {
            /* the pid in the file matches our parent, we should chroot */         
            unlink(scfg->chroot_lock);
            return 1;
         } else {
            /* the pid does not match our parent so
             * we assume the file was not erased when
             * it was supposed to - create the file
             * then
             */
            if (create_chroot_lock(s, p, scfg->chroot_lock) < 0) return -1;
            return 0;
         }
    }
}

#endif

static int sec_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s) {                                                                     
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    apr_status_t rv;

    if (scfg->server_response_token) {    
        ap_add_version_component(p, "mod_security/" MODULE_RELEASE);
    }
    
    /* Although this may seem not to make any sense we are
     * in fact allocating sufficient space in the signature
     * so we can change it later by using brute force.
     */    
    if (scfg->server_signature != NULL) {
        ap_add_version_component(p, scfg->server_signature);
    }
    
    change_server_signature(s, scfg);

    if ((rv = apr_global_mutex_create(&modsec_debuglog_lock, NULL, APR_LOCK_DEFAULT, p)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not create modsec_debuglog_lock");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
                                                                                       
    #ifdef __SET_MUTEX_PERMS
    rv = unixd_set_global_mutex_perms(modsec_debuglog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not set permissions on modsec_debuglog_lock; check User and Group directives");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    #endif
    
    if ((rv = apr_global_mutex_create(&modsec_auditlog_lock, NULL, APR_LOCK_DEFAULT, p)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not create modsec_auditlog_lock");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
                                                                                       
    #ifdef __SET_MUTEX_PERMS
    rv = unixd_set_global_mutex_perms(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not set permissions on modsec_auditlog_lock; check User and Group directives");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    #endif
    
    #if !(defined(WIN32) || defined(NETWARE))
    
    if (scfg->chroot_dir != NULL) {
    
        rv = is_time_to_chroot(s, p);
        if (rv < 0) {
            /* Error already reported */
            exit(1);
        }
    
        if (rv == 1) {
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, 0, s, "mod_security: chroot checkpoint #2 (pid=%i ppid=%i)", getpid(), getppid());
            
            if (chdir(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: chroot failed, unable to chdir to %s, errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
                exit(1);
            }
        
            if (chroot(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: chroot failed, path=%s, errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
                exit(1);
            }
            
            if (chdir("/") < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: chdoot failed, unable to chdir to /, errno=%d(%s)", errno, strerror(errno));
                exit(1);
            }
            
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, 0, s, "mod_security: chroot successful, path=%s", scfg->chroot_dir);
            scfg->chroot_completed = 1;
        } else {
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, 0, s, "mod_security: chroot checkpoint #1 (pid=%i ppid=%i)", getpid(), getppid());
        }
    }
    #endif
    
    apr_pool_cleanup_register(p, (void *)s, locks_remove, apr_pool_cleanup_null);
    
    return OK;
}

static void sec_child_init(apr_pool_t *p, server_rec *s) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    apr_status_t rs;

    if (modsec_debuglog_lock != NULL) {
        rs = apr_global_mutex_child_init(&modsec_debuglog_lock, NULL, p);
        if (rs != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_ERR, rs, s, "Failed to child-init debuglog mutex");
        }
    }
    
    if (modsec_auditlog_lock != NULL) {
        rs = apr_global_mutex_child_init(&modsec_auditlog_lock, NULL, p);
        if (rs != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_ERR, rs, s, "Failed to child-init auditlog mutex");
        }
    }

    #if !(defined(WIN32)||defined(NETWARE))
    
    /* Refuse to work if chroot was requested but
     * not performed. Unfortunately, we can't perform
     * this check earlier, or at a better location.
     */
    if ((scfg->chroot_dir != NULL)&&(scfg->chroot_completed == 0)) {
        ap_log_error(APLOG_MARK, APLOG_EMERG | APLOG_NOERRNO, 0, s, "mod_security: chroot requested but not completed! Exiting.");
        /* This is ugly but better than running a server
         * without a chroot when a chroot was configured.
         * We sleep a little (one sec) to prevent children
         * from dying too fast.
         */
        apr_sleep(1000000);
        exit(1);
    }
    
    #endif

    change_server_signature(s, scfg);
}

void send_error_bucket(ap_filter_t *f, int status) {
    apr_bucket_brigade *brigade;
    apr_bucket *bucket;
                
    brigade = apr_brigade_create(f->r->pool, f->r->connection->bucket_alloc);
    bucket = ap_bucket_error_create(status, NULL, f->r->pool, f->r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(brigade, bucket);
    bucket = apr_bucket_eos_create(f->r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(brigade, bucket);
    
    ap_pass_brigade(f->next, brigade);
}

/* The purpose of this input filter is to break the
 * filter chain in those cases where we have already
 * read the POST data into our buffer. So, instead
 * of getting the data from the next filter in the
 * chain we serve it from our buffer.
 */
static apr_status_t sec_filter_in(ap_filter_t *f, apr_bucket_brigade *pbbOut, ap_input_mode_t eMode, apr_read_type_e eBlock, apr_off_t nBytes) {
    request_rec *r = f->r;
    conn_rec *c = r->connection;
    sec_filter_in_ctx *ctx = NULL;
    
    sec_debug_log(r, 4, "sec_filter_in: start: inputmode=%i, readtype=%i, nBytes=%i", eMode, eBlock, nBytes);

    /* the context will always be supplied to us */
    if (!(ctx = f->ctx)) {
        sec_debug_log(r, 1, "sec_filter_in: context not found!");
        return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
    }
    
    /* Skip the call if there isn't any work left for us */
    if (ctx->done_writing == 1) {
        return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
    }
    
    if (ctx->type == POST_ON_DISK) {
        /* our data is stored on disk, here we create a small
         * buffer and open the file for reading
         */
        if (ctx->tmp_file_fd <= 0) {
            ctx->buffer = ctx->output_ptr = apr_palloc(r->pool, 4000);
            if (ctx->buffer == NULL) {
                sec_debug_log(r, 1, "sec_filter_in: Failed to allocate 4K bytes");
                return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
            }
            
            sec_debug_log(r, 4, "ctx->tmp_file_name \"%s\"", debuglog_escape(r->pool, ctx->tmp_file_name));
            
            ctx->tmp_file_fd = open(ctx->tmp_file_name, O_RDONLY);
            if (ctx->tmp_file_fd < 0) {
                sec_debug_log(r, 1, "sec_filter_in: Failed to open file \"%s\"", debuglog_escape(r->pool, ctx->tmp_file_name));
                return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
            }
        }
    }
           
    /* Send a chunk of data further down the filter chain */
    if (ctx->output_sent < ctx->sofar) {
        apr_bucket *pbktOut;        
        unsigned int len = 4000;
        
        /* We send 4000 bytes out in a go unless a smaller
         * size was requested by the caller. But we never
         * send more than 4000 bytes.
         */
        if (len > nBytes) len = (unsigned int)nBytes;
        
        /* we have fewer than $len bytes left */
        if (ctx->sofar - ctx->output_sent < len) len = ctx->sofar - ctx->output_sent;
        
        if (ctx->type == POST_ON_DISK) {
            /* read a chunk of a file into the buffer */
            if (read(ctx->tmp_file_fd, ctx->output_ptr, len) != (int)len) {
                sec_debug_log(r, 1, "sec_filter_in: Failed to read %i bytes from the tmp file", len);
                return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
            }
            /* the third parameter, NULL, means "make a copy of the data" */
            pbktOut = apr_bucket_heap_create(ctx->output_ptr, len, NULL, c->bucket_alloc);
            
            /* we don't increase ctx->output_ptr here because
             * we are always using the same buffer
             */
            ctx->output_sent += len;
        }
        else {
            /* TODO can we lower memory consumption by using the same data
             * below, by supplying a non-NULL third parameter?
             */ 
            pbktOut = apr_bucket_heap_create(ctx->output_ptr, len, NULL, c->bucket_alloc);
            ctx->output_ptr += len;
            ctx->output_sent += len;
        }
        
        APR_BRIGADE_INSERT_TAIL(pbbOut, pbktOut);        
        sec_debug_log(r, 4, "sec_filter_in: Sent %d bytes (%lu total)", len, ctx->output_sent);
    }
    
    /* are we done yet? */
    if (ctx->output_sent == ctx->sofar) {
        /* send an EOS bucket, we're done */
        apr_bucket *pbktOut = apr_bucket_eos_create(c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbOut, pbktOut);
        
        sec_debug_log(r, 4, "sec_filter_in: Sent EOS bucket");
        ctx->done_writing = 1;
        
        /* nothing left for us to do in this request */
        ap_remove_input_filter(f);
        
        if (ctx->type == POST_ON_DISK) {
            close(ctx->tmp_file_fd);
        }
    }
    
    return APR_SUCCESS;
}

static apr_status_t sec_filter_out(ap_filter_t *f, apr_bucket_brigade *pbbIn) {
    request_rec *r = f->r;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config (r->per_dir_config, &security_module);
    sec_filter_out_ctx *ctx = NULL;
    conn_rec *c = r->connection;
    apr_bucket *pbktIn;
    
    sec_debug_log(r, 3, "sec_filter_out: start");
    
    /* create a context if one does not already exist */
    if (!(ctx = f->ctx)) {
        const char *s;
        
        f->ctx = ctx = apr_pcalloc(r->pool, sizeof(*ctx));
        if (ctx == NULL) {
            sec_debug_log(r, 1, "sec_filter_out: Unable to allocate %i bytes", sizeof(*ctx));
            ap_remove_output_filter(f);
            return ap_pass_brigade(f->next, pbbIn);
        }
        
        sec_debug_log(r, 3, "sec_filter_out: content-type = \"%s\"", debuglog_escape(r->pool, (char *)r->content_type));

        /* if scan_output_mimetypes is not null that means output
         * filtering should be selective, using the content type value        
         */
        if (dcfg->scan_output_mimetypes != NULL) {
            char *content_type;
            
            /* check for the case when the content type is not set */
            if (r->content_type == NULL) {
                if (strstr(dcfg->scan_output_mimetypes, "(null)") == NULL) {
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
            }
            else {
                char *p = NULL;
                
                /* in all other cases, see if the content type appears
                 * on the acceptable mime types list            
                 */
                content_type = apr_psprintf(r->pool, " %s ", r->content_type);
                if (content_type == NULL) {
                    sec_debug_log(r, 1, "sec_filter_out: Unable to allocate %i bytes", strlen(r->content_type) + 2);
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
                strtolower(content_type);
                
                /* Hide the character encoding information
                 * if present. Sometimes the content type header
                 * looks like this "text/html; charset=xyz" ...
                 */
                p = strstr(content_type, ";");
                if (p != NULL) *p = 0;
            
                if (strstr(dcfg->scan_output_mimetypes, content_type) == NULL) {
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
            }
        }

        ctx->buflen = 0;
            
        /* look up the Content-Length header to see if we know
         * the amount of data coming our way
         */
        s = apr_table_get(r->headers_out, "Content-Length");
        
        /* try this too, mod_cgi seems to put headers there */
        if (s == NULL) s = apr_table_get(r->err_headers_out, "Content-Length");
        
        if (s != NULL) {
            long len = strtol(s, NULL, 10);
            if ((len <= 0)||(len + 1 <= 0)) {
                sec_debug_log(r, 1, "sec_filter_out: Invalid Content-Length: %li", len);
                ap_remove_output_filter(f);
                return ap_pass_brigade(f->next, pbbIn);
            }
            
            sec_debug_log(r, 3, "sec_filter_out: got Content-Length %li", len);
            
            ctx->buflen = len;
            if (ctx->buflen >= 1073741824) {
                /* refuse to work past this size */
                ap_remove_output_filter(f);
                return ap_pass_brigade(f->next, pbbIn);
            }
        }

        /* use the default buffer length if everything else fails */
        if (ctx->buflen <= 0) {
            ctx->buflen = 16384;
        }
        
        ctx->buffer = apr_palloc(f->r->pool, ctx->buflen + 1);
        if ((ctx->buffer == NULL)||(ctx->buflen + 1 == 0)) {
            sec_debug_log(r, 1, "sec_filter_out: Failed to allocate buffer [%li]", ctx->buflen + 1);
            ap_remove_output_filter(f);
            return ap_pass_brigade(f->next, pbbIn);
        }        
        
        ctx->input_ptr = ctx->buffer;
    }

    /* read data into our buffer */
    while(!APR_BRIGADE_EMPTY(pbbIn)) {
        const char *data;
        apr_size_t len;
        
        pbktIn = APR_BRIGADE_FIRST(pbbIn);
        
        if (AP_BUCKET_IS_ERROR(pbktIn)) {
            sec_debug_log(r, 2, "sec_filter_out: Got error bucket, abandoning output filtering!");
            ap_remove_output_filter(f);
            return ap_pass_brigade(f->next, pbbIn);
        }
        else if (APR_BUCKET_IS_FLUSH(pbktIn)) {
            /* do nothing */
        }
        else if(APR_BUCKET_IS_EOS(pbktIn)) {
            sec_debug_log(r, 3, "sec_filter_out: done reading");
            ctx->buffer[ctx->bufused] = 0;
            ctx->done_reading = 1;
        }
        else {    
            apr_status_t rv;
            
            rv = apr_bucket_read(pbktIn, &data, &len, APR_BLOCK_READ);
            if (rv != APR_SUCCESS) {
                sec_debug_log(r, 1, "sec_filter_out: apr_bucket_read failed with %i", rv);
                ap_remove_output_filter(f);
                return ap_pass_brigade(f->next, pbbIn);
            }
        
            sec_debug_log(r, 3, "sec_filter_out: got %u bytes, bufused=%u, buflen=%u", len, ctx->bufused, ctx->buflen);

            if(ctx->bufused + len > ctx->buflen) {
                char *newbuffer;
                unsigned long int newsize = ctx->buflen * 2;
            
                if (ctx->buflen >= 1073741824) {
                    /* refuse to work past this size */
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
            
                /* increase the size of the new buffer until the data fits */
                while(ctx->bufused + len >= newsize) newsize = newsize * 2;
            
                sec_debug_log(r, 3, "sec_filter_out: expanding buffer to %lu", newsize);
            
                /* allocate a larger buffer */
                newbuffer = apr_palloc(f->r->pool, newsize + 1);
                if ((newbuffer == NULL)||(newsize + 1 == 0)) {
                    sec_debug_log(r, 1, "sec_filter_out: Failed to allocate buffer [%lu]", newsize + 1);
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
                memcpy(newbuffer, ctx->buffer, ctx->bufused);
            
                ctx->buffer = newbuffer;
                ctx->buflen = newsize;
                ctx->input_ptr = ctx->buffer + ctx->bufused;
            }

            memcpy(ctx->input_ptr, data, len);
            ctx->input_ptr += len;
            ctx->bufused += len;
        }
        
        apr_bucket_delete(pbktIn);
    }

    /* we wait patiently (for the next call) until we get
     * all of the output
     */
    if (ctx->done_reading == 0) return APR_SUCCESS;
    
    /* this shouldn't happen but doesn't hurt us to check */
    if (ctx->done_writing == 1) {
        sec_debug_log(r, 1, "sec_filter_out: internal error, we shouldn't have arrived here");
        return ap_pass_brigade(f->next, pbbIn);
    }
    
    /* TODO check output headers */
    
    /* check the output */
    {
        modsec_rec msr;
        signature **signatures;
        int i, rv;

        signatures = (signature **)dcfg->signatures->elts;
        for (i = 0; i < dcfg->signatures->nelts; i++) {
    
            /* ignore non-output filters */
            if (signatures[i]->is_output != 1) continue;
            
            sec_debug_log(r, 2, "Checking signature \"%s\" at OUTPUT", debuglog_escape(r->pool, signatures[i]->pattern));
            
            /* quick hack, we simulate the msr structure here,
             * it will be ok for the time being,
             * until I get some serious code refactoring and
             * rework the whole output filtering thing
             */
            memset(&msr, 0, sizeof(msr));        
            msr.r = r;
            msr.dcfg = dcfg;
            
            rv = check_sig_against_string(&msr, signatures[i], ctx->buffer, VAR_OUTPUT);
            if ((rv != OK)&&(rv != DECLINED)) {
                if (msr.tmp_message != NULL) {
                    apr_table_setn(r->headers_in, NOTE_MESSAGE, msr.tmp_message);
                    if (msr.tmp_log_message) {
                        sec_debug_log(r, 1, "%s", msr.tmp_message);
                    }
                    else {
                        sec_debug_log(r, 2, "%s", msr.tmp_message);
                    }
                }
    
                if (msr.tmp_redirect_url != NULL) {
                    apr_table_setn(msr.r->headers_out, "Location", msr.tmp_redirect_url);
                }

                /* dirty hack, but filtering doesn't work with mod_deflate
                 * (only when we are rejecting a requests, works fine otherwise)
                 * so we remove it from the filter list; we only do this if
                 * we need to interrupt the request
                 */
                {
                    ap_filter_t *f = r->output_filters;
                    while(f != NULL) {
                        if (strcasecmp(f->frec->name, "deflate") == 0) {
                            ap_remove_output_filter(f);
                            sec_debug_log(r, 2, "sec_filter_out: Removed deflate from the output_filters list");
                            break;
                        }
                        f = f->next;
                    }
                }
            
                /* TODO headers must also be removed from the output!! */
                
                /* it seems that mod_proxy sets the status line
                 * and it later overrides the real status
                 *  in ap_send_error_response; so we kill it here
                 */
                r->status_line = NULL;
                
                send_error_bucket(f, rv);
                
                ctx->done_writing = 1;
                return rv;
            }
        }
    }
    
    /* if we're that means that all went well and that
     * we now need to send the output to the filter chain
     */
    ctx->output_ptr = ctx->buffer;
    ctx->output_sent = 0;
    
    while(ctx->output_sent < ctx->bufused) {
        apr_bucket_brigade *pbbTmp;
        apr_bucket *pbktTmp;
        unsigned int batch = 4000;
        
        /* adjust the chunk size */
        if (ctx->bufused - ctx->output_sent < batch) batch = ctx->bufused - ctx->output_sent;
            
        pbbTmp = apr_brigade_create(r->pool, c->bucket_alloc);
        /* TODO again, we are allocating memory for data that we already have
         * in memory - we should just create a single bucket out of all the
         * data we have.
         */
        pbktTmp = apr_bucket_heap_create(ctx->output_ptr, batch, NULL, c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbTmp, pbktTmp);
    
        ctx->output_ptr += batch;
        ctx->output_sent += batch;
            
        sec_debug_log(r, 3, "sec_filter_out: sent %i bytes", batch);
        
        ap_pass_brigade(f->next, pbbTmp);
    }

    /* TODO don't we have a function for this already */
    /* send the EOS bucket */
    {
        apr_bucket_brigade *pbbTmp;
        apr_bucket *pbktTmp;
        
        pbbTmp = apr_brigade_create(r->pool, c->bucket_alloc);
        pbktTmp = apr_bucket_eos_create(f->r->connection->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbTmp, pbktTmp);
        ap_pass_brigade(f->next, pbbTmp);
        sec_debug_log(r, 3, "sec_filter_out: done writing");
    }
    
    ctx->done_writing = 1;
        
    ap_remove_output_filter(f);
    return APR_SUCCESS;
}

static void sec_pre(request_rec *r) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    
    if (dcfg == NULL) return;
    if (is_filtering_on_here(r, dcfg) == 0) return;    
    
    if (dcfg->scan_output == 1) {    
        sec_debug_log(r, 2, "scan_pre: adding the output filter to the filter list");
        ap_add_output_filter_handle(global_sec_filter_out, NULL, r, r->connection);
    }
    else {
        sec_debug_log(r, 2, "sec_pre: output filtering is off here");
    }
}

static void register_hooks(apr_pool_t *p) {
    ap_hook_post_config(sec_init, NULL, NULL, APR_HOOK_REALLY_LAST);
    ap_hook_log_transaction(sec_logger, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_fixups(sec_check_access, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_child_init(sec_child_init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_error_log(sec_error_log, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_insert_filter(sec_pre, NULL, NULL, APR_HOOK_FIRST);
    global_sec_filter_in = ap_register_input_filter("MODSEC_IN", sec_filter_in, NULL, AP_FTYPE_CONTENT_SET) ;
    global_sec_filter_out = ap_register_output_filter("MODSEC_OUT", sec_filter_out, NULL, AP_FTYPE_CONTENT_SET);
}

char *get_temp_folder(void) {
    char *filename = getenv("TEMP");    
    if (filename != NULL) return filename;
    
    filename = getenv("TMP");
    if (filename != NULL) return filename;
    
    #if defined NETWARE
    return("sys:/tmp/");
    #elif defined WIN32
    return("");
    #else
    return("/tmp/");
    #endif
}

apr_status_t multipart_finish(multipart_data *mpd) {
    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        close(mpd->tmp_file_fd);
    }
    return 1;
}

apr_status_t multipart_cleanup(void *data) {
    multipart_data *mpd = (multipart_data *)data;
    
    if (data == NULL) return -1;    
    sec_debug_log(mpd->r, 4, "multipart_cleanup: Started");

    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        /* delete the file if we are supposed to delete it */
        if (mpd->create_tmp_file == MULTIPART_TMP_FILE_CREATE) {
            if ((mpd->tmp_file_name != NULL)&&(unlink(mpd->tmp_file_name)) < 0) {
                sec_debug_log(mpd->r, 1, "multipart_cleanup: Failed to delete file (request) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, mpd->tmp_file_name), errno, strerror(errno));
            } else {
                sec_debug_log(mpd->r, 2, "multipart_cleanup: Deleted file (request) \"%s\"", debuglog_escape(mpd->r->pool, mpd->tmp_file_name));
            }
        }
        
        /* otherwise leave the file where it is
         * MULTIPART_TMP_FILE_CREATE_LEAVE
         */
    }

    /* loop through the list of parts
     * and delete all temporary files, but only if
     * file storage is not requested
     */
    if (mpd->dcfg->upload_keep_files == 0) {
        multipart_part **parts;
        int i;
        
        parts = (multipart_part **)mpd->parts->elts;
        for(i = 0; i < mpd->parts->nelts; i++) {
            if (parts[i]->type == MULTIPART_FILE) {
                if (parts[i]->tmp_file_name != NULL) {
                    sec_debug_log(mpd->r, 4, "multipart_finish: deleting temporary file (part) [%s]", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    if (unlink(parts[i]->tmp_file_name) < 0) {
                        sec_debug_log(mpd->r, 1, "multipart_cleanup: Failed to delete file (part) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name), errno, strerror(errno));
                    } else {
                        sec_debug_log(mpd->r, 2, "multipart_cleanup: Deleted file (part) \"%s\"", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    }
                }
            }
        }
    } else {
        /* delete empty files only */
        multipart_part **parts;
        int i;
        
        parts = (multipart_part **)mpd->parts->elts;
        for(i = 0; i < mpd->parts->nelts; i++) {
            if ((parts[i]->type == MULTIPART_FILE)&&(parts[i]->tmp_file_size == 0)) {
                if (parts[i]->tmp_file_name != NULL) {
                    sec_debug_log(mpd->r, 4, "multipart_finish: deleting empty temporary file (part) [%s]", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    if (unlink(parts[i]->tmp_file_name) < 0) {
                        sec_debug_log(mpd->r, 1, "multipart_cleanup: Failed to delete empty file (part) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name), errno, strerror(errno));
                    } else {
                        sec_debug_log(mpd->r, 2, "multipart_cleanup: Deleted empty file (part) \"%s\"", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    }
                }
            }
        }
    }
    
    return 1;
}

int multipart_init(multipart_data *mpd, request_rec *r) {
    char *content_type;
    
    mpd->dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    mpd->p = r->pool;
    mpd->r = r;
    
    content_type = (char *)apr_table_get(r->headers_in, "Content-Type");
    if (content_type == NULL) {
        sec_debug_log(r, 1, "multipart_init: Content-Type header not available");
        return -1;
    }
    
    mpd->boundary = strstr(content_type, "boundary=");
    if ((mpd->boundary != NULL)&&(*(mpd->boundary + 9) != 0)) mpd->boundary = mpd->boundary + 9;
    else {
        sec_debug_log(r, 1, "multipart_init: Boundary not found or invalid");
        return -1;
    }
    
    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        char *folder = NULL;

        if (mpd->dcfg->upload_dir != NULL) folder = mpd->dcfg->upload_dir;
        else folder = get_temp_folder();
        
        mpd->tmp_file_name = apr_psprintf(r->pool, "%s/%s-%s-request_body-XXXXXX", folder, current_filetime(mpd->r), mpd->r->connection->remote_ip);
        if (mpd->tmp_file_name == NULL) {
            sec_debug_log(r, 1, "multipart_init: Memory allocation failed");
            return -1;
        }
        
        mpd->tmp_file_fd = sec_mkstemp(mpd->tmp_file_name);        
        if (mpd->tmp_file_fd < 0) {
            sec_debug_log(r, 1, "multipart_init: Failed to create file [%s] because %d(%s)", debuglog_escape(mpd->r->pool, mpd->tmp_file_name), errno, strerror(errno));
            return -1;
        }
    }
    
    mpd->parts = apr_array_make(mpd->p, 10, sizeof(multipart_part *));
    mpd->bufleft = MULTIPART_BUF_SIZE;
    mpd->bufptr = mpd->buf;
    mpd->buf_contains_line = 1;
    mpd->mpp = NULL;

    /* schedule resource cleanup for later */
    apr_pool_cleanup_register(r->pool, (void *)mpd, multipart_cleanup, apr_pool_cleanup_null);
    
    return 1;
}

int multipart_process_chunk(multipart_data *mpd, const char *buf, unsigned int size) {
    char *inptr = (char *)buf;
    unsigned int inleft = size;

    /* write the data to the temporary file if we are
     * required to do so
     */
    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        if (write(mpd->tmp_file_fd, buf, size) != (int)size) {
            sec_debug_log(mpd->r, 1, "multipart_process_chunk: write to tmp file failed");
            return -1;
        }
    }

    /* here we loop through all the data available, byte by byte */
    while(inleft > 0) {
        char c = *inptr++;
        inleft = inleft - 1;
        
        *(mpd->bufptr) = c;
        mpd->bufptr++;
        mpd->bufleft--;

        /* until we either reach the end of the line
         * or the end of our internal buffer        
         */
        if ((c == 0x0a)||(mpd->bufleft == 0)) {
            *(mpd->bufptr) = 0;
            
            /* boundary preconditions: length of the line greater than
             * the length of the boundary + the first two characters
             * are dashes "-"
             */
            if ( mpd->buf_contains_line
                && (strlen(mpd->buf) > strlen(mpd->boundary) + 2)
                && (((*(mpd->buf) == '-'))&&(*(mpd->buf + 1) == '-')) ) {
                    
                if (strncmp(mpd->buf + 2, mpd->boundary, strlen(mpd->boundary)) == 0) {
                    if (multipart_process_boundary(mpd) < 0) return -1;
                }
            }
            else {
                if (multipart_process_data_chunk(mpd) < 0) return -1;
            }

            /* reset the pointer to the beginning of the buffer
             * and continue to accept input data            
             */
            mpd->bufptr = mpd->buf;
            mpd->bufleft = MULTIPART_BUF_SIZE;
            mpd->buf_contains_line = (c == 0x0a) ? 1 : 0;
        }
    }
    
    return 1;
}

char *multipart_construct_filename(multipart_data *mpd) {
    char c, *p, *q = mpd->mpp->filename;
    char *filename;

    /* find the last forward slash and consider the
     * filename to be only what's right from it
     */
    while((p = strstr(q, "\\")) != NULL) {
        q = p;
    }
    
    /* do the same for the slash character, just in case */
    while((p = strstr(q, "/")) != NULL) {
        q = p;
    }

    /* allow letters, digits and dots, replace
     * everything else with underscores    
     */
    p = filename = apr_pstrdup(mpd->p, q);
    while((c = *p) != 0) {    
        if (!( isalnum(c)||(c == '.') )) *p = '_';
        p++;
    }
    
    return filename;
}

int multipart_process_data_chunk(multipart_data *mpd) {
    sec_debug_log(mpd->r, 9, "multipart_process_data_chunk: state=%i, size=%i", mpd->mpp_state, MULTIPART_BUF_SIZE - mpd->bufleft);

    /* while mpp_state equals zero we are still reading
     * part headers
     */
    if (mpd->mpp_state == 0) {
        if (mpd->mpp == NULL) {
            /* the first line in the request must be a line
             * with a boundary but nothing prevents a malicuos
             * user to submit an invalid body
             */
            sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: data found but no boundary");
            /* return -1; */
        }
        else if (mpd->buf[1] == 0x0a) {
            sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: empty line, going to state 1");
            mpd->mpp_state = 1;
            
            /* check that we have all the information
             * we must have the name of the field
             */
            if (mpd->mpp->name == NULL) {
                sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: part name unknown");
                return -1;
            }
        }
        else {
            /* figure out what kind of part we have */
            if (strncasecmp(mpd->buf, "content-disposition: form-data", 30) == 0) {
                char *p1, *p2;
                
                p1 = strstr(mpd->buf + 30, "name=\"");
                if (p1 != NULL) {
                    p2 = p1 = p1 + 6;
                    while((*p2 != 0)&&(*p2 != '"')) p2++;
                    
                    mpd->mpp->name = apr_pcalloc(mpd->p, p2 - p1 + 1);
                    memcpy(mpd->mpp->name, p1, p2 - p1);
                    sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got attribute name \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->name));
                }
                                
                p1 = strstr(mpd->buf + 30, "filename=\"");
                if (p1 != NULL) {
                    p2 = p1 = p1 + 10;
                    while((*p2 != 0)&&(*p2 != '"')) p2++;
                    
                    mpd->mpp->filename = apr_pcalloc(mpd->p, p2 - p1 + 1);
                    memcpy(mpd->mpp->filename, p1, p2 - p1);
                    sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got attribute filename \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->filename));
                    mpd->mpp->type = MULTIPART_FILE;
                }
                
            }
            /* get the content type */
            else if (strncasecmp(mpd->buf, "content-type:", 13) == 0) {
                char *p;
                int i = 13;
                
                if (*(mpd->buf + 13) == 0x20) i = 14;
                mpd->mpp->content_type = apr_pstrdup(mpd->p, mpd->buf + i);
                
                /* get rid of that stuff at the end */
                p = mpd->mpp->content_type;
                while(*p != 0) {
                    if ((*p == 0x0a)||(*p == 0x0d)) *p = 0;
                    p++;
                }
                    
                sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got content_type for part \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->content_type));
            }
            else {
                /* ignore header */
                sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: ignoring header \"%s\"", debuglog_escape(mpd->r->pool, mpd->buf));
            }
        }
    }
    else {
        char *p = mpd->buf + (MULTIPART_BUF_SIZE - mpd->bufleft) - 2;
        char localreserve[2];
        int bytes_reserved = 0;
        
        /* preserve the last two bytes for later */
        if (MULTIPART_BUF_SIZE - mpd->bufleft >= 2) {
            bytes_reserved = 1;
            localreserve[0] = *p;
            localreserve[1] = *(p + 1);
            mpd->bufleft += 2;
            *p = 0;
        }
        
        /* add data to the part we are building */
        if (mpd->mpp->type == MULTIPART_FILE) {
        
            /* only store individual files on disk if we are going
             * to keep them or if we need to have them approved later
             */
            if ((mpd->dcfg->upload_approve_script != NULL)||(mpd->dcfg->upload_keep_files > 0)) {
            
            /* first create a temporary file if we don't have it already */
            if (mpd->mpp->tmp_file_fd == 0) {
                char *filename = multipart_construct_filename(mpd);
                
                /* the temp folder must be chosen in the configuration
                 * create the filename first
                 */
                if (mpd->dcfg->upload_dir != NULL) {
                    mpd->mpp->tmp_file_name = apr_psprintf(mpd->p, "%s/%s-%s-%s", mpd->dcfg->upload_dir, current_filetime(mpd->r), mpd->r->connection->remote_ip, filename);
                }
                else {
                    mpd->mpp->tmp_file_name = apr_psprintf(mpd->p, "%s/%s-%s-%s", get_temp_folder(), current_filetime(mpd->r), mpd->r->connection->remote_ip, filename);
                }
                
                if ((mpd->mpp->tmp_file_fd = open(mpd->mpp->tmp_file_name, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE_UNISTD)) == -1) {
                    /* we've failed while opening the page, so we'll try
                     * again with a more unique filename
                     */
                    mpd->mpp->tmp_file_name = apr_pstrcat(mpd->p, mpd->mpp->tmp_file_name, "_XXXXXX", NULL);
                    mpd->mpp->tmp_file_fd = sec_mkstemp(mpd->mpp->tmp_file_name);
                }
        
                /* do we have an opened file? */
                if (mpd->mpp->tmp_file_fd < 0) {
                    sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: Failed to create file \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
                    return -1;
                }
            }
        
            /* write the reserve first */
            if (mpd->reserve[0] == 1) {
                if (write(mpd->mpp->tmp_file_fd, &mpd->reserve[1], 2) != 2) {
                    sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: writing to \"%s\" failed", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
                }
                mpd->mpp->tmp_file_size += 2;
            }
            
            /* write data to the file */
            if (write(mpd->mpp->tmp_file_fd, mpd->buf, MULTIPART_BUF_SIZE - mpd->bufleft) != (MULTIPART_BUF_SIZE - mpd->bufleft)) {
                sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: writing to \"%s\" failed", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
            }
            mpd->mpp->tmp_file_size += (MULTIPART_BUF_SIZE - mpd->bufleft);
            
            }
            
        }
        else if (mpd->mpp->type == MULTIPART_FORMDATA) {
            
            /* add data to the value we keep in memory */
            if (mpd->mpp->value == NULL) {
                mpd->mpp->value = apr_pstrdup(mpd->p, mpd->buf);
            }
            else {
                if (mpd->reserve[0] == 1) {
                    mpd->mpp->value = apr_pstrcat(mpd->p, mpd->mpp->value, &(mpd->reserve[1]), mpd->buf, NULL);
                }
                else {
                    mpd->mpp->value = apr_pstrcat(mpd->p, mpd->mpp->value, mpd->buf, NULL);
                }
            }
            
            sec_debug_log(mpd->r, 9, "Formdata variable value \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->value));
        }
        else {
            sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: unknown part type %i", mpd->mpp->type);
            return -1;
        }
        
        /* store the reserved bytes to the multipart
         * context so that they don't get lost
         */
        if (bytes_reserved) {
            mpd->reserve[0] = 1;
            mpd->reserve[1] = localreserve[0];
            mpd->reserve[2] = localreserve[1];
        }
        else {
            mpd->reserve[0] = 0;
        }
    }
    
    return 1;
}

int multipart_process_boundary(multipart_data *mpd) {
    sec_debug_log(mpd->r, 4, "multipart_process_boundary");

    /* if there was a part being built finish it */
    if (mpd->mpp != NULL) {
        /* close the temp file */
        if ((mpd->mpp->type == MULTIPART_FILE)&&(mpd->mpp->tmp_file_name != NULL)&&(mpd->mpp->tmp_file_fd != 0)) {
            close(mpd->mpp->tmp_file_fd);
        }

        /* add the part to the list of parts */
        *(multipart_part **)apr_array_push(mpd->parts) = mpd->mpp;
        sec_debug_log(mpd->r, 4, "multipart_process_boundary: added part %x to the list", mpd->mpp);
        mpd->mpp = NULL;
    }

    /* start building a new part */
    mpd->mpp = (multipart_part *)apr_pcalloc(mpd->p, sizeof(multipart_part));
    mpd->mpp->type = MULTIPART_FORMDATA;
    mpd->mpp_state = 0;    
    
    mpd->reserve[0] = 0;
    mpd->reserve[1] = 0;
    mpd->reserve[2] = 0;
    mpd->reserve[3] = 0;
    
    return 1;
}


int verify_uploaded_files(request_rec *r, multipart_data *mpd, char *approver_script, char **error_msg) {
    multipart_part **parts;
    int i;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
        
    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if (parts[i]->type == MULTIPART_FILE) {
            char *script_output = NULL;
            char *file_path = parts[i]->tmp_file_name;
            char **argv;
            
            argv = apr_pcalloc(r->pool, 3 * sizeof(char *));
            argv[0] = approver_script;
            argv[1] = file_path;
            argv[2] = NULL;
        
            if ((sec_exec_child(approver_script, (const char**)argv, r, &script_output) != APR_SUCCESS)||(script_output == NULL)) {
                *error_msg = apr_psprintf(r->pool, "verify_uploaded_files: Execution of the approver script \"%s\" failed", debuglog_escape(mpd->r->pool, approver_script));
                return 0;
            }
            
            if (script_output == NULL) {
                *error_msg = apr_psprintf(r->pool, "verify_uploaded_files: The approver script \"%s\" output is NULL", debuglog_escape(mpd->r->pool, approver_script));
                return 0;
            }
            
            sec_debug_log(r, 2, "Approver script said: %s", script_output);
            
            if (script_output[0] != '1') {
                *error_msg = apr_psprintf(r->pool, "File \"%s\" rejected by the approver script \"%s\"", debuglog_escape(mpd->r->pool, file_path), debuglog_escape(mpd->r->pool, approver_script));
                return 0;
            }
        }
    }
    
    return 1;
}

int multipart_get_variables(multipart_data *mpd, apr_table_t *parsed_args, sec_dir_config *dcfg, char **error_msg) {
    multipart_part **parts;
    char *my_error_msg = NULL;
    int i;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
        
    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if (parts[i]->type == MULTIPART_FORMDATA) {
            char *name = NULL, *value = NULL;
            
            name = normalise_relaxed(mpd->r, dcfg, parts[i]->name, &my_error_msg);
            if (name == NULL) {
                *error_msg = apr_psprintf(mpd->r->pool, "Error normalizing parameter name: %s", my_error_msg);
                return -1;
            }
            
            value = normalise_relaxed(mpd->r, dcfg, parts[i]->value, &my_error_msg);
            if (value == NULL) {
                *error_msg = apr_psprintf(mpd->r->pool, "Error normalizing parameter value: %s", my_error_msg);
                return -1;
            }
            
            apr_table_add(parsed_args, name, value);
        }
    }
    
    return 1;
}

module AP_MODULE_DECLARE_DATA security_module = {
   STANDARD20_MODULE_STUFF,
   sec_create_dir_config,        /* create per-dir    config structures */
   sec_merge_dir_config,         /* merge  per-dir    config structures */
   sec_create_srv_config,        /* create per-server config structures */
   sec_merge_srv_config,         /* merge  per-server config structures */
   sec_cmds,                     /* table of config file commands       */
   register_hooks                /* register hooks                      */
};
