/*
 * Copyright (c) 2005 by Benjamin Herrenschmidt <benh@kernel.crashing.org>
 * Copyright (c) 2006 by Ben Collins <bcollins@ubuntu.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <sound/driver.h>
#include <sound/core.h>
#include <asm/pmac_feature.h>
#include "snd-pmac-gpio.h"
#include "pmac.h"

#undef DEBUG

#ifdef DEBUG
#define DBG(fmt...) printk(fmt)
#else
#define DBG(fmt...)
#endif


static struct pmf_function *get_audio_pfunc(const char *name, const char *altname)
{
	struct device_node *np;
	struct pmf_function *pfunc = NULL;

	if (! (np = find_devices("i2s-a")))
		return NULL;

	pfunc = pmf_find_function(np, name);
	if (pfunc == NULL && altname)
		pfunc = pmf_find_function(np, altname);

	return pfunc;
}

static struct device_node *find_audio_gpio(const char *name,
					   const char *altname)
{
	struct device_node *np;

	if (! (np = find_devices("gpio")))
		return NULL;

	for (np = np->child; np; np = np->sibling) {
		char *property = get_property(np, "audio-gpio", NULL);
		if (property && (strcmp(property, name) == 0 ||
		    (altname && strcmp(property, altname) == 0)))
			break;
		if (device_is_compatible(np, name) ||
		    (altname && device_is_compatible(np, altname)))
			break;
        }

	return np;
}

static int get_audio_gpio(const char *name, const char *altname,
			  snd_pmac_gpio_t *gp)
{
	struct device_node *np;
	u32 *base, addr;

	if (!(np = find_audio_gpio(name, altname)))
		return -ENODEV;

	base = (u32 *)get_property(np, "AAPL,address", NULL);
	if (! base) {
		base = (u32 *)get_property(np, "reg", NULL);
		if (!base) {
			DBG("(E) cannot find address for device %s !\n", name);
			return -ENODEV;
		}
		addr = *base;
		if (addr < 0x50)
			addr += 0x50;
	} else
		addr = *base;

	gp->addr = addr & 0x0000ffff;

	/* Try to find the active state, default to 0 ! */
	base = (u32 *)get_property(np, "audio-gpio-active-state", NULL);
	if (base) {
		gp->active_state = *base;
		gp->active_val = (*base) ? 0x5 : 0x4;
		gp->inactive_val = (*base) ? 0x4 : 0x5;
	} else {
		/* Don't expect this to work. If platform-do isn't
		 * available (pmac_pfunc), and the above didn't work, then
		 * these are probably wrong.
		 */
		gp->active_state = 0;
		gp->active_val = 0x4;
		gp->inactive_val = 0x5;
	}

	DBG("(I) GPIO device %s found, offset: %x, active state: %d !\n",
	    name, gp->addr, gp->active_state);

	gp->irq = (np->n_intrs > 0) ? np->intrs[0].line : 0;

	return 0;
}

int snd_pmac_get_gpio(const char *name, const char *altname,
                            snd_pmac_gpio_t *gp)
{
        memset(gp, 0, sizeof(*gp));

	gp->name = name;
	gp->altname = altname;

        /* Platform functions are prefered */
        if ((gp->pfunc = get_audio_pfunc(name, altname)))
                return 0;

	/* Else, fallback to direct gpio */
	return get_audio_gpio(name, altname, gp);
}

void snd_pmac_free_gpio(snd_pmac_gpio_t *gp)
{
	if (gp->pfunc != NULL) {
		if (gp->irq_client.owner == THIS_MODULE) {
			pmf_unregister_irq_client(&gp->irq_client);
			gp->irq_client.owner = NULL;
		}

		pmf_put_function(gp->pfunc);
		gp->pfunc = NULL;
	} else if (gp->addr) {
		if (gp->irq > 0) {
			free_irq(gp->irq, gp);
			gp->irq = 0;
		}
		gp->addr = 0;
	}
}

int snd_pmac_write_gpio(snd_pmac_gpio_t *gp, u32 val)
{
	int ret = -ENODEV;

	if (gp->pfunc) {
		struct pmf_args args;

		args.count = 1;
		args.u[0].v = val;

		ret = pmf_call_one(gp->pfunc, &args);
	} else if (gp->addr) {
		val = val ? gp->active_val : gp->inactive_val;

		pmac_call_feature(PMAC_FTR_WRITE_GPIO, NULL, gp->addr, val);

		ret = 0;
	}

	if (!ret)
		gp->state = val;

	return -EINVAL;
}

int snd_pmac_read_gpio(snd_pmac_gpio_t *gp, u32 *val)
{
	int ret = -EINVAL;

	if (gp->pfunc) {
		struct pmf_args args;

		args.count = 1;
		args.u[0].p = val;

		ret = pmf_call_one(gp->pfunc, &args);
	} else if (gp->addr) {
		int ret = pmac_call_feature(PMAC_FTR_READ_GPIO, NULL,
					    gp->addr, 0);

		ret = ((ret & 0x02) != 0);
		*val = ret == gp->active_state;

		ret = 0;
	}

	return -EINVAL;
}

u32 snd_pmac_gpio_internal_state(snd_pmac_gpio_t *gp)
{
	return gp->state;
}

static irqreturn_t snd_pmac_intr(int irq, void *data, struct pt_regs *regs)
{
	snd_pmac_gpio_t *gp = data;

	gp->irq_client.handler(gp->irq_client.data);

	return IRQ_HANDLED;
}

int snd_pmac_request_irq(snd_pmac_gpio_t *gp, void (*handler)(void *),
			 void *data)
{
	int ret = -ENODEV;
	struct device_node *np;

	gp->irq_client.handler = handler;
	gp->irq_client.data = data;
	gp->irq_client.owner = NULL;

	if (gp->pfunc) {
		gp->irq_client.owner = THIS_MODULE;

		if ((np = find_devices("i2s-a"))) {
			ret = pmf_register_irq_client(np, gp->name, &gp->irq_client);
			if (ret < 0 && gp->altname)
				ret = pmf_register_irq_client(np, gp->altname, &gp->irq_client);
		}
		if (ret < 0)
			gp->irq_client.owner = NULL;
	} else if (gp->irq > 0) {
		ret = request_irq(gp->irq, snd_pmac_intr, 0, gp->name, gp);
		if (ret < 0)
			gp->irq = 0;
	}

	return ret;
}
