/* $Id: zdusb.c,v 1.18 2005/04/15 22:34:32 sagamore Exp $
 * Implements the functions of the ZyDAS zd1211 MAC
 *
 * Copyright (C) 2004 Zydas Inc.
 * Copyright (C) 2005 Arno WILLIG <akw@users.sourceforge.net>
 * Copyright (C) 2005 Dimitriy KOROVKIN <korovkin@users.sourceforge.net>
 * Copyright (C) 2005 Todor T. ZVISKOV <warderx@users.sourceforge.net>
 * Copyright (C) 2005 Markus KARG <markus-karg@users.sourceforge.net>
 *
 * This file is part of the ZD1211 Wireless USB Driver for Linux.
 *
 * This driver is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This driver is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this driver; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/version.h>

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/usb.h>

#include "zd1205.h"
#include "zdusb.h"
#include "zddebug.h"
#include "zdversion.h"
#include "zd1211.h"
#include "zddevlist.h"

#define ZD1211_DBG_LEVEL	1

static const char driver_name[] = "zd1211";

struct usb_device_id zd1211_ids [] = {
	ZD1211_IDS
};

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE(DRIVER_LICENSE);
MODULE_DEVICE_TABLE(usb, zd1211_ids);
/*
 * This module uses /dev/wlan.  The MODULE_SUPPORTED_DEVICE macro might
 * be used in the future to help automatic configuration of modules, but is
 * currently unused other than for documentation purposes.
 */
MODULE_SUPPORTED_DEVICE("wlan");

extern struct net_device *g_dev;

/**
 * zd1211_probe
 * is called when an appropriate device is detected
 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
static void *zd1211_probe(struct usb_device *usb_dev, unsigned int ifnum,const struct usb_device_id *id)
{
	struct usb_interface *interface = &usb_dev->actconfig->interface[ifnum];
#else
static int zd1211_probe(struct usb_interface *interface,const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(interface);
#endif
	struct net_device *net = NULL;
	struct zd1205_private *macp = NULL;
	int vendor_id, product_id;
	int dev_index = id - zd1211_ids;
	int result = 0;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	usb_get_dev(usb_dev);
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
	vendor_id = le16_to_cpu(usb_dev->descriptor.idVendor);
	product_id = le16_to_cpu(usb_dev->descriptor.idProduct);
#else
	vendor_id = usb_dev->descriptor.idVendor;
	product_id = usb_dev->descriptor.idProduct;
#endif
	printk(KERN_NOTICE "zd1211 device (0x%04x,0x%04x) found.\n", vendor_id,product_id);
	if (usb_dev->speed == USB_SPEED_HIGH)
		printk(KERN_NOTICE "zd1211 device on USB 2.0 Host\n");
	else
		printk(KERN_NOTICE "zd1211 device on USB 1.1 Host\n");

	// memset(serial_number, 0, 30);
	// usb_string(usb_dev, usb_dev->descriptor.iSerialNumber, serial_number, 29);
	// printk("Device serial number is %s\n", serial_number);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))	
	if (usb_set_configuration(usb_dev, usb_dev->config[0].bConfigurationValue)) {
		printk(KERN_ERR "usb_set_configuration() failed\n");
		result = -EIO;
		goto fail;
	}
#endif

	if (!(macp = kmalloc(sizeof(struct zd1205_private), GFP_KERNEL))) {
		printk(KERN_ERR "zd1211: Out of memory while allocating device structure\n");
		result = -ENOMEM;
		goto fail;
	}
	memset(macp, 0, sizeof(struct zd1205_private));

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	usb_inc_dev_use(usb_dev);
#endif

	net = alloc_etherdev(0);
	//net = alloc_etherdev(sizeof (struct zd1205_private)); // kernel 2.4
	if (!net) {
		printk(KERN_ERR "zd1211: Not able to allocate etherdev struct\n");
		result = -ENOMEM;
		goto fail1;
	}

	g_dev = net;			// save this for CBs use
	net->priv = macp;
	strcpy(net->name, "wlan%d");	// @rjp-make it appear as a wlan node
	macp->device = net;
	macp->usb = usb_dev;
	SET_MODULE_OWNER(net);
	macp->dev_index = dev_index;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
	macp->release = le16_to_cpu(usb_dev->descriptor.bcdDevice);
#else
	macp->release = usb_dev->descriptor.bcdDevice;
#endif
	//macp->release = 17200; /* 17200 = 0x4330 Big Endian Hack*/ 
	printk(KERN_NOTICE "Firmware Version = %04x\n", macp->release);
	macp->flags = 0;
	macp->dbg_flag = ZD1211_DBG_LEVEL;
	/* set up the endpoint information */
	/* check out the endpoints */
	macp->interface = interface;

	init_waitqueue_head(&macp->regSet_wait);
	init_waitqueue_head(&macp->iorwRsp_wait);
	init_waitqueue_head(&macp->term_wait);

	if (!zd1211_alloc_all_urbs(macp)) {
		result = -ENOMEM;
		goto fail2;
	}

	//zd1211_DownLoadUSBCode(macp, "WS11Uext.bin", NULL, cFIRMWARE_EXT_CODE);
	if (zd1211_Download_IncludeFile(macp) != 0) {
		printk(KERN_ERR "zd1211_Download_IncludeFile failed\n");
		result = -EIO;
		goto fail3;
	} else { // DEBUG
		printk(KERN_NOTICE "Downloaded firmware.\n");
	}

	// to enable firmware
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	if (usb_set_configuration(usb_dev, usb_dev->config[0].bConfigurationValue)) {
#else
	if (usb_set_interface(usb_dev, interface->altsetting[0].desc.bInterfaceNumber,0)) {
#endif
		printk(KERN_ERR "Failed to enable firmware\n");
		result = -EIO;
		goto fail3;
	}

	set_bit(ZD1211_RUNNING, &macp->flags);
	macp->bUSBDeveiceAttached = 1;
	if (!zd1211_InitSetup(net, macp)) {
		result = -EIO;
		goto fail3;
	} else {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
		usb_set_intfdata(interface, macp);
		SET_NETDEV_DEV(net, &interface->dev);
		// defer_kevent(macp, KEVENT_REGISTER_NET);
#endif

		if (register_netdev(net) != 0) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
			usb_set_intfdata(interface, NULL);
#endif
			goto fail3;
		}
	}
	goto done;

fail3:
	zd1211_free_all_urbs(macp);

fail2:
	free_netdev(net); //kernel 2.6
	//kfree(net);

fail1:
	kfree(macp);

fail:
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	usb_put_dev(usb_dev);
#endif
	macp = NULL;

done: 
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	return macp;
#else
	return result;
#endif	
}

/**
 * zd1211_disconnect
 * USB zd1211 device is removed from bus
 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
static void zd1211_disconnect(struct usb_device *usb_dev, void *ptr)
{
	struct zd1205_private *macp = (struct zd1205_private *)ptr;
#else
static void zd1211_disconnect(struct usb_interface *interface)
{
	struct zd1205_private *macp = (struct zd1205_private *) usb_get_intfdata(interface);
#endif

	if (!macp) {
		printk(KERN_ERR "Unregistering non-existant device!\n");
		return;
	}

	set_bit(ZD1211_UNPLUG, &macp->flags);
	macp->bUSBDeveiceAttached = 0;

	if (macp->driver_isolated) {
		if (macp->device->flags & IFF_UP)
			zd1205_close(macp->device);
	}
	unregister_netdev(macp->device);

	flush_scheduled_work();
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	usb_dec_dev_use(usb_dev);
#else
	usb_put_dev(interface_to_usbdev(interface));
#endif

	zd1211_unlink_all_urbs(macp);
	zd1211_free_all_urbs(macp);
	zd1205_clear_structs(macp->device);
	kfree(macp);
	macp = NULL;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	usb_set_intfdata(interface, NULL);
#endif
}

/**
 *
 */
static struct usb_driver zd1211_driver = {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	.owner		= THIS_MODULE,
#endif
	.name		= driver_name,
	.probe		= zd1211_probe,
	.disconnect	= zd1211_disconnect,
	.id_table	= zd1211_ids,
};

/**
 * zd1211_init
 * Initialization of zd1211 module
 * is called when module is insmodded.
 */
int __init zd1211_init(void)
{
	printk(KERN_NOTICE DRIVER_DESC " v" DRIVER_VERSION " loaded\n");
	printk(KERN_NOTICE "(c) " DRIVER_AUTHOR "\n");
	printk(KERN_NOTICE DRIVER_URL "\n");

	return usb_register(&zd1211_driver);
}

/**
 * zd1211_exit
 * Deregistering zd1211 module
 * is called when module is removed from kernel.
 */
void __exit zd1211_exit(void)
{
	usb_deregister(&zd1211_driver);
	printk(KERN_NOTICE DRIVER_DESC " removed\n");
}

/**
 * Init/Exit Macros
 */
module_init(zd1211_init);
module_exit(zd1211_exit);
