import os, unittest, tempfile, shutil
from linda.funcs import run_external_cmd
from linda.unpack import Unpacker, UnpackException
from sets import Set

class UnpackerBaseTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/base_unpacker.tar.gz' % \
            self.dir)

    def tearDown(self):
        shutil.rmtree(self.dir)

    def test_unpack_bad(self):
        "Check unpack() method raises UnpackException when passed bad args"
        unpacker = Unpacker()
        try:
            unpacker.unpack('foo.tmp', 3)
        except UnpackException:
            pass
        else:
            self.fail
        unpacker.cull_lab()
        
    def test_binary_unpack_level_one(self):
        "Check binary unpack at level one"
        unpacker = Unpacker()
        unpacker.unpack(os.path.join(self.dir, \
            'testcontrol_0.0.1_all.deb'), 1)
        self.assertEqual(os.path.isfile(os.path.join(unpacker.lab, \
            'control', 'control')), True)
        keys = Set(('control', 'collector', 'dsc', 'dir'))
        self.assertEqual(keys.intersection(unpacker.information.keys()), \
            keys)
        control_keys = Set(('self', 'info'))
        self.assertEqual(control_keys.intersection(unpacker.information['control'].keys()), control_keys)
        self.assertEqual(unpacker.information['control']['info'].keys(), \
            ['control'])
        self.assertEqual(unpacker.information['control']['info']['control'], ['root/root', '-rw-r--r--'])
        self.assertEqual(unpacker.information['collector'], None)
        self.assertEqual(unpacker.information['dsc'], [])
        self.assertEqual(unpacker.information['dir'], '')
        fields = Set(('maintainer', 'description', 'package', 'section', \
            'installed-size', 'priority', 'suggests', 'depends', 'version', \
            'architecture', 'provides', 'conflicts'))
        self.assertEqual(fields.intersection(unpacker.information['control']['self'][0].keys()), fields)
        unpacker.cull_lab()
        
    def test_binary_unpack_level_two(self):
        "Check binary unpack at level two"
        unpacker = Unpacker()
        unpacker.unpack(os.path.join(self.dir, \
            'testunpacked_0.0.1_all.deb'), 1)
        unpacker.unpack(os.path.join(self.dir, \
            'testunpacked_0.0.1_all.deb'), 2)
        self.assertEqual(unpacker.lab, unpacker.information['dir'])
        self.assertEqual(unpacker.information['dsc'], [])
        files = Set(('/usr/share/doc/testunpacked/changelog', '/bin/ls', \
            '/lib/libBrokenLocale-2.3.2.so'))
        elf_files = Set(('/bin/ls', '/lib/libBrokenLocale-2.3.2.so'))
        self.assertEqual(files.intersection(unpacker.information['collector']('files', 'files').keys()), files)
        self.assertEqual(unpacker.information['collector']('files', 'files')['/bin/ls'], '-rwxr-xr-x')
        self.assertEqual(unpacker.information['collector']('files', 'files')['/usr/share/doc/testunpacked/changelog'], '-rw-r--r--')
        self.assertEqual(unpacker.information['collector']('files', 'files')['/lib/libBrokenLocale-2.3.2.so'], '-rw-r--r--')
        self.assertEqual(files.intersection(unpacker.information['collector']('output', 'file').keys()), files)
        self.assertEqual(unpacker.information['collector']('output', 'file')['/usr/share/doc/testunpacked/changelog'], 'ASCII text')
        self.assert_(unpacker.information['collector']('output', 'file')['/bin/ls'].startswith('ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV)'), True)
        self.assert_(unpacker.information['collector']('output', 'file')['/lib/libBrokenLocale-2.3.2.so'].startswith('ELF 32-bit LSB shared object, Intel 80386, version 1 (SYSV)'), True)
        self.assertEqual(elf_files.intersection(unpacker.information['collector']('output', 'objdump').keys()), elf_files)
        self.assertEqual(elf_files.intersection(unpacker.information['collector']('files', 'elf')), elf_files)
        unpacker.cull_lab()
        
    def test_source_unpack_level_two(self):
        "Check source unpack at level two"
        unpacker = Unpacker()
        unpacker.unpack(os.path.join(self.dir, \
            'testunpacked_0.0.1.dsc'), 1)
        unpacker.unpack(os.path.join(self.dir, \
            'testunpacked_0.0.1.dsc'), 2)
        self.assertEqual(os.path.isdir(unpacker.information['dir']), True)
        dsc_fields = Set(('files', 'binary', 'maintainer', 'format', \
             'build-depends', 'source', 'version', 'standards-version', \
             'architecture'))
        src_fields = Set(('maintainer', 'section', 'build-depends', \
            'priority', 'source', 'standards-version'))
        bin_fields = Set(('depends', 'description', 'architecture', 'package'))
        self.assertEqual(dsc_fields.intersection(unpacker.information['dsc'].keys()), dsc_fields)
        self.assertEqual(src_fields.intersection(unpacker.information['control']['self'][0].keys()), src_fields)
        self.assertEqual(bin_fields.intersection(unpacker.information['control']['self'][1].keys()), bin_fields)
        files = Set(('testunpacked-0.0.1/libBrokenLocale-2.3.2.so', \
            'testunpacked-0.0.1/debian/files', \
            'testunpacked-0.0.1/debian/rules', 'testunpacked-0.0.1/ls', \
            'testunpacked-0.0.1/debian/changelog', \
            'testunpacked-0.0.1/debian/control', \
            'testunpacked-0.0.1/debian/dirs'))
        self.assertEqual(files.intersection(unpacker.information['collector']('files', 'files').keys()), files)
        self.assertEqual(files.intersection(unpacker.information['collector']('output', 'file').keys()), files)
        unpacker.cull_lab()

    def test_source_unpack_level_two_one_nondir(self):
        "Check source unpack at level two, when orig contains one non-dir member"
        unpacker = Unpacker()
        unpacker.unpack(os.path.join(self.dir, 'foobar_0.0.1.dsc'), 1)
        unpacker.unpack(os.path.join(self.dir, 'foobar_0.0.1.dsc'), 2)
        unpacker.cull_lab()

    def test_source_unpack_multi_file(self):
        "Check source unpack at level two, when orig contains multiple things at top level"
        unpacker= Unpacker()
        unpacker.unpack(os.path.join(self.dir, 'testunpacked2_0.0.1.dsc'), 1)
        unpacker.unpack(os.path.join(self.dir, 'testunpacked2_0.0.1.dsc'), 2)
        unpacker.cull_lab()

