/***************************************************************************
 *
 * knetworkmanager-vpn.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#define SERVICE_DIR "/etc/NetworkManager/VPN"
#define GCONF_PRE   "/.gconf"
#define VPN_PATH    "/system/networking/vpn_connections"

#include <stdlib.h>
#include <kconfig.h>
#include <qdom.h> 

#include "knetworkmanager-vpn.h"
#include "knetworkmanager-vpn_dbus.h"

/*
 * class VPNConnection
 *
 */

QString
VPNConnection::getName () const
{
	return _name;
}

void
VPNConnection::setName (const QString & name)
{
	_name = name;
}

QString
VPNConnection::getServiceName () const
{
	return _serviceName;
}

void
VPNConnection::setServiceName (const QString & serviceName)
{
	_serviceName = serviceName;
}

QString
VPNConnection::getUser () const
{
	return _user;
}

void
VPNConnection::setUser (const QString & user)
{
	_user = user;
}

QStringList
VPNConnection::getRoutes () const
{
	return _routes;
}

void
VPNConnection::addRoute (const QString & route)
{
	_routes.append (route);
}

QStringList
VPNConnection::getData () const
{
	return _data;
}

void
VPNConnection::addData (const QString & data)
{
	_data.append (data);
}

QStringList
VPNConnection::getPasswords () const
{
	return _passwords;
}

void
VPNConnection::addPasswords (const QStringList & passwords)
{
	_passwords = passwords;
}

NMVPNActStage
VPNConnection::getActivationStage () const
{
	return _activationStage;
}

void
VPNConnection::setActivationStage (NMVPNActStage activationStage)
{
	_activationStage = activationStage;
	emit activationStateChanged ();
}

QString
VPNConnection::getAuthHelper () const
{
	return _authHelper;
}

void
VPNConnection::setAuthHelper (const QString & authHelper)
{
	_authHelper = authHelper;
}

bool
VPNConnection::isValid ()
{
	bool status = true;
	
	/* _routes is allowed to be empty for some VPN service */
	if (_name == QString::null || _serviceName == QString::null || _user == QString::null || _data.empty () || _authHelper == QString::null)
		status = false;

	return status;
}

VPNConnection::VPNConnection ( QObject * parent, const char * name ) : QObject( parent, name )
{
	_name            = QString::null;
	_serviceName     = QString::null;
	_user            = getenv ("USER");
	_authHelper      = QString::null;
	_activationStage = NM_VPN_ACT_STAGE_UNKNOWN;
}

VPNConnection::~VPNConnection ()
{ 

}

VPNConnection::VPNConnection (const VPNConnection & vpnConnection) : QObject( vpnConnection.parent(), vpnConnection.name())
{
	_name = vpnConnection.getName ();
	_serviceName = vpnConnection.getServiceName ();
	_user = vpnConnection.getUser();
	_routes = vpnConnection.getRoutes ();
	_data = vpnConnection.getData ();
	_passwords = vpnConnection.getPasswords ();
	_activationStage = vpnConnection.getActivationStage ();
	_authHelper = vpnConnection.getAuthHelper ();
}

/*
 * class VPNConnectionHandler
 *
 */

void
VPNConnectionHandler::receiveAuthenticationData (KProcess* /* authHelper */, char* buffer, int len)
{
	QStringList passwords = QStringList::split (QString ("\n"), QString::fromLatin1 (buffer, len), false);

	_vpnConnection->addPasswords (passwords);
}

void
VPNConnectionHandler::authHelperExited (KProcess* /* authHelper */)
{
	VPNDBus::activateVPNConnection (_vpnConnection);

}

VPNConnectionHandler::VPNConnectionHandler ( QObject * parent, const char * name, VPNConnection* vpnConnection) : QObject( parent, name )
{
	if (!vpnConnection)
		return;

	_vpnConnection = vpnConnection;
}

VPNConnectionHandler::~VPNConnectionHandler ()
{

}

/*
 *  class VPN
 *
 */

void
VPN::activateVPNConnection (VPNConnection* vpnConnection)
{
	KProcess*             authHelper = new KProcess ();
	VPNConnectionHandler* handler    = new VPNConnectionHandler ( this, "vnpconnhandler", vpnConnection);

	*authHelper << vpnConnection->getAuthHelper ()           \
	            << "-n" << vpnConnection->getName ()         \
	            << "-s" << vpnConnection->getServiceName ()  \
	            << "-r";

	connect (authHelper, SIGNAL (receivedStdout             (KProcess*, char*, int)),
		 handler,    SLOT   (receiveAuthenticationData (KProcess*, char*, int)));

	connect (authHelper, SIGNAL (processExited    (KProcess*)),
		 handler,    SLOT   (authHelperExited (KProcess*)));

	authHelper->start (KProcess::NotifyOnExit, KProcess::Stdout);
}

void
VPN::configureVPNExited (KProcess* /* p */)
{
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		delete (*i);
	}

	_vpnList->clear ();
	getConnections  ();
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		VPNDBus::updateVPNConnection (*i);
	}
}

void
VPN::configureVPNConnections (void)
{
	KProcess* configureVPNHelper = new KProcess ();

	*configureVPNHelper << "nm-vpn-properties";

	connect (configureVPNHelper, SIGNAL (processExited      (KProcess*)),
		 this,               SLOT   (configureVPNExited (KProcess*)));

	configureVPNHelper->start (KProcess::NotifyOnExit);
}

void
VPN::disconnectVPNConnection (void)
{
	VPNDBus::disconnectVPNConnection ();
}

bool
VPN::isAvailable (void)
{
	return _available;
}

void
VPN::addConnection (const QString & connection)
{
	VPNConnection* vpnConnection = new VPNConnection ( this, "vpnconnection" );
	QString        name;
	QString        serviceName;
	QStringList    routes;
	QStringList    data;

	QDomDocument doc ("vpnconnection");
	QFile file (_confPath + "/" + connection +  "/%gconf.xml");
	if (file.open (IO_ReadOnly) == false)
		return;
	if (doc.setContent (&file) == false) {
		file.close ();
		return;
	}
	file.close ();

	QDomElement docElem = doc.documentElement ();
	QDomNode n = docElem.firstChild ();
	while (n.isNull () == false) {
		QDomElement e = n.toElement ();
		if (e.isNull () == false) {
			if (e.hasAttribute ("name")) {
				QString nameValue = e.attribute ("name", QString::null);
				if (nameValue == "name") {
					vpnConnection->setName (e.text ());
				} else if (nameValue == "service_name") {
					vpnConnection->setServiceName (e.text ());
				} else if (nameValue == "routes") {
					QDomNode m = n.firstChild ();
					while (m.isNull () == false) {
						QDomElement f = m.toElement ();
						if (f.isNull () == false)
							vpnConnection->addRoute (f.text ());
						m = m.nextSibling ();
					}
				} else if (nameValue == "vpn_data") {
					QDomNode m = n.firstChild ();
					while (m.isNull () == false) {
						QDomElement f = m.toElement ();
						if (f.isNull () == false)
							vpnConnection->addData (f.text ());
						m = m.nextSibling ();
					}
				}
			}
		}
		n = n.nextSibling ();
	}

	if (!vpnConnection->getName () || !vpnConnection->getServiceName ())
		return;

	QDir* serviceDir = new QDir (SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir->entryList ().grep (".name", true);

	for (QStringList::Iterator i = services.begin (); i != services.end (); ++i) {
		QString service = SERVICE_DIR + QString ("/") + *i;
		KConfig* kconfig = new KConfig (service, true, true, "config");
		kconfig->setGroup ("VPN Connection");
		if (kconfig->readEntry ("service", QString::null) == vpnConnection->getServiceName ()) {
			kconfig->setGroup ("GNOME");
			QString helper = kconfig->readEntry ("auth-dialog", QString::null);
			if ( !helper.isEmpty() ) {
				vpnConnection->setAuthHelper (helper);
			} else {
				printf ("Warning: No authentication helper for service \"%s\" found.\n", vpnConnection->getServiceName ().ascii ());
			}
		}
	}

	if (vpnConnection->isValid ())
		_vpnList->append (vpnConnection);
}

void
VPN::getConnections (void)
{
	_confDir = new QDir (_confPath);
	_confDir->setFilter (QDir::Dirs);

	QStringList entryList = _confDir->entryList ();
	for (QStringList::Iterator it = entryList.begin (); it != entryList.end (); ++it) {
		if ((*it) != "." && (*it) != "..")
			addConnection (*it);
	}
}

void
VPN::receiveKeyringData (KProcess* /* keyringDaemon */, char* buffer, int len)
{
	QStringList env = QStringList::split (QString ("\n"), QString::fromLatin1 (buffer, len), false);

	for (QStringList::Iterator it = env.begin (); it != env.end (); ++it) {
		QString* item = &(*it);
		if (item->startsWith ("GNOME_KEYRING_SOCKET")) {
			setenv ("GNOME_KEYRING_SOCKET", item->section ('=', 1, 1).ascii(), 1);
		}
	}
}


void
VPN::initKeyring ()
{
	QString keyringSocket = getenv ("GNOME_KEYRING_SOCKET");

	if (!keyringSocket) {
		KProcess* keyringDaemon = new KProcess ();
		*keyringDaemon << "/opt/gnome/bin/gnome-keyring-daemon";

		connect (keyringDaemon, SIGNAL (receivedStdout     (KProcess*, char*, int)),
			 this,          SLOT   (receiveKeyringData (KProcess*, char*, int)));
		
		keyringDaemon->start (KProcess::NotifyOnExit, KProcess::Stdout);
	}
}

bool
VPN::checkForServices ()
{
	bool status = false;
	
	QDir serviceDir(SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir.entryList ().grep (".name", true);

	if (services.count () > 0)
		status = true;

	return status;
}

VPNConnection*
VPN::getVPNConnection (const QString & name)
{
	VPNConnection* vpnConnection = NULL;
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		if ((*i)->getName () == name) {
			vpnConnection = *i;
			goto out;
		}
	}
out:
	return vpnConnection;
}

bool
VPN::isActive (void)
{
	bool status = false;
	
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		if ((*i)->getActivationStage () == NM_VPN_ACT_STAGE_ACTIVATED) {
			status = true;
			goto out;
		}
	}

out:
	return status;
}

VPNList*
VPN::getVPNList (void)
{
	return _vpnList;
}

void
VPN::push (KNetworkManager* ctx)
{
	VPNDBus::push (ctx);
}

VPN::VPN ( QObject * parent, const char * name ) : QObject( parent, name )
{
	_confPath  = QDir::homeDirPath() + GCONF_PRE + VPN_PATH;
	_available = checkForServices ();

	if (_available == true) {
		initKeyring ();
		_vpnList = new VPNList ();
		getConnections ();
	}
}

VPN::~VPN ()
{

}

#include "knetworkmanager-vpn.moc"
