/* This file is part of the Keep project
   Copyright (C) 2005 Jean-Rémy Falleri <jr.falleri@laposte.net>

   Keep is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   Keep is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Keep; if not, write to the
   Free Software Foundation, Inc.,
   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           */

#include "addbackupwizard.h"

#include <qpushbutton.h>
#include <kiconloader.h>
#include <qcheckbox.h>
#include <kurlrequester.h>
#include <knuminput.h>
#include <kdebug.h>

AddBackupWizard::AddBackupWizard(QWidget *parent,const char* name): KWizard( parent, name, true)
{
	initView();
	resize( minimumSizeHint() );

	setNextEnabled(page1,false);
	setNextEnabled(page2,false);
	setFinishEnabled(page3, false);
	
	this->setCaption("Add a backup");

	initConnections();
}

AddBackupWizard::AddBackupWizard(Backup backup,QWidget *parent,const char* name): KWizard( parent, name, true)
{
	initView();

	this->setCaption("Edit backup");
	
	addBackupWizard1View->source->setURL(backup.source());
	addBackupWizard2View->dest->setURL(backup.dest());
	addBackupWizard3View->interval->setValue(backup.interval());
	addBackupWizard3View->deleteAfter->setValue(backup.deleteAfter());
	addBackupWizard3View->neverDelete->setChecked(backup.neverDelete());
	addBackupWizard3View->useCompression->setChecked(backup.useCompression());
	addBackupWizard3View->excludeSpecialFiles->setChecked(backup.excludeSpecialFiles());
	addBackupWizard3View->useAdvancedConfig->setChecked(backup.useAdvancedConfig());
	advancedBackupConfigDialog->setOptionList(backup.optionList());

	resize( minimumSizeHint() );

	setNextEnabled(page1,true);
	setNextEnabled(page2,true);
	setFinishEnabled(page3, true);

	initConnections();
}

void AddBackupWizard::initView()
{
	advancedBackupConfigDialog = new AdvancedBackupConfigDialog(this);
	KIconLoader icons;
	this->setIcon( icons.loadIcon( "wizard",KIcon::Small ));

	setupPage1();
	setupPage2();
	setupPage3();
}

void AddBackupWizard::initConnections()
{
	connect( addBackupWizard1View->source, SIGNAL( textChanged(const QString&)), this, SLOT( slotPage1Changed() ) );
	connect( addBackupWizard2View->dest, SIGNAL( textChanged(const QString&)), this, SLOT( slotPage2Changed() ) );
	connect( addBackupWizard3View->advancedConfig, SIGNAL( clicked()),this,SLOT(slotShowAdvancedConfig()));
	connect( finishButton(), SIGNAL( clicked()), this, SLOT( slotFinishClicked() ) );
}

void AddBackupWizard::setupPage1()
{
	page1 = new QHBox( this );
	page1->setSpacing(0);
	page1->setMargin(0);

	addBackupWizard1View = new AddBackupWizard1View(page1,"addBackupWizard1View");
	addBackupWizard1View->source->setMode(KFile::Directory);

	addPage( page1, "Directory to backup" );
}

void AddBackupWizard::setupPage2()
{
	page2 = new QHBox( this );
	page2->setSpacing(0);
	page2->setMargin(0);

	addBackupWizard2View = new AddBackupWizard2View(page2,"addBackupWizard2View");
	addBackupWizard2View->dest->setMode(KFile::Directory);

	addPage( page2, "Backup location" );
}

void AddBackupWizard::setupPage3()
{
	page3 = new QHBox( this );
	page3->setSpacing(0);
	page3->setMargin(0);

	addBackupWizard3View = new AddBackupWizard3View(page3,"addBackupWizard3View");
	KIconLoader *icons = KGlobal::iconLoader();
	addBackupWizard3View->advancedConfig->setPixmap(icons->loadIcon("configure",KIcon::Toolbar,16));

	addPage( page3, "Backup options" );
}

void AddBackupWizard::slotPage1Changed()
{
	setNextEnabled(page1,false);
	setFinishEnabled(page3,false);
	if ( addBackupWizard1View->source->url() != "" )
	{
		setNextEnabled(page1,true);
		if ( addBackupWizard2View->dest->url() != "" )
			setFinishEnabled(page3,true);	
	}
}

void AddBackupWizard::slotPage2Changed()
{
	setNextEnabled(page2,false);
	setFinishEnabled(page3,false);
	if ( addBackupWizard2View->dest->url() != "" )
	{
		setNextEnabled(page2,true);
		if (  addBackupWizard1View->source->url() != "" )
			setFinishEnabled(page3,true);	
	}
}

void AddBackupWizard::slotShowAdvancedConfig()
{
	advancedBackupConfigDialog->show();
}

Backup AddBackupWizard::backup()
{
	QString source = addBackupWizard1View->source->url();
	QString dest = addBackupWizard2View->dest->url();
	int interval = addBackupWizard3View->interval->value();
	int deleteAfter = addBackupWizard3View->deleteAfter->value();
	bool neverDelete = addBackupWizard3View->neverDelete->isChecked();
	bool useCompression = addBackupWizard3View->useCompression->isChecked();
	bool excludeSpecialFiles = addBackupWizard3View->excludeSpecialFiles->isChecked();
	bool useAdvancedConfig = addBackupWizard3View->useAdvancedConfig->isChecked();
	QStringList optionList = advancedBackupConfigDialog->optionList();
	Backup backup(source,dest,interval,deleteAfter,neverDelete,useCompression,excludeSpecialFiles,useAdvancedConfig,optionList);
	
	return backup;
}

void AddBackupWizard::slotFinishClicked()
{
	emit backupSetted(backup());
}

#include "addbackupwizard.moc"
