/*
    This file is part of kio_obex.

    Copyright (c) 2003 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <klocale.h>

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <linux/types.h>
#include <linux/irda.h>

#include <qtextstream.h>
#include <qtextcodec.h>
#include <qcstring.h>
#include <qstring.h>
#include <qpushbutton.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qwidgetstack.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qheader.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qlineedit.h>

#include <qvalidator.h>
#include <qregexp.h>

#include "discoverabledevicewidget.h"

DiscoverableDeviceWidget::DiscoverableDeviceWidget( QWidget* parent, const char* name, WFlags fl )
  : QWidget( parent, name, fl )
{
  int margin = 11;
  int space = 6;

  // Prepare page 2 for discoverable transports
  mDiscoveryTransportLayout = new QGridLayout( this, 4, 3, margin, space, "mDiscoveryTransportLayout");


  mButtonGroup = new QButtonGroup( this, "mButtonGroup" );
  mButtonGroup->setLineWidth( 0 );
  mButtonGroup->setTitle( QString::null );
  mButtonGroupLayout = new QVBoxLayout( mButtonGroup, margin, space, "mButtonGroupLayout");

  mDynamicDiscoveryButton = new QRadioButton( mButtonGroup, "mDynamicDiscoveryButton" );
  mDynamicDiscoveryButton->setChecked( TRUE );
  mDynamicDiscoveryButton->setText( i18n( "Dynamically discover device" ) );
  mButtonGroupLayout->addWidget( mDynamicDiscoveryButton );

  mSpecificDeviceButton = new QRadioButton( mButtonGroup, "mSpecificDeviceButton" );
  mSpecificDeviceButton->setText( i18n( "Connect to a specific device" ) );
  mButtonGroupLayout->addWidget( mSpecificDeviceButton );

  mDiscoveryTransportLayout->addMultiCellWidget( mButtonGroup, 0, 0, 0, 1, false );

  mDiscoverButton = new QPushButton( this, "irdaDiscoverButton" );
  mDiscoverButton->setEnabled( FALSE );
  mDiscoverButton->setText( i18n( "Discover Now" ) );
  mDiscoveryTransportLayout->addWidget( mDiscoverButton, 1, 0 );

  mDiscoveryView = new QListView( this, "irdaDiscoveryView" );
  mDiscoveryView->addColumn( i18n( "Name" ) );
  mDiscoveryView->addColumn( i18n( "Address" ) );
  mDiscoveryView->setEnabled( FALSE );
  mDiscoveryView->header()->setLabel( 0, i18n( "Name" ) );
  mDiscoveryView->header()->setLabel( 1, i18n( "Address" ) );
  mDiscoveryTransportLayout->addMultiCellWidget( mDiscoveryView, 1, 2, 1, 2, false );

  mHandInputText = new QLabel( this, "handInputText" );
  mHandInputText->setText( i18n( "Specify an address manually:" ) );
  mHandInputText->setEnabled( FALSE );
  mDiscoveryTransportLayout->addMultiCellWidget( mHandInputText, 3, 3, 0, 1, false );

  mHandInputInput = new QLineEdit( this, "handInputInput" );
  mHandInputInput->setEnabled( FALSE );
  mDiscoveryTransportLayout->addWidget( mHandInputInput, 3, 2 );


  // Enable address input if we dont want to dynamically discover devices
  connect( mSpecificDeviceButton, SIGNAL(toggled(bool)),
	   SLOT(slotEnableSpecific(bool)) );

  connect( mDiscoverButton, SIGNAL(pressed()),
	   SLOT(slotDiscoveryButtonPressed()) );

  connect( mDiscoveryView, SIGNAL( pressed(QListViewItem*) ),
	   SLOT(slotListViewPressed(QListViewItem*)) );
}

DiscoverableDeviceWidget::~DiscoverableDeviceWidget()
{
}

QString DiscoverableDeviceWidget::address() const
{
  if ( mDynamicDiscoveryButton->isChecked() )
    return anyDevice();
  else
    return mHandInputInput->text();
}

void DiscoverableDeviceWidget::setAddress( const QString& addr )
{
  bool specific = mHandInputInput->validateAndSet( addr, 0, 0, 0 );
  specific = specific & addr != anyDevice();
  slotEnableSpecific( specific );
  mDynamicDiscoveryButton->setChecked( !specific );
  mSpecificDeviceButton->setChecked( specific );
}

void DiscoverableDeviceWidget::checkConfiguration()
{
  emit configurationReady( true );
}

void DiscoverableDeviceWidget::slotListViewPressed( QListViewItem *item )
{
  if ( item )
    mHandInputInput->setText( item->text( 1 ) );
}

void DiscoverableDeviceWidget::slotEnableSpecific( bool enable )
{
  mDiscoverButton->setEnabled( enable );
  mDiscoveryView->setEnabled( enable );
  mHandInputText->setEnabled( enable );
  mHandInputInput->setEnabled( enable );
}

void DiscoverableDeviceWidget::clearDiscoveredDevices()
{
  mDiscoveryView->clear();
}

void DiscoverableDeviceWidget::discoveredDevice( const QString& name, const QString& addr, bool enable )
{
  QListViewItem *item = new QListViewItem( mDiscoveryView );
  item->setText( 0, name );
  item->setText( 1, addr );
  item->setEnabled( enable );
  mDiscoveryView->insertItem( item );
  mDiscoveryView->setSelected( item, addr == address() );
}

#include "discoverabledevicewidget.moc"
