
/****************************************************************************
*
* Copyright (C) 2004 Ikke - http://www.eikke.com.  All rights reserved.
*
* This file is part of IVMan (ivm).
*
* This file may be distributed under the terms of the Q Public License
* as defined by Troll Tech AS of Norway and appearing in the file
* LICENSE.QPL included in the packaging of this file.
* 
* See http://www.troll.no/qpl for QPL licensing information.
*
* $Id: manager.c,v 1.55 2005/06/13 02:40:23 ro_han Exp $
*****************************************************************************/

#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <dirent.h>
#include <mntent.h>
#include <errno.h>

#include <linux/cdrom.h>

#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <libhal.h>


#include "common.h"
#include "hal_interface.h"
#include "hal_compat.h"
#include "manager.h"
#include "debug.h"
#include "IvmConfig/IvmConfigProperties.h"
#include "IvmConfig/IvmConfigActions.h"
#include "hal_psi.h"
#include "log.h"
#include "daemonize.h"

#include "userconfigs.h"

#define warn(fmt,arg...) g_warning("%s/%d: " fmt,__FILE__,__LINE__,##arg)

gchar *homeDir;

gboolean pmount_accepts_async = FALSE;
gboolean pmount_accepts_umask = FALSE;
gboolean pmount_accepts_fstype = FALSE;

void discover_pmount_version()
{
#ifdef PMOUNT_PATH

    gchar * argv[4];
    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[3] = 0;
    gint exitStatus;
    GError * error;
    gboolean success;

    GString * pmount_base_command = g_string_new(PMOUNT_PATH);
    pmount_base_command = g_string_append(pmount_base_command,
                                          " --help | ");



    GString * pmount_async_check = g_string_new(pmount_base_command->str);
    pmount_async_check = g_string_append(pmount_async_check,
                                GREP_PATH " -- '--async' &> /dev/null");

    argv[2] = pmount_async_check->str;
    success = g_spawn_sync(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL,
                           &exitStatus, &error);

    g_string_free(pmount_async_check, TRUE);

    if (!success)
    {
        DEBUG("Error while trying to execute pmount: %s!\n",
              error->message);
        g_error_free(error);
        g_string_free(pmount_base_command, TRUE);
        return;
    }

    if (WIFEXITED(exitStatus) && (WEXITSTATUS(exitStatus) == 0))
    {
        DEBUG("pmount accepts --async");
        pmount_accepts_async = TRUE;
    }
    else
    {
        DEBUG("pmount does not accept --async");
        pmount_accepts_async = FALSE;
    }
    
    
    
    GString * pmount_fstype_check = g_string_new(pmount_base_command->str);
    pmount_fstype_check = g_string_append(pmount_fstype_check,
                             GREP_PATH " -- '-u <' &> /dev/null");
    argv[2] = pmount_fstype_check->str;
    success = g_spawn_sync(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL,
                           &exitStatus, &error);

    g_string_free(pmount_fstype_check, TRUE);

    if (!success)
    {
        DEBUG("Error while trying to execute pmount: %s!\n",
              error->message);
        g_error_free(error);
        g_string_free(pmount_base_command, TRUE);
        return;
    }

    if (WIFEXITED(exitStatus) && (WEXITSTATUS(exitStatus) == 0))
    {
        DEBUG("pmount accepts -t <fs>");
        pmount_accepts_fstype = TRUE;
    }
    else
    {
        DEBUG("pmount does not accept -t <fs>");
        pmount_accepts_fstype = FALSE;
    }
    
    
    
    GString * pmount_umask_check = g_string_new(pmount_base_command->str);
    pmount_umask_check = g_string_append(pmount_umask_check,
                               GREP_PATH " -- '-t <' &> /dev/null");
    argv[2] = pmount_umask_check->str;
    success = g_spawn_sync(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL,
                           &exitStatus, &error);

    g_string_free(pmount_umask_check, TRUE);

    if (!success)
    {
        DEBUG("Error while trying to execute pmount: %s!\n",
              error->message);
        g_error_free(error);
        g_string_free(pmount_base_command, TRUE);
        return;
    }

    if (WIFEXITED(exitStatus) && (WEXITSTATUS(exitStatus) == 0))
    {
        DEBUG("pmount accepts -u <umask>");
        pmount_accepts_fstype = TRUE;
    }
    else
    {
        DEBUG("pmount does not accept -u <umask>");
        if (usermode == FALSE)
          DEBUG(
"WARNING: since pmount does not accept -u <umask>, I will refuse to mount\
          volumes (since they would only be accessible by the %s\
          user account, which is probably not what you want).  Please\
          upgrade your version of pmount or run an instance of Ivman as\
          the user for which you wish devices to be accessible.",
          cfg_base->user);
        pmount_accepts_fstype = FALSE;
    }

#endif // defined PMOUNT_PATH
    return;
}


// when passed a filename of a configuration file, returns the full
// path which should be used to load that file
char *ivm_get_config_file(char const *const filename)
{
    GString *configFile;

    if (usermode == TRUE)
    {
        configFile = g_string_new((char *) strdup(homeDir));
        configFile = g_string_append(configFile, "/.ivman/");
        configFile = g_string_append(configFile, filename);
    } else
    {
        configFile = g_string_new(SYSCONFDIR);
        configFile = g_string_append(configFile, "/ivman/");
        configFile = g_string_append(configFile, filename);
    }
    char *ret = (char *) strdup(configFile->str);

    g_string_free(configFile, TRUE);
    return ret;
}

// when running in user mode, tests for config files, and creates them if they
// do not already exist
void ivm_test_configs()
{
    FILE *file;
    GString *filename;
    char *content;

    GString *directory = g_string_new((char *) strdup(homeDir));

    directory = g_string_append(directory, "/.ivman/");
    DIR *dir = opendir(directory->str);

    if (dir == NULL)
    {
        DEBUG
            ("Settings directory does not exist, attempting to create...");
        // make the necessary directory if it does not exist
        GString *command = g_string_new(MKDIR_PATH);

        command = g_string_append(command, " -p ");
        command = g_string_append(command, directory->str);
        ivm_run_command(command->str, NULL, FALSE);
        g_string_free(command, TRUE);

        // wait 5 seconds for the directory to be created
        int i = 0;

        dir = opendir(directory->str);
        while (dir == NULL && i < 5)
        {
            sleep(1);
            dir = opendir(directory->str);
            i++;
        }
        if ((void *) opendir(directory->str) == NULL)
        {
            fprintf(stderr, "Couldn't create %s\n", directory->str);
            exit(1);
        } else
            closedir(dir);
    } else
        closedir(dir);
    g_string_free(directory, TRUE);

    char * filenames[] = {"IvmConfigActions.xml", "IvmConfigBase.xml",
                          "IvmConfigProperties.xml", "IvmConfigConditions.xml", NULL };
    char * contents[] = {IVMCONFIGACTIONS, IVMCONFIGBASE, IVMCONFIGPROPERTIES,
                         IVMCONFIGCONDITIONS, NULL };

    for (int i = 0; filenames[i] != NULL; i++)
    {
        filename = g_string_new((char *) strdup(homeDir));
        filename = g_string_append(filename, "/.ivman/");
        filename = g_string_append(filename, filenames[i]);
        file = fopen(filename->str, "r");
        if (file == NULL)
        {
            DEBUG("%s not found, creating default", filename->str);
            file = fopen(filename->str, "w");
            content = contents[i];
            fwrite(content, 1, strlen(content), file);
            fclose(file);
        } else
        {
            fclose(file);
        }
        g_string_free(filename, TRUE);
    }

}


void ivm_load_config(void)
{
    char *file = ivm_get_config_file("IvmConfigBase.xml");

    cfg_base = parseIvmConfigBase(file);
    free(file);
    if (!cfg_base)
    {
        DEBUG("Could not parse IvmConfigBase.xml!\n\
    Please check this file for errors - read 'man IvmConfigBase.xml' for\n\
    some tips (e.g., how to use special characters such as <,> and &).\n\
    Since I need IvmConfigBase.xml to function, I will now exit.");
        exit(1);
    }
    // if we're not running as root, set pidFile to ~/.ivman/.lock
    if (usermode == TRUE)
    {
        GString *pidFile = g_string_new((char *) strdup(homeDir));

        pidFile = g_string_append(pidFile, "/.ivman/.lock");
        cfg_base->pidFile = (char *) strdup(pidFile->str);
        g_string_free(pidFile, TRUE);

    }

}

void
ivm_free_str_array(char ** strings)
{
    if (strings == NULL)
        return;
    int i = 0;
    while (strings[i] != NULL)
    {
        free(strings[i]);
        i++;
    }
    free(strings);
}

void
ivm_run_commands(char ** commands, char const * udi, gboolean const execun)
{
    if (commands == NULL)
        return;
    int i = 0;
    while (commands[i] != NULL)
    {
        ivm_run_command(commands[i], udi, execun);
        i++;
    }

    if (execun)
        g_hash_table_remove(pss, udi);
}

gboolean
ivm_run_command(char const *const command, char const * udi,
                gboolean const execun)
{
    char *argv[4];
    char *new_command = (char *) strdup(command);
    LibHalPropertySet *hal_ps = NULL;

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    GError *error = NULL;

    GString *currentCommand;

    int length;
    int i;
    char c;
    char *substitution;
    char *propertyname = NULL;
    char *substring = strstr(new_command, "$hal.");
    char *substring2;
    int pos = (int) ((substring - new_command) / sizeof(char));
    char *beginning = (char *) strndup(new_command, pos);

    if (execun)
        hal_ps =
            (LibHalPropertySet *) g_hash_table_lookup(pss, (gpointer) udi);

    while (substring != NULL)
    {
        // get rid of $hal.
        substring[0] = ' ';
        substring[1] = ' ';
        substring[2] = ' ';
        substring[3] = ' ';
        substring[4] = ' ';
        substring = (char *) g_strchug(substring);

        substring2 = strstr(substring, "$");
        length = (int) ((substring2 - substring) / sizeof(char));
        propertyname = (char *) strndup(substring, length);

        i = 0;
        while (i < length)
        {
            substring[i] = ' ';
            i++;
        }
        substring[i] = ' ';

        // hack so we don't eat an extra space
        c = substring[i + 1];
        substring[i + 1] = 'a';

        substring = (char *) g_strchug(substring);

        substring[0] = c;

        if ((execun && psi_property_exists(hal_ps, propertyname))
            || libhal_device_property_exists(hal_ctx, udi,
                                             propertyname, &dbus_error))
        {
            unsigned int type = execun ? psi_get_property_type(hal_ps,
                                                               propertyname)
                : libhal_device_get_property_type(hal_ctx, udi,
                                                  propertyname,
                                                  &dbus_error);

            if (type == DBUS_TYPE_STRING)
            {
                substitution =
                    execun ? (char *)
                    psi_get_property_string(hal_ps,
                                            propertyname)
                    : (char *)
                    libhal_device_get_property_string
                    (hal_ctx, udi, propertyname, &dbus_error);

            }

            else if (type == DBUS_TYPE_BOOLEAN)
            {
                if ((execun
                     && psi_get_property_bool(hal_ps,
                                              propertyname))
                    ||
                    libhal_device_get_property_bool
                    (hal_ctx, udi, propertyname, &dbus_error))
                    substitution = (char *) strdup("true");
                else
                    substitution = (char *) strdup("false");
            }

            else if (type == DBUS_TYPE_INT32 || type == DBUS_TYPE_INT64)
            {

                asprintf(&substitution, "%d",
                         execun ?
                         psi_get_property_int(hal_ps,
                                              propertyname)
                         :
                         libhal_device_get_property_int
                         (hal_ctx, udi, propertyname, &dbus_error));

            } else if (type == DBUS_TYPE_UINT32
                       || type == DBUS_TYPE_UINT64)
            {

                asprintf(&substitution, "%u",
                         execun ? (unsigned int)
                         psi_get_property_uint64(hal_ps,
                                                 propertyname)
                         : (unsigned int)
                         libhal_device_get_property_uint64
                         (hal_ctx, udi, propertyname, &dbus_error));
            }

            else
                substitution = (char *) strdup("NULL");

            // we don't worry about other types of properties,
            // they're pretty uncommon
        } else
        {
            DEBUG
                ("Warning: unhandled HAL type encountered, NULL substituted for value!");
            substitution = (char *) strdup("NULL");
        }

        if (propertyname != NULL)
            free(propertyname);

        currentCommand = g_string_new(substring);

        currentCommand = g_string_prepend(currentCommand, substitution);
        currentCommand = g_string_prepend(currentCommand, beginning);
        new_command = (char *) strdup(currentCommand->str);
        g_string_free(currentCommand, TRUE);

        if (NULL != substitution)
            free(substitution);
        libhal_free_string(beginning);

        substring = strstr(new_command, "$hal.");
        int pos = (int) ((substring - new_command) / sizeof(char));

        beginning = (char *) strndup(new_command, pos);
    }

    ivm_check_dbus_error(&dbus_error);

    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = new_command;
    argv[3] = NULL;
    DEBUG("Running: %s", argv[2]);
    if (!g_spawn_async(NULL, argv, NULL, 0, NULL, NULL,
        NULL, &error))
    {
        warn("failed to exec %s: %s\n", new_command, error->message);
        return FALSE;
    }
    g_free(new_command);
    return TRUE;
}

// despite the name, returns TRUE if DVD is a video DVD, FALSE otherwise
gboolean ivm_is_dvd(char const *const udi)
{
    char *path;
    gboolean retval;

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    // return false if DVD is not mounted
    if (!libhal_device_property_exists
        (hal_ctx, udi, "volume.is_mounted", &dbus_error))
        return FALSE;
    if (libhal_device_get_property_bool
        (hal_ctx, udi, "volume.is_mounted", &dbus_error) == FALSE)
        return FALSE;

    if (!libhal_device_property_exists
        (hal_ctx, udi, "volume.mount_point", &dbus_error))
        return FALSE;
    char *mount_point =
        (char *) libhal_device_get_property_string(hal_ctx, udi,
                                                   "volume.mount_point",
                                                   &dbus_error);

    if (!libhal_device_property_exists
        (hal_ctx, udi, "block.device", &dbus_error))
        return FALSE;
    char *device = libhal_device_get_property_string(hal_ctx, udi,
                                                     "block.device",
                                                     &dbus_error);

    DEBUG("Checking for video DVD in '%s' mounted on '%s'", device,
          mount_point);

    path = g_build_path(G_DIR_SEPARATOR_S, mount_point, "video_ts", NULL);
    retval = g_file_test(path, G_FILE_TEST_IS_DIR);
    g_free(path);

    /* try the other name, if needed */
    if (retval == FALSE)
    {
        path =
            g_build_path(G_DIR_SEPARATOR_S, mount_point, "VIDEO_TS", NULL);
        retval = g_file_test(path, G_FILE_TEST_IS_DIR);
        g_free(path);
    }

    if (retval == TRUE)
        DEBUG("%s is a video DVD", device);
    else
        DEBUG("%s is not a video DVD", device);

    libhal_free_string(mount_point);
    libhal_free_string(device);

    ivm_check_dbus_error(&dbus_error);

    return retval;
}

#ifdef PMOUNT_PATH
void ivm_device_mount(char const *const udi)
#else
void ivm_device_mount(char const *const udi __attribute__ ((__unused__)))
#endif
{
#ifdef PMOUNT_PATH

    if (!usermode && !pmount_accepts_umask)
    {
        DEBUG(
"Refusing to mount %s because I am a system-wide instance of Ivman but\
pmount does not accept -u <umask>, so if I mounted, it would be accessible\
from the %s account only.  Upgrade pmount or start an instance of Ivman\
as the user you wish devices to be accessible from.", udi, cfg_base->user);
        return;
    }


    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    GString * pmount_command = g_string_new(PMOUNT_PATH);

    if (usermode && pmount_accepts_umask)
        // If in usermode, only grant access to the current user
        pmount_command = g_string_append(pmount_command, " -u 0077 ");
    else if (!usermode && pmount_accepts_umask)
        // Otherwise, read access to all, write access to users in
        // plugdev group
        pmount_command = g_string_append(pmount_command, " -u 0002 ");

    // In theory, this shouldn't be needed, but I've seen pmount sometimes
    // mysteriously getting the fs type wrong.
    if (pmount_accepts_fstype)
        pmount_command = g_string_append(pmount_command,
                                         " -t $hal.volume.fstype$ ");

    // If pmount accepts --async, use it.
    if (pmount_accepts_async)
        pmount_command = g_string_append(pmount_command,
                                         " --async ");

    pmount_command = g_string_append(pmount_command, 
                                     " $hal.block.device$ ");

    if (libhal_device_property_exists(hal_ctx, udi,
        "volume.policy.desired_mount_point", &dbus_error))
    {

        char * command = strdup(pmount_command->str);
        pmount_command = g_string_append(pmount_command,
                    " $hal.volume.policy.desired_mount_point$ || ");
        pmount_command = g_string_append(pmount_command, command);
        free(command);
    }

    ivm_run_command(pmount_command->str, udi, FALSE);

    g_string_free(pmount_command, TRUE);

    ivm_check_dbus_error(&dbus_error);

#endif      // defined PMOUNT_PATH
    return;
}



void ivm_media_changed(char const *const udi)
{

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    char *file = ivm_get_config_file("IvmConfigActions.xml");
    IvmConfigActions * cfg = parseIvmConfigActions(file, (char *) udi);

    free(file);
    if (!cfg)
    {
        xmlReadFailed("IvmConfigActions.xml");
        return;
    }

    char *device = NULL;

    if (libhal_device_property_exists
        (hal_ctx, udi, "block.device", &dbus_error))
        device =
            (char *) libhal_device_get_property_string(hal_ctx,
                                                       udi,
                                                       "block.device",
                                                       &dbus_error);

    char *mount_point = NULL;

    if (cfg->mount == TRUE)
    {
        // If we're running in system-wide mode, sleep a few seconds
        // to allow any user-mode instances to mount first.
        // This might seem a little hacky, but I think it's much better
        // than implementing some kind of IPC mechanism for instances
        // of Ivman to talk to each other.
        if (!usermode)
        {
            DEBUG("Waiting for user mode Ivmans to mount...");
            sleep(5);
        }

        if (libhal_device_property_exists
            (hal_ctx, udi, "volume.is_mounted", &dbus_error)
            && !libhal_device_get_property_bool(hal_ctx, udi, "volume.is_mounted",
                                               &dbus_error))
        {
            DEBUG("Attempting to mount %s", device);
            ivm_device_mount(udi);
        }
    }
    // After mounting the filesystem, it takes HAL a while to update,
    // so we poll once every second for five seconds to find out the
    // mount point
    if (cfg->need_mountpoint)
    {
        int i = 0;

        mount_point =
            (char *) libhal_device_get_property_string(hal_ctx,
                                                       udi,
                                                       "volume.mount_point",
                                                       &dbus_error);
        while ((mount_point == NULL || strlen(mount_point) < 1) && i < 5)
        {
            sleep(1);
            mount_point =
                (char *)
                libhal_device_get_property_string(hal_ctx, udi,
                                                  "volume.mount_point",
                                                  &dbus_error);
            i++;
        }
        if (mount_point == NULL || strlen(mount_point) < 1)
        {
            warn("Couldn't get mount point of %s", device);
        }

    }


    if (TRUE == ivm_is_dvd(udi))
        ivm_run_commands(cfg->execdvd, udi, FALSE);
    ivm_free_str_array(cfg->execdvd);

    ivm_run_commands(cfg->exec, udi, FALSE);
    ivm_free_str_array(cfg->exec);

    if (cfg->execun != NULL)
    {

        g_hash_table_insert(execuns, (gpointer) strdup(udi), (gpointer) cfg->execun);
        LibHalPropertySet *hal_ps =
            libhal_device_get_all_properties(hal_ctx,
                                             udi,
                                             &dbus_error);

        g_hash_table_insert(pss, (gpointer) strdup(udi), (gpointer) hal_ps);
    }

    free(cfg);
    libhal_free_string(device);
    libhal_free_string(mount_point);

    ivm_check_dbus_error(&dbus_error);
}

void ivm_sighup_handler(int const sig)
{
    switch (sig)
    {
    case SIGHUP:
        DEBUG("Reloading configuration");
        ivm_load_config();
        signal(SIGHUP, SIG_IGN);
        break;
    }
}

void ivm_umount_if_needed(char const *const mountpoint)
{
#ifndef PMOUNT_PATH
    return;
#else

    if (mountpoint == NULL)
    {
        /* DEBUG("This device (%s) wasn't mounted by me, ignoring...",
           mountpoint); */
        return;
    }
    DEBUG("Umounting %s", mountpoint);
    char *argv[3];
    GError *error = NULL;

    argv[0] = PUMOUNT_PATH;
    argv[1] = (char *) mountpoint;
    argv[2] = NULL;

    g_spawn_async("/", argv, NULL, 0, NULL, NULL, NULL, &error);
    if (error)
    {
        warn("failed to exec " PUMOUNT_PATH ": %s\n", error->message);
        g_error_free(error);
    }

#endif // ! PMOUNT_PATH
}


int main(int argc, char *argv[])
{
    GError *optParseError = NULL;
    gboolean systemOpt = FALSE;
    gboolean noForkOpt = FALSE;
    gboolean debugOpt = FALSE;
    GOptionEntry entries[] = {
        {"system", 's', 0, G_OPTION_ARG_NONE, &systemOpt,
         "Force Ivman to run in system-wide mode.\n\
                   Use this if you want Ivman to read configuration from\n\
                   " SYSCONFDIR "/ivman/ but you're not running it as root.",
         NULL},
        {"debug", 'd', 0, G_OPTION_ARG_NONE, &debugOpt,
         "Force Ivman to show debugging output.", NULL},
        {"nofork", (char) NULL, 0, G_OPTION_ARG_NONE, &noForkOpt,
         "Force Ivman not to daemonize itself.", NULL},
        {(const gchar *) NULL, (gchar) NULL, (gint) NULL,
         (GOptionArg) NULL,
         (gpointer) NULL, (const gchar *) NULL,
         (const gchar *) NULL}
    };

    GOptionContext *context = g_option_context_new("- start ivman");

    g_option_context_add_main_entries(context, entries, NULL);
    if (!g_option_context_parse(context, &argc, &argv, &optParseError))
    {
        fprintf(stderr, "%s\n", optParseError->message);
        exit(1);
    }

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    usermode = TRUE;

    // are we running as root or with --system option?
    if (systemOpt || geteuid() == 0)
        usermode = FALSE;

    homeDir = (gchar *) g_get_home_dir();
    if (usermode)
        ivm_test_configs();

    ivm_load_config();

    if (debugOpt)
        cfg_base->debug = TRUE;

#ifdef HAL_0_4
    LibHalFunctions hal_functions = { hal_mainloop_integration,
        hal_device_added,
        hal_device_removed,
        hal_device_new_capability,
        hal_device_lost_capability,
        hal_device_property_modified,
        hal_device_condition
    };

    hal_ctx = (LibHalContext *) hal_initialize(&hal_functions, FALSE);
    if (!hal_ctx)
    {
        warn("Failed to create HAL context!\n");
        return 1;
    }
#else
    hal_ctx = (LibHalContext *) libhal_ctx_new();
    if (!hal_ctx)
    {
        warn("Failed to create HAL context!\n");
        return 1;
    }


    if (!hal_mainloop_integration(hal_ctx, &dbus_error))
    {
        warn("Couldn't connect to HAL!\n");
        return 1;
    }
    // Set up all the callbacks in hal_interface.c
    libhal_ctx_set_device_added(hal_ctx, hal_device_added);
    libhal_ctx_set_device_removed(hal_ctx, hal_device_removed);
    libhal_ctx_set_device_new_capability(hal_ctx,
                                         hal_device_new_capability);
    libhal_ctx_set_device_lost_capability(hal_ctx,
                                          hal_device_lost_capability);
    libhal_ctx_set_device_property_modified(hal_ctx,
                                            hal_device_property_modified);
    libhal_ctx_set_device_condition(hal_ctx, hal_device_condition);

    if (!libhal_ctx_init(hal_ctx, &dbus_error))
    {
        warn("Couldn't initialise HAL!\n");
        return 1;
    }
#endif

#ifdef HAL_0_4
    if (libhal_device_property_watch_all(hal_ctx, &dbus_error))
#else
    if (!libhal_device_property_watch_all(hal_ctx, &dbus_error))
#endif // HAL_0_4
    {
        warn("Failed to watch all HAL properties!\n%s\n",
             dbus_error.message);
        return 1;
    }


    devices = g_hash_table_new_full(g_str_hash, g_str_equal, free, free);
    execuns = g_hash_table_new_full(g_str_hash, g_str_equal, free, (GDestroyNotify)ivm_free_str_array);
#ifndef HAL_0_4
    pss = g_hash_table_new_full(g_str_hash, g_str_equal, free, (GDestroyNotify)libhal_free_property_set);
#else
    pss = g_hash_table_new_full(g_str_hash, g_str_equal, free, (GDestroyNotify)hal_free_property_set);
#endif

    printf("%s, http://ivman.sourceforge.net\n", PACKAGE_STRING);
#ifdef HAL_0_4
    printf("Compiled against HAL 0.4.x or earlier\n");
#else
    printf("Compiled against HAL 0.5.x or later\n");
#endif
    isdaemon = FALSE;

    if (usermode == FALSE)
    {
        printf("Running in system mode\n");
        log_notice("Ivman started in system mode");
    } else
    {
        printf("Running in user mode\n");
        log_notice("Ivman started in user mode");
    }

    signal(SIGHUP, ivm_sighup_handler);

    if (noForkOpt)
        cfg_base->fork = FALSE;

    if (cfg_base->fork == TRUE)
    {
        // Daemonize if told to do so in IvmConfigBase.xml
        isdaemon = daemonize();
        if (!isdaemon)
        {
            return 1;
        }
    }

    // Permanently unlock CD/DVD drive
    if (geteuid() == 0)
        ivm_run_command("echo 0 > /proc/sys/dev/cdrom/lock", NULL, FALSE);

    // Drop privileges
    if (geteuid() == 0 && !dropPrivileges(cfg_base->user, cfg_base->group))
    {
        DEBUG("Couldn't drop privileges to %s:%s, exiting!", cfg_base->user, cfg_base->group);
        exit(1);
    }

    discover_pmount_version();

    char *file = ivm_get_config_file("IvmConfigProperties.xml");
    IvmConfigProperties * prop_cfg =
        parseIvmConfigProperties(file, NULL, NULL);
    if (!prop_cfg)
    {
        xmlReadFailed("IvmConfigProperties.xml");
    } else
    {
        if (prop_cfg->checkOnInit == TRUE)
        {
            DEBUG("Running through rules for every current property.");
            // run through every device

            int num_devices;
            char **devices = (char **) libhal_get_all_devices(hal_ctx,
                                                              &num_devices,
                                                              &dbus_error);


            int i = 0;
            while (i < num_devices)
            {
                IvmConfigProperties * newProp_cfg =
                    IvmConfigPropertiesAll(file, devices[i]);
                ivm_run_commands(newProp_cfg->exec, devices[i], FALSE);
                ivm_free_str_array(newProp_cfg->exec);
                free(newProp_cfg);
                i++;
            }
            if (devices != NULL)
                libhal_free_string_array(devices);
        }

        free(prop_cfg);
    }
    free(file);

    ivm_check_dbus_error(&dbus_error);

    GMainLoop *loop = g_main_loop_new(NULL, FALSE);
    DEBUG("Entering main loop.\n");
    g_main_loop_run(loop);

    if (cfg_base->fork == TRUE)
    {
        clear_pidfile(cfg_base->pidFile);
    }

    g_hash_table_destroy(devices);

    return 0;
}
