/*
 * netfilter module for userspace packet logging daemons
 *
 * (C) 2000-2004 by Harald Welte <laforge@netfilter.org>
 *
 * 2000/09/22 ulog-cprange feature added
 * 2001/01/04 in-kernel queue as proposed by Sebastian Zander 
 * 						<zander@fokus.gmd.de>
 * 2001/01/30 per-rule nlgroup conflicts with global queue. 
 *            nlgroup now global (sysctl)
 * 2001/04/19 ulog-queue reworked, now fixed buffer size specified at
 * 	      module loadtime -HW
 * 2002/07/07 remove broken nflog_rcv() function -HW
 * 2002/08/29 fix shifted/unshifted nlgroup bug -HW
 * 2002/10/30 fix uninitialized mac_len field - <Anders K. Pedersen>
 * 2004/10/25 fix erroneous calculation of 'len' parameter to NLMSG_PUT
 *	      resulting in bogus 'error during NLMSG_PUT' messages.
 * 2005/02/10 ported to ipv6
 *
 * (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2004 Netfilter Core Team <coreteam@netfilter.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/netfilter_ipv6/ip6_tables.h>
#include <linux/netfilter_ipv4/ipt_ULOG.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jonas Berlin <xkr47@outerspace.dyndns.org>");
MODULE_DESCRIPTION("ip6tables userspace logging module");

#if 0
#define DEBUGP(format, args...) printk("%s:%s:" format, \
                                       __FILE__, __FUNCTION__ , ## args)
#else
#define DEBUGP(format, args...)
#endif

static unsigned int nflog = 1;
module_param(nflog, int, 0400);
MODULE_PARM_DESC(nflog, "register as internal netfilter logging module");

// from ipt_ULOG.c
void ipt_ulog_packet(unsigned int hooknum,
		     const struct sk_buff *skb,
		     const struct net_device *in,
		     const struct net_device *out,
		     const struct ipt_ulog_info *loginfo,
		     const char *prefix);

static unsigned int ip6t_ulog_target(struct sk_buff **pskb,
				    const struct net_device *in,
				    const struct net_device *out,
				    unsigned int hooknum,
				    const void *targinfo, void *userinfo)
{
	const struct ipt_ulog_info *loginfo = (const struct ipt_ulog_info *) targinfo;

	ipt_ulog_packet(hooknum, *pskb, in, out, loginfo, NULL);
 
 	return IP6T_CONTINUE;
}
 
static void ip6t_logfn(unsigned int hooknum,
		      const struct sk_buff *skb,
		      const struct net_device *in,
		      const struct net_device *out,
		      const char *prefix)
{
	struct ipt_ulog_info loginfo = { 
		.nl_group = ULOG_DEFAULT_NLGROUP,
		.copy_range = 0,
		.qthreshold = ULOG_DEFAULT_QTHRESHOLD,
		.prefix = ""
	};

	ipt_ulog_packet(hooknum, skb, in, out, &loginfo, prefix);
}

static int ip6t_ulog_checkentry(const char *tablename,
			       const struct ip6t_entry *e,
			       void *targinfo,
			       unsigned int targinfosize,
			       unsigned int hookmask)
{
	struct ipt_ulog_info *loginfo = (struct ipt_ulog_info *) targinfo;

	if (targinfosize != IP6T_ALIGN(sizeof(struct ipt_ulog_info))) {
		DEBUGP("ip6t_ULOG: targinfosize %u != 0\n", targinfosize);
		return 0;
	}

	if (loginfo->prefix[sizeof(loginfo->prefix) - 1] != '\0') {
		DEBUGP("ip6t_ULOG: prefix term %i\n",
		       loginfo->prefix[sizeof(loginfo->prefix) - 1]);
		return 0;
	}

	if (loginfo->qthreshold > ULOG_MAX_QLEN) {
		DEBUGP("ip6t_ULOG: queue threshold %i > MAX_QLEN\n",
			loginfo->qthreshold);
		return 0;
	}

	return 1;
}

static struct ip6t_target ip6t_ulog_reg = {
	.name		= "ULOG",
	.target		= ip6t_ulog_target,
	.checkentry	= ip6t_ulog_checkentry,
	.me		= THIS_MODULE,
};

static int __init init(void)
{
	DEBUGP("ip6t_ULOG: init module\n");

	if (ip6t_register_target(&ip6t_ulog_reg) != 0) {
		return -EINVAL;
	}
	if (nflog)
		nf_log_register(PF_INET6, &ip6t_logfn);
	
	return 0;
}

static void __exit fini(void)
{
	DEBUGP("ip6t_ULOG: cleanup_module\n");

	if (nflog)
		nf_log_unregister(PF_INET6, &ip6t_logfn);
	ip6t_unregister_target(&ip6t_ulog_reg);
}

module_init(init);
module_exit(fini);
