import commands
import shutil
import os

from twisted.internet import reactor

from imagestore.lib import bpickle
from imagestore.signatureservice import (
    SignatureService, CheckImageSignatureTask, SignatureError)

from imagestore.tests.helpers import ServiceTestCase


class SignatureServiceTest(ServiceTestCase):

    def setUp(self):
        self.keyringDir = self.makeDir()
        self.privateKeyringPath = os.path.join(self.keyringDir, "secring.gpg")
        self.publicKeyringPath = os.path.join(self.keyringDir, "keyring.gpg")
        self.homeDir = self.makeDir()
        self.prepareKeyrings()
        self.service = SignatureService(reactor, self.publicKeyringPath)

    def startWaitStop(self, deferred):
        return self.runServicesAndWaitForDeferred([self.service], deferred)

    def runGPG(self, *args):
        fullArgs = ["gpg", "--batch", "--no-secmem-warning", "--no-options",
                    "--status-fd", "1", "--no-default-keyring",
                    "--homedir", self.homeDir,
                    "--keyring", self.publicKeyringPath,
                    "--secret-keyring", self.privateKeyringPath]
        fullArgs.extend(args)
        command = " ".join(fullArgs) # Should be fine for our use here.
        status, output = commands.getstatusoutput(command)
        self.assertEquals(status, 0, "GPG failed executing:\n%s" % (output,))

    def prepareKeyrings(self):
        # Create the keyring for tests with both the private and
        # the public keys.
        publicKeyPath = self.makeFile(PUBLIC_KEY)
        privateKeyPath = self.makeFile(PRIVATE_KEY)
        for keyPath in (publicKeyPath, privateKeyPath):
            self.runGPG("--import", keyPath)

    def createSignature(self, data):
        signaturePath = self.makeFile()
        dataPath = self.makeFile(data)
        self.runGPG("--armor", "--output", signaturePath,
                    "--detach-sign", dataPath)
        return open(signaturePath).read()

    def signImage(self, image, signatureFields):
        image["signature-fields"] = signatureFields
        imageDict = image.toDict()
        signatureDict = {}
        for key in signatureFields:
            signatureDict[key] = imageDict[key]
        signatureData = bpickle.dumps(signatureDict)
        image["signature"] = self.createSignature(signatureData)

    def testCheckImageSignatureWithGoodSignature(self):
        image = self.createImage(1, withFiles=True)
        self.signImage(image, ["files", "title", "version", "signature-fields"])
        def checkResult(result):
            self.assertEquals(result, image)
            # This means it's correctly using the keyring base path
            # as the --homedir option:
            self.assertIn("trustdb.gpg", os.listdir(self.keyringDir))
        deferred = self.service.addTask(CheckImageSignatureTask(image))
        deferred.addCallback(checkResult)
        return self.startWaitStop(deferred)

    def testCheckImageSignatureWithBadSignature(self):
        image = self.createImage(1, withFiles=True)
        self.signImage(image, ["files", "title", "version", "signature-fields"])

        image["version"] += ".1"

        def checkError(failure):
            failure.trap(SignatureError)
            self.assertEquals(failure.value.message,
                              "Image signature is invalid")
            return failure

        deferred = self.service.addTask(CheckImageSignatureTask(image))
        deferred.addErrback(checkError)
        self.assertFailure(deferred, SignatureError)
        return self.startWaitStop(deferred)

    def testCheckImageSignatureWithMissingSignatureFieldsSelfSignature(self):
        image = self.createImage(1, withFiles=True)
        self.signImage(image, ["files", "title", "version"])

        def checkError(failure):
            failure.trap(SignatureError)
            self.assertEquals(failure.value.message,
                              "signature-fields must be part of the signature")
            return failure

        deferred = self.service.addTask(CheckImageSignatureTask(image))
        deferred.addErrback(checkError)
        self.assertFailure(deferred, SignatureError)
        return self.startWaitStop(deferred)

    def testCheckImageSignatureWithMissingFilesInSignature(self):
        image = self.createImage(1, withFiles=True)
        self.signImage(image, ["title", "version", "signature-fields"])

        def checkError(failure):
            failure.trap(SignatureError)
            self.assertEquals(failure.value.message,
                              "files must be part of the signature")
            return failure

        deferred = self.service.addTask(CheckImageSignatureTask(image))
        deferred.addErrback(checkError)
        self.assertFailure(deferred, SignatureError)
        return self.startWaitStop(deferred)



PUBLIC_KEY = """
-----BEGIN PGP PUBLIC KEY BLOCK-----
Version: GnuPG v1.4.9 (GNU/Linux)

mQGiBEq2ngcRBACvIZ8ri1gspFXqA1DRgsI+4Sfp9lnjl5e2i80waoTpL5IARFHQ
U7nsI9VsrFH5+OIyIvp6/6GFACHI7BAuIz4LGunJqdekEyhIj9TXxPN5/jSwzB27
+qYUpBvxmWycKVT1XOA/lX3D/W2RBF6G3GrT7XMFQ2gLSui/4dwavzvrswCgtK59
hZs8r1Rnd8lv/sOroYMQ+q0EAK6poR6GU8H2GyYpAPXYtRRbX3Db1n8izggXrFv7
ENZYVN3nMps36ctV///NE0n2hhHxdODtOogdBNrtI7fn4UlIJ0MmM6ABopo57nP/
c/evwYhek0ciP8LBQ8H4q97IM+Us2hrq6RtRNzx6Lxia258Sh0I52ZsTrCe4BM6P
u4TZBACqJC4YWwXauF0pOgOf6FOvEy7JVar4WojAHUowUv90F92KqYViRT2e0X27
oKwyDST5IZgbFVu5/j8KUNmgNcIJLgHxtILcbrNPh+FJN7AeVKfEUuyPFONxTVMb
++HbgXDWamUgxttFzfjT7xABceE+levDAm2VVxfEzhJSxH95JLQqSW1hZ2UgU3Rv
cmUgVGVzdHMgPGltYWdlc3RvcmVAZXhhbXBsZS5jb20+iGAEExECACAFAkq2ngcC
GwMGCwkIBwMCBBUCCAMEFgIDAQIeAQIXgAAKCRDwgLa6zTyytM1CAKCiDcOs/XIe
ZvKufb/R5WitXL6dCACeLxGxbiaF2heHpBB2QNp4B4Dx5ia5AQ0ESraeBxAEAOWz
uEUJbDoc8nFA6UQe5KSjWOH5h0mZIbBRc0RAY9NjCpDA6FXFFiHQ6CPlqoGTejDv
QGQ+YVqrcejhHyyh6jvoGhtt6caE2MmazbBWbWqM6D+FLUJVncPIZ5W45p8cOuWn
/CCNJqAdpjEBY+qM+xcIxLrpLGrX8OJTuTRYAdOvAAMFBADdaH9NsElh3gd1EXHa
TbPYcPFBZCg5p5EpMPQzmbjqI3IwSF5oZFHAspBr8KVzBEBW+uVbuJmz5DCX1Qmb
KKg8Jdf3tZvMDZAeYsgx3IYlcm0bjKZ2Rwvg23o/MlLZR2WjUqbSgOt3IzlW/YA0
lHp+n77BRTQ2mUlh7GDXnbyl0ohJBBgRAgAJBQJKtp4HAhsMAAoJEPCAtrrNPLK0
l5oAnAuI6ncnDhZRhKxYyCbUP5e9rvn9AKCMnIJMFaGSujKX1dFtfjy8ofxmNA==
=TPJF
-----END PGP PUBLIC KEY BLOCK-----
"""

PRIVATE_KEY = """
-----BEGIN PGP PRIVATE KEY BLOCK-----
Version: GnuPG v1.4.9 (GNU/Linux)

lQG7BEq2ngcRBACvIZ8ri1gspFXqA1DRgsI+4Sfp9lnjl5e2i80waoTpL5IARFHQ
U7nsI9VsrFH5+OIyIvp6/6GFACHI7BAuIz4LGunJqdekEyhIj9TXxPN5/jSwzB27
+qYUpBvxmWycKVT1XOA/lX3D/W2RBF6G3GrT7XMFQ2gLSui/4dwavzvrswCgtK59
hZs8r1Rnd8lv/sOroYMQ+q0EAK6poR6GU8H2GyYpAPXYtRRbX3Db1n8izggXrFv7
ENZYVN3nMps36ctV///NE0n2hhHxdODtOogdBNrtI7fn4UlIJ0MmM6ABopo57nP/
c/evwYhek0ciP8LBQ8H4q97IM+Us2hrq6RtRNzx6Lxia258Sh0I52ZsTrCe4BM6P
u4TZBACqJC4YWwXauF0pOgOf6FOvEy7JVar4WojAHUowUv90F92KqYViRT2e0X27
oKwyDST5IZgbFVu5/j8KUNmgNcIJLgHxtILcbrNPh+FJN7AeVKfEUuyPFONxTVMb
++HbgXDWamUgxttFzfjT7xABceE+levDAm2VVxfEzhJSxH95JAAAmgMXfGTda9rV
tY89nwqbM7RlnCc/CZ60KkltYWdlIFN0b3JlIFRlc3RzIDxpbWFnZXN0b3JlQGV4
YW1wbGUuY29tPohgBBMRAgAgBQJKtp4HAhsDBgsJCAcDAgQVAggDBBYCAwECHgEC
F4AACgkQ8IC2us08srTNQgCgog3DrP1yHmbyrn2/0eVorVy+nQgAni8RsW4mhdoX
h6QQdkDaeAeA8eYmnQEyBEq2ngcQBADls7hFCWw6HPJxQOlEHuSko1jh+YdJmSGw
UXNEQGPTYwqQwOhVxRYh0Ogj5aqBk3ow70BkPmFaq3Ho4R8soeo76BobbenGhNjJ
ms2wVm1qjOg/hS1CVZ3DyGeVuOafHDrlp/wgjSagHaYxAWPqjPsXCMS66Sxq1/Di
U7k0WAHTrwADBQQA3Wh/TbBJYd4HdRFx2k2z2HDxQWQoOaeRKTD0M5m46iNyMEhe
aGRRwLKQa/ClcwRAVvrlW7iZs+Qwl9UJmyioPCXX97WbzA2QHmLIMdyGJXJtG4ym
dkcL4Nt6PzJS2Udlo1Km0oDrdyM5Vv2ANJR6fp++wUU0NplJYexg1528pdIAAPoC
aEkc4DN2yhW+AYoMWUCX7CqtJvt/lE3h486ZtlokzRAmiEkEGBECAAkFAkq2ngcC
GwwACgkQ8IC2us08srSXmgCghdG//h/KDXvXbLQXCmFlqc448WgAnjig2yd9eEYs
kCBQUmKWzuhF+011
=QBOM
-----END PGP PRIVATE KEY BLOCK-----
"""
