#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.tests import SQLiteTestBase
from imagestore.lib.sqlitepatcher import SQLitePatcher, RepeatedVersionError


class SQLitePatcherTest(SQLiteTestBase):

    def testNoSchemaNoPatches(self):
        patcher = SQLitePatcher()
        patcher.patch(self.filename)

        # There are no patches, so the table should stay empty.
        cursor = self.execute("SELECT * FROM patch")
        self.assertEquals(cursor.fetchall(), [])

    def testPatches(self):
        patcher = SQLitePatcher()
        patcher.patch(self.filename)

        @patcher.addPatch(3)
        def patch(cursor):
            cursor.execute("CREATE TABLE foo (id SERIAL)")

        @patcher.addPatch(5)
        def patch(cursor):
            cursor.execute("CREATE TABLE bar (id SERIAL)")

        patcher.patch(self.filename)

        cursor = self.execute("SELECT * FROM foo")
        self.assertEquals(cursor.fetchall(), [])
        cursor = self.execute("SELECT * FROM bar")
        self.assertEquals(cursor.fetchall(), [])

        cursor = self.execute("SELECT * FROM patch ORDER BY version")
        self.assertEquals(cursor.fetchall(), [(3,), (5,)])

    def testPatchesAppliedInOrder(self):
        patcher = SQLitePatcher()
        patcher.patch(self.filename)

        applied = []
        versions = [5134, 1324, 7423, 3245, 2782, 9187]
        for version in versions:
            @patcher.addPatch(version)
            def patch(cursor, version=version):
                applied.append(version)

        patcher.patch(self.filename)
        self.assertEquals(applied, sorted(versions))

    def testSkipAppliedPatches(self):
        patcher = SQLitePatcher()
        patcher.patch(self.filename)

        self.execute("INSERT INTO patch VALUES (2782)")
        self.execute("INSERT INTO patch VALUES (7423)")
        self.close()

        applied = []
        versions = [5134, 1324, 7423, 3245, 2782, 9187]
        for version in versions:
            @patcher.addPatch(version)
            def patch(cursor, version=version):
                applied.append(version)

        versions.remove(2782)
        versions.remove(7423)

        patcher.patch(self.filename)
        self.assertEquals(applied, sorted(versions))

    def testApplyFinalSchema(self):
        patcher = SQLitePatcher()
        patcher.setFinalSchema([
            "CREATE TABLE foo (id SERIAL)",
            "CREATE TABLE bar (id SERIAL)",
        ])

        patcher.patch(self.filename)

        # This should have created the two tables.
        cursor = self.execute("SELECT * FROM foo")
        self.assertEquals(cursor.fetchall(), [])
        cursor = self.execute("SELECT * FROM bar")
        self.assertEquals(cursor.fetchall(), [])

        # And also the patch table.  There are still no patches.
        cursor = self.execute("SELECT * FROM patch")
        self.assertEquals(cursor.fetchall(), [])

    def testInitialPatchingWithoutSchema(self):
        """
        If there's no final schema, all patches are applied for
        creating the initial schema.
        """
        patcher = SQLitePatcher()

        applied = []
        versions = [5134, 1324, 7423, 3245, 2782, 9187]
        for version in versions:
            @patcher.addPatch(version)
            def patch(cursor, version=version):
                applied.append(version)

        patcher.patch(self.filename)

        self.assertEquals(applied, sorted(versions))

    def testInitialPatchingWithFinalSchema(self):
        """
        When there's a known final schema, the schema should contain the
        changes equivalent to adding all known patches.
        """
        patcher = SQLitePatcher()
        patcher.setFinalSchema([])

        applied = []
        versions = [5134, 1324, 7423, 3245, 2782, 9187]
        for version in versions:
            @patcher.addPatch(version)
            def patch(cursor, version=version):
                applied.append(version)

        patcher.patch(self.filename)

        cursor = self.execute("SELECT * FROM patch ORDER BY version")
        self.assertEquals([v for (v,) in cursor.fetchall()], sorted(versions))
        self.assertEquals(applied, [])

    def testRefusesRepeatedPatchVersions(self):
        patcher = SQLitePatcher()
        patcher.addPatch(123)(lambda cursor: None)
        self.assertRaises(RepeatedVersionError, patcher.addPatch, 123)
