#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.tests import TestCase
from imagestore.lib.signer import Signer, DEFAULT_TIMEOUT


class SignerTest(TestCase):

    def setUp(self):
        self.signer = Signer("s3kr3t")

    def testSignatureVerisonAndMethod(self):
        params = self.signer.sign("GET", "localhost", "/path")
        self.assertEquals(params.get("SignatureVersion"), ["2"])
        self.assertEquals(params.get("SignatureMethod"), ["HmacSHA256"])

    def testDefaultExpiresHeader(self):
        timeMock = self.mocker.replace("time.time")
        timeMock()
        self.mocker.result(12340.678)
        self.mocker.replay()

        try:
            params = self.signer.sign("GET", "localhost", "/path")
            self.assertEquals(params.get("Expires"),
                              [str(12340 + DEFAULT_TIMEOUT)])
        except:
            # The testing environment uses time.time().
            self.mocker.reset()
            raise
        else:
            self.mocker.verify()
            self.mocker.reset()

    def testExpiresHeaderWithTimeout(self):
        timeMock = self.mocker.replace("time.time")
        timeMock()
        self.mocker.result(12340.678)
        self.mocker.replay()

        try:
            params = self.signer.sign("GET", "localhost", "/path", timeout=10)
            self.assertEquals(params.get("Expires"), ["12350"])
        except:
            # The testing environment uses time.time().
            self.mocker.reset()
            raise
        else:
            self.mocker.verify()
            self.mocker.reset()

    def testNonce(self):
        params = self.signer.sign("GET", "localhost", "/path")
        nonce1 = params.get("Nonce")
        params = self.signer.sign("GET", "localhost", "/path")
        nonce2 = params.get("Nonce")
        self.assertEquals(type(nonce1[0]), str)
        self.assertEquals(type(nonce2[0]), str)
        self.assertNotEquals(nonce1, nonce2)

    def testSignature(self):
        params = self.signer.sign("GET", "localhost", "/path",
                                  expires=False, nonce=False)
        self.assertEquals(params, {
            "Signature": ["k3PmB5Zm7PvhELJvPGnIGVqF4KqL4BTphnwhVbQTB5U="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            })

    def testSignatureWithNonce(self):
        params = self.signer.sign("GET", "localhost", "/path",
                                  expires=False, nonce="nonce")
        self.assertEquals(params, {
            "Signature": ["tjliLdjo7NcPUwEyquH3BPKiNRdIdbc44xIjPs8IWs0="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "Nonce": ["nonce"],
            })

    def testSignatureWithExpires(self):
        params = self.signer.sign("GET", "localhost", "/path",
                                  nonce=False, expires=12345)
        self.assertEquals(params, {
            "Signature": ["bGi+IlnPh+n1lEVbZ7OGJLxY4sw51WuhEVtNo2DX7QM="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "Expires": ["12345"],
            })

    def testSignatureWithParameters(self):
        params = self.signer.sign("GET", "localhost", "/path",
                                  {"param1": "value1",
                                   "param2": ["value2"],
                                   "param3": "value3"},
                                  nonce=False, expires=False)
        self.assertEquals(params, {
            "Signature": ["UHuAE+F68d1SibfikVlwNNzPHGZhzUWt7T5i8agpze4="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "param1": ["value1"],
            "param2": ["value2"],
            "param3": ["value3"],
            })

    def testSignatureWithEscaping(self):
        params = self.signer.sign("GET", "localhost", "/path",
                                  nonce="+ +", expires=False)
        self.assertEquals(params, {
            "Signature": ["P7liHXrtQWxr2FV0dGEgNsKpD61AxxfOXxJa+3lNYDQ="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "Nonce": ["+ +"],
            })

    def testCheck(self):
        params = {
            "Signature": ["k3PmB5Zm7PvhELJvPGnIGVqF4KqL4BTphnwhVbQTB5U="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
        }
        valid = self.signer.check("GET", "localhost", "/path", params)
        self.assertTrue(valid)

    def testCheckWithWrongSignature(self):
        params = {
            "Signature": ["K3PmB5Zm7PvhELJvPGnIGVqF4KqL4BTphnwhVbQTB5U="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
        }
        valid = self.signer.check("GET", "localhost", "/path", params)
        self.assertFalse(valid)

    def testCheckWithWrongSignatureVersion(self):
        # Note that the signature is actually right for the bad data
        # (including the version 3 itself), except it was made with
        # version 2 of the algorithm.
        params = {
            "Signature": ["kPuws4c/AJG6Yt6wCYlviLUWqSui+HwFgnIah6cfr7E="],
            "SignatureVersion": ["3"],
            "SignatureMethod": ["HmacSHA256"],
        }
        valid = self.signer.check("GET", "localhost", "/path", params)
        self.assertFalse(valid)

    def testCheckWithWrongSignatureMethod(self):
        # Note that the signature is actually right for the bad data
        # (including the method), except it was made with the method
        # HmacSHA256.
        params = {
            "Signature": ["Y7FgW74M8dRBeS8OAH99BMt1Nt8QdRsll8trVzMY/Nk="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA1"],
        }
        valid = self.signer.check("GET", "localhost", "/path", params)
        self.assertFalse(valid)

    def testSignatureExample1(self):
        signer = Signer("secret")
        params = signer.sign("GET", "sdb.amazonaws.com", "/",
                             {"Timestamp": "2009-02-01T12:53:20+00:00",
                              "Version": "2007-11-07",
                              "AWSAccessKeyId": "access",
                              "Action": "ListDomains",
                             }, nonce=False, expires=False)
        self.assertEquals(params, {
            "Signature": ["okj96/5ucWBSc1uR2zXVfm6mDHtgfNv657rRtt/aunQ="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "AWSAccessKeyId": ["access"],
            "Action": ["ListDomains"],
            "Timestamp": ["2009-02-01T12:53:20+00:00"],
            "Version": ["2007-11-07"],
            })

    def testCheckSignatureExample1(self):
        signer = Signer("secret")
        valid = signer.check("GET", "sdb.amazonaws.com", "/", {
            "Signature": ["okj96/5ucWBSc1uR2zXVfm6mDHtgfNv657rRtt/aunQ="],
            "SignatureVersion": ["2"],
            "SignatureMethod": ["HmacSHA256"],
            "AWSAccessKeyId": ["access"],
            "Action": ["ListDomains"],
            "Timestamp": ["2009-02-01T12:53:20+00:00"],
            "Version": ["2007-11-07"],
            })
        self.assertTrue(valid)
