#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite


class RepeatedVersionError(Exception):
    """Raised when trying to add the same version to a patcher twice."""


class SQLitePatcher(object):

    def __init__(self):
        self._schema = None
        self._patches = {}

    def setFinalSchema(self, schema):
        assert type(schema) is list
        self._schema = schema

    def addPatch(self, version):
        if version in self._patches:
            raise RepeatedVersionError("Version %r was already added."
                                       % (version,))
        def realAddPatch(method):
            self._patches[version] = method
        return realAddPatch

    def patch(self, filename):
        connection = sqlite.connect(filename)
        cursor = connection.cursor()
        try:
            try:
                # Is there any schema at all?
                cursor.execute("SELECT version FROM patch")
            except sqlite.DatabaseError:
                # Schema does not exist.
                connection.rollback()
                self._createSchema(cursor)
            else:
                # Yes, there's a schema already, bring it up-to-date.
                self._applyPatches(cursor)
            connection.commit()
        finally:
            cursor.close()
            connection.close()

    def _createSchema(self, cursor):
        cursor.execute("CREATE TABLE patch (version INTEGER NOT NULL)")
        if self._schema is not None:
            for query in self._schema:
                cursor.execute(query)
            for version in sorted(self._patches):
                cursor.execute("INSERT INTO patch VALUES (?)", (version,))
        else:
            self._applyPatches(cursor)

    def _applyPatches(self, cursor):
        cursor.execute("SELECT version FROM patch")
        applied = set(v for (v,) in cursor.fetchall())
        for version, patch in sorted(self._patches.iteritems()):
            if version not in applied:
                patch(cursor)
                cursor.execute("INSERT INTO patch VALUES (?)",
                               (version,))


