/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include <linux/config.h>

#ifdef CONFIG_USB_DEBUG
#define DEBUG 1
#endif

#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>
#include <linux/time.h>

#include <sound/driver.h>
#include <sound/core.h>
#include <sound/control.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>

#include "audio.h"
#include "capture.h"
#include "driver.h"
#include "playback.h"


/* trigger callback */
int snd_podxtpro_trigger(snd_pcm_substream_t *substream, int cmd)
{
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	struct list_head *pos;
	snd_pcm_substream_t *s;
	int err;
	unsigned long flags;

	spin_lock_irqsave(&chip->lock_trigger, flags);

	snd_pcm_group_for_each(pos, substream) {
		s = snd_pcm_group_substream_entry(pos);

		switch(s->stream) {
		case SNDRV_PCM_STREAM_PLAYBACK:
			err = snd_podxtpro_playback_trigger(s, cmd);

			if(err < 0) {
				spin_unlock_irqrestore(&chip->lock_trigger, flags);
				return err;
			}

			break;

		case SNDRV_PCM_STREAM_CAPTURE:
			err = snd_podxtpro_capture_trigger(s, cmd);

			if(err < 0) {
				spin_unlock_irqrestore(&chip->lock_trigger, flags);
				return err;
			}

			break;

		default:
			dev_err(s2m(substream), "Unknown stream direction %d\n", s->stream);
		}
	}

	spin_unlock_irqrestore(&chip->lock_trigger, flags);
	return 0;
}

/* control info callback */
static int snd_podxtpro_control_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t *uinfo) {
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 256;
	return 0;
}

/* control get callback */
static int snd_podxtpro_control_get(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol) {
	int i;
	snd_podxtpro_pcm_t *chip = snd_kcontrol_chip(kcontrol);

	for(i = 2; i--;)
		ucontrol->value.integer.value[i] = chip->volume[i];

	return 0;
}

/* control put callback */
static int snd_podxtpro_control_put(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol) {
	int i, changed = 0;
	snd_podxtpro_pcm_t *chip = snd_kcontrol_chip(kcontrol);

	for(i = 2; i--;)
		if(chip->volume[i] != ucontrol->value.integer.value[i]) {
			chip->volume[i] = ucontrol->value.integer.value[i];
			changed = 1;
		}

	return changed;
}

/* control definition */
static snd_kcontrol_new_t podxtpro_control = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "PCM Playback Volume",
	.index = 0,
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE,
	.info = snd_podxtpro_control_info,
	.get = snd_podxtpro_control_get,
	.put = snd_podxtpro_control_put
};

/*
	Cleanup the PODxt Pro PCM device.
*/
static void podxtpro_cleanup_pcm(snd_pcm_t *pcm)
{
	int i;
	struct snd_podxtpro_pcm *chip = snd_pcm_chip(pcm);

	for(i = PODXTPRO_ISO_BUFFERS; i--;) {
		if(chip->urb_audio_out[i]) {
			usb_kill_urb(chip->urb_audio_out[i]);
			usb_free_urb(chip->urb_audio_out[i]);
		}
		if(chip->urb_audio_in[i]) {
			usb_kill_urb(chip->urb_audio_in[i]);
			usb_free_urb(chip->urb_audio_in[i]);
		}
	}
}

/* create a PCM device */
static int snd_podxtpro_new_pcm(struct snd_podxtpro_pcm *chip)
{
	snd_pcm_t *pcm;
	int err;

	if((err = snd_pcm_new(chip->podxtpro->card, (char *)chip->podxtpro->devname, 0, 1, 1, &pcm)) < 0) 
		return err;

	pcm->private_data = chip;
	pcm->private_free = podxtpro_cleanup_pcm;
	chip->pcm = pcm;
	strcpy(pcm->name, chip->podxtpro->devname);

	/* set operators */
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_podxtpro_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_podxtpro_capture_ops);

	/* pre-allocation of buffers */
	snd_pcm_lib_preallocate_pages_for_all(pcm,
																				SNDRV_DMA_TYPE_CONTINUOUS,
																				snd_dma_continuous_data(GFP_KERNEL),
																				64 * 1024, 128 * 1024);

	return 0;
}

/* PCM device destructor */
static int snd_podxtpro_pcm_free(snd_device_t *device)
{
	return 0;
}

/*
	Create and register the PCM device and mixer entries.
	Create URBs for playback and capture.
*/
int podxtpro_init_pcm(struct usb_podxtpro *podxtpro)
{
	static snd_device_ops_t pcm_ops = {
		.dev_free = snd_podxtpro_pcm_free,
	};

	int err;
	snd_podxtpro_pcm_t *chip;
	chip = kmalloc(sizeof(snd_podxtpro_pcm_t), GFP_KERNEL);

	if(chip == NULL)
		return -ENOMEM;

	memset(chip, 0, sizeof(snd_podxtpro_pcm_t));
	chip->volume[0] = chip->volume[1] = 128;
	chip->podxtpro = podxtpro;
	podxtpro->chip = chip;

	/* PCM device: */
	if((err = snd_device_new(podxtpro->card, SNDRV_DEV_PCM, podxtpro, &pcm_ops)) < 0)
		return err;

	if((err = snd_podxtpro_new_pcm(chip)) < 0)
		return err;

	spin_lock_init(&chip->lock_audio_out);
	spin_lock_init(&chip->lock_audio_in);
	spin_lock_init(&chip->lock_trigger);

	if((err = create_audio_out_urbs(chip)) < 0)
		return err;

	if((err = create_audio_in_urbs(chip)) < 0)
		return err;

	/* mixer: */
	if((err = snd_ctl_add(podxtpro->card, snd_ctl_new1(&podxtpro_control, chip))) < 0)
		return err;

	return 0;
}
