/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Searches")]
private interface SearchesInterface : Object {
        public abstract string Get (string id) throws GLib.IOError;
        public abstract string List () throws GLib.IOError;
        public abstract string Create (string query) throws GLib.IOError;
        public abstract void Delete (string id) throws GLib.IOError;
        public signal void Updated (string account);
        public signal void Created (string account);
        public signal void Deleted (string account);
}

namespace Gwibber
{
    public class Searches : Object
    {
        private const string searches_name  = "com.Gwibber.Searches";
        private const string searches_path  = "/com/gwibber/Searches";

        private SearchesInterface searches;
	private Gwibber.Utils utils;

	/** 
            Searches::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Searches availability state changes
	*/
	public signal void is_available (bool is_up);
        public signal void updated (string id);
        public signal void deleted (string id);
        public signal void created (string id);

    public Searches ()
    {
            try
            {
                searches = Bus.get_proxy_sync(BusType.SESSION,
                                                          searches_name,
                                                          searches_path);
                utils = new Gwibber.Utils();
                utils.setup(searches_name);
                utils.available.connect(searches_available);

                searches.Updated.connect((source) => {
                    search_updated (source);
                });
                searches.Created.connect((source) => {
                    search_created (source);
                });
                searches.Deleted.connect((source) => {
                    search_deleted (source);
                });

            }
            catch (GLib.IOError e)
            {
                warning ("Unable to get Gwibber Searches: "+e.message);
            }
        }


        /**
         * com.Gwibber.Searches
         **/
        public Gee.HashMap <string?, string?> lookup(string id)
        {
            try
            {
                var result = searches.Get(id);
                var parser = new Json.Parser();
                parser.load_from_data(result, -1);
                Json.Object obj = parser.get_root().get_object();
                var members = obj.get_members();
                var searches_map = new Gee.HashMap <string?, string?> ();
                foreach (string member in members)
                {
                  searches_map[member] = obj.get_string_member(member);
                }
                return searches_map;


            }
            catch (GLib.IOError e)
            {
                warning (e.message);
                return null;
            }
          
        }   

        /**
         * com.Gwibber.Searches
         **/
        public void delete(string id)
        {
            try
            {
                searches.Delete(id);
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
            }
            return;
        }


        /**
         * com.Gwibber.Searches
         **/
        public string create(string name, string query)
        {
            var searches_list = list ();
            foreach (Gee.HashMap<string?, string?> s in searches_list )
            {
              if (s["name"] == name && s["query"] == query)
              {
                debug ("search stream exists, updating");
                updated(s["id"]);
                return s["id"];
              }
            }

            try
            {
                return searches.Create("{\"query\": \"" + query + "\", \"name\": \""+ name + "\"}");
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
                return "";
            }
        }


        /**
         * com.Gwibber.Searches
         **/
        public List<Gee.HashMap?> list()
        {
            try
            {
                var searches_list = new List<Gee.HashMap<string?, string?>> ();
                var result = searches.List();
                var parser = new Json.Parser();
                parser.load_from_data(result, -1);
                Json.Array nodeArray = parser.get_root().get_array();
                for(int i = 0; i < nodeArray.get_length(); i++) {
                  var searches_map = new Gee.HashMap <string?, string?> ();
                  var members = nodeArray.get_element(i).get_object().get_members();
                  foreach (string member in members)
                  {
                    searches_map[member] = nodeArray.get_element(i).get_object().get_string_member(member);
                  }
                  searches_list.append(searches_map);
                }
                return searches_list;
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
                return null; 
            }

        }

        public void search_deleted(string data)
        {
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                deleted(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public void search_created(string data)
        {
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                created(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public void search_updated(string data)
        {
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                updated(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }


	public void searches_available(bool is_up)
	{
		is_available(is_up);
	}
    }
}
