/*
 * Copyright (C) 2010 Neil Jagdish Patel
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

using Gwibber;

namespace GwibberGtk
{
  public class TileBox : Gtk.VBox
  {
    public int last_height { get; set; }
    public TileBox ()
    {
      Object (homogeneous:false, spacing:0);
    }
    
    construct
    {
      last_height = 200;
    }

    public override void get_preferred_height_for_width (int width,
                                                         out int mh,
                                                         out int nh)
    {
      mh = nh = last_height;
    }

    public override void get_preferred_height (out int mh, out int nh)
    {
      mh = nh = last_height;
    }

  }
  public class StreamView : Gtk.HBox
  {
    private Gtk.Adjustment     adjustment;
    private Gtk.VScrollbar     scrollbar;
    private TileBox            view_box;

    private Dee.Model? _model = null;
    private Dee.Model? _stream_filter_model = null;
    private Dee.Filter *stream_filter;
    private Dee.Filter *sort_filter;
    private int _position = 0;
    private string _stream = "home";
    private int _sort_order = 1;
    private uint refresh_id = 0;
    private int tiles_visible = 0;
    private bool _prepared = false;
    private bool _showing = false;
    private bool _show_fullname;
    public Gwibber.Utils utils;
    public Gwibber.Service service;
    private GLib.Settings settings;

    public Dee.Model model {
      get { return _model; }
      set {
        if (_model is Dee.Model)
          {
            _model.row_added.disconnect (on_row_added);
            _model.row_removed.disconnect (on_row_removed);
            _model.row_changed.disconnect (on_row_changed);
            if (_model is Dee.SharedModel)
              _model.notify["synchronized"].disconnect (on_synchronized);
          }

        _model = value;
        _model.row_added.connect (on_row_added);
        _model.row_removed.connect (on_row_removed);
        _model.row_changed.connect (on_row_changed);
        if (_model is Dee.SharedModel)
          _model.notify["synchronized"].connect (on_synchronized);
      }
    }

    public Dee.Model stream_filter_model {
      get { return _stream_filter_model; }
      set {
        if (sort_order == 0)
          sort_filter = new Dee.Filter.collator(StreamModelColumn.TIMESTAMP);
        else
          sort_filter = new Dee.Filter.collator_desc(StreamModelColumn.TIMESTAMP);
        _stream_filter_model = new Dee.FilterModel (sort_filter, model);
        
        refresh ();
        Idle.add (() => {
          adjustment.set_upper ((double)(_stream_filter_model.get_n_rows ()));
          return false;
          });
      }
    }

    public int position {
      get { return _position; }
      set {
        if (_position != value)
          {
            _position = value;
            _position = _position.clamp (0, (int)stream_filter_model.get_n_rows ());

            if (_position != adjustment.get_value ())
            {
              adjustment.set_value (_position);
              refresh ();
            }
          }
      }
    }

    public string stream {
      get { return _stream; }
      set {
          _stream = value;
          stream_filter_model = model;
      }
    }


    public int sort_order {
      get { return _sort_order; }
      set {
        if (value != _sort_order)
        {
          _sort_order = value;
          stream_filter_model = model;
        }
      }
    }

    public bool show_fullname {
      get { return _show_fullname; }
      set {
        if (value != _show_fullname)
        {
          _show_fullname = value;
        }
      }
    }

    public bool prepared {
      get { return _prepared; }
      set {
        if (value != _prepared)
        {
          _prepared = value;
        }
      }
    }

    public bool showing {
      get { return _showing; }
      set {
        if (value != _showing)
        {
          _showing = value;
        }
        if (_showing)
        {
          adjustment.set_upper ((double)(_stream_filter_model.get_n_rows ()));
          service.update_indicators (stream);
          refresh ();
        }
      }
    }

    private SList<StreamViewTile> tiles;

    private int last_width = 0;
    private int last_height = 0;

    public StreamView ()
    {
      Object (homogeneous:false, spacing:0);
    }

    ~StreamView ()
    {
      _stream_filter_model = null;
      _model = null;
    }

    construct
    {
      Intl.setlocale(LocaleCategory.COLLATE, "C");
      utils = new Gwibber.Utils ();
      service = new Gwibber.Service ();
      settings = new GLib.Settings ("org.gwibber.preferences");

      show_fullname = settings.get_boolean("show-fullname");
      settings.changed.connect ((key) => {
      if (key == "show-fullname") {
         if (show_fullname != settings.get_boolean("show-fullname"))
           show_fullname = settings.get_boolean("show-fullname");
      }
      });

      adjustment = new Gtk.Adjustment (0, 0, 1, 1, 1, 1);
      adjustment.notify["value"].connect (()=> {
        double v = adjustment.get_value ();
        if ((double)position != v) 
        {
          if ((double)position < v && v < (position + 1))
            position += 1;
          else if ((double)position > v && v > (position - 1))
            position -= 1;
          else
            position = (int)adjustment.get_value ();
        }
      });

      view_box = new TileBox ();
      pack_start (view_box, true, true, 0);

      scrollbar = new Gtk.VScrollbar (adjustment);
      pack_start (scrollbar, false, false, 0);   
      
      set_events(Gdk.EventMask.KEY_PRESS_MASK | Gdk.EventMask.SCROLL_MASK);
      scroll_event.connect ((source, event) => {
        Gdk.ScrollDirection direction = event.direction;
        if (event.direction == Gdk.ScrollDirection.UP)
        {
          if (position > 0)
          {
            position -= 1;
          }
        }
        if (event.direction == Gdk.ScrollDirection.DOWN)
        {
          if (position < (adjustment.upper - tiles_visible))
          {
            position += 1;
          }
        }
        return true;
      });

      key_press_event.connect ((source, event) => {
        var key = event.keyval;

        switch (event.keyval)
        {
          case 0xff55: // *_Page_Up
          case 0xff9a:
            position -= tiles_visible;
            break;
          case 0xff56: // *_Page_Down
          case 0xff9b:
            position += tiles_visible;
            break;
          case 0xff54: // *_Down
          case 0xff99:
            position += 1;
            break;
          case 0xff52: // *_Up
          case 0xff97:
            position -= 1;
            break;
          default:
            return false;
        }
        return true;
      });


      for (int i = 0; i < 15; i++)
        {
          var tile = new StreamViewTile (utils, service, show_fullname);
          tile.show_all ();
          view_box.pack_start (tile, false, false, 0);
          tiles.append (tile);
          tile.reply.connect((mid, account, sender) => {
            reply (mid, account, sender);
            });
        }

      view_box.size_allocate.connect_after ((a) => {
        if (last_width != a.width || last_height != a.height)
        {
          last_width = a.width;
          last_height = a.height;
          view_box.last_height = last_height;

          refresh ();
        }
      });
    }

    public override void get_preferred_height_for_width (int width,
                                                         out int mh,
                                                         out int nh)
    {
      mh = nh = last_height;
    }

    public override void get_preferred_height (out int mh, out int nh)
    {
      mh = nh = last_height;
    }

    /*
     * This is a wrapper method to ensure we don't get DOS'd by row-added signals
     */
    private void refresh ()
    {
      if (refresh_id == 0 && showing)
      {
        refresh_id = Timeout.add (0, () => {
          do_refresh ();
          refresh_id = 0;
          return false;
        });
      }
    }

    /*
     * This refreshes the view with the latest model and position
     */
    private void do_refresh ()
    {
      int i = 0;

      if (!(stream_filter_model is Dee.FilterModel))
      {
        stream_filter_model = model;
      }
      uint n_rows = stream_filter_model.get_n_rows ();

      fill_up_remaining_space ();

      unowned Dee.ModelIter? iter;
      iter = stream_filter_model.get_iter_at_row (position);
      
      foreach (StreamViewTile tile in tiles)
      {
        if (iter != null && position + i < n_rows)
        {
          tile.set_details (stream_filter_model.get_value (iter, StreamModelColumn.ACCOUNTS) as string[],
                            stream_filter_model.get_string (iter, StreamModelColumn.STREAM),
                            stream_filter_model.get_string (iter, StreamModelColumn.SENDER),
                            stream_filter_model.get_string (iter, StreamModelColumn.SENDER_NICK),
                            stream_filter_model.get_bool (iter, StreamModelColumn.FROM_ME),
                            stream_filter_model.get_string (iter, StreamModelColumn.TIMESTAMP),
                            stream_filter_model.get_string (iter, StreamModelColumn.MESSAGE),
                            stream_filter_model.get_string (iter, StreamModelColumn.HTML),
                            stream_filter_model.get_string (iter, StreamModelColumn.ICON_URI),
                            stream_filter_model.get_string (iter, StreamModelColumn.URL),
                            stream_filter_model.get_string (iter, StreamModelColumn.SOURCE),
                            stream_filter_model.get_string (iter, StreamModelColumn.TIMESTRING),
                            stream_filter_model.get_string (iter, StreamModelColumn.REPLY_NICK),
                            stream_filter_model.get_string (iter, StreamModelColumn.REPLY_NAME),
                            stream_filter_model.get_string (iter, StreamModelColumn.REPLY_URL),
                            stream_filter_model.get_double (iter, StreamModelColumn.LIKES),
                            stream_filter_model.get_string (iter, StreamModelColumn.RETWEET_NICK),
                            stream_filter_model.get_string (iter, StreamModelColumn.RETWEET_NAME),
                            stream_filter_model.get_string (iter, StreamModelColumn.RETWEET_ID),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_PICTURE),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_NAME),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_URL),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_DESC),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_CAPTION),
                            stream_filter_model.get_string (iter, StreamModelColumn.LINK_ICON),
                            stream_filter_model.get_string (iter, StreamModelColumn.IMG_URL),
                            stream_filter_model.get_string (iter, StreamModelColumn.IMG_SRC),
                            stream_filter_model.get_string (iter, StreamModelColumn.IMG_THUMB),
                            stream_filter_model.get_string (iter, StreamModelColumn.IMG_NAME),
                            stream_filter_model.get_string (iter, StreamModelColumn.VIDEO_PIC),
                            stream_filter_model.get_string (iter, StreamModelColumn.VIDEO_SRC),
                            stream_filter_model.get_string (iter, StreamModelColumn.VIDEO_URL),
                            stream_filter_model.get_string (iter, StreamModelColumn.VIDEO_NAME),
                            stream_filter_model.get_string (iter, StreamModelColumn.COMMENTS));
            tiles_visible = i + 1;
        }
        else
        {
            tile.reset ();
            tiles_visible = i - 1;
        }
        i++;
        if (position + i > n_rows)
          iter = null;
        else
          iter = stream_filter_model.get_iter_at_row (position + i);
      }
 
      /* FIXME
      Idle.add (()=>{
        queue_draw ();
        return false;
      });
      */
      if (!prepared)
        prepared = true;
    }

    private void fill_up_remaining_space ()
    {
      Gtk.Allocation alloc;
      int overall_height = 0;
      int largest_tile = 0;
      bool one_was_hidden = false;

      view_box.get_allocation (out alloc);

      foreach (StreamViewTile tile in tiles)
      {
        Gtk.Allocation a;

        if (tile.get_visible ())
        {
            tile.get_allocation (out a);
            overall_height += a.height;
            if (a.height > largest_tile)
              largest_tile = a.height;
        }
        else
        {
          one_was_hidden = true;
        }
      }

      if (alloc.height > overall_height && !one_was_hidden)
      {
          var tile = new StreamViewTile (utils, service, show_fullname);
          tile.show_all ();
          view_box.pack_start (tile, false, false, 0);
          tiles.append (tile);
          tile.reply.connect((mid, account, sender) => {
            reply (mid, account, sender);
          });
          Idle.add (()=>{
            queue_draw ();
            do_refresh ();
            return false;
          });
          return;
      }

      if (alloc.height < (overall_height - (largest_tile * 3)))
      {
        var last_tile = tiles.nth_data (tiles.length () - 1);
        last_tile.reset ();
        last_tile.destroy ();
        tiles.remove (last_tile);
        Idle.add (()=>{
          queue_draw ();
          do_refresh ();
          return false;
        });
        return;
      }

      /* FIXME
      Idle.add (()=>{
        queue_draw ();
        return false;
      });
      return;
      */
    }

    [Signal (action=true)]
    public virtual signal void reply (string mid, string account, string sender)
    {
    }

    private void on_row_added ()
    {
      //debug ("on_row_added");
      refresh ();
    }

    private void on_row_removed ()
    {
      //debug ("on_row_removed");
      refresh ();
    }

    private void on_row_changed ()
    {
      //debug ("on_row_changed");
      refresh ();
    }

    private void on_synchronized ()
    {
      debug ("on_synchronized");
      stream_filter_model = model;
      adjustment.set_upper ((double)(stream_filter_model.get_n_rows ()));
      refresh ();
    }
  }
}
