/* grepmap
 *
 * ieee1394map.c - parse modules.ieee1394map
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/* Flags specifying what to match */
typedef enum {
	IEEE1394_MATCH_VENDOR        = (1 << 0),
	IEEE1394_MATCH_MODEL         = (1 << 1),
	IEEE1394_MATCH_SPECIFIER     = (1 << 2),
	IEEE1394_MATCH_VERSION       = (1 << 3)
} Ieee1394MatchFlags;

/* IEEE1394 Device structure */
typedef struct {
	Ieee1394MatchFlags match_flags;
	unsigned int       vendor, model, specifier, version;
} Ieee1394Device;


/**
 * ieee1394map_modules:
 * @mapf: open map file to read from.
 * @file: filename or map file, or '-' for standard input.
 * @args: arguments (vendor, specifier, version)
 *
 * Handles the modules.ieee1394map file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
ieee1394map_modules (FILE       *mapf,
		const char *file,
		char       *argv[])
{
	Ieee1394Device  dev;
	char      *line;
	int        lineno = 0, ret = 0;

	ret |= parse_hex (argv[0], &dev.vendor);
	ret |= parse_hex (argv[1], &dev.specifier);
	ret |= parse_hex (argv[2], &dev.version);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 2;
	}

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		Ieee1394Device map;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		if (parse_line (line, "xxxxx",
				(unsigned int *)&map.match_flags,
				&map.vendor, &map.model,
				&map.specifier, &map.version))
		{
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, IEEE1394_MATCH_VENDOR)
		    && (map.vendor != dev.vendor))
			continue;

		/* NOTE: hotplug doesn't seem to have the model id handy,
		 * that's why we don't look for it here.  Also hotplug
		 * never checks that the match_flags require a model to
		 * be matched -- bug for bug compatible again.
		 */

		if (FLAG_SET(map.match_flags, IEEE1394_MATCH_SPECIFIER)
		    && (map.specifier != dev.specifier))
			continue;

		if (FLAG_SET(map.match_flags, IEEE1394_MATCH_VERSION)
		    && (map.version != dev.version))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
