/***************************************************************************
 *
 * gnome-mount-properties.c : Nautilus properties page for gnome-mount
 *
 * Copyright (C) 2006 David Zeuthen, <david@fubar.dk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 **************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libhal.h>
#include <libhal-storage.h>

#include <libnautilus-extension/nautilus-extension-types.h>
#include <libnautilus-extension/nautilus-info-provider.h>
#include <libnautilus-extension/nautilus-property-page-provider.h>

#include "gnome-mount-properties-view.h"

static DBusConnection *dbus_connection;
static LibHalContext *hal_ctx;

static GType gmp_type = 0;
static void property_page_provider_iface_init (NautilusPropertyPageProviderIface *iface);
static GList *gmp_properties_get_pages (NautilusPropertyPageProvider *provider, GList *files);

/* Implementation of the NautilusInfoProvider interface */

/* nautilus_info_provider_update_file_info 
 * This function is called by Nautilus when it wants the extension to 
 * fill in data about the file.  It passes a NautilusFileInfo object,
 * which the extension can use to read data from the file, and which
 * the extension should add data to.
 *
 * If the data can be added immediately (without doing blocking IO), 
 * the extension can do so, and return NAUTILUS_OPERATION_COMPLETE.  
 * In this case the 'update_complete' and 'handle' parameters can be 
 * ignored.
 * 
 * If waiting for the deata would block the UI, the extension should
 * perform the task asynchronously, and return 
 * NAUTILUS_OPERATION_IN_PROGRESS.  The function must also set the 
 * 'handle' pointer to a value unique to the object, and invoke the
 * 'update_complete' closure when the update is done.
 * 
 * If the extension encounters an error, it should return 
 * NAUTILUS_OPERATION_FAILED.
 */
static NautilusOperationResult
update_file_info (NautilusInfoProvider *provider,
		  NautilusFileInfo *file,
		  GClosure *update_complete,
		  NautilusOperationHandle **handle)
{
	char *volume_hal_udi = NULL;
	char *drive_hal_udi = NULL;
	GnomeVFSDrive *drive = NULL;
	GnomeVFSVolume *volume = NULL;
	NautilusOperationResult result;


	result = NAUTILUS_OPERATION_COMPLETE;


	drive = nautilus_file_info_get_drive (file);
	if (drive != NULL)
		gnome_vfs_drive_ref (drive);

	volume = nautilus_file_info_get_volume (file);
	if (volume != NULL) {
		gnome_vfs_volume_ref (volume);
	} else {
		if (drive != NULL)
			volume = gnome_vfs_drive_get_mounted_volume (drive);
	}

	if (drive == NULL && volume != NULL)
		drive = gnome_vfs_volume_get_drive (volume);

	g_message ("drive = %x", drive);
	g_message ("volume = %x", volume);

	if (drive != NULL)
		drive_hal_udi = gnome_vfs_drive_get_hal_udi (drive);
	if (volume != NULL)
		volume_hal_udi = gnome_vfs_volume_get_hal_udi (volume);

	if (volume != NULL && volume_hal_udi != NULL) {
		char *backing_udi;
		DBusError error;

		dbus_error_init (&error);
		backing_udi = libhal_device_get_property_string (
			hal_ctx,
			volume_hal_udi,
			"volume.crypto_luks.clear.backing_volume",
			&error);
		if (backing_udi != NULL) {
			nautilus_file_info_add_emblem (file, "lock");
			g_message ("setting lock for %x", file);
			libhal_free_string (backing_udi);
		}

	}

	if (drive_hal_udi != NULL)
		g_free (drive_hal_udi);
	if (volume_hal_udi != NULL)
		g_free (volume_hal_udi);
	if (drive != NULL)
		gnome_vfs_drive_unref (drive);
	if (volume != NULL)
		gnome_vfs_volume_unref (volume);

	return result;

#if 0
	NautilusCvsDirectory *directory;
	NautilusOperationResult result;
	
	directory = get_cvs_directory (file);

	/* Keep the directory alive for as long as the file is around */
	watch_directory (directory, file);

	if (nautilus_cvs_directory_is_ready (directory)) {
		update_file_info (file, directory);
		
		result = NAUTILUS_OPERATION_COMPLETE;
	} else {
		NautilusCvsHandle *cvs_handle;
		
		cvs_handle = g_new0 (NautilusCvsHandle, 1);

		cvs_handle->provider = provider;
		cvs_handle->file = g_object_ref (file);

		/* We need to keep the closure around until we call it
		 * back. */
		cvs_handle->update_complete = g_closure_ref (update_complete);

		nautilus_cvs_directory_call_when_ready 
			(directory, directory_ready_callback, cvs_handle);

		*handle = (NautilusOperationHandle*)cvs_handle;

		result = NAUTILUS_OPERATION_IN_PROGRESS;
	}

	g_object_unref (directory);

	return result;
#endif
}

static void
cancel_update (NautilusInfoProvider *provider,
	       NautilusOperationHandle *handle)
{
#if 0
	NautilusCvsHandle *cvs_handle;
	
	cvs_handle = (NautilusCvsHandle*)handle;
	
	cvs_handle->cancelled = TRUE;
#endif
}

static void 
info_provider_iface_init (NautilusInfoProviderIface *iface)
{
	iface->update_file_info = update_file_info;
	iface->cancel_update = cancel_update;
}


static void
gnome_mount_properties_plugin_register_type (GTypeModule *module)
{
	static const GTypeInfo info = {
		sizeof (GObjectClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) NULL,
		NULL,
		NULL,
		sizeof (GObject),
		0,
		(GInstanceInitFunc) NULL
	};

	static const GInterfaceInfo property_page_provider_iface_info = {
		(GInterfaceInitFunc) property_page_provider_iface_init,
		NULL,
		NULL
	};

	static const GInterfaceInfo info_provider_iface_info = {
		(GInterfaceInitFunc) info_provider_iface_init,
		NULL,
		NULL
	};

	gmp_type = g_type_module_register_type (module, G_TYPE_OBJECT,
			"GnomeMountPropertiesPlugin",
			&info, 0);

	g_type_module_add_interface (module,
				     gmp_type,
				     NAUTILUS_TYPE_PROPERTY_PAGE_PROVIDER,
				     &property_page_provider_iface_info);

	g_type_module_add_interface (module,
				     gmp_type,
				     NAUTILUS_TYPE_INFO_PROVIDER,
				     &info_provider_iface_info);
}

static void
property_page_provider_iface_init (NautilusPropertyPageProviderIface *iface)
{
	iface->get_pages = gmp_properties_get_pages;
}


static GList *
gmp_properties_get_pages (NautilusPropertyPageProvider *provider,
			  GList *files)
{
	char *mime;
	GError *error = NULL;
	GList *pages = NULL;
	char *uri = NULL;
	NautilusFileInfo *file;
	GnomeVFSDrive *drive = NULL;
	GnomeVFSVolume *volume = NULL;
	GtkWidget *page;
	GtkWidget *label;
	NautilusPropertyPage *property_page;
	char *volume_hal_udi = NULL;
	char *drive_hal_udi = NULL;
	gboolean added_volume;

	/* only add properties page if a single file is selected */
	if (files == NULL || files->next != NULL)
		goto out;
	file = files->data;

	/*g_message ("name = '%s'", nautilus_file_info_get_name (file));*/

	drive = nautilus_file_info_get_drive (file);
	if (drive != NULL)
		gnome_vfs_drive_ref (drive);

	volume = nautilus_file_info_get_volume (file);
	if (volume != NULL) {
		gnome_vfs_volume_ref (volume);
	} else {
		if (drive != NULL)
			volume = gnome_vfs_drive_get_mounted_volume (drive);
	}

	if (drive == NULL && volume != NULL)
		drive = gnome_vfs_volume_get_drive (volume);

	if (drive != NULL)
		drive_hal_udi = gnome_vfs_drive_get_hal_udi (drive);
	if (volume != NULL)
		volume_hal_udi = gnome_vfs_volume_get_hal_udi (volume);

	added_volume = FALSE;
	if (volume != NULL && volume_hal_udi != NULL) {
		char *udi;
		LibHalVolume *vol;
		udi = gnome_vfs_volume_get_hal_udi (volume);
		vol = libhal_volume_from_udi (hal_ctx, udi);
		g_free (udi);
		if (vol != NULL) {
			g_debug ("volume=%p vol=%p, vol->udi=%s", 
				 volume, vol, vol != NULL ? libhal_volume_get_udi (vol) : "none");
			page = gm_properties_view_new ();
			gm_properties_view_set_info_volume (GM_PROPERTIES_VIEW (page), vol, hal_ctx);
			label = gtk_label_new ("Volume");
			gtk_widget_show (page);
			property_page = nautilus_property_page_new ("gnome-mount-volume-properties", label, page);
			pages = g_list_prepend (pages, property_page);
			libhal_volume_free (vol);
			added_volume = TRUE;
		}
	}

	if (drive != NULL && drive_hal_udi) {
		char *udi;
		char *udi2;
		LibHalDrive *drv;

		udi = gnome_vfs_drive_get_hal_udi (drive);
		drv = libhal_drive_from_udi (hal_ctx, udi);
		if (drv == NULL) {
			LibHalVolume *vol;
			vol = libhal_volume_from_udi (hal_ctx, udi);
			if (vol != NULL) {
				udi2 = libhal_volume_get_storage_device_udi (vol);
				drv = libhal_drive_from_udi (hal_ctx, udi2);

				/* oh, so we actually do stem from a volume, it's just not mounted...
				 * show volume dialog anyway then!
				 */
				if (!added_volume) {
					g_debug ("volume=%p vol=%p, vol->udi=%s", 
						 volume, vol, vol != NULL ? libhal_volume_get_udi (vol) : "none");
					page = gm_properties_view_new ();
					gm_properties_view_set_info_volume (GM_PROPERTIES_VIEW (page), vol, hal_ctx);
					label = gtk_label_new ("Volume");
					gtk_widget_show (page);
					property_page = nautilus_property_page_new ("gnome-mount-volume-properties", label, page);
					pages = g_list_prepend (pages, property_page);
				}

				libhal_volume_free (vol);
			}
		}
		g_free (udi);

		if (drv != NULL) {
			g_debug ("drive=%p  drv=%p, drv->udi=%s", 
				 drive,  drv, drv != NULL ? libhal_drive_get_udi (drv) : "none");

			page = gm_properties_view_new ();
			gm_properties_view_set_info_drive (GM_PROPERTIES_VIEW (page), drv, hal_ctx);
			label = gtk_label_new ("Drive");
			gtk_widget_show (page);
			property_page = nautilus_property_page_new ("gnome-mount-drive-properties", label, page);
			pages = g_list_prepend (pages, property_page);

			libhal_drive_free (drv);
		}
	}

out:

	if (drive_hal_udi != NULL)
		g_free (drive_hal_udi);
	if (volume_hal_udi != NULL)
		g_free (volume_hal_udi);
	if (drive != NULL)
		gnome_vfs_drive_unref (drive);
	if (volume != NULL)
		gnome_vfs_volume_unref (volume);

	return pages;
}

/** Internal HAL initialization function
 *
 * @return			The LibHalContext of the HAL connection or
 *				NULL on error.
 */
static LibHalContext *
do_hal_init (void)
{
	LibHalContext *ctx;
	DBusError error;
	char **devices;
	int nr;
	
	ctx = libhal_ctx_new ();
	if (ctx == NULL) {
		g_warning ("Failed to get libhal context");
		goto error;
	}
	
	dbus_error_init (&error);
	dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	if (dbus_error_is_set (&error)) {
		g_warning ("Cannot connect to system bus: %s : %s", error.name, error.message);
		dbus_error_free (&error);
		goto error;
	}
	
        dbus_connection_setup_with_g_main (dbus_connection, NULL);	
	libhal_ctx_set_dbus_connection (ctx, dbus_connection);
	
	if (!libhal_ctx_init (ctx, &error)) {
		g_warning ("Failed to initialize libhal context: %s : %s", error.name, error.message);
		dbus_error_free (&error);
		goto error;
	}

	return ctx;

error:
	if (ctx != NULL)
		libhal_ctx_free (ctx);
	return NULL;
}

/* Extension module functions.  These functions are defined in 
 * nautilus-extensions-types.h, and must be implemented by all 
 * extensions. */

/* Initialization function.  In addition to any module-specific 
 * initialization, any types implemented by the module should 
 * be registered here. */
void
nautilus_module_initialize (GTypeModule  *module)
{
	g_print ("Initializing gnome-mount extension\n");

	hal_ctx = do_hal_init ();
	if (hal_ctx == NULL) {
		g_warning ("Could not initialize hal context\n");
		goto error;
	}

	/* set up translation catalog */
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");

	gnome_mount_properties_plugin_register_type (module);
	gm_properties_view_register_type (module);

error:
	;
}

/* Perform module-specific shutdown. */
void
nautilus_module_shutdown (void)
{
	g_print ("Shutting down gnome-mount extension\n");
}

/* List all the extension types.  */
void 
nautilus_module_list_types (const GType **types,
			    int          *num_types)
{
	static GType type_list[1];

	type_list[0] = gmp_type;
	*types = type_list;
	*num_types = G_N_ELEMENTS (type_list);
}




