/** -*- C++ -*-
    @file libept/lister.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <klistview.h>
#include <qtooltip.h>
#include <qvbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qmap.h>
#include <kiconloader.h>
#include <set>

#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/actor.h>
#include <apt-front/predicate/matchers.h>
#include <apt-front/predicate/combinators.h>
#include <apt-front/utils/range.h>

#include <libept/extendablelist.h>
#include <libept/listerextenderui.h>
#include <libept/actor.h>

#ifndef EPT_LISTER_H
#define EPT_LISTER_H

class KLineEdit;
class QPushButton;
class QLabel;

namespace ept {

using namespace aptFront;
using namespace aptFront::cache;
class ListerTooltip;
class ListerItemExtender;
class ListerItem;
class Lister;

struct Timer;

class Lister: public ExtendableList, public Observer
{
    Q_OBJECT
public:
    typedef std::vector< entity::Entity > Vector;
    typedef utils::VectorRange< entity::Entity > VectorRange;
    typedef std::set< entity::Entity > Set;
    typedef std::map< entity::Entity, ListerItem * > Map;
    typedef actor::Actor< entity::Package > Actor;
    typedef predicate::Predicate< entity::Entity > Predicate;

    Lister (QWidget *parent = 0, const char *name = 0);
    ~Lister();
    void setSource();
    virtual void notifyPostChange( component::Base * );
    virtual void notifyPreRebuild( component::Base * );
    virtual void notifyPostRebuild( component::Base * );
    VectorRange selection();
    VectorRange content();
    int itemCount() { return m_itemCount; }

    bool busy() { return m_rebuildScheduled || m_inRebuild; }

    Timer *creationTimer;
signals:
    // this is ept::Lister because of braindead moc
    void actionsChanged( ept::Lister * );
    // void filterChanged( Lister::Predicate );

public slots:
    virtual void cleanRebuild();
    virtual void rebuild();
    virtual void baseAnd( Lister::Predicate );
    virtual void interactiveDrop( Lister::Predicate );
    virtual void interactiveAnd( Lister::Predicate );
    virtual void baseSet( Lister::Predicate );
    virtual void updateActions();
    virtual void scheduleRebuild();
    virtual void reallyUpdate();
protected slots:
    void contextMenu( QListViewItem *, const QPoint &, int );
    void contextActivated( int );
protected:
    ListerTooltip *m_tip;

    static entity::Entity extractKey( Map::value_type );
    static bool itemSelected( Map::value_type );

    struct CreateItem {
        CreateItem( Lister * );
        ~CreateItem();
        Map::value_type operator()( entity::Entity );
        Lister *l;
        int time;
        int items;
        ListerItem *last;
    };

    ListerItem *m_context;
    QMap< QString, QString > m_icons;
    Map m_items;
    Vector m_all;
    Predicate m_baseF, m_interactiveF;
    int m_itemCount;
    bool m_rebuildScheduled:1;
    bool m_inRebuild:1;
    bool m_inDtor:1;
};

class ListerTooltip : public QToolTip
{
public:
    ListerTooltip (QWidget *v, Lister *p) : QToolTip (v, 0), m_parent (p) {};
protected:
    QString format( const QString &, const QString &, bool = true );
    virtual void maybeTip (const QPoint &p);
    Lister *m_parent;
};

class ListerItemExtender : public ListerItemExtenderUi,
                           public cache::Observer {
public:
    ListerItemExtender( QWidget *parent = 0, const char * n = 0 );
    ~ListerItemExtender();
    virtual void resize( int w, int h );
    void setItem( ExtendableItem *i );
    ListerItem *item(); 
    entity::Entity entity() const { return m_entity; }
    void notifyPostChange( component::Base * );
protected:
    bool eventFilter( QObject *o, QEvent *e );
    QString format( const QString &, const QString &, bool = true );
    entity::Entity m_entity;
    unsigned m_leftHeight;
};

class ListerItem: public ExtendableItem
{
public:
    ListerItem(Lister *v, entity::Entity e)
        : ExtendableItem (v), m_entity( e )
        {}
    ListerItem(Lister *v, ListerItem *i, entity::Entity e)
        : ExtendableItem( v, i ), m_entity( e )
        {
        }
    virtual ItemExtender *createExtender() {
        return new ListerItemExtender();
    }
    virtual QString text( int column ) const;
    entity::Entity entity() { return m_entity; }
    const entity::Entity entity() const { return m_entity; }
    virtual void paintCell (QPainter *p, const QColorGroup &cg,
                            int column, int width, int alignment);
    virtual bool less( const ExtendableItem * ) const;
    Lister *lister() {
        return dynamic_cast< Lister* >( listView() );
    }
protected:
    entity::Entity m_entity;
    ListerItemExtender *m_extender;
};

}

#endif /* ifndef PKGLIST_H */
