# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import common

import pgm
from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.utils import image as image_utils

plugin_registry = common.application.plugin_registry
BaseListView = plugin_registry.get_component_class('raval:list_view')

class Preview(BaseListView, Group):
    
    def create_widgets(self):
        super(Preview, self).create_widgets()

        viewport = self.frontend.context.viewport_handle
        canvas = viewport.get_canvas()
        
        # preview drawable
        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.visible = True

        # reflection drawable
        self.reflection = Image()
        self.add(self.reflection)
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.opacity = 150
        self.reflection.visible = True

        self._image_path = None
        self.image.connect('pixbuf-loaded', lambda x: self._update_reflection())

        # 2px spacing
        spacing = (canvas.height / viewport.height) * 2.
        self.reflection.y += self.image.height + spacing
        
        self.reflection.z = 0.01
        
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix

    def _update_image_path(self, image_path):
        self._image_path = image_path

    def _update_reflection(self):
        if self._image_path != None:
            image_utils.cairo_gradient(self._image_path, self.reflection, 0.4)
        else:
            self.reflection.clear()

    def _update_from_theme(self, model):
        path = self.frontend.theme.get_media(model.theme_icon)
        self._update_image_path(path)
        self.image.set_from_file(path)
        return path
        
    def update(self):
        # FIXME: this view should be connected to the same controller as its
        # parent
        model = self.parent.controller.model[self.parent.selected_item]

        if model.thumbnail_source:
            # white borders            
            self.image.border_outer_color = (255, 255, 255, 255)
            self.image.border_inner_color = (255, 255, 255, 220)
#            self.image.border_width = self.image * 0.005

            dfr = self._model_thumbnail_to_image(model, self.image)
            dfr.addCallback(self._update_image_path)
            dfr.addErrback(lambda x: self._update_from_theme(model))
        else:
            self._update_from_theme(model)


