# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Manuel de la Pena <manuel.delapena@canonical.com>
"""Test the linux keyring implementation."""
import sys

from mocker import MockerTestCase, ANY
# pylint does not understand when a decorator is used, lame :(

# pylint: disable=W0611
from ubuntuone.devtools.testcase import skipIf
# pylint: enable=W0611
from desktopcouch.application.platform.linux.keyring import Keyring

# pylint: disable=C0103
try:
    import gnomekeyring
except ImportError, e:
    if not sys.platform.startswith('linux'):
        gnomekeyring = None
    else:
        raise e
# pylint: enable=C0103


# pylint: disable=W0104
@skipIf(gnomekeyring is None,
    'Can only be ran in a platform where the gnomekeyring is available')
class TestGnomeKeyring(MockerTestCase):
    """Test the interaction with the gnome keyring."""

    def setUp(self):
        super(TestGnomeKeyring, self).setUp()
        self._gnomekeyring = self.mocker.mock()
        self._random_string = self.mocker.mock()
        self._data = self.mocker.mock()
        self._keyring = Keyring(make_random_string_fn=self._random_string,
            keyring=self._gnomekeyring)

    def test_get_user_name_password_keyring(self):
        """Test the retrieval of the data when is present in the ring."""
        user_name = 'mandel'
        password = 'test'
        self._gnomekeyring.ITEM_GENERIC_SECRET
        self.mocker.result(0)
        self._gnomekeyring.find_items_sync(0, {'desktopcouch': 'basic'})
        self.mocker.result(self._data)
        self._data[0].secret
        self.mocker.result("%s:%s" % (user_name, password))
        self.mocker.replay()
        result_user, result_pass = self._keyring.get_user_name_password()
        self.assertEqual(result_user, user_name)
        self.assertEqual(result_pass, password)

    def test_get_user_name_password_not_in_keyring(self):
        """Test the retrieval of the data when missing from the ring."""
        user_name = 'mandel'
        password = 'test'
        self._gnomekeyring.ITEM_GENERIC_SECRET
        self.mocker.result(0)
        self._gnomekeyring.find_items_sync(0, {'desktopcouch': 'basic'})
        self.mocker.throw(gnomekeyring.NoMatchError)
        self._random_string(10)
        self.mocker.result(user_name)
        self._random_string(10)
        self.mocker.result(password)
        self._gnomekeyring.item_create_sync(None, 0, ANY,
            {'desktopcouch': 'basic'}, "%s:%s" % (user_name, password), True)
        self.mocker.replay()
        result_user, result_pass = self._keyring.get_user_name_password()
        self.assertEqual(result_user, user_name)
        self.assertEqual(result_pass, password)

    def test_get_user_name_password_user_cancel(self):
        """Test the retrieva of the data when the user denies the access."""
        user_name = 'mandel'
        password = 'test'
        self._gnomekeyring.ITEM_GENERIC_SECRET
        self.mocker.result(0)
        self._gnomekeyring.find_items_sync(0, {'desktopcouch': 'basic'})
        self.mocker.throw(gnomekeyring.NoMatchError)
        self._random_string(10)
        self.mocker.result(user_name)
        self._random_string(10)
        self.mocker.result(password)
        self._gnomekeyring.item_create_sync(None, 0, ANY,
            {'desktopcouch': 'basic'}, "%s:%s" % (user_name, password), True)
        self.mocker.throw(gnomekeyring.NoKeyringDaemonError)
        self.mocker.replay()
        result_user, result_pass = self._keyring.get_user_name_password()
        self.assertEqual(result_user, user_name)
        self.assertEqual(result_pass, password)

    def test_get_user_name_password_no_daemon(self):
        """Test the retrieval when the keyring daemon is not running."""
        user_name = 'mandel'
        password = 'test'
        self._gnomekeyring.ITEM_GENERIC_SECRET
        self.mocker.result(0)
        self._gnomekeyring.find_items_sync(0, {'desktopcouch': 'basic'})
        self.mocker.throw(gnomekeyring.NoMatchError)
        self._random_string(10)
        self.mocker.result(user_name)
        self._random_string(10)
        self.mocker.result(password)
        self._gnomekeyring.item_create_sync(None, 0, ANY,
            {'desktopcouch': 'basic'}, "%s:%s" % (user_name, password), True)
        self.mocker.throw(gnomekeyring.NoKeyringDaemonError)
        self.mocker.replay()
        result_user, result_pass = self._keyring.get_user_name_password()
        self.assertEqual(result_user, user_name)
        self.assertEqual(result_pass, password)

    def test_get_oauth_data_stored(self):
        """Test the storeage of the oath data."""
        consumer_key = 'consumer_key'
        consumer_secret = 'consumer_secret'
        token = 'token'
        token_secret = 'token_secret'
        self._random_string(10)
        self.mocker.result(consumer_key)
        self._random_string(10)
        self.mocker.result(consumer_secret)
        self._random_string(10)
        self.mocker.result(token)
        self._random_string(10)
        self.mocker.result(token_secret)
        self._gnomekeyring.item_create_sync(
            None, ANY, ANY,
            {'desktopcouch': 'oauth'},
            "%s:%s:%s:%s" %
                (consumer_key, consumer_secret, token, token_secret), True)
        self.mocker.replay()
        result_consumer_key, result_consumer_secret, result_token, \
            result_token_secret = self._keyring.get_oauth_data()
        self.assertEqual(consumer_key, result_consumer_key)
        self.assertEqual(consumer_secret, result_consumer_secret)
        self.assertEqual(token, result_token)
        self.assertEqual(token_secret, result_token_secret)

    def test_get_aouth_no_keyring(self):
        """Test the retrieval of the oauth data without a keyring."""
        consumer_key = 'consumer_key'
        consumer_secret = 'consumer_secret'
        token = 'token'
        token_secret = 'token_secret'
        self._random_string(10)
        self.mocker.result(consumer_key)
        self._random_string(10)
        self.mocker.result(consumer_secret)
        self._random_string(10)
        self.mocker.result(token)
        self._random_string(10)
        self.mocker.result(token_secret)
        self._gnomekeyring.item_create_sync(
            None, ANY, ANY,
            {'desktopcouch': 'oauth'},
            "%s:%s:%s:%s" %
                (consumer_key, consumer_secret, token, token_secret), True)
        self.mocker.throw(gnomekeyring.NoKeyringDaemonError)
        self.mocker.replay()
        result_consumer_key, result_consumer_secret, result_token, \
            result_token_secret = self._keyring.get_oauth_data()
        self.assertEqual(consumer_key, result_consumer_key)
        self.assertEqual(consumer_secret, result_consumer_secret)
        self.assertEqual(token, result_token)
        self.assertEqual(token_secret, result_token_secret)

    def test_get_oauth_canceled_store(self):
        """Test the retrieval of the oauth data when the user cancels it."""
        consumer_key = 'consumer_key'
        consumer_secret = 'consumer_secret'
        token = 'token'
        token_secret = 'token_secret'
        self._random_string(10)
        self.mocker.result(consumer_key)
        self._random_string(10)
        self.mocker.result(consumer_secret)
        self._random_string(10)
        self.mocker.result(token)
        self._random_string(10)
        self.mocker.result(token_secret)
        self._gnomekeyring.item_create_sync(
            None, ANY, ANY,
            {'desktopcouch': 'oauth'},
            "%s:%s:%s:%s" %
                (consumer_key, consumer_secret, token, token_secret), True)
        self.mocker.throw(gnomekeyring.CancelledError)
        self.mocker.replay()
        result_consumer_key, result_consumer_secret, result_token, \
            result_token_secret = self._keyring.get_oauth_data()
        self.assertEqual(consumer_key, result_consumer_key)
        self.assertEqual(consumer_secret, result_consumer_secret)
        self.assertEqual(token, result_token)
        self.assertEqual(token_secret, result_token_secret)
