#!/usr/bin/python
# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Stuart Langridge <stuart.langridge@canonical.com>
#         Eric Casteleijn <eric.casteleijn@canonical.com>

"""
Start local CouchDB server.
Steps:
    1. Work out which folders to use (running from source tree, or installed)
    2. Actually start CouchDB
    3. check design documents in central folder against design documents in
       Couch, and overwrite any new ones
    4. Write an HTML "bookmark" file which directs people to the local
       running CouchDB.

This script is normally called by advertisePort.py, which advertises the local
CouchDB port over D-Bus. That advertisePort script is started by D-Bus
activation.
"""

from __future__ import with_statement
import os, subprocess, sys, glob, random, string
import desktopcouch
from desktopcouch import local_files
import xdg.BaseDirectory
import errno
import time, gnomekeyring
from desktopcouch.records.server import CouchDatabase

ACCEPTABLE_USERNAME_PASSWORD_CHARS = string.lowercase + string.uppercase

def dump_ini(data, filename):
    """Dump INI data with sorted sections and keywords"""
    fd = open(filename, 'w')
    sections = data.keys()
    sections.sort()
    for section in sections:
        fd.write("[%s]\n" % (section))
        keys = data[section].keys()
        keys.sort()
        for key in keys:
            fd.write("%s=%s\n" % (key, data[section][key]))
        fd.write("\n")
    fd.close()

def create_ini_file(port="0"):
    """Write CouchDB ini file if not already present"""
    if os.path.exists(local_files.FILE_INI):
        # load the username and password from the keyring
        try:
            data = gnomekeyring.find_items_sync(gnomekeyring.ITEM_GENERIC_SECRET,
                                            {'desktopcouch': 'basic'})
        except gnomekeyring.NoMatchError:
            data = None
        if data:
            username, password = data[0].secret.split(":")
            return username, password
        # otherwise fall through; for some reason the access details aren't
        # in the keyring, so re-create the ini file and do it all again

    # randomly generate tokens and usernames
    def make_random_string(count):
        return ''.join([
             random.SystemRandom().choice(ACCEPTABLE_USERNAME_PASSWORD_CHARS)
                        for x in range(count)])

    admin_account_username = make_random_string(10)
    admin_account_basic_auth_password = make_random_string(10)
    consumer_key = make_random_string(10)
    consumer_secret = make_random_string(10)
    token = make_random_string(10)
    token_secret = make_random_string(10)
    db_dir = local_files.DIR_DB
    local = {
        'couchdb': {
            'database_dir': db_dir,
            'view_index_dir': db_dir,
        },
        'httpd': {
            'bind_address': '127.0.0.1',
            'port': port,
        },
        'log': {
            'file': local_files.FILE_LOG,
            'level': 'info',
        },
        'admins': {
            admin_account_username: admin_account_basic_auth_password
        },
        'oauth_consumer_secrets': {
            consumer_key: consumer_secret
        },
        'oauth_token_secrets': {
            token: token_secret
        },
        'oauth_token_users': {
            token: admin_account_username
        },
        'couch_httpd_auth': {
            'require_valid_user': 'true'
        }
    }

    dump_ini(local, local_files.FILE_INI)
    # save admin account details in keyring
    item_id = gnomekeyring.item_create_sync(
            None,
            gnomekeyring.ITEM_GENERIC_SECRET,
            'Desktop Couch user authentication',
            {'desktopcouch': 'basic'},
            "%s:%s" % (
            admin_account_username, admin_account_basic_auth_password),
            True)
    # and oauth tokens
    item_id = gnomekeyring.item_create_sync(
            None,
            gnomekeyring.ITEM_GENERIC_SECRET,
            'Desktop Couch user authentication',
            {'desktopcouch': 'oauth'},
            "%s:%s:%s:%s" % (
            consumer_key, consumer_secret, token, token_secret),
            True)
    return (admin_account_username, admin_account_basic_auth_password)


def process_is_couchdb__linux(pid):
    if pid is None:
        return False
    pid = int(pid)  # ensure it's a number
    proc_dir = "/proc/%d" % (pid,)
    try:
        # check to make sure it is actually a desktop-couch instance
        with open(os.path.join(proc_dir, 'cmdline')) as cmd_file:
            cmd = cmd_file.read()
        #if '/desktop-couch' not in cmd:
        if 'beam' not in cmd:
            return False

        # make sure it's our process.
        if not os.access(os.path.join(proc_dir, "mem"), os.W_OK):
            return False

    except IOError:
        return False

    return True

import platform
os_name = platform.system()
try:
    process_is_couchdb = {
        "Linux": process_is_couchdb__linux
        } [os_name]
except KeyError:
    raise NotImplementedError("os %r is not yet supported" % (os_name,))

def read_pidfile():
    try:
        pid_file = local_files.FILE_PID
        with open(pid_file) as fp:
            try:
                contents = fp.read()
                return int(contents)
            except ValueError:
                return None
    except IOError:
        return None

def run_couchdb():
    """Actually start the CouchDB process.  Return its PID."""
    pid = read_pidfile()
    if pid is not None and not process_is_couchdb(pid):
        print "Removing stale, deceptive pid file."
        os.remove(local_files.FILE_PID)
    local_exec = local_files.COUCH_EXEC_COMMAND + ['-b']
    try:
        # subprocess is buggy.  Chad patched, but that takes time to propagate.
        proc = subprocess.Popen(local_exec)
        while True:
            try:
                retcode = proc.wait()
                break
            except OSError, e:
                if e.errno == errno.EINTR:
                    continue
                raise
        if retcode < 0:
            print >> sys.stderr, "Child was terminated by signal", -retcode
        elif retcode > 0:
            print >> sys.stderr, "Child returned", retcode
    except OSError, e:
        print >> sys.stderr, "Execution failed: %s: %s" % (e, local_exec)
        exit(1)

    # give the process a chance to start
    for timeout in xrange(1000):
        pid = read_pidfile()
        time.sleep(0.3)
        if pid is not None and process_is_couchdb(pid):
            break
        print "...waiting for couchdb to start..."
    return pid

def update_design_documents():
    """Check system design documents and update any that need updating

    A database should be created if
      $XDG_DATA_DIRs/desktop-couch/databases/dbname/database.cfg exists
    Design docs are defined by the existence of
      $XDG_DATA_DIRs/desktop-couch/databases/dbname/_design/designdocname/views/viewname/map.js
      reduce.js may also exist in the same folder.
    """
    for base in xdg.BaseDirectory.xdg_data_dirs:
        db_spec = os.path.join(
            base, "desktop-couch", "databases", "*", "database.cfg")
        for database_path in glob.glob(db_spec):
            database_root = os.path.split(database_path)[0]
            database_name = os.path.split(database_root)[1]
            # Just the presence of database.cfg is enough to create the database
            db = CouchDatabase(database_name, create=True)
            # look for design documents
            dd_spec = os.path.join(
                database_root, "_design", "*", "views", "*", "map.js")
            for dd_path in glob.glob(dd_spec):
                view_root = os.path.split(dd_path)[0]
                view_name = os.path.split(view_root)[1]
                dd_root = os.path.split(os.path.split(view_root)[0])[0]
                dd_name = os.path.split(dd_root)[1]

                def load_js_file(filename_no_extension):
                    fn = os.path.join(
                        view_root, "%s.js" % (filename_no_extension))
                    if not os.path.isfile(fn): return None
                    fp = open(fn)
                    data = fp.read()
                    fp.close()
                    return data

                mapjs = load_js_file("map")
                reducejs = load_js_file("reduce")

                # XXX check whether this already exists or not, rather
                # than inefficiently just overwriting it regardless
                db.add_view(view_name, mapjs, reducejs, dd_name)

def write_bookmark_file(username, password, pid):
    """Write out an HTML document that the user can bookmark to find their DB"""
    bookmark_file = os.path.join(local_files.DIR_DB, "couchdb.html")

    if os.path.exists(
            os.path.join(os.path.split(__file__)[0], "../data/couchdb.tmpl")):
        bookmark_template = os.path.join(
            os.path.split(__file__)[0], "../data/couchdb.tmpl")
    else:
        for base in xdg.BaseDirectory.xdg_data_dirs:
            template_path = os.path.join(base, "desktopcouch", "couchdb.tmpl")
            if os.path.exists(template_path):
                bookmark_template = os.path.join(
                    os.path.split(__file__)[0], template_path)

    fp = open(bookmark_template)
    html = fp.read()
    fp.close()

    port = desktopcouch.find_port(pid=pid)
    if port is None:
        print ("We couldn't find desktop-CouchDB's network port.  Bookmark "
               "file not written.")
        try:
            os.remove(bookmark_file)
        except OSError:
            pass
    else:
        fp = open(bookmark_file, "w")
        out = html.replace("[[COUCHDB_PORT]]", str(port))
        out = out.replace("[[COUCHDB_USERNAME]]", username)
        out = out.replace("[[COUCHDB_PASSWORD]]", password)
        fp.write(out)
        fp.close()
        print "Browse your desktop CouchDB at file://%s" % \
          os.path.realpath(bookmark_file)

def start_couchdb():
    """Execute each step to start a desktop CouchDB."""
    username, password = create_ini_file()
    pid = run_couchdb()
    # Note that we do not call update_design_documents here. This is because
    # Couch won't actually have started yet, so when update_design_documents
    # calls the Records API, that will call back into get_port and we end up
    # starting Couch again. Instead, get_port calls update_design_documents
    # *after* Couch startup has occurred.
    write_bookmark_file(username, password, pid)
    return pid


if __name__ == "__main__":
    start_couchdb()
    print "Desktop CouchDB started"

